"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StatelessProgrammableSwitchCreator = void 0;
const hap_1 = require("../hap");
const z2mModels_1 = require("../z2mModels");
const monitor_1 = require("./monitor");
const helpers_1 = require("../helpers");
const action_helper_1 = require("./action_helper");
class StatelessProgrammableSwitchCreator {
    createServicesFromExposes(accessory, exposes) {
        const actionExposes = exposes.filter(e => (0, z2mModels_1.exposesIsPublished)(e) && (0, z2mModels_1.exposesHasEnumProperty)(e) && e.name === 'action'
            && !accessory.isPropertyExcluded(e.property))
            .map(e => e);
        for (const expose of actionExposes) {
            // Each action expose can map to multiple instances of a Stateless Programmable Switch,
            // depending on the values provided.
            try {
                const allowedValues = expose.values.filter(v => accessory.isValueAllowedForProperty(expose.property, v));
                const mappings = action_helper_1.SwitchActionHelper.getInstance().valuesToNumberedMappings(allowedValues).filter(m => m.isValidMapping());
                const logEntries = [`Mapping of property '${expose.property}' of device '${accessory.displayName}':`];
                for (const mapping of mappings) {
                    try {
                        const id = StatelessProgrammableSwitchHandler.generateIdentifier(expose.endpoint, mapping.subType);
                        if (!accessory.isServiceHandlerIdKnown(id)) {
                            const handler = new StatelessProgrammableSwitchHandler(accessory, expose, mapping);
                            accessory.registerServiceHandler(handler);
                        }
                        const logEntry = mapping.toString();
                        if (logEntry !== undefined) {
                            logEntries.push(logEntry);
                        }
                    }
                    catch (error) {
                        accessory.log.error(`Failed to setup stateless programmable switch for accessory ${accessory.displayName} ` +
                            `from expose "${JSON.stringify(expose)}" and mapping "${JSON.stringify(mapping)}", error: ${error}`);
                    }
                }
                accessory.log.info(logEntries.join('\n'));
            }
            catch (error) {
                accessory.log.error(`Failed to setup stateless programmable switch for accessory ${accessory.displayName} ` +
                    `from expose "${JSON.stringify(expose)}", error: ${error}`);
            }
        }
    }
}
exports.StatelessProgrammableSwitchCreator = StatelessProgrammableSwitchCreator;
class StatelessProgrammableSwitchHandler {
    constructor(accessory, actionExpose, mapping) {
        var _a;
        this.actionExpose = actionExpose;
        this.identifier = StatelessProgrammableSwitchHandler.generateIdentifier(actionExpose.endpoint, mapping.subType);
        // Create service
        let subType = mapping.subType;
        if (actionExpose.endpoint !== undefined) {
            if (subType !== undefined) {
                subType += '.' + actionExpose.endpoint;
            }
            else {
                subType = actionExpose.endpoint;
            }
        }
        const serviceName = accessory.getDefaultServiceDisplayName(subType);
        const service = accessory.getOrAddService(new hap_1.hap.Service.StatelessProgrammableSwitch(serviceName, subType));
        // Setup monitor and characteristic
        (0, helpers_1.getOrAddCharacteristic)(service, hap_1.hap.Characteristic.ServiceLabelIndex)
            .updateValue((_a = mapping.serviceLabelIndex) !== null && _a !== void 0 ? _a : 0);
        const eventCharacteristic = (0, helpers_1.getOrAddCharacteristic)(service, hap_1.hap.Characteristic.ProgrammableSwitchEvent);
        const valueMap = new Map();
        if (mapping.valueSinglePress !== undefined) {
            valueMap.set(mapping.valueSinglePress, hap_1.hap.Characteristic.ProgrammableSwitchEvent.SINGLE_PRESS);
        }
        if (mapping.valueDoublePress !== undefined) {
            valueMap.set(mapping.valueDoublePress, hap_1.hap.Characteristic.ProgrammableSwitchEvent.DOUBLE_PRESS);
        }
        if (mapping.valueLongPress !== undefined) {
            valueMap.set(mapping.valueLongPress, hap_1.hap.Characteristic.ProgrammableSwitchEvent.LONG_PRESS);
        }
        eventCharacteristic.setProps(StatelessProgrammableSwitchHandler.generateValueConfigForProgrammableSwitchEvents([...valueMap.values()]));
        this.monitor = new monitor_1.MappingCharacteristicMonitor(actionExpose.property, service, hap_1.hap.Characteristic.ProgrammableSwitchEvent, valueMap);
    }
    static generateValueConfigForProgrammableSwitchEvents(events) {
        const result = {
            validValues: events,
        };
        if (events.length > 0) {
            result.minValue = Math.min(...events);
            result.maxValue = Math.max(...events);
        }
        return result;
    }
    get getableKeys() {
        if ((0, z2mModels_1.exposesCanBeGet)(this.actionExpose)) {
            return [this.actionExpose.property];
        }
        return [];
    }
    updateState(state) {
        this.monitor.callback(state);
    }
    static generateIdentifier(endpoint, mappingSubType) {
        let identifier = hap_1.hap.Service.StatelessProgrammableSwitch.UUID;
        if (endpoint !== undefined) {
            identifier += '_' + endpoint.trim();
        }
        if (mappingSubType) {
            identifier += '#' + mappingSubType;
        }
        return identifier;
    }
}
//# sourceMappingURL=action.js.map