/*
 / _____)             _              | |
( (____  _____ ____ _| |_ _____  ____| |__
 \____ \| ___ |    (_   _) ___ |/ ___)  _ \
 _____) ) ____| | | || |_| ____( (___| | | |
(______/|_____)_|_|_| \__)_____)\____)_| |_|
  (C)2013 Semtech-Cycleo

Description:
    Functions used to handle FPGA register access for LoRa concentrator.
    Registers are addressed by name.
    Multi-bytes registers are handled automatically.
    Read-modify-write is handled automatically.

License: Revised BSD License, see LICENSE.TXT file include in the project
Maintainer: Michael Coracin
*/

#ifndef _LORAGW_FPGA_REG_H
#define _LORAGW_FPGA_REG_H

/* -------------------------------------------------------------------------- */
/* --- DEPENDANCIES --------------------------------------------------------- */

#include <stdint.h>     /* C99 types */
#include <stdbool.h>    /* bool type */

/* -------------------------------------------------------------------------- */
/* --- PUBLIC CONSTANTS ----------------------------------------------------- */

#define LGW_REG_SUCCESS 0
#define LGW_REG_ERROR -1

#define LGW_MIN_NOTCH_FREQ      126000U /* 126 KHz */
#define LGW_MAX_NOTCH_FREQ      250000U /* 250 KHz */
#define LGW_DEFAULT_NOTCH_FREQ  129000U /* 129 KHz */

/*
auto generated register mapping for C code
this file contains autogenerated C struct used to access the FPGA registers
this file is autogenerated from registers description
*/

#define LGW_FPGA_SOFT_RESET 0
#define LGW_FPGA_FEATURE 1
#define LGW_FPGA_LBT_INITIAL_FREQ 2
#define LGW_FPGA_VERSION 3
#define LGW_FPGA_STATUS 4
#define LGW_FPGA_CTRL_FEATURE_START 5
#define LGW_FPGA_CTRL_RADIO_RESET 6
#define LGW_FPGA_CTRL_INPUT_SYNC_I 7
#define LGW_FPGA_CTRL_INPUT_SYNC_Q 8
#define LGW_FPGA_CTRL_OUTPUT_SYNC 9
#define LGW_FPGA_CTRL_INVERT_IQ 10
#define LGW_FPGA_CTRL_ACCESS_HISTO_MEM 11
#define LGW_FPGA_CTRL_CLEAR_HISTO_MEM 12
#define LGW_FPGA_HISTO_RAM_ADDR 13
#define LGW_FPGA_HISTO_RAM_DATA 14
#define LGW_FPGA_HISTO_NB_READ 15
#define LGW_FPGA_LBT_TIMESTAMP_CH 16
#define LGW_FPGA_LBT_TIMESTAMP_SELECT_CH 17
#define LGW_FPGA_LBT_CH0_FREQ_OFFSET 18
#define LGW_FPGA_LBT_CH1_FREQ_OFFSET 19
#define LGW_FPGA_LBT_CH2_FREQ_OFFSET 20
#define LGW_FPGA_LBT_CH3_FREQ_OFFSET 21
#define LGW_FPGA_LBT_CH4_FREQ_OFFSET 22
#define LGW_FPGA_LBT_CH5_FREQ_OFFSET 23
#define LGW_FPGA_LBT_CH6_FREQ_OFFSET 24
#define LGW_FPGA_LBT_CH7_FREQ_OFFSET 25
#define LGW_FPGA_SCAN_FREQ_OFFSET 26
#define LGW_FPGA_LBT_SCAN_TIME_CH0 27
#define LGW_FPGA_LBT_SCAN_TIME_CH1 28
#define LGW_FPGA_LBT_SCAN_TIME_CH2 29
#define LGW_FPGA_LBT_SCAN_TIME_CH3 30
#define LGW_FPGA_LBT_SCAN_TIME_CH4 31
#define LGW_FPGA_LBT_SCAN_TIME_CH5 32
#define LGW_FPGA_LBT_SCAN_TIME_CH6 33
#define LGW_FPGA_LBT_SCAN_TIME_CH7 34
#define LGW_FPGA_RSSI_TARGET 35
#define LGW_FPGA_HISTO_SCAN_FREQ 36
#define LGW_FPGA_NOTCH_FREQ_OFFSET 37
#define LGW_FPGA_TOTALREGS 38

/* -------------------------------------------------------------------------- */
/* --- PUBLIC FUNCTIONS PROTOTYPES ------------------------------------------ */

/**
@brief LoRa concentrator TX notch filter delay
@return delay in microseconds introduced by TX notch filter
*/
float lgw_fpga_get_tx_notch_delay(void);

/**
@brief LoRa concentrator FPGA configuration
@param tx_notch_freq TX notch filter frequency, in Hertz
@return status of register operation (LGW_REG_SUCCESS/LGW_REG_ERROR)
*/
int lgw_fpga_configure(uint32_t tx_notch_freq);

/**
@brief LoRa concentrator FPGA register write
@param register_id register number in the data structure describing registers
@param reg_value signed value to write to the register (for u32, use cast)
@return status of register operation (LGW_REG_SUCCESS/LGW_REG_ERROR)
*/
int lgw_fpga_reg_w(uint16_t register_id, int32_t reg_value);

/**
@brief LoRa concentrator FPGA register read
@param register_id register number in the data structure describing registers
@param reg_value pointer to a variable where to write register read value
@return status of register operation (LGW_REG_SUCCESS/LGW_REG_ERROR)
*/
int lgw_fpga_reg_r(uint16_t register_id, int32_t *reg_value);

/**
@brief LoRa concentrator FPGA register burst write
@param register_id register number in the data structure describing registers
@param data pointer to byte array that will be sent to the LoRa concentrator
@param size size of the transfer, in byte(s)
@return status of register operation (LGW_REG_SUCCESS/LGW_REG_ERROR)
*/
int lgw_fpga_reg_wb(uint16_t register_id, uint8_t *data, uint16_t size);

/**
@brief LoRa concentrator FPGA register burst read
@param register_id register number in the data structure describing registers
@param data pointer to byte array that will be written from the LoRa concentrator
@param size size of the transfer, in byte(s)
@return status of register operation (LGW_REG_SUCCESS/LGW_REG_ERROR)
*/
int lgw_fpga_reg_rb(uint16_t register_id, uint8_t *data, uint16_t size);

#endif
/* --- EOF ------------------------------------------------------------------ */
