"""
The MIT License (MIT)

Copyright (c) 2023 Arduino SA

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

Basic example usage:

import time
from bmi270 import BMI270
from machine import Pin, SPI, I2C

# Init in I2C mode.
imu = BMI270(I2C(1, scl=Pin(15), sda=Pin(14)))

# Or init in SPI mode.
# TODO: Not supported yet.
# imu = BMI270(SPI(5), cs=Pin(10))

while (True):
    print('Accelerometer: x:{:>6.3f}  y:{:>6.3f}  z:{:>6.3f}'.format(*imu.accel()))
    print('Gyroscope:     x:{:>6.3f}  y:{:>6.3f}  z:{:>6.3f}'.format(*imu.gyro()))
    print('Magnetometer:  x:{:>8.3f}  y:{:>8.3f}  z:{:>8.3f}'.format(*imu.magnet()))
    print("")
    time.sleep_ms(100)
"""

import array
import time
from micropython import const

_DEFAULT_ADDR = const(0x68)
_CHIP_ID = const(0x00)
_STATUS = const(0x21)
_INIT_ADDR_0 = const(0x5B)
_INIT_ADDR_1 = const(0x5C)
_DATA_8 = const(0x0C)
_DATA_14 = const(0x12)
_CMD = const(0x7E)
_CONFIG_DATA = const(
    b"\xc8\x2e\x00\x2e\x80\x2e\x3d\xb1\xc8\x2e\x00\x2e\x80\x2e\x91\x03\x80\x2e\xbc"
    b"\xb0\x80\x2e\xa3\x03\xc8\x2e\x00\x2e\x80\x2e\x00\xb0\x50\x30\x21\x2e\x59\xf5"
    b"\x10\x30\x21\x2e\x6a\xf5\x80\x2e\x3b\x03\x00\x00\x00\x00\x08\x19\x01\x00\x22"
    b"\x00\x75\x00\x00\x10\x00\x10\xd1\x00\xb3\x43\x80\x2e\x00\xc1\x80\x2e\x00\xc1"
    b"\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00"
    b"\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e"
    b"\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80"
    b"\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1"
    b"\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00"
    b"\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e"
    b"\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80"
    b"\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1"
    b"\x80\x2e\x00\xc1\x80\x2e\x00\xc1\xe0\x5f\x00\x00\x00\x00\x01\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x92\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x08\x19\x00\x00\x88\x00\x00\x00\x00\x00\x00\x00\x05"
    b"\xe0\xaa\x38\x05\xe0\x90\x30\xfa\x00\x96\x00\x4b\x09\x11\x00\x11\x00\x02\x00"
    b"\x2d\x01\xd4\x7b\x3b\x01\xdb\x7a\x04\x00\x3f\x7b\xcd\x6c\xc3\x04\x85\x09\xc3"
    b"\x04\xec\xe6\x0c\x46\x01\x00\x27\x00\x19\x00\x96\x00\xa0\x00\x01\x00\x0c\x00"
    b"\xf0\x3c\x00\x01\x01\x00\x03\x00\x01\x00\x0e\x00\x00\x00\x32\x00\x05\x00\xee"
    b"\x06\x04\x00\xc8\x00\x00\x00\x04\x00\xa8\x05\xee\x06\x00\x04\xbc\x02\xb3\x00"
    b"\x85\x07\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\xb4\x00\x01\x00\xb9\x00\x01\x00\x98\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x01\x00\x80\x00\x04\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x2e\x00\xc1\xfd\x2d\xde"
    b"\x00\xeb\x00\xda\x00\x00\x0c\xff\x0f\x00\x04\xc0\x00\x5b\xf5\xc9\x01\x1e\xf2"
    b"\x80\x00\x3f\xff\x19\xf4\x58\xf5\x66\xf5\x64\xf5\xc0\xf1\xf0\x00\xe0\x00\xcd"
    b"\x01\xd3\x01\xdb\x01\xff\x7f\xff\x01\xe4\x00\x74\xf7\xf3\x00\xfa\x00\xff\x3f"
    b"\xca\x03\x6c\x38\x56\xfe\x44\xfd\xbc\x02\xf9\x06\x00\xfc\x12\x02\xae\x01\x58"
    b"\xfa\x9a\xfd\x77\x05\xbb\x02\x96\x01\x95\x01\x7f\x01\x82\x01\x89\x01\x87\x01"
    b"\x88\x01\x8a\x01\x8c\x01\x8f\x01\x8d\x01\x92\x01\x91\x01\xdd\x00\x9f\x01\x7e"
    b"\x01\xdb\x00\xb6\x01\x70\x69\x26\xd3\x9c\x07\x1f\x05\x9d\x00\x00\x08\xbc\x05"
    b"\x37\xfa\xa2\x01\xaa\x01\xa1\x01\xa8\x01\xa0\x01\xa8\x05\xb4\x01\xb4\x01\xce"
    b"\x00\xd0\x00\xfc\x00\xc5\x01\xff\xfb\xb1\x00\x00\x38\x00\x30\xfd\xf5\xfc\xf5"
    b"\xcd\x01\xa0\x00\x5f\xff\x00\x40\xff\x00\x00\x80\x6d\x0f\xeb\x00\x7f\xff\xc2"
    b"\xf5\x68\xf7\xb3\xf1\x67\x0f\x5b\x0f\x61\x0f\x80\x0f\x58\xf7\x5b\xf7\x83\x0f"
    b"\x86\x00\x72\x0f\x85\x0f\xc6\xf1\x7f\x0f\x6c\xf7\x00\xe0\x00\xff\xd1\xf5\x87"
    b"\x0f\x8a\x0f\xff\x03\xf0\x3f\x8b\x00\x8e\x00\x90\x00\xb9\x00\x2d\xf5\xca\xf5"
    b"\xcb\x01\x20\xf2\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x30\x50\x98\x2e"
    b"\xd7\x0e\x50\x32\x98\x2e\xfa\x03\x00\x30\xf0\x7f\x00\x2e\x00\x2e\xd0\x2e\x00"
    b"\x2e\x01\x80\x08\xa2\xfb\x2f\x98\x2e\xba\x03\x21\x2e\x19\x00\x01\x2e\xee\x00"
    b"\x00\xb2\x07\x2f\x01\x2e\x19\x00\x00\xb2\x03\x2f\x01\x50\x03\x52\x98\x2e\x07"
    b"\xcc\x01\x2e\xdd\x00\x00\xb2\x27\x2f\x05\x2e\x8a\x00\x05\x52\x98\x2e\xc7\xc1"
    b"\x03\x2e\xe9\x00\x40\xb2\xf0\x7f\x08\x2f\x01\x2e\x19\x00\x00\xb2\x04\x2f\x00"
    b"\x30\x21\x2e\xe9\x00\x98\x2e\xb4\xb1\x01\x2e\x18\x00\x00\xb2\x10\x2f\x05\x50"
    b"\x98\x2e\x4d\xc3\x05\x50\x98\x2e\x5a\xc7\x98\x2e\xf9\xb4\x98\x2e\x54\xb2\x98"
    b"\x2e\x67\xb6\x98\x2e\x17\xb2\x10\x30\x21\x2e\x77\x00\x01\x2e\xef\x00\x00\xb2"
    b"\x04\x2f\x98\x2e\x7a\xb7\x00\x30\x21\x2e\xef\x00\x01\x2e\xd4\x00\x04\xae\x0b"
    b"\x2f\x01\x2e\xdd\x00\x00\xb2\x07\x2f\x05\x52\x98\x2e\x8e\x0e\x00\xb2\x02\x2f"
    b"\x10\x30\x21\x2e\x7d\x00\x01\x2e\x7d\x00\x00\x90\x90\x2e\xf1\x02\x01\x2e\xd7"
    b"\x00\x00\xb2\x04\x2f\x98\x2e\x2f\x0e\x00\x30\x21\x2e\x7b\x00\x01\x2e\x7b\x00"
    b"\x00\xb2\x12\x2f\x01\x2e\xd4\x00\x00\x90\x02\x2f\x98\x2e\x1f\x0e\x09\x2d\x98"
    b"\x2e\x81\x0d\x01\x2e\xd4\x00\x04\x90\x02\x2f\x50\x32\x98\x2e\xfa\x03\x00\x30"
    b"\x21\x2e\x7b\x00\x01\x2e\x7c\x00\x00\xb2\x90\x2e\x09\x03\x01\x2e\x7c\x00\x01"
    b"\x31\x01\x08\x00\xb2\x04\x2f\x98\x2e\x47\xcb\x10\x30\x21\x2e\x77\x00\x81\x30"
    b"\x01\x2e\x7c\x00\x01\x08\x00\xb2\x61\x2f\x03\x2e\x89\x00\x01\x2e\xd4\x00\x98"
    b"\xbc\x98\xb8\x05\xb2\x0f\x58\x23\x2f\x07\x90\x09\x54\x00\x30\x37\x2f\x15\x41"
    b"\x04\x41\xdc\xbe\x44\xbe\xdc\xba\x2c\x01\x61\x00\x0f\x56\x4a\x0f\x0c\x2f\xd1"
    b"\x42\x94\xb8\xc1\x42\x11\x30\x05\x2e\x6a\xf7\x2c\xbd\x2f\xb9\x80\xb2\x08\x22"
    b"\x98\x2e\xc3\xb7\x21\x2d\x61\x30\x23\x2e\xd4\x00\x98\x2e\xc3\xb7\x00\x30\x21"
    b"\x2e\x5a\xf5\x18\x2d\xe1\x7f\x50\x30\x98\x2e\xfa\x03\x0f\x52\x07\x50\x50\x42"
    b"\x70\x30\x0d\x54\x42\x42\x7e\x82\xe2\x6f\x80\xb2\x42\x42\x05\x2f\x21\x2e\xd4"
    b"\x00\x10\x30\x98\x2e\xc3\xb7\x03\x2d\x60\x30\x21\x2e\xd4\x00\x01\x2e\xd4\x00"
    b"\x06\x90\x18\x2f\x01\x2e\x76\x00\x0b\x54\x07\x52\xe0\x7f\x98\x2e\x7a\xc1\xe1"
    b"\x6f\x08\x1a\x40\x30\x08\x2f\x21\x2e\xd4\x00\x20\x30\x98\x2e\xaf\xb7\x50\x32"
    b"\x98\x2e\xfa\x03\x05\x2d\x98\x2e\x38\x0e\x00\x30\x21\x2e\xd4\x00\x00\x30\x21"
    b"\x2e\x7c\x00\x18\x2d\x01\x2e\xd4\x00\x03\xaa\x01\x2f\x98\x2e\x45\x0e\x01\x2e"
    b"\xd4\x00\x3f\x80\x03\xa2\x01\x2f\x00\x2e\x02\x2d\x98\x2e\x5b\x0e\x30\x30\x98"
    b"\x2e\xce\xb7\x00\x30\x21\x2e\x7d\x00\x50\x32\x98\x2e\xfa\x03\x01\x2e\x77\x00"
    b"\x00\xb2\x24\x2f\x98\x2e\xf5\xcb\x03\x2e\xd5\x00\x11\x54\x01\x0a\xbc\x84\x83"
    b"\x86\x21\x2e\xc9\x01\xe0\x40\x13\x52\xc4\x40\x82\x40\xa8\xb9\x52\x42\x43\xbe"
    b"\x53\x42\x04\x0a\x50\x42\xe1\x7f\xf0\x31\x41\x40\xf2\x6f\x25\xbd\x08\x08\x02"
    b"\x0a\xd0\x7f\x98\x2e\xa8\xcf\x06\xbc\xd1\x6f\xe2\x6f\x08\x0a\x80\x42\x98\x2e"
    b"\x58\xb7\x00\x30\x21\x2e\xee\x00\x21\x2e\x77\x00\x21\x2e\xdd\x00\x80\x2e\xf4"
    b"\x01\x1a\x24\x22\x00\x80\x2e\xec\x01\x10\x50\xfb\x7f\x98\x2e\xf3\x03\x57\x50"
    b"\xfb\x6f\x01\x30\x71\x54\x11\x42\x42\x0e\xfc\x2f\xc0\x2e\x01\x42\xf0\x5f\x80"
    b"\x2e\x00\xc1\xfd\x2d\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x01"
    b"\x34\x03\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x20\x50\xe7\x7f\xf6\x7f\x06\x32\x0f\x2e\x61\xf5\xfe\x09\xc0\xb3\x04"
    b"\x2f\x17\x30\x2f\x2e\xef\x00\x2d\x2e\x61\xf5\xf6\x6f\xe7\x6f\xe0\x5f\xc8\x2e"
    b"\x20\x50\xe7\x7f\xf6\x7f\x46\x30\x0f\x2e\xa4\xf1\xbe\x09\x80\xb3\x06\x2f\x0d"
    b"\x2e\xd4\x00\x84\xaf\x02\x2f\x16\x30\x2d\x2e\x7b\x00\x86\x30\x2d\x2e\x60\xf5"
    b"\xf6\x6f\xe7\x6f\xe0\x5f\xc8\x2e\x01\x2e\x77\xf7\x09\xbc\x0f\xb8\x00\xb2\x10"
    b"\x50\xfb\x7f\x10\x30\x0b\x2f\x03\x2e\x8a\x00\x96\xbc\x9f\xb8\x40\xb2\x05\x2f"
    b"\x03\x2e\x68\xf7\x9e\xbc\x9f\xb8\x40\xb2\x07\x2f\x03\x2e\x7e\x00\x41\x90\x01"
    b"\x2f\x98\x2e\xdc\x03\x03\x2c\x00\x30\x21\x2e\x7e\x00\xfb\x6f\xf0\x5f\xb8\x2e"
    b"\x20\x50\xe0\x7f\xfb\x7f\x00\x2e\x27\x50\x98\x2e\x3b\xc8\x29\x50\x98\x2e\xa7"
    b"\xc8\x01\x50\x98\x2e\x55\xcc\xe1\x6f\x2b\x50\x98\x2e\xe0\xc9\xfb\x6f\x00\x30"
    b"\xe0\x5f\x21\x2e\x7e\x00\xb8\x2e\x73\x50\x01\x30\x57\x54\x11\x42\x42\x0e\xfc"
    b"\x2f\xb8\x2e\x21\x2e\x59\xf5\x10\x30\xc0\x2e\x21\x2e\x4a\xf1\x90\x50\xf7\x7f"
    b"\xe6\x7f\xd5\x7f\xc4\x7f\xb3\x7f\xa1\x7f\x90\x7f\x82\x7f\x7b\x7f\x98\x2e\x35"
    b"\xb7\x00\xb2\x90\x2e\x97\xb0\x03\x2e\x8f\x00\x07\x2e\x91\x00\x05\x2e\xb1\x00"
    b"\x3f\xba\x9f\xb8\x01\x2e\xb1\x00\xa3\xbd\x4c\x0a\x05\x2e\xb1\x00\x04\xbe\xbf"
    b"\xb9\xcb\x0a\x4f\xba\x22\xbd\x01\x2e\xb3\x00\xdc\x0a\x2f\xb9\x03\x2e\xb8\x00"
    b"\x0a\xbe\x9a\x0a\xcf\xb9\x9b\xbc\x01\x2e\x97\x00\x9f\xb8\x93\x0a\x0f\xbc\x91"
    b"\x0a\x0f\xb8\x90\x0a\x25\x2e\x18\x00\x05\x2e\xc1\xf5\x2e\xbd\x2e\xb9\x01\x2e"
    b"\x19\x00\x31\x30\x8a\x04\x00\x90\x07\x2f\x01\x2e\xd4\x00\x04\xa2\x03\x2f\x01"
    b"\x2e\x18\x00\x00\xb2\x0c\x2f\x19\x50\x05\x52\x98\x2e\x4d\xb7\x05\x2e\x78\x00"
    b"\x80\x90\x10\x30\x01\x2f\x21\x2e\x78\x00\x25\x2e\xdd\x00\x98\x2e\x3e\xb7\x00"
    b"\xb2\x02\x30\x01\x30\x04\x2f\x01\x2e\x19\x00\x00\xb2\x00\x2f\x21\x30\x01\x2e"
    b"\xea\x00\x08\x1a\x0e\x2f\x23\x2e\xea\x00\x33\x30\x1b\x50\x0b\x09\x01\x40\x17"
    b"\x56\x46\xbe\x4b\x08\x4c\x0a\x01\x42\x0a\x80\x15\x52\x01\x42\x00\x2e\x01\x2e"
    b"\x18\x00\x00\xb2\x1f\x2f\x03\x2e\xc0\xf5\xf0\x30\x48\x08\x47\xaa\x74\x30\x07"
    b"\x2e\x7a\x00\x61\x22\x4b\x1a\x05\x2f\x07\x2e\x66\xf5\xbf\xbd\xbf\xb9\xc0\x90"
    b"\x0b\x2f\x1d\x56\x2b\x30\xd2\x42\xdb\x42\x01\x04\xc2\x42\x04\xbd\xfe\x80\x81"
    b"\x84\x23\x2e\x7a\x00\x02\x42\x02\x32\x25\x2e\x62\xf5\x05\x2e\xd6\x00\x81\x84"
    b"\x25\x2e\xd6\x00\x02\x31\x25\x2e\x60\xf5\x05\x2e\x8a\x00\x0b\x50\x90\x08\x80"
    b"\xb2\x0b\x2f\x05\x2e\xca\xf5\xf0\x3e\x90\x08\x25\x2e\xca\xf5\x05\x2e\x59\xf5"
    b"\xe0\x3f\x90\x08\x25\x2e\x59\xf5\x90\x6f\xa1\x6f\xb3\x6f\xc4\x6f\xd5\x6f\xe6"
    b"\x6f\xf7\x6f\x7b\x6f\x82\x6f\x70\x5f\xc8\x2e\xc0\x50\x90\x7f\xe5\x7f\xd4\x7f"
    b"\xc3\x7f\xb1\x7f\xa2\x7f\x87\x7f\xf6\x7f\x7b\x7f\x00\x2e\x01\x2e\x60\xf5\x60"
    b"\x7f\x98\x2e\x35\xb7\x02\x30\x63\x6f\x15\x52\x50\x7f\x62\x7f\x5a\x2c\x02\x32"
    b"\x1a\x09\x00\xb3\x14\x2f\x00\xb2\x03\x2f\x09\x2e\x18\x00\x00\x91\x0c\x2f\x43"
    b"\x7f\x98\x2e\x97\xb7\x1f\x50\x02\x8a\x02\x32\x04\x30\x25\x2e\x64\xf5\x15\x52"
    b"\x50\x6f\x43\x6f\x44\x43\x25\x2e\x60\xf5\xd9\x08\xc0\xb2\x36\x2f\x98\x2e\x3e"
    b"\xb7\x00\xb2\x06\x2f\x01\x2e\x19\x00\x00\xb2\x02\x2f\x50\x6f\x00\x90\x0a\x2f"
    b"\x01\x2e\x79\x00\x00\x90\x19\x2f\x10\x30\x21\x2e\x79\x00\x00\x30\x98\x2e\xdc"
    b"\x03\x13\x2d\x01\x2e\xc3\xf5\x0c\xbc\x0f\xb8\x12\x30\x10\x04\x03\xb0\x26\x25"
    b"\x21\x50\x03\x52\x98\x2e\x4d\xb7\x10\x30\x21\x2e\xee\x00\x02\x30\x60\x7f\x25"
    b"\x2e\x79\x00\x60\x6f\x00\x90\x05\x2f\x00\x30\x21\x2e\xea\x00\x15\x50\x21\x2e"
    b"\x64\xf5\x15\x52\x23\x2e\x60\xf5\x02\x32\x50\x6f\x00\x90\x02\x2f\x03\x30\x27"
    b"\x2e\x78\x00\x07\x2e\x60\xf5\x1a\x09\x00\x91\xa3\x2f\x19\x09\x00\x91\xa0\x2f"
    b"\x90\x6f\xa2\x6f\xb1\x6f\xc3\x6f\xd4\x6f\xe5\x6f\x7b\x6f\xf6\x6f\x87\x6f\x40"
    b"\x5f\xc8\x2e\xc0\x50\xe7\x7f\xf6\x7f\x26\x30\x0f\x2e\x61\xf5\x2f\x2e\x7c\x00"
    b"\x0f\x2e\x7c\x00\xbe\x09\xa2\x7f\x80\x7f\x80\xb3\xd5\x7f\xc4\x7f\xb3\x7f\x91"
    b"\x7f\x7b\x7f\x0b\x2f\x23\x50\x1a\x25\x12\x40\x42\x7f\x74\x82\x12\x40\x52\x7f"
    b"\x00\x2e\x00\x40\x60\x7f\x98\x2e\x6a\xd6\x81\x30\x01\x2e\x7c\x00\x01\x08\x00"
    b"\xb2\x42\x2f\x03\x2e\x89\x00\x01\x2e\x89\x00\x97\xbc\x06\xbc\x9f\xb8\x0f\xb8"
    b"\x00\x90\x23\x2e\xd8\x00\x10\x30\x01\x30\x2a\x2f\x03\x2e\xd4\x00\x44\xb2\x05"
    b"\x2f\x47\xb2\x00\x30\x2d\x2f\x21\x2e\x7c\x00\x2b\x2d\x03\x2e\xfd\xf5\x9e\xbc"
    b"\x9f\xb8\x40\x90\x14\x2f\x03\x2e\xfc\xf5\x99\xbc\x9f\xb8\x40\x90\x0e\x2f\x03"
    b"\x2e\x49\xf1\x25\x54\x4a\x08\x40\x90\x08\x2f\x98\x2e\x35\xb7\x00\xb2\x10\x30"
    b"\x03\x2f\x50\x30\x21\x2e\xd4\x00\x10\x2d\x98\x2e\xaf\xb7\x00\x30\x21\x2e\x7c"
    b"\x00\x0a\x2d\x05\x2e\x69\xf7\x2d\xbd\x2f\xb9\x80\xb2\x01\x2f\x21\x2e\x7d\x00"
    b"\x23\x2e\x7c\x00\xe0\x31\x21\x2e\x61\xf5\xf6\x6f\xe7\x6f\x80\x6f\xa2\x6f\xb3"
    b"\x6f\xc4\x6f\xd5\x6f\x7b\x6f\x91\x6f\x40\x5f\xc8\x2e\x60\x51\x0a\x25\x36\x88"
    b"\xf4\x7f\xeb\x7f\x00\x32\x31\x52\x32\x30\x13\x30\x98\x2e\x15\xcb\x0a\x25\x33"
    b"\x84\xd2\x7f\x43\x30\x05\x50\x2d\x52\x98\x2e\x95\xc1\xd2\x6f\x27\x52\x98\x2e"
    b"\xd7\xc7\x2a\x25\xb0\x86\xc0\x7f\xd3\x7f\xaf\x84\x29\x50\xf1\x6f\x98\x2e\x4d"
    b"\xc8\x2a\x25\xae\x8a\xaa\x88\xf2\x6e\x2b\x50\xc1\x6f\xd3\x6f\xf4\x7f\x98\x2e"
    b"\xb6\xc8\xe0\x6e\x00\xb2\x32\x2f\x33\x54\x83\x86\xf1\x6f\xc3\x7f\x04\x30\x30"
    b"\x30\xf4\x7f\xd0\x7f\xb2\x7f\xe3\x30\xc5\x6f\x56\x40\x45\x41\x28\x08\x03\x14"
    b"\x0e\xb4\x08\xbc\x82\x40\x10\x0a\x2f\x54\x26\x05\x91\x7f\x44\x28\xa3\x7f\x98"
    b"\x2e\xd9\xc0\x08\xb9\x33\x30\x53\x09\xc1\x6f\xd3\x6f\xf4\x6f\x83\x17\x47\x40"
    b"\x6c\x15\xb2\x6f\xbe\x09\x75\x0b\x90\x42\x45\x42\x51\x0e\x32\xbc\x02\x89\xa1"
    b"\x6f\x7e\x86\xf4\x7f\xd0\x7f\xb2\x7f\x04\x30\x91\x6f\xd6\x2f\xeb\x6f\xa0\x5e"
    b"\xb8\x2e\x03\x2e\x97\x00\x1b\xbc\x60\x50\x9f\xbc\x0c\xb8\xf0\x7f\x40\xb2\xeb"
    b"\x7f\x2b\x2f\x03\x2e\x7f\x00\x41\x40\x01\x2e\xc8\x00\x01\x1a\x11\x2f\x37\x58"
    b"\x23\x2e\xc8\x00\x10\x41\xa0\x7f\x38\x81\x01\x41\xd0\x7f\xb1\x7f\x98\x2e\x64"
    b"\xcf\xd0\x6f\x07\x80\xa1\x6f\x11\x42\x00\x2e\xb1\x6f\x01\x42\x11\x30\x01\x2e"
    b"\xfc\x00\x00\xa8\x03\x30\xcb\x22\x4a\x25\x01\x2e\x7f\x00\x3c\x89\x35\x52\x05"
    b"\x54\x98\x2e\xc4\xce\xc1\x6f\xf0\x6f\x98\x2e\x95\xcf\x04\x2d\x01\x30\xf0\x6f"
    b"\x98\x2e\x95\xcf\xeb\x6f\xa0\x5f\xb8\x2e\x03\x2e\xb3\x00\x02\x32\xf0\x30\x03"
    b"\x31\x30\x50\x8a\x08\x08\x08\xcb\x08\xe0\x7f\x80\xb2\xf3\x7f\xdb\x7f\x25\x2f"
    b"\x03\x2e\xca\x00\x41\x90\x04\x2f\x01\x30\x23\x2e\xca\x00\x98\x2e\x3f\x03\xc0"
    b"\xb2\x05\x2f\x03\x2e\xda\x00\x00\x30\x41\x04\x23\x2e\xda\x00\x98\x2e\x92\xb2"
    b"\x10\x25\xf0\x6f\x00\xb2\x05\x2f\x01\x2e\xda\x00\x02\x30\x10\x04\x21\x2e\xda"
    b"\x00\x40\xb2\x01\x2f\x23\x2e\xc8\x01\xdb\x6f\xe0\x6f\xd0\x5f\x80\x2e\x95\xcf"
    b"\x01\x30\xe0\x6f\x98\x2e\x95\xcf\x11\x30\x23\x2e\xca\x00\xdb\x6f\xd0\x5f\xb8"
    b"\x2e\xd0\x50\x0a\x25\x33\x84\x55\x50\xd2\x7f\xe2\x7f\x03\x8c\xc0\x7f\xbb\x7f"
    b"\x00\x30\x05\x5a\x39\x54\x51\x41\xa5\x7f\x96\x7f\x80\x7f\x98\x2e\xd9\xc0\x05"
    b"\x30\xf5\x7f\x20\x25\x91\x6f\x3b\x58\x3d\x5c\x3b\x56\x98\x2e\x67\xcc\xc1\x6f"
    b"\xd5\x6f\x52\x40\x50\x43\xc1\x7f\xd5\x7f\x10\x25\x98\x2e\xfe\xc9\x10\x25\x98"
    b"\x2e\x74\xc0\x86\x6f\x30\x28\x92\x6f\x82\x8c\xa5\x6f\x6f\x52\x69\x0e\x39\x54"
    b"\xdb\x2f\x19\xa0\x15\x30\x03\x2f\x00\x30\x21\x2e\x81\x01\x0a\x2d\x01\x2e\x81"
    b"\x01\x05\x28\x42\x36\x21\x2e\x81\x01\x02\x0e\x01\x2f\x98\x2e\xf3\x03\x57\x50"
    b"\x12\x30\x01\x40\x98\x2e\xfe\xc9\x51\x6f\x0b\x5c\x8e\x0e\x3b\x6f\x57\x58\x02"
    b"\x30\x21\x2e\x95\x01\x45\x6f\x2a\x8d\xd2\x7f\xcb\x7f\x13\x2f\x02\x30\x3f\x50"
    b"\xd2\x7f\xa8\x0e\x0e\x2f\xc0\x6f\x53\x54\x02\x00\x51\x54\x42\x0e\x10\x30\x59"
    b"\x52\x02\x30\x01\x2f\x00\x2e\x03\x2d\x50\x42\x42\x42\x12\x30\xd2\x7f\x80\xb2"
    b"\x03\x2f\x00\x30\x21\x2e\x80\x01\x12\x2d\x01\x2e\xc9\x00\x02\x80\x05\x2e\x80"
    b"\x01\x11\x30\x91\x28\x00\x40\x25\x2e\x80\x01\x10\x0e\x05\x2f\x01\x2e\x7f\x01"
    b"\x01\x90\x01\x2f\x98\x2e\xf3\x03\x00\x2e\xa0\x41\x01\x90\xa6\x7f\x90\x2e\xe3"
    b"\xb4\x01\x2e\x95\x01\x00\xa8\x90\x2e\xe3\xb4\x5b\x54\x95\x80\x82\x40\x80\xb2"
    b"\x02\x40\x2d\x8c\x3f\x52\x96\x7f\x90\x2e\xc2\xb3\x29\x0e\x76\x2f\x01\x2e\xc9"
    b"\x00\x00\x40\x81\x28\x45\x52\xb3\x30\x98\x2e\x0f\xca\x5d\x54\x80\x7f\x00\x2e"
    b"\xa1\x40\x72\x7f\x82\x80\x82\x40\x60\x7f\x98\x2e\xfe\xc9\x10\x25\x98\x2e\x74"
    b"\xc0\x62\x6f\x05\x30\x87\x40\xc0\x91\x04\x30\x05\x2f\x05\x2e\x83\x01\x80\xb2"
    b"\x14\x30\x00\x2f\x04\x30\x05\x2e\xc9\x00\x73\x6f\x81\x40\xe2\x40\x69\x04\x11"
    b"\x0f\xe1\x40\x16\x30\xfe\x29\xcb\x40\x02\x2f\x83\x6f\x83\x0f\x22\x2f\x47\x56"
    b"\x13\x0f\x12\x30\x77\x2f\x49\x54\x42\x0e\x12\x30\x73\x2f\x00\x91\x0a\x2f\x01"
    b"\x2e\x8b\x01\x19\xa8\x02\x30\x6c\x2f\x63\x50\x00\x2e\x17\x42\x05\x42\x68\x2c"
    b"\x12\x30\x0b\x25\x08\x0f\x50\x30\x02\x2f\x21\x2e\x83\x01\x03\x2d\x40\x30\x21"
    b"\x2e\x83\x01\x2b\x2e\x85\x01\x5a\x2c\x12\x30\x00\x91\x2b\x25\x04\x2f\x63\x50"
    b"\x02\x30\x17\x42\x17\x2c\x02\x42\x98\x2e\xfe\xc9\x10\x25\x98\x2e\x74\xc0\x05"
    b"\x2e\xc9\x00\x81\x84\x5b\x30\x82\x40\x37\x2e\x83\x01\x02\x0e\x07\x2f\x5f\x52"
    b"\x40\x30\x62\x40\x41\x40\x91\x0e\x01\x2f\x21\x2e\x83\x01\x05\x30\x2b\x2e\x85"
    b"\x01\x12\x30\x36\x2c\x16\x30\x15\x25\x81\x7f\x98\x2e\xfe\xc9\x10\x25\x98\x2e"
    b"\x74\xc0\x19\xa2\x16\x30\x15\x2f\x05\x2e\x97\x01\x80\x6f\x82\x0e\x05\x2f\x01"
    b"\x2e\x86\x01\x06\x28\x21\x2e\x86\x01\x0b\x2d\x03\x2e\x87\x01\x5f\x54\x4e\x28"
    b"\x91\x42\x00\x2e\x82\x40\x90\x0e\x01\x2f\x21\x2e\x88\x01\x02\x30\x13\x2c\x05"
    b"\x30\xc0\x6f\x08\x1c\xa8\x0f\x16\x30\x05\x30\x5b\x50\x09\x2f\x02\x80\x2d\x2e"
    b"\x82\x01\x05\x42\x05\x80\x00\x2e\x02\x42\x3e\x80\x00\x2e\x06\x42\x02\x30\x90"
    b"\x6f\x3e\x88\x01\x40\x04\x41\x4c\x28\x01\x42\x07\x80\x10\x25\x24\x40\x00\x40"
    b"\x00\xa8\xf5\x22\x23\x29\x44\x42\x7a\x82\x7e\x88\x43\x40\x04\x41\x00\xab\xf5"
    b"\x23\xdf\x28\x43\x42\xd9\xa0\x14\x2f\x00\x90\x02\x2f\xd2\x6f\x81\xb2\x05\x2f"
    b"\x63\x54\x06\x28\x90\x42\x85\x42\x09\x2c\x02\x30\x5b\x50\x03\x80\x29\x2e\x7e"
    b"\x01\x2b\x2e\x82\x01\x05\x42\x12\x30\x2b\x2e\x83\x01\x45\x82\x00\x2e\x40\x40"
    b"\x7a\x82\x02\xa0\x08\x2f\x63\x50\x3b\x30\x15\x42\x05\x42\x37\x80\x37\x2e\x7e"
    b"\x01\x05\x42\x12\x30\x01\x2e\xc9\x00\x02\x8c\x40\x40\x84\x41\x7a\x8c\x04\x0f"
    b"\x03\x2f\x01\x2e\x8b\x01\x19\xa4\x04\x2f\x2b\x2e\x82\x01\x98\x2e\xf3\x03\x12"
    b"\x30\x81\x90\x61\x52\x08\x2f\x65\x42\x65\x42\x43\x80\x39\x84\x82\x88\x05\x42"
    b"\x45\x42\x85\x42\x05\x43\x00\x2e\x80\x41\x00\x90\x90\x2e\xe1\xb4\x65\x54\xc1"
    b"\x6f\x80\x40\x00\xb2\x43\x58\x69\x50\x44\x2f\x55\x5c\xb7\x87\x8c\x0f\x0d\x2e"
    b"\x96\x01\xc4\x40\x36\x2f\x41\x56\x8b\x0e\x2a\x2f\x0b\x52\xa1\x0e\x0a\x2f\x05"
    b"\x2e\x8f\x01\x14\x25\x98\x2e\xfe\xc9\x4b\x54\x02\x0f\x69\x50\x05\x30\x65\x54"
    b"\x15\x2f\x03\x2e\x8e\x01\x4d\x5c\x8e\x0f\x3a\x2f\x05\x2e\x8f\x01\x98\x2e\xfe"
    b"\xc9\x4f\x54\x82\x0f\x05\x30\x69\x50\x65\x54\x30\x2f\x6d\x52\x15\x30\x42\x8c"
    b"\x45\x42\x04\x30\x2b\x2c\x84\x43\x6b\x52\x42\x8c\x00\x2e\x85\x43\x15\x30\x24"
    b"\x2c\x45\x42\x8e\x0f\x20\x2f\x0d\x2e\x8e\x01\xb1\x0e\x1c\x2f\x23\x2e\x8e\x01"
    b"\x1a\x2d\x0e\x0e\x17\x2f\xa1\x0f\x15\x2f\x23\x2e\x8d\x01\x13\x2d\x98\x2e\x74"
    b"\xc0\x43\x54\xc2\x0e\x0a\x2f\x65\x50\x04\x80\x0b\x30\x06\x82\x0b\x42\x79\x80"
    b"\x41\x40\x12\x30\x25\x2e\x8c\x01\x01\x42\x05\x30\x69\x50\x65\x54\x84\x82\x43"
    b"\x84\xbe\x8c\x84\x40\x86\x41\x26\x29\x94\x42\xbe\x8e\xd5\x7f\x19\xa1\x43\x40"
    b"\x0b\x2e\x8c\x01\x84\x40\xc7\x41\x5d\x29\x27\x29\x45\x42\x84\x42\xc2\x7f\x01"
    b"\x2f\xc0\xb3\x1d\x2f\x05\x2e\x94\x01\x99\xa0\x01\x2f\x80\xb3\x13\x2f\x80\xb3"
    b"\x18\x2f\xc0\xb3\x16\x2f\x12\x40\x01\x40\x92\x7f\x98\x2e\x74\xc0\x92\x6f\x10"
    b"\x0f\x20\x30\x03\x2f\x10\x30\x21\x2e\x7e\x01\x0a\x2d\x21\x2e\x7e\x01\x07\x2d"
    b"\x20\x30\x21\x2e\x7e\x01\x03\x2d\x10\x30\x21\x2e\x7e\x01\xc2\x6f\x01\x2e\xc9"
    b"\x00\xbc\x84\x02\x80\x82\x40\x00\x40\x90\x0e\xd5\x6f\x02\x2f\x15\x30\x98\x2e"
    b"\xf3\x03\x41\x91\x05\x30\x07\x2f\x67\x50\x3d\x80\x2b\x2e\x8f\x01\x05\x42\x04"
    b"\x80\x00\x2e\x05\x42\x02\x2c\x00\x30\x00\x30\xa2\x6f\x98\x8a\x86\x40\x80\xa7"
    b"\x05\x2f\x98\x2e\xf3\x03\xc0\x30\x21\x2e\x95\x01\x06\x25\x1a\x25\xe2\x6f\x76"
    b"\x82\x96\x40\x56\x43\x51\x0e\xfb\x2f\xbb\x6f\x30\x5f\xb8\x2e\x01\x2e\xb8\x00"
    b"\x01\x31\x41\x08\x40\xb2\x20\x50\xf2\x30\x02\x08\xfb\x7f\x01\x30\x10\x2f\x05"
    b"\x2e\xcc\x00\x81\x90\xe0\x7f\x03\x2f\x23\x2e\xcc\x00\x98\x2e\x55\xb6\x98\x2e"
    b"\x1d\xb5\x10\x25\xfb\x6f\xe0\x6f\xe0\x5f\x80\x2e\x95\xcf\x98\x2e\x95\xcf\x10"
    b"\x30\x21\x2e\xcc\x00\xfb\x6f\xe0\x5f\xb8\x2e\x00\x51\x05\x58\xeb\x7f\x2a\x25"
    b"\x89\x52\x6f\x5a\x89\x50\x13\x41\x06\x40\xb3\x01\x16\x42\xcb\x16\x06\x40\xf3"
    b"\x02\x13\x42\x65\x0e\xf5\x2f\x05\x40\x14\x30\x2c\x29\x04\x42\x08\xa1\x00\x30"
    b"\x90\x2e\x52\xb6\xb3\x88\xb0\x8a\xb6\x84\xa4\x7f\xc4\x7f\xb5\x7f\xd5\x7f\x92"
    b"\x7f\x73\x30\x04\x30\x55\x40\x42\x40\x8a\x17\xf3\x08\x6b\x01\x90\x02\x53\xb8"
    b"\x4b\x82\xad\xbe\x71\x7f\x45\x0a\x09\x54\x84\x7f\x98\x2e\xd9\xc0\xa3\x6f\x7b"
    b"\x54\xd0\x42\xa3\x7f\xf2\x7f\x60\x7f\x20\x25\x71\x6f\x75\x5a\x77\x58\x79\x5c"
    b"\x75\x56\x98\x2e\x67\xcc\xb1\x6f\x62\x6f\x50\x42\xb1\x7f\xb3\x30\x10\x25\x98"
    b"\x2e\x0f\xca\x84\x6f\x20\x29\x71\x6f\x92\x6f\xa5\x6f\x76\x82\x6a\x0e\x73\x30"
    b"\x00\x30\xd0\x2f\xd2\x6f\xd1\x7f\xb4\x7f\x98\x2e\x2b\xb7\x15\xbd\x0b\xb8\x02"
    b"\x0a\xc2\x6f\xc0\x7f\x98\x2e\x2b\xb7\x15\xbd\x0b\xb8\x42\x0a\xc0\x6f\x08\x17"
    b"\x41\x18\x89\x16\xe1\x18\xd0\x18\xa1\x7f\x27\x25\x16\x25\x98\x2e\x79\xc0\x8b"
    b"\x54\x90\x7f\xb3\x30\x82\x40\x80\x90\x0d\x2f\x7d\x52\x92\x6f\x98\x2e\x0f\xca"
    b"\xb2\x6f\x90\x0e\x06\x2f\x8b\x50\x14\x30\x42\x6f\x51\x6f\x14\x42\x12\x42\x01"
    b"\x42\x00\x2e\x31\x6f\x98\x2e\x74\xc0\x41\x6f\x80\x7f\x98\x2e\x74\xc0\x82\x6f"
    b"\x10\x04\x43\x52\x01\x0f\x05\x2e\xcb\x00\x00\x30\x04\x30\x21\x2f\x51\x6f\x43"
    b"\x58\x8c\x0e\x04\x30\x1c\x2f\x85\x88\x41\x6f\x04\x41\x8c\x0f\x04\x30\x16\x2f"
    b"\x84\x88\x00\x2e\x04\x41\x04\x05\x8c\x0e\x04\x30\x0f\x2f\x82\x88\x31\x6f\x04"
    b"\x41\x04\x05\x8c\x0e\x04\x30\x08\x2f\x83\x88\x00\x2e\x04\x41\x8c\x0f\x04\x30"
    b"\x02\x2f\x21\x2e\xad\x01\x14\x30\x00\x91\x14\x2f\x03\x2e\xa1\x01\x41\x90\x0e"
    b"\x2f\x03\x2e\xad\x01\x14\x30\x4c\x28\x23\x2e\xad\x01\x46\xa0\x06\x2f\x81\x84"
    b"\x8d\x52\x48\x82\x82\x40\x21\x2e\xa1\x01\x42\x42\x5c\x2c\x02\x30\x05\x2e\xaa"
    b"\x01\x80\xb2\x02\x30\x55\x2f\x03\x2e\xa9\x01\x92\x6f\xb3\x30\x98\x2e\x0f\xca"
    b"\xb2\x6f\x90\x0f\x00\x30\x02\x30\x4a\x2f\xa2\x6f\x87\x52\x91\x00\x85\x52\x51"
    b"\x0e\x02\x2f\x00\x2e\x43\x2c\x02\x30\xc2\x6f\x7f\x52\x91\x0e\x02\x30\x3c\x2f"
    b"\x51\x6f\x81\x54\x98\x2e\xfe\xc9\x10\x25\xb3\x30\x21\x25\x98\x2e\x0f\xca\x32"
    b"\x6f\xc0\x7f\xb3\x30\x12\x25\x98\x2e\x0f\xca\x42\x6f\xb0\x7f\xb3\x30\x12\x25"
    b"\x98\x2e\x0f\xca\xb2\x6f\x90\x28\x83\x52\x98\x2e\xfe\xc9\xc2\x6f\x90\x0f\x00"
    b"\x30\x02\x30\x1d\x2f\x05\x2e\xa1\x01\x80\xb2\x12\x30\x0f\x2f\x42\x6f\x03\x2e"
    b"\xab\x01\x91\x0e\x02\x30\x12\x2f\x52\x6f\x03\x2e\xac\x01\x91\x0f\x02\x30\x0c"
    b"\x2f\x21\x2e\xaa\x01\x0a\x2c\x12\x30\x03\x2e\xcb\x00\x8d\x58\x08\x89\x41\x40"
    b"\x11\x43\x00\x43\x25\x2e\xa1\x01\xd4\x6f\x8f\x52\x00\x43\x3a\x89\x00\x2e\x10"
    b"\x43\x10\x43\x61\x0e\xfb\x2f\x03\x2e\xa0\x01\x11\x1a\x02\x2f\x02\x25\x21\x2e"
    b"\xa0\x01\xeb\x6f\x00\x5f\xb8\x2e\x91\x52\x10\x30\x02\x30\x95\x56\x52\x42\x4b"
    b"\x0e\xfc\x2f\x8d\x54\x88\x82\x93\x56\x80\x42\x53\x42\x40\x42\x42\x86\x83\x54"
    b"\xc0\x2e\xc2\x42\x00\x2e\xa3\x52\x00\x51\x52\x40\x47\x40\x1a\x25\x01\x2e\x97"
    b"\x00\x8f\xbe\x72\x86\xfb\x7f\x0b\x30\x7c\xbf\xa5\x50\x10\x08\xdf\xba\x70\x88"
    b"\xf8\xbf\xcb\x42\xd3\x7f\x6c\xbb\xfc\xbb\xc5\x0a\x90\x7f\x1b\x7f\x0b\x43\xc0"
    b"\xb2\xe5\x7f\xb7\x7f\xa6\x7f\xc4\x7f\x90\x2e\x1c\xb7\x07\x2e\xd2\x00\xc0\xb2"
    b"\x0b\x2f\x97\x52\x01\x2e\xcd\x00\x82\x7f\x98\x2e\xbb\xcc\x0b\x30\x37\x2e\xd2"
    b"\x00\x82\x6f\x90\x6f\x1a\x25\x00\xb2\x8b\x7f\x14\x2f\xa6\xbd\x25\xbd\xb6\xb9"
    b"\x2f\xb9\x80\xb2\xd4\xb0\x0c\x2f\x99\x54\x9b\x56\x0b\x30\x0b\x2e\xb1\x00\xa1"
    b"\x58\x9b\x42\xdb\x42\x6c\x09\x2b\x2e\xb1\x00\x8b\x42\xcb\x42\x86\x7f\x73\x84"
    b"\xa7\x56\xc3\x08\x39\x52\x05\x50\x72\x7f\x63\x7f\x98\x2e\xc2\xc0\xe1\x6f\x62"
    b"\x6f\xd1\x0a\x01\x2e\xcd\x00\xd5\x6f\xc4\x6f\x72\x6f\x97\x52\x9d\x5c\x98\x2e"
    b"\x06\xcd\x23\x6f\x90\x6f\x99\x52\xc0\xb2\x04\xbd\x54\x40\xaf\xb9\x45\x40\xe1"
    b"\x7f\x02\x30\x06\x2f\xc0\xb2\x02\x30\x03\x2f\x9b\x5c\x12\x30\x94\x43\x85\x43"
    b"\x03\xbf\x6f\xbb\x80\xb3\x20\x2f\x06\x6f\x26\x01\x16\x6f\x6e\x03\x45\x42\xc0"
    b"\x90\x29\x2e\xce\x00\x9b\x52\x14\x2f\x9b\x5c\x00\x2e\x93\x41\x86\x41\xe3\x04"
    b"\xae\x07\x80\xab\x04\x2f\x80\x91\x0a\x2f\x86\x6f\x73\x0f\x07\x2f\x83\x6f\xc0"
    b"\xb2\x04\x2f\x54\x42\x45\x42\x12\x30\x04\x2c\x11\x30\x02\x2c\x11\x30\x11\x30"
    b"\x02\xbc\x0f\xb8\xd2\x7f\x00\xb2\x0a\x2f\x01\x2e\xfc\x00\x05\x2e\xc7\x01\x10"
    b"\x1a\x02\x2f\x21\x2e\xc7\x01\x03\x2d\x02\x2c\x01\x30\x01\x30\xb0\x6f\x98\x2e"
    b"\x95\xcf\xd1\x6f\xa0\x6f\x98\x2e\x95\xcf\xe2\x6f\x9f\x52\x01\x2e\xce\x00\x82"
    b"\x40\x50\x42\x0c\x2c\x42\x42\x11\x30\x23\x2e\xd2\x00\x01\x30\xb0\x6f\x98\x2e"
    b"\x95\xcf\xa0\x6f\x01\x30\x98\x2e\x95\xcf\x00\x2e\xfb\x6f\x00\x5f\xb8\x2e\x83"
    b"\x86\x01\x30\x00\x30\x94\x40\x24\x18\x06\x00\x53\x0e\x4f\x02\xf9\x2f\xb8\x2e"
    b"\xa9\x52\x00\x2e\x60\x40\x41\x40\x0d\xbc\x98\xbc\xc0\x2e\x01\x0a\x0f\xb8\xab"
    b"\x52\x53\x3c\x52\x40\x40\x40\x4b\x00\x82\x16\x26\xb9\x01\xb8\x41\x40\x10\x08"
    b"\x97\xb8\x01\x08\xc0\x2e\x11\x30\x01\x08\x43\x86\x25\x40\x04\x40\xd8\xbe\x2c"
    b"\x0b\x22\x11\x54\x42\x03\x80\x4b\x0e\xf6\x2f\xb8\x2e\x9f\x50\x10\x50\xad\x52"
    b"\x05\x2e\xd3\x00\xfb\x7f\x00\x2e\x13\x40\x93\x42\x41\x0e\xfb\x2f\x98\x2e\xa5"
    b"\xb7\x98\x2e\x87\xcf\x01\x2e\xd9\x00\x00\xb2\xfb\x6f\x0b\x2f\x01\x2e\x69\xf7"
    b"\xb1\x3f\x01\x08\x01\x30\xf0\x5f\x23\x2e\xd9\x00\x21\x2e\x69\xf7\x80\x2e\x7a"
    b"\xb7\xf0\x5f\xb8\x2e\x01\x2e\xc0\xf8\x03\x2e\xfc\xf5\x15\x54\xaf\x56\x82\x08"
    b"\x0b\x2e\x69\xf7\xcb\x0a\xb1\x58\x80\x90\xdd\xbe\x4c\x08\x5f\xb9\x59\x22\x80"
    b"\x90\x07\x2f\x03\x34\xc3\x08\xf2\x3a\x0a\x08\x02\x35\xc0\x90\x4a\x0a\x48\x22"
    b"\xc0\x2e\x23\x2e\xfc\xf5\x10\x50\xfb\x7f\x98\x2e\x56\xc7\x98\x2e\x49\xc3\x10"
    b"\x30\xfb\x6f\xf0\x5f\x21\x2e\xcc\x00\x21\x2e\xca\x00\xb8\x2e\x03\x2e\xd3\x00"
    b"\x16\xb8\x02\x34\x4a\x0c\x21\x2e\x2d\xf5\xc0\x2e\x23\x2e\xd3\x00\x03\xbc\x21"
    b"\x2e\xd5\x00\x03\x2e\xd5\x00\x40\xb2\x10\x30\x21\x2e\x77\x00\x01\x30\x05\x2f"
    b"\x05\x2e\xd8\x00\x80\x90\x01\x2f\x23\x2e\x6f\xf5\xc0\x2e\x21\x2e\xd9\x00\x11"
    b"\x30\x81\x08\x01\x2e\x6a\xf7\x71\x3f\x23\xbd\x01\x08\x02\x0a\xc0\x2e\x21\x2e"
    b"\x6a\xf7\x30\x25\x00\x30\x21\x2e\x5a\xf5\x10\x50\x21\x2e\x7b\x00\x21\x2e\x7c"
    b"\x00\xfb\x7f\x98\x2e\xc3\xb7\x40\x30\x21\x2e\xd4\x00\xfb\x6f\xf0\x5f\x03\x25"
    b"\x80\x2e\xaf\xb7\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00"
    b"\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e"
    b"\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80"
    b"\x2e\x00\xc1\x80\x2e\x00\xc1\x01\x2e\x5d\xf7\x08\xbc\x80\xac\x0e\xbb\x02\x2f"
    b"\x00\x30\x41\x04\x82\x06\xc0\xa4\x00\x30\x11\x2f\x40\xa9\x03\x2f\x40\x91\x0d"
    b"\x2f\x00\xa7\x0b\x2f\x80\xb3\xb3\x58\x02\x2f\x90\xa1\x26\x13\x20\x23\x80\x90"
    b"\x10\x30\x01\x2f\xcc\x0e\x00\x2f\x00\x30\xb8\x2e\xb5\x50\x18\x08\x08\xbc\x88"
    b"\xb6\x0d\x17\xc6\xbd\x56\xbc\xb7\x58\xda\xba\x04\x01\x1d\x0a\x10\x50\x05\x30"
    b"\x32\x25\x45\x03\xfb\x7f\xf6\x30\x21\x25\x98\x2e\x37\xca\x16\xb5\x9a\xbc\x06"
    b"\xb8\x80\xa8\x41\x0a\x0e\x2f\x80\x90\x02\x2f\x2d\x50\x48\x0f\x09\x2f\xbf\xa0"
    b"\x04\x2f\xbf\x90\x06\x2f\xb7\x54\xca\x0f\x03\x2f\x00\x2e\x02\x2c\xb7\x52\x2d"
    b"\x52\xf2\x33\x98\x2e\xd9\xc0\xfb\x6f\xf1\x37\xc0\x2e\x01\x08\xf0\x5f\xbf\x56"
    b"\xb9\x54\xd0\x40\xc4\x40\x0b\x2e\xfd\xf3\xbf\x52\x90\x42\x94\x42\x95\x42\x05"
    b"\x30\xc1\x50\x0f\x88\x06\x40\x04\x41\x96\x42\xc5\x42\x48\xbe\x73\x30\x0d\x2e"
    b"\xd8\x00\x4f\xba\x84\x42\x03\x42\x81\xb3\x02\x2f\x2b\x2e\x6f\xf5\x06\x2d\x05"
    b"\x2e\x77\xf7\xbd\x56\x93\x08\x25\x2e\x77\xf7\xbb\x54\x25\x2e\xc2\xf5\x07\x2e"
    b"\xfd\xf3\x42\x30\xb4\x33\xda\x0a\x4c\x00\x27\x2e\xfd\xf3\x43\x40\xd4\x3f\xdc"
    b"\x08\x43\x42\x00\x2e\x00\x2e\x43\x40\x24\x30\xdc\x0a\x43\x42\x04\x80\x03\x2e"
    b"\xfd\xf3\x4a\x0a\x23\x2e\xfd\xf3\x61\x34\xc0\x2e\x01\x42\x00\x2e\x60\x50\x1a"
    b"\x25\x7a\x86\xe0\x7f\xf3\x7f\x03\x25\xc3\x52\x41\x84\xdb\x7f\x33\x30\x98\x2e"
    b"\x16\xc2\x1a\x25\x7d\x82\xf0\x6f\xe2\x6f\x32\x25\x16\x40\x94\x40\x26\x01\x85"
    b"\x40\x8e\x17\xc4\x42\x6e\x03\x95\x42\x41\x0e\xf4\x2f\xdb\x6f\xa0\x5f\xb8\x2e"
    b"\xb0\x51\xfb\x7f\x98\x2e\xe8\x0d\x5a\x25\x98\x2e\x0f\x0e\xcb\x58\x32\x87\xc4"
    b"\x7f\x65\x89\x6b\x8d\xc5\x5a\x65\x7f\xe1\x7f\x83\x7f\xa6\x7f\x74\x7f\xd0\x7f"
    b"\xb6\x7f\x94\x7f\x17\x30\xc7\x52\xc9\x54\x51\x7f\x00\x2e\x85\x6f\x42\x7f\x00"
    b"\x2e\x51\x41\x45\x81\x42\x41\x13\x40\x3b\x8a\x00\x40\x4b\x04\xd0\x06\xc0\xac"
    b"\x85\x7f\x02\x2f\x02\x30\x51\x04\xd3\x06\x41\x84\x05\x30\x5d\x02\xc9\x16\xdf"
    b"\x08\xd3\x00\x8d\x02\xaf\xbc\xb1\xb9\x59\x0a\x65\x6f\x11\x43\xa1\xb4\x52\x41"
    b"\x53\x41\x01\x43\x34\x7f\x65\x7f\x26\x31\xe5\x6f\xd4\x6f\x98\x2e\x37\xca\x32"
    b"\x6f\x75\x6f\x83\x40\x42\x41\x23\x7f\x12\x7f\xf6\x30\x40\x25\x51\x25\x98\x2e"
    b"\x37\xca\x14\x6f\x20\x05\x70\x6f\x25\x6f\x69\x07\xa2\x6f\x31\x6f\x0b\x30\x04"
    b"\x42\x9b\x42\x8b\x42\x55\x42\x32\x7f\x40\xa9\xc3\x6f\x71\x7f\x02\x30\xd0\x40"
    b"\xc3\x7f\x03\x2f\x40\x91\x15\x2f\x00\xa7\x13\x2f\x00\xa4\x11\x2f\x84\xbd\x98"
    b"\x2e\x79\xca\x55\x6f\xb7\x54\x54\x41\x82\x00\xf3\x3f\x45\x41\xcb\x02\xf6\x30"
    b"\x98\x2e\x37\xca\x35\x6f\xa4\x6f\x41\x43\x03\x2c\x00\x43\xa4\x6f\x35\x6f\x17"
    b"\x30\x42\x6f\x51\x6f\x93\x40\x42\x82\x00\x41\xc3\x00\x03\x43\x51\x7f\x00\x2e"
    b"\x94\x40\x41\x41\x4c\x02\xc4\x6f\xd1\x56\x63\x0e\x74\x6f\x51\x43\xa5\x7f\x8a"
    b"\x2f\x09\x2e\xd8\x00\x01\xb3\x21\x2f\xcb\x58\x90\x6f\x13\x41\xb6\x6f\xe4\x7f"
    b"\x00\x2e\x91\x41\x14\x40\x92\x41\x15\x40\x17\x2e\x6f\xf5\xb6\x7f\xd0\x7f\xcb"
    b"\x7f\x98\x2e\x00\x0c\x07\x15\xc2\x6f\x14\x0b\x29\x2e\x6f\xf5\xc3\xa3\xc1\x8f"
    b"\xe4\x6f\xd0\x6f\xe6\x2f\x14\x30\x05\x2e\x6f\xf5\x14\x0b\x29\x2e\x6f\xf5\x18"
    b"\x2d\xcd\x56\x04\x32\xb5\x6f\x1c\x01\x51\x41\x52\x41\xc3\x40\xb5\x7f\xe4\x7f"
    b"\x98\x2e\x1f\x0c\xe4\x6f\x21\x87\x00\x43\x04\x32\xcf\x54\x5a\x0e\xef\x2f\x15"
    b"\x54\x09\x2e\x77\xf7\x22\x0b\x29\x2e\x77\xf7\xfb\x6f\x50\x5e\xb8\x2e\x10\x50"
    b"\x01\x2e\xd4\x00\x00\xb2\xfb\x7f\x51\x2f\x01\xb2\x48\x2f\x02\xb2\x42\x2f\x03"
    b"\x90\x56\x2f\xd7\x52\x79\x80\x42\x40\x81\x84\x00\x40\x42\x42\x98\x2e\x93\x0c"
    b"\xd9\x54\xd7\x50\xa1\x40\x98\xbd\x82\x40\x3e\x82\xda\x0a\x44\x40\x8b\x16\xe3"
    b"\x00\x53\x42\x00\x2e\x43\x40\x9a\x02\x52\x42\x00\x2e\x41\x40\x15\x54\x4a\x0e"
    b"\x3a\x2f\x3a\x82\x00\x30\x41\x40\x21\x2e\x85\x0f\x40\xb2\x0a\x2f\x98\x2e\xb1"
    b"\x0c\x98\x2e\x45\x0e\x98\x2e\x5b\x0e\xfb\x6f\xf0\x5f\x00\x30\x80\x2e\xce\xb7"
    b"\xdd\x52\xd3\x54\x42\x42\x4f\x84\x73\x30\xdb\x52\x83\x42\x1b\x30\x6b\x42\x23"
    b"\x30\x27\x2e\xd7\x00\x37\x2e\xd4\x00\x21\x2e\xd6\x00\x7a\x84\x17\x2c\x42\x42"
    b"\x30\x30\x21\x2e\xd4\x00\x12\x2d\x21\x30\x00\x30\x23\x2e\xd4\x00\x21\x2e\x7b"
    b"\xf7\x0b\x2d\x17\x30\x98\x2e\x51\x0c\xd5\x50\x0c\x82\x72\x30\x2f\x2e\xd4\x00"
    b"\x25\x2e\x7b\xf7\x40\x42\x00\x2e\xfb\x6f\xf0\x5f\xb8\x2e\x70\x50\x0a\x25\x39"
    b"\x86\xfb\x7f\xe1\x32\x62\x30\x98\x2e\xc2\xc4\xb5\x56\xa5\x6f\xab\x08\x91\x6f"
    b"\x4b\x08\xdf\x56\xc4\x6f\x23\x09\x4d\xba\x93\xbc\x8c\x0b\xd1\x6f\x0b\x09\xcb"
    b"\x52\xe1\x5e\x56\x42\xaf\x09\x4d\xba\x23\xbd\x94\x0a\xe5\x6f\x68\xbb\xeb\x08"
    b"\xbd\xb9\x63\xbe\xfb\x6f\x52\x42\xe3\x0a\xc0\x2e\x43\x42\x90\x5f\xd1\x50\x03"
    b"\x2e\x25\xf3\x13\x40\x00\x40\x9b\xbc\x9b\xb4\x08\xbd\xb8\xb9\x98\xbc\xda\x0a"
    b"\x08\xb6\x89\x16\xc0\x2e\x19\x00\x62\x02\x10\x50\xfb\x7f\x98\x2e\x81\x0d\x01"
    b"\x2e\xd4\x00\x31\x30\x08\x04\xfb\x6f\x01\x30\xf0\x5f\x23\x2e\xd6\x00\x21\x2e"
    b"\xd7\x00\xb8\x2e\x01\x2e\xd7\x00\x03\x2e\xd6\x00\x48\x0e\x01\x2f\x80\x2e\x1f"
    b"\x0e\xb8\x2e\xe3\x50\x21\x34\x01\x42\x82\x30\xc1\x32\x25\x2e\x62\xf5\x01\x00"
    b"\x22\x30\x01\x40\x4a\x0a\x01\x42\xb8\x2e\xe3\x54\xf0\x3b\x83\x40\xd8\x08\xe5"
    b"\x52\x83\x42\x00\x30\x83\x30\x50\x42\xc4\x32\x27\x2e\x64\xf5\x94\x00\x50\x42"
    b"\x40\x42\xd3\x3f\x84\x40\x7d\x82\xe3\x08\x40\x42\x83\x42\xb8\x2e\xdd\x52\x00"
    b"\x30\x40\x42\x7c\x86\xb9\x52\x09\x2e\x70\x0f\xbf\x54\xc4\x42\xd3\x86\x54\x40"
    b"\x55\x40\x94\x42\x85\x42\x21\x2e\xd7\x00\x42\x40\x25\x2e\xfd\xf3\xc0\x42\x7e"
    b"\x82\x05\x2e\x7d\x00\x80\xb2\x14\x2f\x05\x2e\x89\x00\x27\xbd\x2f\xb9\x80\x90"
    b"\x02\x2f\x21\x2e\x6f\xf5\x0c\x2d\x07\x2e\x71\x0f\x14\x30\x1c\x09\x05\x2e\x77"
    b"\xf7\xbd\x56\x47\xbe\x93\x08\x94\x0a\x25\x2e\x77\xf7\xe7\x54\x50\x42\x4a\x0e"
    b"\xfc\x2f\xb8\x2e\x50\x50\x02\x30\x43\x86\xe5\x50\xfb\x7f\xe3\x7f\xd2\x7f\xc0"
    b"\x7f\xb1\x7f\x00\x2e\x41\x40\x00\x40\x48\x04\x98\x2e\x74\xc0\x1e\xaa\xd3\x6f"
    b"\x14\x30\xb1\x6f\xe3\x22\xc0\x6f\x52\x40\xe4\x6f\x4c\x0e\x12\x42\xd3\x7f\xeb"
    b"\x2f\x03\x2e\x86\x0f\x40\x90\x11\x30\x03\x2f\x23\x2e\x86\x0f\x02\x2c\x00\x30"
    b"\xd0\x6f\xfb\x6f\xb0\x5f\xb8\x2e\x40\x50\xf1\x7f\x0a\x25\x3c\x86\xeb\x7f\x41"
    b"\x33\x22\x30\x98\x2e\xc2\xc4\xd3\x6f\xf4\x30\xdc\x09\x47\x58\xc2\x6f\x94\x09"
    b"\xeb\x58\x6a\xbb\xdc\x08\xb4\xb9\xb1\xbd\xe9\x5a\x95\x08\x21\xbd\xf6\xbf\x77"
    b"\x0b\x51\xbe\xf1\x6f\xeb\x6f\x52\x42\x54\x42\xc0\x2e\x43\x42\xc0\x5f\x50\x50"
    b"\xf5\x50\x31\x30\x11\x42\xfb\x7f\x7b\x30\x0b\x42\x11\x30\x02\x80\x23\x33\x01"
    b"\x42\x03\x00\x07\x2e\x80\x03\x05\x2e\xd3\x00\x23\x52\xe2\x7f\xd3\x7f\xc0\x7f"
    b"\x98\x2e\xb6\x0e\xd1\x6f\x08\x0a\x1a\x25\x7b\x86\xd0\x7f\x01\x33\x12\x30\x98"
    b"\x2e\xc2\xc4\xd1\x6f\x08\x0a\x00\xb2\x0d\x2f\xe3\x6f\x01\x2e\x80\x03\x51\x30"
    b"\xc7\x86\x23\x2e\x21\xf2\x08\xbc\xc0\x42\x98\x2e\xa5\xb7\x00\x2e\x00\x2e\xd0"
    b"\x2e\xb0\x6f\x0b\xb8\x03\x2e\x1b\x00\x08\x1a\xb0\x7f\x70\x30\x04\x2f\x21\x2e"
    b"\x21\xf2\x00\x2e\x00\x2e\xd0\x2e\x98\x2e\x6d\xc0\x98\x2e\x5d\xc0\xed\x50\x98"
    b"\x2e\x44\xcb\xef\x50\x98\x2e\x46\xc3\xf1\x50\x98\x2e\x53\xc7\x35\x50\x98\x2e"
    b"\x64\xcf\x10\x30\x98\x2e\xdc\x03\x20\x26\xc0\x6f\x02\x31\x12\x42\xab\x33\x0b"
    b"\x42\x37\x80\x01\x30\x01\x42\xf3\x37\xf7\x52\xfb\x50\x44\x40\xa2\x0a\x42\x42"
    b"\x8b\x31\x09\x2e\x5e\xf7\xf9\x54\xe3\x08\x83\x42\x1b\x42\x23\x33\x4b\x00\xbc"
    b"\x84\x0b\x40\x33\x30\x83\x42\x0b\x42\xe0\x7f\xd1\x7f\x98\x2e\x58\xb7\xd1\x6f"
    b"\x80\x30\x40\x42\x03\x30\xe0\x6f\xf3\x54\x04\x30\x00\x2e\x00\x2e\x01\x89\x62"
    b"\x0e\xfa\x2f\x43\x42\x11\x30\xfb\x6f\xc0\x2e\x01\x42\xb0\x5f\xc1\x4a\x00\x00"
    b"\x6d\x57\x00\x00\x77\x8e\x00\x00\xe0\xff\xff\xff\xd3\xff\xff\xff\xe5\xff\xff"
    b"\xff\xee\xe1\xff\xff\x7c\x13\x00\x00\x46\xe6\xff\xff\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x2e\x00\xc1\x80"
    b"\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1"
    b"\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00"
    b"\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e"
    b"\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80"
    b"\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1"
    b"\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00"
    b"\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e"
    b"\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80"
    b"\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1"
    b"\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00"
    b"\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e"
    b"\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80\x2e\x00\xc1\x80"
    b"\x2e\x00\xc1"
)


class BMI270:
    def __init__(
        self,
        bus,
        cs=None,
        address=_DEFAULT_ADDR,
        gyro_odr=100,
        gyro_scale=2000,
        accel_odr=100,
        accel_scale=4,
        bmm_magnet=None,
    ):
        """Initalizes Gyro and Accelerometer.
        bus: IMU bus
        address: I2C address (in I2C mode).
        cs: SPI CS pin (in SPI mode).
        gyro_odr:  (0.78, 1.5Hz, 3.1Hz, 6.25Hz, 12.5Hz, 25Hz, 50Hz, 100Hz, 200Hz, 400Hz, 800Hz, 1600Hz)
        gyro_scale:  (125dps, 250dps, 500dps, 1000dps, 2000dps)
        accel_odr: (0.78, 1.5Hz, 3.1Hz, 6.25Hz, 12.5Hz, 25Hz, 50Hz, 100Hz, 200Hz, 400Hz, 800Hz, 1600Hz)
        accel_scale: (+/-2g, +/-4g, +/-8g, +-16g)
        """
        self.bus = bus
        self.bmm_magnet = bmm_magnet
        self.cs = cs
        self.address = address
        self._use_i2c = hasattr(self.bus, "readfrom_mem")

        ACCEL_SCALE = (2, 4, 8, 16)
        GYRO_SCALE = (2000, 1000, 500, 250, 125)
        ODR = (0.78, 1.5, 3.1, 6.25, 12.5, 25, 50, 100, 200, 400, 800, 1200)

        # Sanity checks
        if not self._use_i2c:
            raise ValueError("SPI mode is not supported")
        if not gyro_odr in ODR:
            raise ValueError("Invalid gyro sampling rate: %d" % gyro_odr)
        if not gyro_scale in GYRO_SCALE:
            raise ValueError("Invalid gyro scaling: %d" % gyro_scale)
        if not accel_odr in ODR:
            raise ValueError("Invalid accelerometer sampling rate: %d" % accel_odr)
        if not accel_scale in ACCEL_SCALE:
            raise ValueError("Invalid accelerometer scaling: %d" % accel_scale)
        if self._read_reg(_CHIP_ID) != 0x24:
            raise OSError("No BMI270 device was found at address 0x%x" % (self.address))

        # Perform initialization sequence.
        # 0. Soft-reset
        self._write_reg(_CMD, 0xB6)
        time.sleep_ms(250)

        # 1. Disable power save mode.
        self._write_reg(0x7C, 0x00)
        time.sleep_ms(10)

        # 2. Prepare config load.
        self._write_reg(0x59, 0x00)

        # 3. Load config data.
        self._write_burst(0x5E, _CONFIG_DATA)

        # 4. Finish config load.
        self._write_reg(0x59, 0x01)

        # 5. Check correct initialization status.
        if not self._poll_reg(_STATUS, 0x01):
            raise OSError("Init sequence failed")

        # 6. Configure the device in normal power mode
        # FIFO Reset
        self._write_reg(_CMD, 0xB0)
        # Enable accel, gyro and temperature data.
        self._write_reg(0x7D, 0x0E)
        # acc_filter_perf | acc_bwp normal mode | ODR
        self._write_reg(0x40, 0xA | (ODR.index(accel_odr) + 1))
        # gyr_filter_perf | gyr_bwp normal mode | ODR
        self._write_reg(0x42, 0xA | (ODR.index(gyro_odr) + 1))
        # Disable adv_power_save | Enable fifo_self_wakeup.
        self._write_reg(0x7C, 0x02)

        # Set accelerometer scale and range.
        self.accel_scale = 32768 / accel_scale
        self._write_reg(0x41, ACCEL_SCALE.index(accel_scale))

        # Set gyroscope scale and range.
        self.gyro_scale = 32768 / gyro_scale
        self._write_reg(0x43, GYRO_SCALE.index(gyro_scale))

        # Allocate scratch buffer and set scale.
        self.scratch = memoryview(array.array("h", [0, 0, 0]))

    def _read_reg(self, reg, size=1):
        buf = self.bus.readfrom_mem(self.address, reg, size)
        if size == 1:
            return int(buf[0])
        return buf

    def _read_reg_into(self, reg, buf):
        self.bus.readfrom_mem_into(self.address, reg, buf)

    def _write_reg(self, reg, val):
        if isinstance(val, int):
            val = bytes([val])
        self.bus.writeto_mem(self.address, reg, val)
        time.sleep_ms(1)

    def _write_burst(self, reg, data, chunk=16):
        self._write_reg(_INIT_ADDR_0, 0)
        self._write_reg(_INIT_ADDR_1, 0)
        for i in range(0, len(data) // chunk):
            offs = i * chunk
            self._write_reg(reg, data[offs : offs + chunk])
            init_addr = ((i + 1) * chunk) // 2
            self._write_reg(_INIT_ADDR_0, (init_addr & 0x0F))
            self._write_reg(_INIT_ADDR_1, (init_addr >> 4) & 0xFF)

    def _poll_reg(self, reg, mask, retry=10, delay=100):
        for i in range(0, retry):
            if self._read_reg(reg) & mask:
                return True
            time.sleep_ms(delay)
        return False

    def reset(self):
        self._write_reg(_CMD, 0xB6)

    def gyro(self):
        """Returns gyroscope vector in degrees/sec."""
        f = self.gyro_scale
        self._read_reg_into(_DATA_14, self.scratch)
        return (self.scratch[0] / f, self.scratch[1] / f, self.scratch[2] / f)

    def accel(self):
        """Returns acceleration vector in gravity units (9.81m/s^2)."""
        f = self.accel_scale
        self._read_reg_into(_DATA_8, self.scratch)
        return (self.scratch[0] / f, self.scratch[1] / f, self.scratch[2] / f)

    def magnet(self):
        """Returns magnetometer vector."""
        if self.bmm_magnet is not None:
            return self.bmm_magnet.magnet()
        return (0.0, 0.0, 0.0)
