"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const HostType_1 = __importDefault(require("./types/HostType"));
const hap_nodejs_1 = require("hap-nodejs");
const logger_1 = require("@nrchkb/logger");
module.exports = (RED) => {
    const nrchkbConfigCompatibilityOverride = function () {
        const self = this;
        const log = logger_1.logger('NRCHKB', 'HAPServiceNode', self.config.name, self);
        if (self.config.isParent === undefined) {
            log.trace(`nrchkbConfigCompatibilityOverride => self.config.isParent=${self.config.isParent} value changed to true`);
            self.config.isParent = true;
        }
        if (self.config.hostType === undefined) {
            log.trace(`nrchkbConfigCompatibilityOverride => self.config.hostType=${self.config.hostType} value changed to HostType.BRIDGE`);
            self.config.hostType = HostType_1.default.BRIDGE;
        }
    };
    const preInit = function (config) {
        const self = this;
        self.config = config;
        self.name = self.config.name;
        const log = logger_1.logger('NRCHKB', 'HAPServiceNode', self.config.name, self);
        self.RED = RED;
        self.publishTimers = {};
        nrchkbConfigCompatibilityOverride.call(self);
        RED.nodes.createNode(self, self.config);
        const ServiceUtils = require('./utils/ServiceUtils')(self);
        new Promise((resolve) => {
            if (self.config.waitForSetupMsg) {
                log.debug('Waiting for Setup message. It should be of format {"payload":{"nrchkb":{"setup":{}}}}');
                self.setupDone = false;
                self.status({
                    fill: 'blue',
                    shape: 'dot',
                    text: 'Waiting for Setup',
                });
                self.handleWaitForSetup = (msg) => ServiceUtils.handleWaitForSetup(self.config, msg, resolve);
                self.on('input', self.handleWaitForSetup);
            }
            else {
                resolve(self.config);
            }
        }).then((newConfig) => {
            init.call(self, newConfig);
        });
    };
    const init = function (config) {
        const self = this;
        self.config = config;
        const log = logger_1.logger('NRCHKB', 'HAPServiceNode', self.config.name, self);
        const ServiceUtils = require('./utils/ServiceUtils')(self);
        if (self.config.isParent) {
            log.debug('Starting Parent Service');
            configure.call(self);
            self.configured = true;
        }
        else {
            const serviceType = config.serviceName === 'CameraControl' ? 'Camera' : 'Linked';
            ServiceUtils.waitForParent()
                .then(() => {
                log.debug(`Starting  ${serviceType} Service`);
                configure.call(self);
                self.configured = true;
            })
                .catch((error) => {
                log.error(`Error while starting ${serviceType} Service due to ${error}`);
            });
        }
    };
    const configure = function () {
        const self = this;
        const log = logger_1.logger('NRCHKB', 'HAPServiceNode', self.config.name, self);
        const Utils = require('./utils')(self);
        const AccessoryUtils = Utils.AccessoryUtils;
        const BridgeUtils = Utils.BridgeUtils;
        const CharacteristicUtils = Utils.CharacteristicUtils;
        const ServiceUtils = Utils.ServiceUtils;
        let parentNode;
        if (self.config.isParent) {
            const hostId = self.config.hostType == HostType_1.default.BRIDGE
                ? self.config.bridge
                : self.config.accessoryId;
            self.hostNode = RED.nodes.getNode(hostId);
            if (!self.hostNode) {
                log.error('Host Node not found', false);
                throw Error('Host Node not found');
            }
            self.childNodes = [];
            self.childNodes.push(self);
        }
        else {
            parentNode = RED.nodes.getNode(self.config.parentService);
            if (!parentNode) {
                log.error('Parent Node not assigned', false);
                throw Error('Parent Node not assigned');
            }
            self.parentService = parentNode.service;
            if (!self.parentService) {
                log.error('Parent Service not assigned', false);
                throw Error('Parent Service not assigned');
            }
            self.hostNode = parentNode.hostNode;
            parentNode.childNodes.push(self);
            self.accessory = parentNode.accessory;
        }
        self.name = self.config.name;
        const subtypeUUID = hap_nodejs_1.uuid.generate(self.id);
        if (self.config.hostType == HostType_1.default.BRIDGE) {
            if (self.config.isParent) {
                const accessoryUUID = hap_nodejs_1.uuid.generate('A' +
                    self.id +
                    self.name +
                    self.config.manufacturer +
                    self.config.serialNo +
                    self.config.model);
                self.accessory = AccessoryUtils.getOrCreate(self.hostNode.host, {
                    name: self.name,
                    UUID: accessoryUUID,
                    manufacturer: self.config.manufacturer,
                    serialNo: self.config.serialNo,
                    model: self.config.model,
                    firmwareRev: self.config.firmwareRev,
                    hardwareRev: self.config.hardwareRev,
                    softwareRev: self.config.softwareRev,
                }, subtypeUUID);
                self.onIdentify = AccessoryUtils.onIdentify;
                self.accessory.on('identify', self.onIdentify);
            }
        }
        else {
            log.debug('Binding Service accessory as Standalone Accessory');
            self.accessory = self.hostNode.host;
        }
        self.service = ServiceUtils.getOrCreate(self.accessory, {
            name: self.name,
            UUID: subtypeUUID,
            serviceName: self.config.serviceName,
            config: self.config,
        }, self.parentService);
        self.characteristicProperties = CharacteristicUtils.load(self.service, self.config);
        if (self.config.isParent) {
            self.publishTimers = BridgeUtils.delayedPublish(self);
        }
        self.status({
            fill: 'yellow',
            shape: 'ring',
            text: self.hostNode.config.pinCode,
        });
        self.supported = CharacteristicUtils.subscribeAndGetSupported(self.service);
        self.on('input', ServiceUtils.onInput);
        self.on('close', ServiceUtils.onClose);
    };
    return {
        preInit,
        init,
    };
};
