"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const HostType_1 = __importDefault(require("../types/HostType"));
const logger_1 = require("@nrchkb/logger");
const NRCHKBError_1 = __importDefault(require("../NRCHKBError"));
module.exports = function (node) {
    const log = (0, logger_1.logger)('NRCHKB', 'ServiceUtils', node.config.name, node);
    const HapNodeJS = require('hap-nodejs');
    const Service = HapNodeJS.Service;
    const Characteristic = HapNodeJS.Characteristic;
    const CameraSource = require('../cameraSource').Camera;
    const NO_RESPONSE_MSG = 'NO_RESPONSE';
    const prepareHapData = (context, connection) => {
        const hap = {};
        if (connection) {
            hap.session = {
                sessionID: connection.sessionID,
                username: connection.username,
                remoteAddress: connection.remoteAddress,
                localAddress: connection.localAddress,
                httpPort: connection.remotePort,
            };
            hap.context = {};
        }
        if (context) {
            hap.context = context;
        }
        return hap;
    };
    const onCharacteristicGet = function (callback, context, connection) {
        log.debug(`onCharacteristicGet with status: ${this.statusCode}, value: ${this.value}, reachability is ${node.accessory.reachable} with context ${JSON.stringify(context)} on connection ${connection === null || connection === void 0 ? void 0 : connection.sessionID}`);
        if (callback) {
            try {
                callback(node.accessory.reachable
                    ? this.statusCode
                    : new Error(NO_RESPONSE_MSG), this.value);
            }
            catch (_) { }
        }
    };
    const onValueChange = function (allCharacteristics, outputNumber, { oldValue, newValue, context }, connection) {
        const topic = node.config.topic ? node.config.topic : node.topic_in;
        const msg = { payload: {}, hap: {}, name: node.name, topic: topic };
        const key = this.constructor.name;
        msg.payload[key] = newValue;
        msg.hap = prepareHapData(context, connection);
        msg.hap.allChars = allCharacteristics.reduce((allChars, singleChar) => {
            allChars[singleChar.displayName] = singleChar.value;
            return allChars;
        }, {});
        if (oldValue !== undefined) {
            msg.hap.oldValue = oldValue;
        }
        msg.hap.newValue = newValue;
        const statusId = node.setStatus({
            fill: 'yellow',
            shape: 'dot',
            text: key + ': ' + newValue,
        });
        setTimeout(function () {
            node.clearStatus(statusId);
        }, 3000);
        log.debug(`${node.name} received ${key} : ${newValue}`);
        if (connection ||
            context ||
            node.hostNode.config.allowMessagePassthrough) {
            if (outputNumber === 0) {
                node.send(msg);
            }
            else if (outputNumber === 1) {
                node.send([null, msg]);
            }
        }
    };
    const onCharacteristicSet = (allCharacteristics) => function (newValue, callback, context, connection) {
        log.debug(`onCharacteristicSet with status: ${this.statusCode}, value: ${this.value}, reachability is ${node.accessory.reachable} 
            with context ${JSON.stringify(context)} on connection ${connection === null || connection === void 0 ? void 0 : connection.sessionID}`);
        try {
            if (callback) {
                callback(node.accessory.reachable
                    ? null
                    : new Error(NO_RESPONSE_MSG));
            }
        }
        catch (_) { }
        onValueChange.call(this, allCharacteristics, 1, {
            newValue,
            context,
        }, connection);
    };
    const onCharacteristicChange = (allCharacteristics) => function (change) {
        const { oldValue, newValue, context, originator, reason } = change;
        log.debug(`onCharacteristicChange with reason: ${reason}, oldValue: ${oldValue}, newValue: ${newValue}, reachability is ${node.accessory.reachable} 
            with context ${JSON.stringify(context)} on connection ${originator === null || originator === void 0 ? void 0 : originator.sessionID}`);
        if (oldValue != newValue) {
            onValueChange.call(this, allCharacteristics, 0, {
                oldValue,
                newValue,
                context,
            }, originator);
        }
    };
    const onInput = function (msg) {
        if (msg.hasOwnProperty('payload')) {
            const type = typeof msg.payload;
            if (type !== 'object') {
                log.error(`Invalid payload type: ${type}`);
                return;
            }
        }
        else {
            log.error('Invalid message (payload missing)');
            return;
        }
        const topic = node.config.topic ? node.config.topic : node.name;
        if (node.config.filter && msg.topic !== topic) {
            log.debug("msg.topic doesn't match configured value and filter is enabled. Dropping message.");
            return;
        }
        let context = null;
        if (msg.payload.hasOwnProperty('Context')) {
            context = msg.payload.Context;
            delete msg.payload.Context;
        }
        node.topic_in = msg.topic ? msg.topic : '';
        Object.keys(msg.payload).map((key) => {
            if (node.supported.indexOf(key) < 0) {
                log.error(`Instead of ${key} try one of these characteristics: ${node.supported.join(', ')}`);
            }
            else {
                if ((node.config.isParent &&
                    node.config.hostType == HostType_1.default.BRIDGE) ||
                    (!node.config.isParent &&
                        node.hostNode.hostType == HostType_1.default.BRIDGE)) {
                    node.accessory.updateReachability(msg.payload[key] !== NO_RESPONSE_MSG);
                }
                const characteristic = node.service.getCharacteristic(Characteristic[key]);
                if (context !== null) {
                    characteristic.setValue(msg.payload[key], undefined, context);
                }
                else {
                    characteristic.setValue(msg.payload[key]);
                }
            }
        });
    };
    const onClose = function (removed, done) {
        const characteristics = node.service.characteristics.concat(node.service.optionalCharacteristics);
        characteristics.forEach(function (characteristic) {
            characteristic.removeListener('get', node.onCharacteristicGet);
            characteristic.removeListener('set', node.onCharacteristicSet);
            characteristic.removeListener('change', node.onCharacteristicChange);
        });
        if (node.config.isParent) {
            node.accessory.removeListener('identify', node.onIdentify);
        }
        if (removed) {
            if (node.config.isParent) {
                node.hostNode.host.removeBridgedAccessories([node.accessory]);
                node.accessory.destroy();
            }
            else {
                node.accessory.removeService(node.service);
                node.parentService.removeLinkedService(node.service);
            }
        }
        done();
    };
    const getOrCreate = function (accessory, serviceInformation, parentService) {
        const newService = new Service[serviceInformation.serviceName](serviceInformation.name, serviceInformation.UUID);
        log.debug(`Looking for service with UUID ${serviceInformation.UUID} ...`);
        let service = accessory.services.find((service) => {
            return newService.subtype === service.subtype;
        });
        if (service && newService.UUID !== service.UUID) {
            log.debug('... service type changed! Removing the old service.');
            accessory.removeService(service);
            service = undefined;
        }
        if (!service) {
            log.debug(`... didn't find it. Adding new ${serviceInformation.serviceName} service.`);
            if (serviceInformation.serviceName === 'CameraControl') {
                configureCameraSource(accessory, newService, serviceInformation.config);
                service = newService;
            }
            else {
                service = accessory.addService(newService);
            }
        }
        else {
            log.debug('... found it! Updating it.');
            service
                .getCharacteristic(Characteristic.Name)
                .setValue(serviceInformation.name);
        }
        if (parentService) {
            if (serviceInformation.serviceName === 'CameraControl') {
                log.debug('... and adding service to accessory.');
            }
            else if (service) {
                log.debug('... and linking service to parent.');
                parentService.addLinkedService(service);
            }
        }
        return service;
    };
    const configureCameraSource = function (accessory, service, config) {
        if (config.cameraConfigSource) {
            log.debug('Configuring Camera Source');
            if (!config.cameraConfigVideoProcessor) {
                log.error('Missing configuration for CameraControl: videoProcessor cannot be empty!');
            }
            else {
                accessory.configureCameraSource(new CameraSource(service, config, node));
            }
        }
        else {
            log.error('Missing configuration for CameraControl.');
        }
    };
    const waitForParent = () => {
        log.debug('Waiting for Parent Service');
        return new Promise((resolve) => {
            node.setStatus({
                fill: 'blue',
                shape: 'dot',
                text: 'Waiting for Parent Service',
            });
            const checkAndWait = () => {
                const parentNode = node.RED.nodes.getNode(node.config.parentService);
                if (parentNode && parentNode.configured) {
                    resolve(parentNode);
                }
                else {
                    setTimeout(checkAndWait, 1000);
                }
            };
            checkAndWait();
        }).catch((error) => {
            log.error(`Waiting for Parent Service failed due to: ${error}`);
            throw new NRCHKBError_1.default(error);
        });
    };
    const handleWaitForSetup = (config, msg, resolve) => {
        if (node.setupDone) {
            return;
        }
        if (msg.hasOwnProperty('payload') &&
            msg.payload.hasOwnProperty('nrchkb') &&
            msg.payload.nrchkb.hasOwnProperty('setup')) {
            node.setupDone = true;
            const newConfig = Object.assign(Object.assign({}, config), msg.payload.nrchkb.setup);
            node.removeListener('input', node.handleWaitForSetup);
            resolve(newConfig);
        }
        else {
            log.error('Invalid message (required {"payload":{"nrchkb":{"setup":{}}}})');
        }
    };
    return {
        getOrCreate,
        onCharacteristicGet,
        onCharacteristicSet,
        onCharacteristicChange,
        onInput,
        onClose,
        waitForParent,
        handleWaitForSetup,
    };
};
