// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.
//
// Forked from github.com/certifi/gocertifi <https://git.io/JJjmG>.
//
// This script should not be invoked directly, rather it should be
// executed by running go generate ./... from toplevel dir.

package main

import (
	"context"
	"crypto/x509"
	"log"
	"net/http"
	"os"
	"strings"
	"text/template"
	"time"

	"github.com/ooni/probe-cli/v3/internal/netxlite"
)

var tmpl = template.Must(template.New("").Parse(`// Code generated by go generate; DO NOT EDIT.
// {{ .Timestamp }}
// {{ .URL }}

package netxlite

//{{ .GoGenerate }} go run ./internal/gencertifi/ "{{ .URL }}"

const pemcerts string = ` + "`" + `
{{ .Bundle }}
` + "`" + `
`))

func main() {
	if len(os.Args) != 2 || !strings.HasPrefix(os.Args[1], "https://") {
		log.Fatal("usage: go run ./generator/ <url>")
	}
	url := os.Args[1]

	resp, err := http.Get(url)
	if err != nil {
		log.Fatal(err)
	}
	if resp.StatusCode != 200 {
		log.Fatal("expected 200, got", resp.StatusCode)
	}
	defer resp.Body.Close()

	bundle, err := netxlite.ReadAllContext(context.Background(), resp.Body)
	if err != nil {
		log.Fatal(err)
	}

	pool := x509.NewCertPool()
	if !pool.AppendCertsFromPEM(bundle) {
		log.Fatalf("can't parse certificates from %s", url)
	}

	fp, err := os.Create("certifi.go")
	if err != nil {
		log.Fatal(err)
	}

	err = tmpl.Execute(fp, struct {
		Bundle     string
		GoGenerate string
		Timestamp  time.Time
		URL        string
	}{
		Bundle:     string(bundle),
		GoGenerate: "go:generate",
		Timestamp:  time.Now(),
		URL:        url,
	})
	if err != nil {
		log.Fatal(err)
	}

	if err := fp.Close(); err != nil {
		log.Fatal(err)
	}
}
