#!/bin/sh
#
# Array manipulation in scripts for shells that do not support arrays (ash)
# Copyright (c) 2022 RIPE NCC, portions Michel Stam (c) 2013
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
# 
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
# 
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
# 

# Global variables
_array_instance_num=0
_array_max_instances=4096

#
# Internal function to guess the delimiter to use on array elements
# Usage: _array_guess_delim <variable1>..<variableN>
# any variables passed are scanned for special characters. The
# delimiter found will be written to stdout, or ''
#
_array_guess_delim( )
{
	local content="${*}"
	local delim='~ # $ % ^ \* + - ?'
	local oldifs="${IFS}"

	IFS=' '
	set -- ${delim}

	while [ "${content//${1}/}" != "${content}" ]; do
		shift
	done

	IFS="${oldifs}"
	echo ${1}
}

#
# Method to retrieve data from the array
# Usage: array_get <handle> <index>
# <handle> generated by array_constructor()
# <index> member of array to retrieve, or 0
#  to get the size of the array
#
array_get( )
{
	local oldifs="${IFS}"
	local instance="${1}"
	local index="${2}"
	local content

	eval "content=\${_array_data_${instance}}"

	if [ -z "${content}" ]; then
		content='~0'
	fi

	IFS="${content:0:1}"
	set -- ${content}
	eval echo \${$(( index + 2 ))}
	IFS="${oldifs}"
}

#
# Method to set the contents of the array
# Usage: array_set <handle> <variable1>..<variableN>
# <handle> generated by array_constructor()
# <variable*> values to add into array
#
array_set( )
{
	local instance="${1}"
	local delim
	local result

	shift
	delim=$(_array_guess_delim "${*}")
	result="${delim}${#}"

	while [ -n "${1}" ]; do
		result="${result}${delim}${1}"
		shift
	done

        eval "export _array_data_${instance}='${result}'"
}

#
# Method to append a value to the array
# Usage: array_append <handle> <variable>
# <handle> generated by array_constructor()
# <variable> value to append to end of array
#
array_append( )
{
	local oldifs="${IFS}"
	local instance="${1}"
	local value="${2}"
	local content

	shift
	eval "content=\${_array_data_${instance}}"

	if [ -z "${content}" ]; then
		content='~0'
	fi

	IFS="${content:0:1}"
	set -- ${content}
	shift
	shift
	IFS="${oldifs}"
	array_set "${instance}" "${@}" ${value}
}

#
# Method to locate the offset of a value in the array
# Usage: array_find <handle> <value>
# <handle> generated by array_constructor()
# <value> value to locate in array
# Only the first value is located, and its position
#  in the array (1-based) is written to stdout. -1
#  is returned if not found
#
array_find( )
{
	local oldifs="${IFS}"
	local instance="${1}"
	local token="${2}"
	local index=1
	local content

	eval "content=\${_array_data_${instance}}"

	if [ -z "${content}" ]; then
		content='~0'
	fi

	IFS="${content:0:1}"
	set -- ${content}
	shift
	shift
	while [ ${#} -gt 0 ]; do
		if [ "${1}" = "${token}" ]; then
			break
		fi
		index=$(( index + 1))
		shift
	done
	if [ ${#} -eq 0 ]; then
		index=-1
	fi
	IFS="${oldifs}"

	echo ${index}
}

#
# Method to expand the array contents into a space
# separated list, with shell quoting
# Usage: array_expand <handle>
# <handle> generated by array_constructor()
#
array_expand( )
{
	local oldifs="${IFS}"
	local instance="${1}"
	local content

	eval "content=\${_array_data_${instance}}"

	if [ -z "${content}" ]; then
		content='~0'
	fi

	IFS="${content:0:1}"
	set -- ${content}
	shift
	shift
	while [ ${#} -gt 0 ]; do
		echo -n "'${1}' "
		shift
	done
	echo
	IFS="${oldifs}"
}

#
# Internal function to invoke class methods
# Usage: _array_call <handle> <method> <method args>
# NOTE: Only called from the function entry point,
#  not intended to be called directly.
# <handle> generated by array_constructor()
# <method> name of the class method to invoke
# <method args> any arguments to be passed to class method
#
_array_call( )
{
	local instance="${1}"
	shift
	local func="${1}"
	shift

	eval "array_${func}" "${instance}" \"\${@}\"
}

#
# Method to create an array object
# Usage: array_constructor <variable>
# Newly generated entry point to class will be exported to <variable>
# in the global environment.
#
array_constructor( )
{
        local _entry="_array_call_${_array_instance_num}"
        local _data="_array_data_${_array_instance_num}"
	local _variable="${1}"

        eval "${_entry}() { _array_call \"${_array_instance_num}\" \"\${@}\"; }"
        export _array_instance_num=$(( ( _array_instance_num + 1 ) % _array_max_instances ))
        eval "export ${_variable}=${_entry}"
        eval "export ${_data}=''"
}

#
# Method to clean up data of an array object
# Usage: array_destructor <instance number>
# <instance number> generated by array_constructor()
#
array_destructor( )
{
	local instance="${1}"
        local entry="_array_call_${instance}"
        local data="_array_data_${instance}"

	unset "${entry}"
	unset "${data}"
}
