package outbound

import (
	"context"
	"io"

	core "github.com/v2fly/v2ray-core/v5"
	"github.com/v2fly/v2ray-core/v5/app/proxyman"
	"github.com/v2fly/v2ray-core/v5/common"
	"github.com/v2fly/v2ray-core/v5/common/buf"
	"github.com/v2fly/v2ray-core/v5/common/errors"
	"github.com/v2fly/v2ray-core/v5/common/mux"
	"github.com/v2fly/v2ray-core/v5/common/net"
	"github.com/v2fly/v2ray-core/v5/common/net/packetaddr"
	"github.com/v2fly/v2ray-core/v5/common/net/pingproto"
	"github.com/v2fly/v2ray-core/v5/common/serial"
	"github.com/v2fly/v2ray-core/v5/common/session"
	"github.com/v2fly/v2ray-core/v5/features/dns"
	"github.com/v2fly/v2ray-core/v5/features/outbound"
	"github.com/v2fly/v2ray-core/v5/features/ping"
	"github.com/v2fly/v2ray-core/v5/features/policy"
	"github.com/v2fly/v2ray-core/v5/features/stats"
	"github.com/v2fly/v2ray-core/v5/proxy"
	"github.com/v2fly/v2ray-core/v5/transport"
	"github.com/v2fly/v2ray-core/v5/transport/internet"
	"github.com/v2fly/v2ray-core/v5/transport/internet/tls"
	"github.com/v2fly/v2ray-core/v5/transport/internet/xtls"
	"github.com/v2fly/v2ray-core/v5/transport/pipe"
)

func getStatCounter(v *core.Instance, tag string) (stats.Counter, stats.Counter) {
	var uplinkCounter stats.Counter
	var downlinkCounter stats.Counter

	policy := v.GetFeature(policy.ManagerType()).(policy.Manager)
	if len(tag) > 0 && policy.ForSystem().Stats.OutboundUplink {
		statsManager := v.GetFeature(stats.ManagerType()).(stats.Manager)
		name := "outbound>>>" + tag + ">>>traffic>>>uplink"
		c, _ := stats.GetOrRegisterCounter(statsManager, name)
		if c != nil {
			uplinkCounter = c
		}
	}
	if len(tag) > 0 && policy.ForSystem().Stats.OutboundDownlink {
		statsManager := v.GetFeature(stats.ManagerType()).(stats.Manager)
		name := "outbound>>>" + tag + ">>>traffic>>>downlink"
		c, _ := stats.GetOrRegisterCounter(statsManager, name)
		if c != nil {
			downlinkCounter = c
		}
	}

	return uplinkCounter, downlinkCounter
}

// Handler is an implements of outbound.Handler.
type Handler struct {
	tag               string
	senderSettings    *proxyman.SenderConfig
	streamSettings    *internet.MemoryStreamConfig
	proxy             proxy.Outbound
	outboundManager   outbound.Manager
	dnsClient         dns.NewClient
	mux               *mux.ClientManager
	uplinkCounter     stats.Counter
	downlinkCounter   stats.Counter
	muxPacketEncoding packetaddr.PacketAddrType
	pingManager       ping.Manager
}

// NewHandler create a new Handler based on the given configuration.
func NewHandler(ctx context.Context, config *core.OutboundHandlerConfig) (outbound.Handler, error) {
	v := core.MustFromContext(ctx)
	uplinkCounter, downlinkCounter := getStatCounter(v, config.Tag)
	h := &Handler{
		tag:             config.Tag,
		outboundManager: v.GetFeature(outbound.ManagerType()).(outbound.Manager),
		dnsClient:       v.GetFeature(dns.ClientType()).(dns.NewClient),
		uplinkCounter:   uplinkCounter,
		downlinkCounter: downlinkCounter,
	}
	if pingManager := v.GetFeature(ping.ManagerType()); pingManager != nil {
		h.pingManager = pingManager.(ping.Manager)
	}

	if config.SenderSettings != nil {
		senderSettings, err := serial.GetInstanceOf(config.SenderSettings)
		if err != nil {
			return nil, err
		}
		switch s := senderSettings.(type) {
		case *proxyman.SenderConfig:
			h.senderSettings = s
			mss, err := internet.ToMemoryStreamConfig(s.StreamSettings)
			if err != nil {
				return nil, newError("failed to parse stream settings").Base(err).AtWarning()
			}
			h.streamSettings = mss
		default:
			return nil, newError("settings is not SenderConfig")
		}
	}

	proxyConfig, err := serial.GetInstanceOf(config.ProxySettings)
	if err != nil {
		return nil, err
	}

	rawProxyHandler, err := common.CreateObject(ctx, proxyConfig)
	if err != nil {
		return nil, err
	}

	proxyHandler, ok := rawProxyHandler.(proxy.Outbound)
	if !ok {
		return nil, newError("not an outbound handler")
	}

	if h.senderSettings != nil && h.senderSettings.MultiplexSettings != nil {
		config := h.senderSettings.MultiplexSettings
		if config.Concurrency < 1 || config.Concurrency > 1024 {
			return nil, newError("invalid mux concurrency: ", config.Concurrency).AtWarning()
		}
		h.muxPacketEncoding = h.senderSettings.MultiplexSettings.PacketEncoding
		h.mux = &mux.ClientManager{
			Enabled: h.senderSettings.MultiplexSettings.Enabled,
			Picker: &mux.IncrementalWorkerPicker{
				Factory: mux.NewDialingWorkerFactory(
					ctx,
					proxyHandler,
					h,
					mux.ClientStrategy{
						MaxConcurrency: config.Concurrency,
						MaxConnection:  128,
					},
				),
			},
		}
	}

	h.proxy = proxyHandler
	return h, nil
}

// Tag implements outbound.Handler.
func (h *Handler) Tag() string {
	return h.tag
}

// Dispatch implements proxy.Outbound.Dispatch.
func (h *Handler) Dispatch(ctx context.Context, link *transport.Link) {
	outbound := session.OutboundFromContext(ctx)
	destination := outbound.Target

	var domainString string
	switch {
	case destination.Address.Family().IsDomain():
		domainString = destination.Address.Domain()
	default:
		domainString = ""
	}

	var domainStrategy proxyman.DomainStrategy
	if h.senderSettings != nil {
		domainStrategy = h.senderSettings.DomainStrategy
	}

	if domainStrategy == proxyman.DomainStrategy_AS_IS && proxyman.PreferUseIPFromContext(ctx) {
		domainStrategy = proxyman.DomainStrategy_USE_IP
		ctx = proxyman.SetPreferUseIP(ctx, false)
	}

	if domainStrategy != proxyman.DomainStrategy_AS_IS && domainString != "" {
		ctx, cancel := context.WithTimeout(ctx, dns.DefaultTimeout)
		defer cancel()
		ips, _, err := h.dnsClient.Lookup(ctx, domainString, dns.QueryStrategy(domainStrategy-1))
		if err == nil {
			destination.Address = net.IPAddress(ips[0])
			outbound.Target = destination
		}
	}

	if h.mux != nil && h.mux.Enabled {
		if destination.Network == net.Network_UDP {
			switch h.muxPacketEncoding {
			case packetaddr.PacketAddrType_None:
				link.Reader = &buf.EndpointErasureReader{Reader: link.Reader}
				link.Writer = &buf.EndpointErasureWriter{Writer: link.Writer}
			case packetaddr.PacketAddrType_XUDP:
				break
			case packetaddr.PacketAddrType_Packet:
				link.Reader = packetaddr.NewReversePacketReader(link.Reader, destination)
				link.Writer = packetaddr.NewReversePacketWriter(link.Writer)
				outbound.Target = net.Destination{
					Network: net.Network_UDP,
					Address: net.DomainAddress(packetaddr.SeqPacketMagicAddress),
					Port:    0,
				}
			}
		}
		if err := h.mux.DispatchSync(ctx, link); err != nil {
			cause := errors.Cause(err)
			if !(cause == io.EOF || cause == context.Canceled || cause == io.ErrClosedPipe) {
				err := newError("failed to process mux outbound traffic").Base(err)
				err.WriteToLog(session.ExportIDToError(ctx))
			}
			session.SubmitOutboundErrorToOriginator(ctx, err)
			common.Interrupt(link.Writer)
		}
	} else {
		err := h.proxy.Process(ctx, link, h)
		if err != nil {
			cause := errors.Cause(err)
			if cause == context.Canceled || cause == io.ErrClosedPipe || cause == io.EOF {
				err = nil
			}
		}
		if err != nil {
			err := newError("failed to process outbound traffic").Base(err)
			err.WriteToLog(session.ExportIDToError(ctx))
			session.SubmitOutboundErrorToOriginator(ctx, err)
			common.Interrupt(link.Writer)
		} else {
			common.Close(link.Writer)
		}
		common.Interrupt(link.Reader)
	}
}

func (h *Handler) IsConnDispatcher() bool {
	_, ok := h.proxy.(proxy.RawOutbound)
	return ok
}

func (h *Handler) DispatchConn(ctx context.Context, conn net.Conn) {
	outbound := session.OutboundFromContext(ctx)
	destination := outbound.Target

	var domainString string
	switch {
	case destination.Address.Family().IsDomain():
		domainString = destination.Address.Domain()
	default:
		domainString = ""
	}

	var domainStrategy proxyman.DomainStrategy
	if h.senderSettings != nil {
		domainStrategy = h.senderSettings.DomainStrategy
	}

	if domainStrategy == proxyman.DomainStrategy_AS_IS && proxyman.PreferUseIPFromContext(ctx) {
		domainStrategy = proxyman.DomainStrategy_USE_IP
		ctx = proxyman.SetPreferUseIP(ctx, false)
	}

	if domainStrategy != proxyman.DomainStrategy_AS_IS && domainString != "" {
		ctx, cancel := context.WithTimeout(ctx, dns.DefaultTimeout)
		defer cancel()
		ips, _, err := h.dnsClient.Lookup(ctx, domainString, dns.QueryStrategy(domainStrategy-1))
		if err == nil {
			destination.Address = net.IPAddress(ips[0])
			outbound.Target = destination
		}
	}

	if h.mux != nil && h.mux.Enabled {
		link := &transport.Link{
			Reader: buf.NewReader(conn),
			Writer: buf.NewWriter(conn),
		}
		if destination.Network == net.Network_UDP {
			switch h.muxPacketEncoding {
			case packetaddr.PacketAddrType_None:
				link.Reader = &buf.EndpointErasureReader{Reader: link.Reader}
				link.Writer = &buf.EndpointErasureWriter{Writer: link.Writer}
			case packetaddr.PacketAddrType_XUDP:
				break
			case packetaddr.PacketAddrType_Packet:
				link.Reader = packetaddr.NewReversePacketReader(link.Reader, destination)
				link.Writer = packetaddr.NewReversePacketWriter(link.Writer)
				outbound.Target = net.Destination{
					Network: net.Network_UDP,
					Address: net.DomainAddress(packetaddr.SeqPacketMagicAddress),
					Port:    0,
				}
			}
		}
		if err := h.mux.DispatchSync(ctx, link); err != nil {
			cause := errors.Cause(err)
			if !(cause == io.EOF || cause == context.Canceled || cause == io.ErrClosedPipe) {
				err := newError("failed to process mux outbound traffic").Base(err)
				err.WriteToLog(session.ExportIDToError(ctx))
			}
			session.SubmitOutboundErrorToOriginator(ctx, err)
			common.Interrupt(link.Writer)
		}
		return
	}

	err := h.proxy.(proxy.RawOutbound).ProcessConn(ctx, conn, h)
	if err != nil {
		cause := errors.Cause(err)
		if !(cause == context.Canceled || cause == io.ErrClosedPipe || cause == io.EOF) {
			err := newError("failed to process outbound traffic").Base(err)
			err.WriteToLog(session.ExportIDToError(ctx))
		}
		session.SubmitOutboundErrorToOriginator(ctx, err)
	}
	common.Close(conn)
}

// Address implements internet.Dialer.
func (h *Handler) Address() net.Address {
	if h.senderSettings == nil || h.senderSettings.Via == nil {
		return nil
	}
	return h.senderSettings.Via.AsAddress()
}

// Dial implements internet.Dialer.
func (h *Handler) Dial(ctx context.Context, dest net.Destination) (internet.Connection, error) {
	if h.senderSettings != nil {
		if h.senderSettings.ProxySettings.HasTag() && !h.senderSettings.ProxySettings.TransportLayerProxy {
			ctx = proxyman.SetPreferUseIP(ctx, true)
			tag := h.senderSettings.ProxySettings.Tag
			handler := h.outboundManager.GetHandler(tag)
			if handler != nil {
				newError("proxying to ", tag, " for dest ", dest).AtDebug().WriteToLog(session.ExportIDToError(ctx))
				ctx = session.ContextWithOutbound(ctx, &session.Outbound{
					Target: dest,
				})

				opts := pipe.OptionsFromContext(ctx)
				uplinkReader, uplinkWriter := pipe.New(opts...)
				downlinkReader, downlinkWriter := pipe.New(opts...)

				go handler.Dispatch(ctx, &transport.Link{Reader: uplinkReader, Writer: downlinkWriter})
				conn := buf.NewConnection(buf.ConnectionInputMulti(uplinkWriter), buf.ConnectionOutputMulti(downlinkReader))

				if config := tls.ConfigFromStreamSettings(h.streamSettings); config != nil {
					tlsConfig := config.GetTLSConfig(tls.WithDestination(dest))
					conn = tls.Client(conn, tlsConfig)
				} else if config := xtls.ConfigFromStreamSettings(h.streamSettings); config != nil {
					return xtls.Client(conn, config.GetXTLSConfig(xtls.WithDestination(dest))), nil
				}

				return h.getStatCouterConnection(conn), nil
			}

			newError("failed to get outbound handler with tag: ", tag).AtWarning().WriteToLog(session.ExportIDToError(ctx))
		}

		if h.senderSettings.Via != nil {
			outbound := session.OutboundFromContext(ctx)
			if outbound == nil {
				outbound = new(session.Outbound)
				ctx = session.ContextWithOutbound(ctx, outbound)
			}
			outbound.Gateway = h.senderSettings.Via.AsAddress()
		}
	}
	enablePacketAddrCapture := true
	if h.senderSettings != nil && h.senderSettings.ProxySettings != nil && h.senderSettings.ProxySettings.HasTag() && h.senderSettings.ProxySettings.TransportLayerProxy {
		tag := h.senderSettings.ProxySettings.Tag
		newError("transport layer proxying to ", tag, " for dest ", dest).AtDebug().WriteToLog(session.ExportIDToError(ctx))
		ctx = session.SetTransportLayerProxyTagToContext(ctx, tag)
		ctx = proxyman.SetPreferUseIP(ctx, true)
		enablePacketAddrCapture = false
	}

	if isStream, err := packetaddr.GetDestinationSubsetOf(dest); err == nil && enablePacketAddrCapture {
		packetConn, err := internet.ListenSystemPacket(ctx, &net.UDPAddr{IP: net.AnyIP.IP(), Port: 0}, h.streamSettings.SocketSettings)
		if err != nil {
			return nil, newError("unable to listen socket").Base(err)
		}
		conn := packetaddr.ToPacketAddrConnWrapper(packetConn, isStream)
		return h.getStatCouterConnection(conn), nil
	}

	if pingproto.GetDestinationIsSubsetOf(dest) {
		if h.pingManager == nil {
			return nil, newError("ping support not enabled")
		}
		pingConn, err := h.pingManager.Dial(dest)
		if err != nil {
			return nil, newError("failed to listen icmp connection").Base(err)
		}
		return h.getStatCouterConnection(pingConn), nil
	}

	conn, err := internet.Dial(ctx, dest, h.streamSettings)
	return h.getStatCouterConnection(conn), err
}

func (h *Handler) getStatCouterConnection(conn internet.Connection) internet.Connection {
	if h.uplinkCounter != nil || h.downlinkCounter != nil {
		return &internet.StatCounterConn{
			Connection:   conn,
			ReadCounter:  h.downlinkCounter,
			WriteCounter: h.uplinkCounter,
		}
	}
	return conn
}

// GetOutbound implements proxy.GetOutbound.
func (h *Handler) GetOutbound() proxy.Outbound {
	return h.proxy
}

// Start implements common.Runnable.
func (h *Handler) Start() error {
	return nil
}

// Close implements common.Closable.
func (h *Handler) Close() error {
	common.Close(h.mux)
	return nil
}
