/* 
 * Copyright 2016-2020 The Regents of the University of California
 * All rights reserved.
 * 
 * This file is part of Spoofer.
 * 
 * Spoofer is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Spoofer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Spoofer.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef MANAGER_COMMON_BLOCKREADER_H
#define MANAGER_COMMON_BLOCKREADER_H

#include <QBuffer>
#include <QDataStream>
#include <QIODevice>

// BlockReader provides a stream interface to read data written by a
// BlockWriter, blocking if needed to wait for transmission delays.  All the
// data written by a single BlockWriter must be read by a single BlockReader.
class BlockReader
{
public:
    BlockReader(QIODevice *dev) : buffer(), stream()
    {
        buffer.open(QIODevice::ReadWrite);
        stream.setDevice(&buffer);

	// Read size.
	quint32 size;
	readn(dev, sizeof(size));
	buffer.seek(0);
	stream >> size;

	// Read data from dev into buffer.
	readn(dev, size);
	buffer.seek(sizeof(size)); // beginning of data
    }

    template <class T> BlockReader &operator>>(T &data)
	{ stream >> data; return *this; }

private:
    // Read from dev into buffer until buffer contains exactly n bytes.
    void readn(QIODevice *dev, quint32 n)
    {
        while (buffer.size() < n && dev->isOpen()) {
            if (!dev->bytesAvailable())
                dev->waitForReadyRead(1000);
            buffer.write(dev->read(n - buffer.size()));
        }
    }

    QBuffer buffer;
    QDataStream stream;
};

#endif // MANAGER_COMMON_BLOCKREADER_H
