/* 
 * Copyright 2015-2020 The Regents of the University of California
 * All rights reserved.
 * 
 * This file is part of Spoofer.
 * 
 * Spoofer is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Spoofer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Spoofer.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <time.h>
#include "spoof_qt.h"
#include <QCommandLineParser>
#include <QtGlobal>
#include <QDir>
#include <QProcess>
#include <QProcessEnvironment>
#ifdef VISIT_URL
 #ifdef Q_OS_WIN32
  #include <windows.h> // for shellapi.h
  #include <shellapi.h> // for ShellExecute()
 #endif
 #ifdef Q_OS_UNIX
  #include <sys/wait.h> // WIFEXITED(), WEXITSTATUS()
 #endif
#endif

#ifdef Q_OS_UNIX
 #include <unistd.h> // execv() in upgrade()
#endif

#include "../../config.h"
#include "SpooferUI.h"
#include "FileTailThread.h"
#include "BlockReader.h"
static const char cvsid[] ATR_USED = "$Id: SpooferUI.cpp,v 1.46 2020/07/16 19:23:40 kkeys Exp $";

#ifdef VISIT_URL
// Visit a URL using the default web browser.  (Unlike
// QDesktopServices::openUrl(), this does not require linking with QtGui.)
bool SpooferUI::visitURL(const char *url)
{
#if defined(Q_OS_WIN32)
    return (int)ShellExecuteA(nullptr, nullptr, url, nullptr, nullptr,
	SW_SHOWNORMAL) > 32;
#else
    char buf[1024];
#if defined(Q_OS_MACOS)
    snprintf(buf, sizeof(buf), "open '%s'", url);
#elif defined(Q_OS_UNIX)
    snprintf(buf, sizeof(buf), "xdg-open '%s'", url);
#endif
    int rc = system(buf);
    return rc != -1 && WIFEXITED(rc) && WEXITSTATUS(rc) == 0;
#endif
}

void SpooferUI::visitEmbeddedURLs(const QString *text)
{
    // If text contains a URL, open it in a web browser.
    static QString prefix("http://");
    int i = 0;
    while (i < text->size()) {
	if ((*text)[i].isSpace()) {
	    i++; // skip leading space on a line
	} else {
	    int n = text->indexOf("\n", i);
	    if (n < 0) n = text->size();
	    QStringRef line = text->midRef(i, n-i);
	    if (line.startsWith(prefix))
		visitURL(qPrintable(line.toString()));
	    i = n+1; // jump to next line
	}
    }
}
#endif

void SpooferUI::printNextProberStart()
{
    time_t when = nextProberStart.when;
    if (when)
	spout << "Next prober scheduled for " << qPrintable(ftime(QString(), &when)) << Qt_endl;
    else
	spout << "No prober scheduled." << Qt_endl;
}

void SpooferUI::readScheduler()
{
    qint32 type;
    sc_msg_text msg;
    QStringList keys;
    static bool rcvdHello = false;
    while (!scheduler->atEnd()) {
	BlockReader in(scheduler);
	in >> type;
	if (!rcvdHello && type != SC_HELLO) {
	    qCritical().noquote() << "UI version" << PACKAGE_VERSION <<
		"does not match Scheduler version (missing HELLO)";
	    scheduler->abort();
	}
	switch (type) {
	    case SC_DONE_CMD:
		doneCmd(0);
		break;
	    case SC_CONFIG_CHANGED:
		spout << "Settings changed." << Qt_endl;
		config->sync();
		configChanged();
		break;
	    case SC_ERROR:
		in >> msg;
		qCritical() << "Error:" << qPrintable(msg.text);
		doneCmd(1);
		break;
	    case SC_TEXT:
		in >> msg;
		spout << "Scheduler: " << msg.text << Qt_endl;
		doneCmd(0);
		break;
	    case SC_PROBER_STARTED:
		in >> msg;
		spout << "Prober started; log: " <<
		    QDir::toNativeSeparators(msg.text) << Qt_endl;
		nextProberStart.when = 0;
		proberExitCode = 0;
		proberExitStatus = QProcess::CrashExit;
		startFileTail(msg.text);
		break;
	    case SC_PROBER_FINISHED:
		// Don't print exit code/status, just store them, and ask
		// fileTail to stop.  When fileTail is done reading the log,
		// it will signal finished, and finishProber() will run.
		in >> proberExitCode >> proberExitStatus;
		if (fileTail && fileTail->isRunning())
		    fileTail->requestInterruption();
		else // fileTail failed to start or exited early
		    finishProber();
		break;
	    case SC_PROBER_ERROR:
		in >> msg;
		qCritical() << "Prober error:" << qPrintable(msg.text);
		proberError(msg.text);
		doneCmd(1);
		break;
	    case SC_SCHEDULED:
		in >> nextProberStart;
		if (!fileTail || !fileTail->isRunning()) printNextProberStart();
		    // else, wait until finishProber()
		break;
	    case SC_PAUSED:
		spout << "Scheduler paused." << Qt_endl;
		nextProberStart.when = 0;
		schedulerPaused = true;
		printNextProberStart();
		break;
	    case SC_RESUMED:
		spout << "Scheduler resumed." << Qt_endl;
		schedulerPaused = false;
		break;
	    case SC_NEED_CONFIG:
		for (auto m : config->members) {
		    if (!m->isSet() && m->required)
			keys << m->key;
		}
		spout << "The following required settings must be set: " <<
		    keys.join(QSL(", ")) << Qt_endl;
		schedulerNeedsConfig = true;
		needConfig();
		break;
	    case SC_CONFIGED:
		spout << "Scheduler is configured." << Qt_endl;
		schedulerNeedsConfig = false;
		break;
	    case SC_UPGRADE_AVAILABLE:
		upgradeInfo = new sc_msg_upgrade_available();
		in >> *upgradeInfo;
		qDebug() << "SC_UPGRADE_AVAILABLE:" << upgradeInfo->vnum;
		promptForUpgrade();
		break;
#ifdef AUTOUPGRADE_ENABLED
	    case SC_UPGRADE_ERROR:
		in >> msg;
		if (upgradeInfo) upgradeInfo->autoTime = -1;
		spout << msg.text << Qt_endl;
		showUpgradeResult(msg.text);
		cancelUpgradePrompt();
		break;
	    case SC_UPGRADE_PROGRESS:
		in >> msg;
		qDebug() << "SC_UPGRADE_PROGRESS";
		cancelUpgradePrompt();
		showUpgradeProgress(msg.text);
		break;
	    case SC_UPGRADE_INSTALLING:
		qDebug() << "SC_UPGRADE_INSTALLING";
		cancelUpgradePrompt();
		upgrade();
		break;
#endif
	    case SC_HELLO:
		in >> msg;
		if (msg.text.compare(QSL(PACKAGE_VERSION)) != 0) {
		    qCritical().noquote() << "UI version" << PACKAGE_VERSION <<
			"does not match Scheduler version" << msg.text;
		    scheduler->abort();
		}
		rcvdHello = true;
		break;
	    default:
		qCritical() << "Illegal message" << type << "from scheduler.";
		scheduler->abort();
	}
    }
}

void SpooferUI::handleProberText(QString *text)
{
    spout << *text << Qt_flush;
#ifdef VISIT_URL
    visitEmbeddedURLs(text);
#endif
    delete text;
}

bool SpooferUI::finishProber()
{
    if (proberExitStatus == QProcess::NormalExit) {
	spout << "prober exited normally, exit code " << proberExitCode << Qt_endl;
    } else {
	sperr << "prober exited abnormally" << Qt_endl;
    }
    if (fileTail) fileTail->deleteLater();
    fileTail = nullptr;
    printNextProberStart();
    doneCmd(0);
    return true;
}

#ifdef AUTOUPGRADE_ENABLED
void SpooferUI::upgrade()
{
    QStringList appArgs = QCoreApplication::arguments();
    appArgs << additionalArgs();

#ifdef Q_OS_UNIX
    char **argv = new char *[appArgs.length() + 7];
    int argc = 0;
    argv[argc++] = strdup("/bin/sh");
    argv[argc++] = strdup("-c");
    argv[argc++] = strdup(
	"dir=\"$1\"; shift; "
	"from=\"$1\"; shift; "
	"to=\"$1\"; shift; "
	"echo \"Upgrading Spoofer\"; "
	"echo \"  from: ${from}\"; "
	"echo \"  to:   ${to}\"; "
	"echo \"  time: `date +'%Y-%m-%d %H:%M:%S'`\"; "
	"echo \"Please wait...\"; "
	"cd /; "
	"finished=\"$dir/upgrade-finished\"; "
	"while ! test -f \"${finished}\"; do "
	    "sleep 1; "
	"done; "
	"cd \"$dir\"; "
	"cat \"${finished}\"; "
	"echo \"Restarting Spoofer UI: $@\"; "
	"exec \"$@\";");
    argv[argc++] = strdup("spoofer-upgrade-ui"); // $0 for script
    argv[argc++] = strdup(config->dataDir().toLocal8Bit().constData()); // $1 for script
    argv[argc++] = strdup(PACKAGE_VERSION); // $2 for script
    argv[argc++] = strdup(upgradeInfo ? upgradeInfo->vstr.toLocal8Bit().constData() : "new-version"); // $3 for script
    // appArgs[0] is not necessarily accurate or full path of this UI
    argv[argc++] = strdup(appFile.toLocal8Bit().constData());
    for (int i = 1; i < appArgs.length(); i++)
	argv[argc++] = strdup(appArgs[i].toLocal8Bit().constData());
    argv[argc++] = nullptr;
    execv("/bin/sh", (char *const *)argv);
    // UNREACHABLE
    qCritical() << "failed to exec sh script";
#endif

#ifdef Q_OS_WIN32
    // Windows script can't be done entirely on command line; we must write it
    // to a batch file and execute that.
    QString body = QSL(
	"@echo off\n"
	"cd \\\n" // get out of the installer's way
	"set dir=%1\n"
	"set from=%2\n"
	"set to=%3\n"
	"echo Upgrading Spoofer\n"
	"echo   from: %from%\n"
	"echo   to:   %to%\n"
	"echo   time: !DATE! !TIME!\n"
	"echo Please wait...\n"
	"set finished=%dir%\\upgrade-finished\n"
	":loop\n"
	"if not exist %finished% (\n"
	"    timeout 1 /nobreak >nul\n"
	"    goto loop\n"
	")\n"
	"cd %dir%\n"
	"type %finished%\n"
	"echo Restarting Spoofer UI: %4 %5 %6 %7 %8 %9\n"
	"start %4 %4 %5 %6 %7 %8 %9\n"
	//     ^^ not a typo.  First %4 is window title, second %4 is command.
	"del %0\n"
	);

    QFile scriptFile(QSL("%1/spoofer-upgrade-ui-%2.bat").
	arg(QDir::tempPath()).arg(QCoreApplication::applicationPid()));
    scriptFile.open(QIODevice::WriteOnly | QIODevice::Text);
    scriptFile.write(body.toLocal8Bit());
    scriptFile.close();

    QStringList args;
    args << QSL("/V:ON"); // enable !VAR! runtime subs
    args << QSL("/S"); // more consistent quoting
    args << QSL("/C");
    args << QDir::toNativeSeparators(scriptFile.fileName());
    args << QDir::toNativeSeparators(config->dataDir()); // %1 in batch script
    args << QSL(PACKAGE_VERSION); // %2 in batch script
    args << (upgradeInfo ? upgradeInfo->vstr : QSL("new-version")); // %3 in batch script
    // appArgs[0] is not necessarily accurate or full path of this UI
    args << QDir::toNativeSeparators(appFile); // %4
    args.append(appArgs.mid(1, -1)); // %5...
    auto env = QProcessEnvironment::systemEnvironment();
    QString comspec = env.value(QSL("ComSpec"), QSL("cmd.exe"));
    if (QProcess::startDetached(comspec, args)) {
	qDebug() << "executing" << comspec;
	QCoreApplication::quit();
    } else {
	qCritical() << "failed to execute" << comspec;
    }
#endif
}
#endif // AUTOUPGRADE_ENABLED

void SpooferUI::preExec()
{
    QFile file(config->dataDir() + QSL("/upgrade-finished"));
    if (file.open(QIODevice::ReadOnly | QIODevice::Text)) {
	QTextStream in(&file);
	sperr << Qt_endl;
	while (!in.atEnd()) {
	    QString str = in.readLine();
	    showUpgradeResult(str);
	}
    } else {
	qDebug() << "###" << file.fileName() << "not found";
    }
}
