/* 
 * Copyright 2015-2020 The Regents of the University of California
 * All rights reserved.
 * 
 * This file is part of Spoofer.
 * 
 * Spoofer is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Spoofer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Spoofer.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <time.h>
#include <locale.h>
#include <cstdio>
#include <errno.h>
#include <unistd.h> // unlink()
#include "spoof_qt.h"
#include <QCommandLineParser>
#include <QtGlobal>
#include <QDir>
#include <QUrl>
#ifdef Q_OS_WIN32
# include <windows.h> // RegGetValue()
# include <psapi.h> // GetProcessImageFileName()
#endif
#ifdef Q_OS_UNIX
# include <sys/types.h>
# include <signal.h> // kill()
#endif
#include "../../config.h"
#include "common.h"
static const char cvsid[] ATR_USED = "$Id: common.cpp,v 1.91 2020/07/16 19:23:40 kkeys Exp $";

SpooferBase::OnDemandDevice SpooferBase::outdev(stdout);
SpooferBase::OnDemandDevice SpooferBase::errdev(stderr);
QTextStream SpooferBase::spout(&SpooferBase::outdev);
QTextStream SpooferBase::sperr(&SpooferBase::errdev);
const QStringList *SpooferBase::args;
QString SpooferBase::optSettings;
SpooferBase::Config *SpooferBase::config;
QSettings *SpooferBase::Config::settings;
QList<SpooferBase::Config::MemberBase*> SpooferBase::Config::members;

// We avoid ".log" Suffix because OSX "open" would launch a log reader app
// we don't want.
const QString SpooferBase::proberLogFtime = QSL("'spoofer-prober-'yyyy~MM~dd-HH~mm~ss'.txt'");
const QString SpooferBase::proberLogGlob = QSL("spoofer-prober-\?\?\?\?\?\?\?\?-\?\?\?\?\?\?.txt"); // backslashes prevent trigraphs
const QString SpooferBase::proberLogRegex = QSL("spoofer-prober-(\\d{4})(\\d{2})(\\d{2})-(\\d{2})(\\d{2})(\\d{2}).txt$");

#ifdef Q_OS_WIN32
unsigned long getLastErr() { return GetLastError(); }
#endif

QString getErrmsg(error_t err)
{
#if defined(Q_OS_UNIX)
    QString msg = QString::fromLocal8Bit(strerror(err));
#elif defined(Q_OS_WIN32)
    wchar_t buf[1024];
    FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS,
	nullptr, err, 0, buf, sizeof(buf), nullptr);
    QString msg = QString::fromWCharArray(buf);
#endif
    return QString(QSL("%2 (error %1)")).arg(err).arg(msg.trimmed());
}

// Returns false if process does not exist.  Otherwise, returns true and
// writes the process name (if possible) or an empty string into buf.
// Unix: process name may include a path, and includes arguments.
// Windows: process name does not included path or arguments.
bool getProcessName(qint64 pid, char *buf, unsigned len)
{
#if defined(Q_OS_UNIX)
    if (kill(static_cast<pid_t>(pid), 0) < 0 && errno == ESRCH) {
	// kill() is more reliable than the "ps" below
	return false;
    } else {
	// POSIX "ps -ocomm=" prints just the name of the command, but some
	// platforms truncate it.  So we use "ps -oargs=" to print the full
	// argv (space-delimited and unquoted, making spaces ambiguous).
	// (Either form _might_ include a path in the command name.)
	snprintf(buf, len, "ps -p%lu -oargs=", static_cast<unsigned long>(pid));
	FILE *ps = popen(buf, "r");
	if (!ps) {
	    buf[0] = '\0';
	    return true;
	}
	if (fgets(buf, static_cast<int>(len), ps)) {
	    char *p = buf + strlen(buf);
	    if (p > buf && *--p == '\n') *p = '\0'; // strip trailing newline
	} else {
	    buf[0] = '\0';
	}
	fclose(ps);
	return true;
    }

#elif defined(Q_OS_WIN32)
    HANDLE hProc;
    // Note: QueryProcessImageName() and GetProcessImageFileName() require
    // only PROCESS_QUERY_LIMITED_INFORMATION.
    // GetModuleBaseName() and GetModuleFileNameEx() require
    // PROCESS_QUERY_INFORMATION, which apparently isn't allowed for Services.
    if (!(hProc = OpenProcess(PROCESS_QUERY_LIMITED_INFORMATION, false, static_cast<DWORD>(pid)))) {
	buf[0] = '\0';
	return GetLastError() != ERROR_INVALID_PARAMETER; // false if process did not exist
    }
    if (GetProcessImageFileNameA(hProc, buf, len) > 0) {
	char *p = strrchr(buf, '\\');
	if (p)
	    memmove(buf, p+1, strlen(p)); // strip path
    } else {
	buf[0] = '\0';
    }
    CloseHandle(hProc);
    return true;
#endif
}

QString processErrorMessage(const QProcess &proc)
{
    QString msg;
    switch (proc.error()) {
	case QProcess::FailedToStart: msg = QSL("failed to start"); break;
	case QProcess::Crashed:       msg = QSL("crashed");         break;
	case QProcess::Timedout:      msg = QSL("timed out");       break;
	case QProcess::WriteError:    msg = QSL("write error");     break;
	case QProcess::ReadError:     msg = QSL("read error");      break;
	case QProcess::UnknownError:  msg = QSL("unknown error");   break;
	default:                      msg = QSL("error");           break;
    }
    return msg % QSL(": ") % proc.errorString();
}

SpooferBase::Config::Config() :
    forWriting(false),
    // Internal
    dataDir(QSL("dataDir"), QString(),
	QSL("Use <dir> as data directory"), true),
    schedulerSocketName(QSL("schedulerSocketName")),
    paused(QSL("paused"), false,
	QSL("Start with scheduled prober runs disabled"), true),
#if DEBUG
    // Debug
    useDevServer(QSL("useDevServer"), true,
	QSL("use development test server")),
    spooferProtocolVersion(QSL("spooferProtocolVersion"), 0, 0, INT_MAX,
	QSL("force spoofer protocol version number (0 for default)")),
    pretendMode(QSL("pretendMode"), false,
	QSL("pretend mode - don't send any probe packets")),
    standaloneMode(QSL("standaloneMode"), false,
	QSL("standalone debugging mode - run a test without server")),
    installerAddTaint(QSL("installerAddTaint"), true,
 #if defined(Q_OS_WIN32)
	QSL("add Windows MOTW to downloaded installer")),
 #elif defined(Q_OS_MACOS)
	QSL("add OSX quarantine attribute to downloaded installer")),
 #else
	QSL("not used")),
 #endif // Q_OS_MACOS
    installerVerifySig(QSL("installerVerifySig"), true,
	QSL("verify signature of downloaded installer")),
    installerKeep(QSL("installerKeep"), false,
	QSL("don't delete downloaded installer after use")),
#endif // DEBUG

    // General
#ifdef AUTOUPGRADE_ENABLED
    autoUpgrade(QSL("autoUpgrade"), true,
	QSL("Enable automatic software upgrades")),
#endif
    enableIPv4(QSL("enableIPv4"), true,
	QSL("Enable testing on IPv4 interfaces (if available)")),
    enableIPv6(QSL("enableIPv6"), true,
	QSL("Enable testing on IPv6 interfaces (if available)")),
    keepLogs(QSL("keepLogs"), 60, 0, INT_MAX,
	QSL("Number of prober log files to keep (0 means unlimited)")),
    sharePublic(QSL("sharePublic"), true,
	QSL(DESC_SHARE_PUBLIC)),
    shareRemedy(QSL("shareRemedy"), true,
	QSL(DESC_SHARE_REMEDY)),
    enableTLS(QSL("enableTLS"), true,
	QSL("Use SSL/TLS to connect to server (recommended unless blocked by your provider)")),
    // Probing
    netPollInterval(QSL("netPollInterval"), 2*60, 1, 86400,
	QSL("Wait to check for a network change (seconds)")),
    delayInterval(QSL("delayInterval"), 60, 1, 3600,
	QSL("Wait to run a test after detecting a network change (seconds)")),
    // odd proberInterval helps prevent many clients from synchronizing
    proberInterval(QSL("proberInterval"), 7*24*60*60 + 65*60, 3600, INT_MAX,
	QSL("Wait to run a test after a successful run on the same network (seconds)")),
    proberRetryInterval(QSL("proberRetryInterval"), 10*60, 60, INT_MAX,
	QSL("Wait to retry after first incomplete run (seconds) (doubles each time)")),
    maxRetries(QSL("maxRetries"), 3, 0, INT_MAX,
	QSL("Maximum number of retries after an incomplete run")),
    // Permissions:  "Allow unprivileged users on this computer to..."
    unprivView(QSL("unprivView"), true,
	QSL("Observe a test in progress and view results of past tests")),
    unprivTest(QSL("unprivTest"), false,
	QSL("Run a test")),
    unprivPref(QSL("unprivPref"), false,
	QSL("Change preferences"))
{
    sharePublic.required = true;
    shareRemedy.required = true;
#ifdef Q_OS_UNIX
    // We don't want these user-desktop-specific environment variables
    // affecting our defaults for dataDir or settings file; we want the
    // same defaults for all users, including system daemon launchers.
    unsetenv("XDG_CONFIG_DIRS");
    unsetenv("XDG_DATA_DIRS");
#endif
}

void SpooferBase::Config::initSettings(bool _forWriting, bool debug)
{
    config->forWriting = _forWriting;
    if (!optSettings.isEmpty()) {
	settings = new QSettings(optSettings, QSettings::IniFormat);
    } else {
	settings = findDefaultSettings(debug);
	settings->setFallbacksEnabled(false);
    }
}

QString SpooferBase::Config::lockFileName()
{
    // Note: QSettings may generate a temporary lock file by appending ".lock"
    // to the file name, so we must use a different name for our lock.
    if (isFile()) return settings->fileName() % QSL(".write-lock");

    QString path;
#ifdef Q_OS_WIN32
    // We want the system's %TEMP%, not the user's.
    LONG err;
    char buf[1024];
    DWORD size = sizeof(buf) - 1;
    const char *subkeyName =
	"System\\CurrentControlSet\\Control\\Session Manager\\Environment";
    err = RegGetValueA(HKEY_LOCAL_MACHINE, subkeyName, "TEMP", RRF_RT_REG_SZ,
	nullptr, buf, &size);
    if (err != ERROR_SUCCESS) {
	qDebug() << "RegGetValue:" << getErrmsg(static_cast<DWORD>(err));
	goto doneReg;
    }
    path = QString::fromLocal8Bit(buf).trimmed();
    qDebug() << "TEMP:" << qPrintable(path);
doneReg:
#endif
    if (path.isEmpty()) path = QDir::tempPath();
    return (path % QSL("/") % QCoreApplication::applicationName() % QSL(".lock"));
}

bool SpooferBase::Config::error(const char *label)
{
    if (settings->status() == QSettings::NoError)
	return false;

    logError(label,
	(settings->status() == QSettings::AccessError) ? QSL("AccessErrror") :
	(settings->status() == QSettings::FormatError) ? QSL("FormatError") :
	QSL("unknown error %1").arg(int(settings->status())));

    return true;
}

void SpooferBase::Config::logError(const char *label, QString msg, QString msg2)
{
    msg = QSL("%1 in \"%2\"").arg(msg,
	QDir::toNativeSeparators(settings->fileName()));

    if (isFile()) {
	// Use the standard library to get a more informative error message.
	FILE *f = fopen(qPrintable(settings->fileName()),
	    forWriting ? "r+" : "r");
	if (!f)
	    msg = QSL("%1 (%2)").arg(msg,
		QString::fromLocal8Bit(strerror(errno)));
	else
	    fclose(f);
    }
    qCritical().nospace().noquote() << label << ": " << msg << ". " << msg2;
}

void SpooferBase::Config::remove()
{
    if (!settings) return;
    QString name = isFile() ? settings->fileName() : QString();
    settings->clear();
    delete settings;
    settings = nullptr;
    if (!name.isEmpty()) {
	if (unlink(name.toStdString().c_str()) == 0)
	    sperr << name << " removed." << Qt_endl;
	else
	    sperr << "Error removing " << name << ": " << strerror(errno) << Qt_endl;
    }
}

SpooferBase::SpooferBase() :
    appDir(QCoreApplication::applicationDirPath()),
    appFile(QCoreApplication::applicationFilePath())
{
    setlocale(LC_NUMERIC, "C");

    // for QStandardPaths::standardLocations() and QSettings
    QCoreApplication::setOrganizationName(QSL(ORG_NAME));
    QCoreApplication::setOrganizationDomain(QSL(ORG_DOMAIN));
    QCoreApplication::setApplicationName(QSL("Spoofer")); // may change later

    qInstallMessageHandler(logHandler);

    config = new Config();
}

qint64 SpooferBase::OnDemandDevice::writeData(const char *data, qint64 maxSize)
{
    if (newdev) {
	QFile *file = dynamic_cast<QFile*>(dev);
	QFile *newfile = dynamic_cast<QFile*>(newdev);
	if (newfile) {
	    // Make sure filename is clean and absolute for comparison below.
	    QDir newpath(QDir::cleanPath(newfile->fileName()));
	    newfile->setFileName(newpath.absolutePath());
	}
	if (file && file->isOpen() && newfile && file->fileName() ==
	    newfile->fileName())
	{
	    // Old dev is open, and newdev is the same file; ignore newdev.
	} else {
	    char buf[2048];
	    if (dev && dev->isOpen() && !newname.isEmpty()) {
		snprintf(buf, sizeof(buf), "Redirecting output to %s\n",
		    qPrintable(newname));
		dev->write(buf);
	    }
	    if (!newdev->open(WriteOnly|Unbuffered|Append|Text)) {
		if (dev && dev->isOpen()) {
		    snprintf(buf, sizeof(buf), "Redirection failed: %s.\n",
			qPrintable(newdev->errorString()));
		    dev->write(buf);
		}
		delete newdev;
	    } else { // success
		if (dev) delete dev;
		dev = newdev;
	    }
	}
	newdev = nullptr;
    }

    if (!dev) {
	if (!fallback) {
	    setErrorString(QSL("output device is not set"));
	    return maxSize; // *dev failed, but *this can still work
	}
	// E.g., dev was nulled in a previous call, and fallback was set later.
	newdev = fallback;
	fallback = nullptr;
	return writeData(data, maxSize); // Try again with the fallback.
    }

    if (timestampEnabled) {
	char tbuf[40];
	time_t t = time(nullptr);
	struct tm *tm = gmtime(&t);
	strftime(tbuf, sizeof(tbuf), "[%Y-%m-%d %H:%M:%S] ", tm);
	dev->write(tbuf, safe_int<qint64>(strlen(tbuf)));
    }
    qint64 retval = dev->write(data, maxSize);
    if (retval < 0) {
	// E.g., stderr is closed when running as a Windows service.
	if (!fallback) { // There was no fallback.
	    delete dev; dev = nullptr;
	    return maxSize; // *dev failed, but *this can still work
	}
	newdev = fallback;
	fallback = nullptr;
	return writeData(data, maxSize); // Try again with the fallback.
    }
    fallback = nullptr; // Dev worked; we won't need the fallback.
    return retval;
}

void SpooferBase::logHandler(QtMsgType type, const QMessageLogContext &ctx,
    const QString &msg)
{
    Q_UNUSED(ctx);

#if !DEBUG
    if (type == QtDebugMsg) return;
#endif

    const char *prefix =
	(type == QtDebugMsg)    ? "Debug"    :
	(type == QtWarningMsg)  ? "Warning"  :
	(type == QtCriticalMsg) ? "Critical" :
	(type == QtFatalMsg)    ? "Fatal"    :
	nullptr;

    static int depth = 0;
    if (++depth > 1) { // should not happen
	fprintf(stderr, "INTERNAL ERROR: logHandler recursion\n");
	fprintf(stderr, "%s: %s\n", prefix, qPrintable(msg));
	::fflush(stderr);
    } else {
	if (prefix)
	    sperr << prefix << ": " << msg << Qt_endl;
	else
	    sperr << msg << Qt_endl;
    }
    depth--;
}

// Caller can add additional options before calling parseCommandLine(), and
// inspect them after.
bool SpooferBase::parseCommandLine(QCommandLineParser &clp, QString desc)
{
    clp.setApplicationDescription(desc);

    QSettings *ds = findDefaultSettings(false);
    QString format = QSL("");
#ifdef Q_OS_WIN32
    if (ds->format() == QSettings::NativeFormat) format = QSL("registry ");
#endif
    QCommandLineOption cloSettings(QSL("settings"),
	QSL("Use settings in <file> [%1\"%2\"].").arg(format).arg(ds->fileName()),
	QSL("file"));
    clp.addOption(cloSettings);
    delete ds;

    QCommandLineOption cloVersion(QStringList() << QSL("v") << QSL("version"),
	QSL("Display version information."));
    clp.addOption(cloVersion);

    // clp.addHelpOption() wouldn't include "-?" on non-Windows.
    QCommandLineOption cloHelp(QStringList() << QSL("?") << QSL("h") << QSL("help"),
	QSL("Display this help."));
    clp.addOption(cloHelp);

    if (!clp.parse(SpooferBase::args ? *SpooferBase::args : QCoreApplication::arguments())) {
	qCritical() << qPrintable(clp.errorText());
	return false;
    }

    if (clp.isSet(cloHelp)) {
	qInfo() << qPrintable(clp.helpText());
	return false;
    }

    if (clp.isSet(cloVersion)) {
	qInfo() << PACKAGE_NAME << "version" << PACKAGE_VERSION << Qt_endl << 
	    "Qt version" << qVersion();
	return false;
    }

    if (clp.isSet(cloSettings))
	optSettings = QDir::current().absoluteFilePath(clp.value(cloSettings));

    return true;
}

// Format is like QDateTime::toString(), with the addition that '~' will be
// removed after formatting, allowing you to create adjacent non-separated
// fields in output by inserting '~' between them in the input format.
QString SpooferBase::ftime_zone(const QString &fmt, const time_t *tp, const Qt::TimeSpec &spec)
{
    time_t t;
    if (!tp) { time(&t); tp = &t; }
    // QDateTime::fromTime_t() is not available in Qt >= 5.8(?);
    // QDateTime::fromSecsSinceEpoch() is not available in Qt < 5.8.
    return QDateTime::fromMSecsSinceEpoch(qint64(*tp) * 1000, spec)
	.toString(!fmt.isEmpty() ? fmt : QSL("yyyy-MM-dd HH:mm:ss t"))
	.remove(QLatin1Char('~'));
}

sc_msg_upgrade_available::sc_msg_upgrade_available(bool wantAuto,
    bool _mandatory, int32_t _vnum, const QString &_vstr, const QString &_file) :
    autoTime(wantAuto ? 60 : -1), mandatory(_mandatory),
    vnum(_vnum), vstr(_vstr), file(_file), warning()
{
#if 0 // these checks aren't needed since we have TLS and signed installer
    if (file.isEmpty()) return;
    QStringList warnings;
#if defined(UPGRADE_KEY) && !defined(UPGRADE_WITHOUT_DOWNLOAD)
    QUrl u(file);
    if (!u.isValid()) {
	warnings << QSL("invalid URL: ") % u.errorString();
    } else {
	if (!file.startsWith(QSL(UPGRADE_KEY)))
	    warnings << QSL("URL does not start with \"" UPGRADE_KEY "\"");
    }
#endif
    if (!warnings.isEmpty()) {
	warning = QSL("WARNING: ") % warnings.join(QSL("; ")) % QSL(".");
#if !DEBUG
	// Don't autoupgrade if URL looks fishy
	autoTime = -1;
#endif
    }
#endif // 0
}

