/* 
 * Copyright 2015-2020 The Regents of the University of California
 * All rights reserved.
 * 
 * This file is part of Spoofer.
 * 
 * Spoofer is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Spoofer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Spoofer.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef SCHEDULER_APP_H
#define SCHEDULER_APP_H

#include <QProcess>
#include <QSet>
#include <QFile>
#include <QTimer>
#include <QHash>
#include <QHostAddress>
#include "common.h"
#include "SubnetAddr.h"
#include "downloader.h"

QT_BEGIN_NAMESPACE
class QLocalSocket;
class QLocalServer;
class QNetworkConfigurationManager;
class QLockFile;
class QCommandLineOption;
class QFileSystemWatcher;
QT_END_NAMESPACE

#define APPNAME "spoofer-scheduler"

class App : public QCoreApplication, public SpooferBase {
    Q_OBJECT

    App(const App&) NO_METHOD; // no copy-ctor
    void operator=(const App&) NO_METHOD; // no copy-assign

    enum UpgradeState { US_NONE, US_PROMPTED, US_DOWNLOADING, US_VERIFYING,
	US_INSTALLING };
protected:
    class AppLog : public QFile {
    public:
	static QString makeName();
	AppLog() : QFile(makeName()) {}
    };

    struct RunRecord {
	time_t t;
	int errors;
    };

    // Don't remove the "upgrade-finished" file until at least one UI has
    // connected AND a prober has started, to make sure a user gets a chance
    // to see it.
    class UpFinRemover {
	const unsigned int all = ((1U<<uiConnected) | (1U<<proberStarted));
	const App *app;
	unsigned int state;
    public:
	enum UFType { uiConnected, proberStarted };
	UpFinRemover(const App *_app) : app(_app), state(0) {}
	void set(UFType ufType) {
	    state |= (1U << ufType);
	    if (state == all) {
		qDebug() << "removing file" << app->upFinName();
		QFile::remove(app->upFinName());
	    }
	}
	void reset() { state = 0; }
    };

    App(int &argc, char **argv) :
	QCoreApplication(argc, argv), SpooferBase(),
	optSharePublic(-1), optShareRemedy(-1), optInitOnly(false),
	optDeleteUpgrade(false), optDeleteData(false), optDeleteSettings(false),
	optSaveSettings(false), optRestoreSettings(false), altSettingsFile(),
	optDumpPaths(false), optDumpSettings(false), optCheckSettings(false),
	optDetach(false), optLogfile(false), optStartPaused(-1), optDataDir(),
	isInteractive(false), paused(false), prober(), upgradeInfo(),
	pApplabel(),
	settingLockFile(), dataLockFile(), privServer(), unprivServer(),
	uiSet(), proberOutputFileName(), scheduledSubnets(),
	nextProberStart(), proberTimer(), netPollTimer(), pastRuns(),
	hangTimer(), proberWatcher(),
	upFinRemover(this), upgradeState(US_NONE)
#ifdef AUTOUPGRADE_ENABLED
	, upgradePromptTimer(), upgradeIfGreater(NVERSION),
	downloader(), installerTimer(), installerTicker()
#endif
	{
	    errdev.setTimestampEnabled(true);
	    outdev.setTimestampEnabled(true);
#ifdef AUTOUPGRADE_ENABLED
	    upgradePromptTimer.setSingleShot(true);
	    connect(&upgradePromptTimer, &QTimer::timeout, this, &App::startUpgradeSlot);
#endif
	}
public:
    virtual ~App();
    static App *newapp(int &argc, char **argv);
    static QString lastErrorString();
    virtual bool init(int &exitCode);
    virtual void readyService(int exitCode) const { Q_UNUSED(exitCode); }
    virtual void endService(int exitCode) { Q_UNUSED(exitCode); }
    virtual void end() const {}
    QString applabel() { return pApplabel; }
protected:
    void cleanup(void);
    bool parseCommandLine(QCommandLineParser &clp);
    bool parseOptionFlag(int &opt, QCommandLineParser &clp,
	const QCommandLineOption &clo);
    virtual void dumpPaths(void) const;
    virtual void dumpSettings(void) const;
    virtual QString chooseDataDir();
    QLockFile *lockLockFile(QString name);
    virtual bool prestart(int &exitCode) { Q_UNUSED(exitCode); return true; }
    virtual bool initSignals() = 0;
    virtual void pause();
    virtual void resume();

// static members
public:
    static const QString appname;
    static const QString schedulerLogFtime;
    static const QString schedulerLogGlob;
protected:
    static QString defaultDataDir;
    static QString dataDir;

// members
protected:
    int optSharePublic;
    int optShareRemedy;
    bool optInitOnly;
    bool optDeleteUpgrade;
    bool optDeleteData;
    bool optDeleteSettings;
    bool optSaveSettings;
    bool optRestoreSettings;
    QString altSettingsFile;
    bool optDumpPaths;
    bool optDumpSettings;
    bool optCheckSettings;
    bool optDetach;
    bool optLogfile;
    int optStartPaused;
    QString optDataDir;
    bool isInteractive;
    bool paused; // actual current state (cf. config->paused(), desired state)
    QProcess *prober;
    sc_msg_upgrade_available *upgradeInfo;
    QString pApplabel;
private:
    QLockFile *settingLockFile, *dataLockFile;
    QLocalServer *privServer;
    QLocalServer *unprivServer;
    QSet<QLocalSocket *> uiSet;
    QString proberOutputFileName;
    QSet<SubnetAddr> scheduledSubnets;
    sc_msg_scheduled nextProberStart;
    QTimer proberTimer;
    QTimer netPollTimer;
    // pastRuns is indexed not by the full address of interfaces, but by their
    // subnet prefixes, so that:  1) a new prober run is not triggered when
    // the host moves within a subnet or uses a different temporary IPv6
    // address; 2) history isn't cluttered with random temporary IPv6 addrs.
    QHash<SubnetAddr, RunRecord> pastRuns;
    QTimer *hangTimer;
    QFileSystemWatcher *proberWatcher;
    UpFinRemover upFinRemover;
#ifndef AUTOUPGRADE_ENABLED
    const UpgradeState upgradeState;
#else // AUTOUPGRADE_ENABLED
    UpgradeState upgradeState;
    QTimer upgradePromptTimer;
    int32_t upgradeIfGreater;
    Downloader *downloader;
    QTimer *installerTimer;
    int installerTicker;
#endif

// internal methods
protected:
    QLocalServer *listen(bool privileged);
    bool opAllowed(QLocalSocket *ui, const Config::MemberBase &cfgItem);
    bool opAllowedVerbose(QLocalSocket *ui, const Config::MemberBase &cfgItem);
    void recordRun(bool success);
    void startProber() { startProber(false); }
    virtual void startProber(bool manual);
    void deleteProber();
    virtual void killProber();
    virtual void shutdown() { this->exit(0); }
    QList<SubnetAddr> getAddresses() const;
#ifdef ENABLE_QNetworkConfigurationManager
    QNetworkConfigurationManager *ncm; // experimental
#endif
#ifdef AUTOUPGRADE_ENABLED
    void cancelUpgrade();
    void startUpgrade(QLocalSocket *ui = nullptr);
#endif

private slots:
    void uiRead();
    void uiAccept();
    void uiDelete();
    void proberStarted();
    void proberError(QProcess::ProcessError);
    void proberFinished(int exitCode, QProcess::ExitStatus exitStatus);
    void dumpNetCfg(QLocalSocket *ui);
    void handleNetChange();
    void scheduleNextProber();
#ifdef AUTOUPGRADE_ENABLED
    void startUpgradeSlot() { startUpgrade(); }
    void enterUpgradeVerifyState();
    void downloadFinished();
    void downloadFailed();
    void installerCheck();
#endif

private:
    void parseProberTextForUpgrade();
    void promptForUpgrade();
    void recordUpgrade();
    void sendUpgradeProgress(QLocalSocket *ui, const QString &text);
    void sendUpgradeError(QLocalSocket *ui, const QString &text);
    void startInstaller(const QString &installerName);
protected:
#ifdef AUTOUPGRADE_ENABLED
    void abortInstallation(const QString &err);
    virtual bool installerIsRunning() = 0;
    virtual void killInstaller() = 0;
    virtual void executeInstaller(const QString &installerName) = 0;
#endif
    QString upFinName() const { return dataDir % QSL("/upgrade-finished"); }
};

#endif // SCHEDULER_APP_H
