/* 
 * Copyright 2015-2020 The Regents of the University of California
 * All rights reserved.
 * 
 * This file is part of Spoofer.
 * 
 * Spoofer is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Spoofer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Spoofer.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h> // umask()
#include <syslog.h> // syslog()
#include <sys/wait.h> // waitpid()
#include "spoof_qt.h"
#include <QSocketNotifier>
#include <QLockFile>
#include <QCoreApplication>
#include "../../config.h"
#include "app.h"
#include "appunix.h"
#include "common.h"
static const char cvsid[] ATR_USED = "$Id: appunix.cpp,v 1.33 2020/07/16 19:23:43 kkeys Exp $";

int AppUnix::psPipe[2]; // pipe for reporting posix signals

QString App::lastErrorString()
{
    return QSL("%1 (error %2)")
	.arg(QString::fromLocal8Bit(strerror(errno)))
	.arg(errno);
}

bool AppUnix::Syslog::open(QIODevice::OpenMode mode) {
    Q_UNUSED(mode);
    openlog(APPNAME, LOG_PID, LOG_USER);
    return QIODevice::open(WriteOnly|Unbuffered);
}

qint64 AppUnix::Syslog::writeData(const char *data, qint64 maxSize)
{
    syslog(LOG_NOTICE, "%.*s", (int)maxSize, data);
    return maxSize;
}

AppUnix::AppUnix(int &argc, char **argv) :
    App(argc, argv), installerPid(0), psNotifier()
{
    isInteractive = isatty(STDIN_FILENO) && isatty(STDOUT_FILENO);
    // Note: verifyDaemon() ensures the parent doesn't exit too quickly, so
    // we'll see the real parent and not 1.
    if (getppid() == 1) { // started by unix init or mac osx launchd
	errdev.setDevice(new Syslog(), QSL("syslog"));
	errdev.setTimestampEnabled(false);
    }
}

bool AppUnix::verifyDaemon(pid_t childPid)
{
    QLockFile lockFile(config->lockFileName());
    qDebug() << "waiting for child to lock" << config->lockFileName();
    qint64 qpid;
    while (!lockFile.getLockInfo(&qpid, nullptr, nullptr) ||
	qpid != childPid) // child is not ready
    {
	if (waitpid(childPid, nullptr, WNOHANG)) { // child exited
	    sperr << "Daemon exited prematurely" << Qt_endl;
	    return false;
	}
	usleep(100000);
    }
    return true;
}

bool AppUnix::prestart(int &exitCode)
{
    if (optDetach) {
	// NB: we must NOT detach (daemonize) if started from Mac OSX launchd
	pid_t pid;
	if ((pid = fork()) < 0) {
	    exitCode = errno;
	    sperr << "can't fork: " << strerror(errno) << Qt_endl;
	    return false; // skip app.exec()
	} else if (pid > 0) {
	    // parent
	    pApplabel = QSL(" parent");
	    exitCode = verifyDaemon(pid) ? SP_EXIT_OK : SP_EXIT_DAEMON_FAILED;
	    return false; // skip app.exec() in parent
	}
	// child
	pApplabel = QSL(" daemon");
	if (errdev.type() == typeid(Syslog)) {
	    // parent was using syslog; child continues to use syslog
	} else {
	    // parent was using logfile or stderr; child opens new logfile
	    errdev.setDevice(new AppLog());
	}
	setsid(); // become session leader
    }

    umask(022);
    return true;
}

void AppUnix::psHandler(int sig)
{
    if (::write(psPipe[1], &sig, sizeof(sig)) < 0)
	return; // shouldn't happen, and there's nothing we can do anyway
}

void AppUnix::psSlot()
{
    int sig = 0;
    psNotifier->setEnabled(false);
    if (::read(psPipe[0], &sig, sizeof(sig)) < 0)
	sig = 0; // shouldn't happen
    sperr << "Scheduler caught signal " << sig << Qt_endl;
    this->exit(EINTR);
    psNotifier->setEnabled(true);
}

bool AppUnix::initSignals()
{
    // It's not safe to call Qt functions from POSIX signal handlers.  So we
    // convert POSIX signals to Qt signals using a POSIX signal handler that
    // safely writes to a pipe and a QSocketNotifier that emits a Qt signal
    // when the pipe has something to read.
    if (::socketpair(AF_UNIX, SOCK_STREAM, 0, psPipe) < 0)
	sperr << "socketpair " << strerror(errno) << Qt_endl;
    psNotifier = new QSocketNotifier(psPipe[0], QSocketNotifier::Read, this);
    connect(psNotifier, &QSocketNotifier::activated, this, &AppUnix::psSlot);

    struct sigaction act;
    act.sa_handler = AppUnix::psHandler;
    sigemptyset(&act.sa_mask);
    act.sa_flags = SA_RESTART;
    for (int sig : {SIGTERM, SIGINT, SIGHUP}) {
	if (sigaction(sig, &act, nullptr) < 0)
	    sperr << "sigaction(" << sig << "): " << strerror(errno) << Qt_endl;
    }
    return true;
}

#ifndef EVERYONE_IS_PRIVILEGED
void AppUnix::startProber(bool manual)
{
    mode_t oldmask = 07000;
    if (!config->unprivView.variant().toBool())
	oldmask = umask(0077);
    App::startProber(manual);
    if (oldmask != 07000)
	umask(oldmask);
}
#endif

#ifdef AUTOUPGRADE_ENABLED
bool AppUnix::installerIsRunning()
{
    int status;
    pid_t pid = waitpid(installerPid, &status, WNOHANG);
    if (pid == -1) return false;
    if (pid == 0) return true;
    if (WIFEXITED(status))
	qWarning() << "installer exit code:" << WEXITSTATUS(status);
    else if (WIFSIGNALED(status))
	qWarning() << "installer killed by signal" << WTERMSIG(status);
    return false;
}

void AppUnix::killInstaller()
{
    if (installerPid > 0) {
	// use negative pid to kill every process in the group
	if (kill(-installerPid, SIGTERM) < 0)
	    qDebug() << "failed to kill" << installerPid << strerror(errno);
	else
	    qDebug() << "killed" << installerPid;
    }
}

void AppUnix::executeInstaller(const QString &installerName)
{
    QString script = QSL("{ "
	"echo \"Installing $1\"; "
	"FromTo=\"from " PACKAGE_VERSION " to $4\"; "
	UPGRADE_CMD "; "
	"exitcode=$?; "
	"if test $exitcode -eq 0; then ") %
#ifndef UPGRADE_WITHOUT_DOWNLOAD
	(config->installerKeep() ? QSL("") : QSL("rm \"$1\"; ")) %
#endif
	QSL("msg=\"Successfully upgraded $FromTo\"; "
	"else "
	"msg=\"Failed to upgrade $FromTo (exit code $exitcode)\"; "
	"fi; "
	"echo \"$msg at `date +'%Y-%m-%d %H:%M:%S'`\" >\"$2\"; "
	"echo \"$msg\"; "
	"exit $exitcode; "
	"} >\"$3/upgrade-log.txt\" 2>&1");

    qDebug().noquote() << QSL("script:") << script;
    installerPid = fork();
    if (installerPid < 0) { // error
	abortInstallation(QSL("fork: ") % lastErrorString());
	return;
    } else if (installerPid == 0) { // child
	setsid(); // so the child won't get the kill signal meant for the parent
	execl("/bin/sh", "/bin/sh", "-c", script.toStdString().data(),
	    "spoofer-scheduler-upgrade",            // $0
	    installerName.toStdString().data(),     // $1
	    upFinName().toStdString().data(),       // $2
	    dataDir.toStdString().data(),           // $3
	    upgradeInfo->vstr.toStdString().data(), // $4
	    nullptr);
	// UNREACHABLE
	qCritical() << "Failed to exec sh script:" << lastErrorString();
	exit(1);
    } else { // parent
	qDebug() << "executing sh script, pid" << installerPid;
    }
}
#endif // AUTOUPGRADE_ENABLED
