/* 
 * Copyright 2017-2020 The Regents of the University of California
 * All rights reserved.
 * 
 * This file is part of Spoofer.
 * 
 * Spoofer is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Spoofer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Spoofer.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "spoof_qt.h"
#include <QDebug>
#include <QtDebug>
#include <QFile>
#include <QTemporaryFile>
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QTimer>
#include <QDir>
#ifdef Q_OS_WIN32
#include <windows.h> // CreateFile(), WriteFile()
#include <softpub.h> // WinVerifyTrust()
#endif
#ifdef Q_OS_MACOS
#include <QProcess>
#include <sys/xattr.h> // setxattr()
#endif
#include "downloader.h"

void Downloader::start()
{
    destpath = QString(dir % QSL("/") % url.fileName());
    if (QFile::exists(destpath))
	QFile::remove(destpath); // XXX check error
    manager = new QNetworkAccessManager(this);
    tmp = new QTemporaryFile(dir % QSL("/XXXXXX.tmp"), this);
    if (!tmp->open()) {
	abort(QSL("Error opening temporary file %1: %2").arg(
	    tmp->fileName(), tmp->errorString()));
	return;
    }
    qDebug().noquote() << "Temporary file:" << tmp->fileName();

    reqTimer = new QTimer(this);
    reqTimer->start(30000);

    reply = manager->get(QNetworkRequest(url));
    reply->setParent(this); // so reply will be destroyed by ~Downloader()

    connect(reply, &QNetworkReply::readyRead, this, &Downloader::replyRead);
    connect(reply, QNETWORKREPLY_ERROR_OCCURRED, this, &Downloader::replyError);
    connect(reply, &QNetworkReply::finished, this, &Downloader::replyFinished);
    connect(reply, &QNetworkReply::downloadProgress,
	reqTimer, SIGCAST(QTimer, start, ()));
    connect(reqTimer, &QTimer::timeout, this, &Downloader::timedout);
}

void Downloader::replyRead()
{
    qint64 readable = reply->bytesAvailable();
    if (readable == 0) {
	qDebug().noquote() << "0 bytes available from" << url.toString();
	return;
    }
    QByteArray data = reply->read(readable);
    qint64 written = tmp->write(data);
    if (written < 0) {
	abort(QSL("Error writing to %1: %2").arg(
	    tmp->fileName(), tmp->errorString()));
    } else {
	total += written;
    }
}

#ifdef Q_OS_WIN32
// see: https://docs.microsoft.com/en-us/windows/desktop/SecCrypto/example-c-program--verifying-the-signature-of-a-pe-file
static bool verifySignature(LPCWSTR filename)
{
    WINTRUST_FILE_INFO FileData;
    memset(&FileData, 0, sizeof(FileData));
    FileData.cbStruct = sizeof(FileData);
    FileData.pcwszFilePath = filename;

    // Use the Authenticode policy provider.
    GUID actionId = WINTRUST_ACTION_GENERIC_VERIFY_V2;

    WINTRUST_DATA wtd;
    memset(&wtd, 0, sizeof(wtd));
    wtd.cbStruct = sizeof(wtd);

    wtd.dwUnionChoice = WTD_CHOICE_FILE; // Verify signature on pFile
    wtd.pFile = &FileData;

    wtd.dwStateAction = WTD_STATEACTION_VERIFY;

    wtd.dwUIChoice = WTD_UI_NONE;
    HWND hwnd = (HWND)INVALID_HANDLE_VALUE; // no UI

    LONG status = WinVerifyTrust(hwnd, &actionId, &wtd);
    bool success = (status == 0);
    if (!success)
	SetLastError((error_t)status);

    // Free any hWVTStateData allocated by WTD_STATEACTION_VERIFY.
    wtd.dwStateAction = WTD_STATEACTION_CLOSE;
    WinVerifyTrust(nullptr, &actionId, &wtd);

    return success;
}
#endif // Q_OS_WIN32

void Downloader::replyFinished()
{
    if (proc) return; // QNetworkReply failed, but external process is trying
    tmp->close();
    if (reqTimer) reqTimer->stop();
    if (aborted) return;
    qDebug().noquote() << "Downloaded" << total << "bytes from" << url.toString();

    tmp->setAutoRemove(false);
    if (!tmp->rename(destpath)) {
	abort(QSL("Error renaming %1 to %2: %3").arg(
	    tmp->fileName(), destpath, tmp->errorString()));
	return;
    }
    tmp = nullptr;

    finalize();
}

void Downloader::finalize()
{
#ifdef Q_OS_WIN32
    if (addTaint) {
	// Set the Windows "Mark of the Web" on the file in an "Alternate Data
	// Stream"
	qDebug() << "adding Mark of the Web";
	QString attrname = destpath % QSL(":Zone.Identifier");
	HANDLE attrfile = CreateFile(attrname.toStdWString().c_str(), GENERIC_WRITE,
	    FILE_SHARE_READ | FILE_SHARE_WRITE | FILE_SHARE_DELETE,
	    nullptr, OPEN_ALWAYS, FILE_ATTRIBUTE_NORMAL, nullptr);
	if (attrfile == INVALID_HANDLE_VALUE) {
	    abort(QSL("Error creating MOTW on %1: %2").arg(
		destpath, getLastErrmsg()));
	    return;
	}
	const char body[] = "[ZoneTransfer]\r\nZoneId=3"; // 3=URLZONE_INTERNET
	DWORD written = 0, len = strlen(body);
	BOOL result = WriteFile(attrfile, body, len, &written, nullptr);
	CloseHandle(attrfile);
	if (!result || written != len) {
	    abort(QSL("Error writing MOTW on %1: %2").arg(
		destpath, getLastErrmsg()));
	    return;
	}
    }
#endif

#ifdef Q_OS_MACOS
    if (addTaint) {
	// Apple doesn't document this very well, but a decent reverse
	// engineering can be found at
	// https://github.com/Homebrew/homebrew-cask/issues/22388.
	// The QuarantineEventsV2 database entry is not strictly necessary, so
	// we skip it.
	char value[256];
	sprintf(value, "0000;%08x;%.240s;", safe_int<unsigned>(time(nullptr)),
	    PACKAGE_NAME);
	qDebug() << "adding com.apple.quarantine attribute: " << value;
	if (setxattr(destpath.toStdString().c_str(),
	    "com.apple.quarantine", value, strlen(value), 0, 0) < 0)
	{
	    abort(QSL("Error adding quarantine on %1: %2").arg(
		destpath, getLastErrmsg()));
	    return;
	}
    }
#endif

#ifdef Q_OS_WIN
    if (verifySig) {
	qDebug().noquote() << "Verifying installer " <<
	    QDir::toNativeSeparators(destpath);
	emit verifying();
	if (!verifySignature(destpath.toStdWString().c_str())) {
	    abort(QSL("Could not verify installer signature: ") % getLastErrmsg());
	    return;
	}
	qDebug() << "Verified installer signature.";
    }
#endif

#ifdef Q_OS_MACOS
    if (verifySig) {
	// run "pkgutil --check-signature $installerName".
	// if it fails, save stdout/stderr into an error message
	qDebug().noquote() << "Verifying installer " << destpath;
	emit verifying();
	QStringList args;
	args << QSL("--check-signature") << destpath;
	QProcess verifier;
	verifier.setProcessChannelMode(QProcess::MergedChannels); // ... 2>&1
	verifier.setStandardInputFile(QProcess::nullDevice()); // ... </dev/null
	verifier.start(QSL("pkgutil"), args);
	if (!verifier.waitForStarted() || !verifier.waitForFinished()) {
	    abort(QSL("Could not verify installer signature: %1: %2").arg(
		verifier.program(), processErrorMessage(verifier)));
	    return;
	}
	if (verifier.exitStatus() != QProcess::NormalExit) {
	    abort(QSL("Could not verify installer signature: verifier exited abnormally"));
	    return;
	} else if (verifier.exitCode() != 0) {
	    QByteArray output = verifier.readAll();
	    abort(QSL("Could not verify installer signature: (%1) %2")
		.arg(verifier.exitCode())
		.arg(QString::fromUtf8(output).trimmed()));
	    return;
	}
	qDebug() << "Verified installer signature.";
    }
#endif

    emit finished();
}

void Downloader::replyError()
{
#ifdef Q_OS_MACOS
    // On old OSX (10.7), QNetworkReply fails on modern HTTPS with "Failed to
    // set protocol version" (error 99), so we fall back to external "curl".
    if (total == 0 && !aborted) {
	qWarning().noquote() << QSL("Internal downloader failed: %1 (%2)")
	    .arg(reply->errorString()).arg(reply->error());
	proc = new QProcess(this);
	proc->setProcessChannelMode(QProcess::MergedChannels); // ... 2>&1
	proc->setStandardInputFile(QProcess::nullDevice()); // ... </dev/null
	QString procname = QSL("/usr/bin/curl");
	QStringList args;
	args << QSL("--fail") << QSL("--silent") << QSL("--show-error") <<
	    QSL("--output") << destpath << url.toString(QUrl::FullyEncoded);
	qDebug().noquote() << "Starting external downloader: " << procname <<
	    args.join(QSL(" "));
	connect(proc, &QProcess::started, this, &Downloader::procStarted);
	connect(proc, QPROCESS_ERROR_OCCURRED, this, &Downloader::procError);
	connect(proc, SIGCAST(QProcess, finished, (int, QProcess::ExitStatus)),
	    this, &Downloader::procFinished);
	proc->start(procname, args);
	return;
    }
#endif
    abort(QSL("Download failed: %1 (%2)")
	.arg(reply->errorString()).arg((int)reply->error()));
}

#ifdef Q_OS_MACOS
void Downloader::procStarted()
{
    qDebug() << "External downloader started";
}

void Downloader::procError()
{
    abort(QSL("Download failed:  Internal: %1 (%2).  External: %3")
	.arg(reply->errorString()).arg(reply->error())
	.arg(processErrorMessage(*proc)));
}

void Downloader::procFinished()
{
    if (proc->exitStatus() != QProcess::NormalExit) {
	abort(QSL("External download failed: process exited abnormally"));
	return;
    } else if (proc->exitCode() != 0) {
	QByteArray output = proc->readAll();
	abort(QSL("External download failed: (%1) %2").arg(proc->exitCode())
	    .arg(QString::fromUtf8(output).trimmed()));
	return;
    }
    qDebug() << "External download successful";
    finalize();
}
#endif

void Downloader::timedout()
{
    abort(QSL("Download timed out"));
}

void Downloader::abort(const QString &str)
{
    if (reply) {
	reply->disconnect();
	reply->abort();
	reply = nullptr; // ~Downloader() will delete reply
    }
    if (proc) {
	proc->disconnect();
	proc->terminate();
	proc = nullptr; // ~Downloader() will delete proc
    }
    manager = nullptr; // ~Downloader() will delete manager
    errorStr = str;
    aborted = true;
    if (reqTimer) reqTimer->stop();
    qDebug().noquote() << "Aborting download after" << total << "bytes:" << str;
    emit error();
}
