/* 
 * Copyright 2015-2022 The Regents of the University of California
 * All rights reserved.
 * 
 * This file is part of Spoofer.
 * 
 * Spoofer is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Spoofer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Spoofer.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef SPOOFER_MANAGER_COMMON_COMMON_H
#define SPOOFER_MANAGER_COMMON_COMMON_H

#include "../../config.h"
#include <typeinfo>
#include <ctime>

#include "spoof_qt.h"

#include <QDataStream>
#include <QCoreApplication>
#include <QFile>
#include <QTextStream>
//#include <QHostAddress>
#include <QDateTime>
#include <QSettings>
#include <QList>

#include "port.h"

#ifdef UPGRADE_KEY
 #define AUTOUPGRADE_ENABLED
 // UPGRADE_CMD is a C++ string literal that will be executed in a shell
 // script.  $1 is the package file name.
 #ifndef UPGRADE_CMD
  #if defined(Q_OS_MACOS)
   #define UPGRADE_CMD "/usr/sbin/installer -package \"$1\" -target /"
  #elif defined(Q_OS_WIN32)
   // UPGRADE_CMD is not used on Windows
  #endif
 #endif
#endif

// forward declarations
QT_BEGIN_NAMESPACE
class QCommandLineParser;
QT_END_NAMESPACE

#define EVERYONE_IS_PRIVILEGED 1

// scheduler messages
enum sc_msg_type {
    SC_ERROR,
    SC_TEXT,
    SC_SCHEDULED,
    SC_PROBER_STARTED,
    SC_PROBER_FINISHED,
    SC_PROBER_ERROR,
    SC_PAUSED,
    SC_RESUMED,
    SC_DONE_CMD,
    SC_CONFIG_CHANGED,
    SC_NEED_CONFIG,
    SC_CONFIGED,
    SC_HELLO,
    SC_UPGRADE_AVAILABLE,
    SC_UPGRADE_PROGRESS,
    SC_UPGRADE_ERROR,
    SC_UPGRADE_INSTALLING
};

struct sc_msg_text {
    QString text;
    sc_msg_text(const QString &_text = QString()) : text(_text) {}
};

inline QDataStream &operator<<(QDataStream &out, const sc_msg_text &msg) {
    return out << msg.text;
}

inline QDataStream &operator>>(QDataStream &in, sc_msg_text &msg) {
    return in >> msg.text;
}

struct sc_msg_scheduled {
    // QHostAddress addr;
    qint32 when;
    sc_msg_scheduled() : /*addr(),*/ when(0) {}
};

inline QDataStream &operator<<(QDataStream &out, const sc_msg_scheduled &msg) {
    return out /* << msg.addr */ << msg.when;
}

inline QDataStream &operator>>(QDataStream &in, sc_msg_scheduled &msg) {
    return in /* >> msg.addr */ >> msg.when;
}

struct sc_msg_upgrade_available {
    int autoTime;
    bool mandatory;
    int32_t vnum;
    QString vstr;
    QString file;
    QString warning;
    sc_msg_upgrade_available(bool wantAuto = false, bool _mandatory = false,
	int32_t _vnum = 0,
	const QString &_vstr = QString(), const QString &_file = QString());
};

inline QDataStream &operator<<(QDataStream &out, const sc_msg_upgrade_available &msg) {
    return out << msg.autoTime << msg.mandatory << msg.vnum << msg.vstr << msg.file << msg.warning;
}

inline QDataStream &operator>>(QDataStream &in, sc_msg_upgrade_available &msg) {
    return in >> msg.autoTime >> msg.mandatory >> msg.vnum >> msg.vstr >> msg.file >> msg.warning;
}

#if defined(Q_OS_UNIX)
typedef int error_t;
inline error_t getLastErr() { return errno; }
#elif defined(Q_OS_WIN32)
typedef unsigned long error_t;
error_t getLastErr();
#endif

QString getErrmsg(error_t err);
inline QString getLastErrmsg() { return getErrmsg(getLastErr()); }
bool getProcessName(qint64 pid, char *buf, unsigned len);

// base class for Spoofer applications
class SpooferBase {
public:
    // A write-only QIODevice that wraps another but doesn't open it until
    // needed, and can be switched to a different device at any time.
    class OnDemandDevice Q_DECL_FINAL : public QIODevice {
    private:
	QIODevice *dev, *newdev, *fallback;
	QString newname;
	bool timestampEnabled;
	OnDemandDevice(OnDemandDevice &) NO_METHOD; // no copy-ctor
	OnDemandDevice operator=(OnDemandDevice &) NO_METHOD; // no copy-assign
    public:
	OnDemandDevice(FILE *file) :
	    QIODevice(), dev(), newdev(), fallback(), newname(), timestampEnabled()
	{
	    QFile *qfile = new QFile();
	    if (qfile) qfile->open(file, WriteOnly|Unbuffered);
	    dev = qfile;
	    this->open(WriteOnly);
	}
	qint64 readData(char *data, qint64 maxSize) OVERRIDE
	    { Q_UNUSED(data); Q_UNUSED(maxSize); return -1; }
	qint64 writeData(const char *data, qint64 maxSize) OVERRIDE;
	void setDevice(QIODevice *device, const QString &name) {
	    newname = name;
	    newdev = device;
	}
	void setDevice(QFileDevice *device) {
	    setDevice(device, device->fileName());
	}
	void setFallbackDevice(QFileDevice *device) {
	    fallback = device;
	}
	void close() OVERRIDE {
	    QIODevice::close();
	    if (dev) { delete dev; dev = nullptr; }
	    if (newdev) { delete newdev; newdev = nullptr; }
	    if (fallback) { delete fallback; fallback = nullptr; }
	}
	const std::type_info& type() { return dev ? typeid(*dev) : typeid(0); }
	void setTimestampEnabled(bool flag) { timestampEnabled = flag; }
	bool getTimestampEnabled() const { return timestampEnabled; }
    };

    class Config {
	bool forWriting;
	ATR_UNUSED_MEMBER uint8_t unused_padding[3];
    public:
	struct MemberBase {
	    const QString key;
	    QVariant defaultVal;
	    bool required;
	    const QString desc;
	    MemberBase(QString _key, QVariant _defaultVal, QString _desc = QString(), bool _hidden = false) :
		key(_key), defaultVal(_defaultVal), required(false), desc(_desc)
	    {
		if (_hidden || desc.isNull()) return;
#ifdef EVERYONE_IS_PRIVILEGED
		if (_key.startsWith(QSL("unpriv"))) return;
#endif
		members.push_back(this);
	    }
	    virtual ~MemberBase() {}
	    QVariant variant() const
		{ return settings ? settings->value(key, defaultVal) : defaultVal; }
	    virtual bool setFromString(QString value, QString &errmsg) = 0;
	    void remove()            // remove: config->foo.remove()
		{ if (settings) settings->remove(key); }
	    void setDefault(QVariant d) { defaultVal = d; }
	    bool isSet() const
		{ return settings && settings->contains(key); }
	    virtual QString optionHelpString() = 0;
	    virtual void enforce() {}
	    virtual bool validate(QVariant var, QString &errmsg)
		{ Q_UNUSED(var); Q_UNUSED(errmsg); return true; }
	};
	template <class T> struct Member : public MemberBase {
	    Member(QString _key, T _defaultVal = T(), QString _desc = QString(), bool _hidden = false) :
		MemberBase(_key, QVariant(_defaultVal), _desc, _hidden) {}
	    ~Member() {}
	    T operator()() const // get: config->foo()
		{ return variant().template value<T>(); }
	    void operator()(T value) // set: config->foo(value)
		{ if (settings) settings->setValue(key, QVariant(value)); }
	    bool setFromString(QString value, QString &errmsg) OVERRIDE {
		if (!settings) return false;
		QVariant var(value);
		int qtypeid = qMetaTypeId<T>();
		if (!var.convert(qtypeid)) {
		    errmsg = QSL("%1: can not convert \"%2\" to %3.").
			arg(key).arg(value).
			arg(QString::fromLocal8Bit(QMetaType::typeName(qtypeid)));
		    qDebug() << errmsg;
		    return false;
		}
		if (!validate(var, errmsg)) return false;
		settings->setValue(key, var);
		return true;
	    }
	    QString optionHelpString() OVERRIDE
		{ return optionHelpString(*this); }
	private:
	    template <typename U> QString optionHelpString(const Member<U> &member) {
		Q_UNUSED(member);
		return QSL("%1 from now on [\"%2\" setting or \"%3\"].").
		    arg(desc).arg(key).arg(defaultVal.toString());
	    }
	    // C++ doesn't allow an explicit template specialization inside a
	    // class, but does allow an overload.  (The unused parameter is
	    // just for overload resolution.)
	    QString optionHelpString(const Member<bool> &member) {
		Q_UNUSED(member);
		return QSL("%1 from now on (yes/no) [\"%2\" setting or %3].").
		    arg(desc).arg(key).arg(defaultVal.toInt());
	    }
	};
	struct MemberInt Q_DECL_FINAL : public Member<int> {
	    int minVal, maxVal;
	    MemberInt(QString _key, int _defaultVal, int _minVal, int _maxVal,
		QString _desc = QString(), bool _hidden = false) :
		Member(_key, _defaultVal, _desc, _hidden),
		minVal(_minVal), maxVal(_maxVal)
		{}
	private:
	    bool validate(QVariant var, QString &errmsg) OVERRIDE {
		int val = var.value<int>();
		if (val >= minVal && val <= maxVal) return true;
		errmsg = QSL("%1: value %2 out of range [%3, %4]").
		    arg(key).arg(val).arg(minVal).arg(maxVal);
		qDebug() << errmsg;
		return false;
	    }
	    void enforce() OVERRIDE {
		if (!settings) return;
		int val = variant().value<int>();
		if (val < minVal)
		    settings->setValue(key, QVariant(minVal));
		if (val > maxVal)
		    settings->setValue(key, QVariant(maxVal));
	    }
	};
	static QSettings *settings;
    public:
	static QList<MemberBase*> members;

	Member<QString> dataDir;
	Member<QString> schedulerSocketName;
	Member<bool> paused;
#if DEBUG
	Member<bool> useDevServer;
	MemberInt spooferProtocolVersion;
	Member<bool> pretendMode;
	Member<bool> standaloneMode;
	Member<bool> installerAddTaint;
	Member<bool> installerVerifySig;
	Member<bool> installerKeep;
#else // DEBUG
	bool useDevServer() { return false; }
	bool pretendMode() { return false; }
	bool standaloneMode() { return false; }
	bool installerAddTaint() { return true; }
	bool installerVerifySig() { return true; }
	bool installerKeep() { return false; }
#endif // DEBUG

#ifdef AUTOUPGRADE_ENABLED
	Member<bool> autoUpgrade;
#endif
	Member<bool> enableIPv4;
	Member<bool> enableIPv6;
	MemberInt keepLogs;
	Member<bool> sharePublic;
	Member<bool> shareRemedy;
	Member<bool> enableTLS;
	MemberInt netPollInterval;
	MemberInt delayInterval;
	MemberInt proberInterval;
	MemberInt proberRetryInterval;
	MemberInt maxRetries;
	Member<bool> unprivView;
	Member<bool> unprivTest;
	Member<bool> unprivPref;

	Config();
	void initSettings(bool forWriting = false, bool debug = false);
	~Config() { if (settings) delete settings; settings = nullptr; }

	bool error(const char *label);
	void logError(const char *label, QString msg, QString msg2 = QString());
	QString lockFileName();
	void remove();

	QString fileName() {
	    return settings ? settings->fileName() : QString();
	}

	bool isFile() {
	    if (!settings) return false;
#ifdef Q_OS_WIN32
	    if (settings->format() == QSettings::NativeFormat)
		return false; // windows registry
#endif
	    return true;
	}

	bool sync() {
	    if (!settings) return false;
	    settings->sync();
	    return !error("Config sync:");
	}

	MemberBase *find(const QString &key) {
	    for (int i = 0; i < members.size(); i++) {
		if (key.compare(members[i]->key, Qt::CaseInsensitive) == 0)
		    return members[i];
	    }
	    return nullptr;
	}

	bool hasRequiredSettings() {
	    for (auto m : members) {
		if (!m->isSet() && m->required)
		    return false;
	    }
	    return true;
	}

	void enforce() {
	    for (auto m : members) {
		m->enforce();
	    }
	}
    };

    QString appDir;
    QString appFile;
    static const QStringList *args; // alternate command line arguments
    static QString optSettings;
    static Config *config;
    static OnDemandDevice outdev;
    static OnDemandDevice errdev;
    static QTextStream spout;
    static QTextStream sperr;
    static const QString proberLogFtime;
    static const QString proberLogGlob;
    static const QString proberLogRegex;

    SpooferBase();

    virtual ~SpooferBase() {
	if (config) delete config;
    }

    static void logHandler(QtMsgType type,
	const QMessageLogContext &ctx, const QString &msg);
    static bool parseCommandLine(QCommandLineParser &clp, QString desc);
    static QString ftime_zone(const QString &fmt, const time_t *tp, const Qt::TimeSpec &spec);
    static QString ftime(const QString &fmt = QString(), const time_t *tp = nullptr)
	{ return ftime_zone(fmt, tp, Qt::LocalTime); }
    static QString ftime_utc(const QString &fmt = QString(), const time_t *tp = nullptr)
	{ return ftime_zone(fmt, tp, Qt::UTC); }

    static QSettings *findDefaultSettings(bool debug)
    {
	QSettings::Scope scope = debug ? QSettings::UserScope :
	    QSettings::SystemScope;
	// Mac expects a domain here where others expect a name.
	// QSettings() with no parameters would correctly auto-pick
	// the domain or name, but doesn't let us specify the scope.
	return new QSettings(scope,
#ifdef Q_OS_MACOS
	    QCoreApplication::organizationDomain(),
#else
	    QCoreApplication::organizationName(),
#endif
	    QCoreApplication::applicationName(), nullptr);
    }

    bool initConfig(bool _forWriting = false) {
	config->initSettings(_forWriting);
	if (config->error("Config")) // XXX ???
	    return false;
	if (!_forWriting && config->dataDir().isEmpty()) {
	    // QSettings doesn't consider a missing file an error.  But if
	    // we want read-only and dataDir is missing, something is wrong.
	    config->logError("Config", QSL("Missing \"dataDir\""),
		QSL("Make sure the scheduler is running and using the same configuration."));
	    return false;
	}
	return true;
    }
};

#endif // SPOOFER_MANAGER_COMMON_COMMON_H
