/*
 *  Tvheadend - Linux DVB Multiplex
 *
 *  Copyright (C) 2013 Adam Sutton
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "tvheadend.h"
#include "input.h"
#include "mpegts_dvb.h"
#include "queue.h"
#include "settings.h"

#include <sys/types.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <stdlib.h>
#include <dirent.h>
#include <fcntl.h>
#include <math.h>

/* **************************************************************************
 * Class definition
 * *************************************************************************/

static void
dvb_mux_delete ( mpegts_mux_t *mm, int delconf );

extern const idclass_t mpegts_mux_class;

/*
 * Generic
 */

/* Macro to define mux class str get/set */
#define dvb_mux_class_R(c, f, l, ...)\
static const void * \
dvb_mux_##c##_class_##l##_get (void *o)\
{\
  static const char *s;\
  dvb_mux_t *lm = o;\
  s = dvb_##l##2str(lm->lm_tuning.dmc_fe_##f);\
  return &s;\
}\
static int \
dvb_mux_##c##_class_##l##_set (void *o, const void *v)\
{\
  dvb_mux_t *lm = o;\
  lm->lm_tuning.dmc_fe_##f = dvb_str2##l ((const char*)v);\
  return 1;\
}\
static htsmsg_t *\
dvb_mux_##c##_class_##l##_enum (void *o)\
{\
  static const int     t[] = { __VA_ARGS__ };\
  int i;\
  htsmsg_t *m = htsmsg_create_list();\
  for (i = 0; i < ARRAY_SIZE(t); i++)\
    htsmsg_add_str(m, NULL, dvb_##l##2str(t[i]));\
  return m;\
}
#define dvb_mux_class_X(c, f, p, l, ...)\
static const void * \
dvb_mux_##c##_class_##l##_get (void *o)\
{\
  static const char *s;\
  dvb_mux_t *lm = o;\
  s = dvb_##l##2str(lm->lm_tuning.u.dmc_fe_##f.p);\
  return &s;\
}\
static int \
dvb_mux_##c##_class_##l##_set (void *o, const void *v)\
{\
  dvb_mux_t *lm = o;\
  lm->lm_tuning.u.dmc_fe_##f.p = dvb_str2##l ((const char*)v);\
  return 1;\
}\
static htsmsg_t *\
dvb_mux_##c##_class_##l##_enum (void *o)\
{\
  static const int     t[] = { __VA_ARGS__ };\
  int i;\
  htsmsg_t *m = htsmsg_create_list();\
  for (i = 0; i < ARRAY_SIZE(t); i++)\
    htsmsg_add_str(m, NULL, dvb_##l##2str(t[i]));\
  return m;\
}
#define MUX_PROP_STR(_id, _name, t, l, d)\
  .type  = PT_STR,\
  .id    = _id,\
  .name  = _name,\
  .get   = dvb_mux_##t##_class_##l##_get,\
  .set   = dvb_mux_##t##_class_##l##_set,\
  .list  = dvb_mux_##t##_class_##l##_enum,\
  .def.s = d

static const void *
dvb_mux_class_delsys_get (void *o)
{
  static const char *s;
  dvb_mux_t *lm = o;
  s = dvb_delsys2str(lm->lm_tuning.dmc_fe_delsys);
  return &s;
}

static int
dvb_mux_class_delsys_set (void *o, const void *v)
{
  const char *s = v;
  int delsys = dvb_str2delsys(s);
  dvb_mux_t *lm = o;
  if (delsys != lm->lm_tuning.dmc_fe_delsys) {
    lm->lm_tuning.dmc_fe_delsys = dvb_str2delsys(s);
    return 1;
  }
  return 0;
}

const idclass_t dvb_mux_class =
{
  .ic_super      = &mpegts_mux_class,
  .ic_class      = "dvb_mux",
  .ic_caption    = "Linux DVB Multiplex",
  .ic_properties = (const property_t[]){
    {}
  }
};

/*
 * DVB-T
 */

dvb_mux_class_X(dvbt, ofdm, bandwidth,             bw,
                     DVB_BANDWIDTH_AUTO,  DVB_BANDWIDTH_10_MHZ,
                     DVB_BANDWIDTH_8_MHZ, DVB_BANDWIDTH_7_MHZ,
                     DVB_BANDWIDTH_6_MHZ, DVB_BANDWIDTH_5_MHZ,
                     DVB_BANDWIDTH_1_712_MHZ);
dvb_mux_class_R(dvbt, modulation,                  qam,
                     DVB_MOD_QAM_AUTO, DVB_MOD_QPSK, DVB_MOD_QAM_16,
                     DVB_MOD_QAM_64, DVB_MOD_QAM_256);
dvb_mux_class_X(dvbt, ofdm, transmission_mode,     mode,
                     DVB_TRANSMISSION_MODE_AUTO, DVB_TRANSMISSION_MODE_32K,
                     DVB_TRANSMISSION_MODE_16K, DVB_TRANSMISSION_MODE_8K,
                     DVB_TRANSMISSION_MODE_2K, DVB_TRANSMISSION_MODE_1K);
dvb_mux_class_X(dvbt, ofdm, guard_interval,        guard,
                     DVB_GUARD_INTERVAL_AUTO, DVB_GUARD_INTERVAL_1_32,
                     DVB_GUARD_INTERVAL_1_16, DVB_GUARD_INTERVAL_1_8,
                     DVB_GUARD_INTERVAL_1_4, DVB_GUARD_INTERVAL_1_128,
                     DVB_GUARD_INTERVAL_19_128, DVB_GUARD_INTERVAL_19_256);
dvb_mux_class_X(dvbt, ofdm, hierarchy_information, hier,
                     DVB_HIERARCHY_AUTO, DVB_HIERARCHY_NONE,
                     DVB_HIERARCHY_1, DVB_HIERARCHY_2, DVB_HIERARCHY_4);
dvb_mux_class_X(dvbt, ofdm, code_rate_HP,          fechi,
                     DVB_FEC_AUTO, DVB_FEC_1_2, DVB_FEC_2_3, DVB_FEC_3_4,
                     DVB_FEC_3_5,  DVB_FEC_4_5, DVB_FEC_5_6, DVB_FEC_7_8);
dvb_mux_class_X(dvbt, ofdm, code_rate_LP,          feclo,
                     DVB_FEC_AUTO, DVB_FEC_1_2, DVB_FEC_2_3, DVB_FEC_3_4,
                     DVB_FEC_3_5,  DVB_FEC_4_5, DVB_FEC_5_6, DVB_FEC_7_8);

#define dvb_mux_dvbt_class_delsys_get dvb_mux_class_delsys_get
#define dvb_mux_dvbt_class_delsys_set dvb_mux_class_delsys_set

static htsmsg_t *
dvb_mux_dvbt_class_delsys_enum (void *o)
{
  htsmsg_t *list = htsmsg_create_list();
  htsmsg_add_str(list, NULL, dvb_delsys2str(DVB_SYS_DVBT));
  htsmsg_add_str(list, NULL, dvb_delsys2str(DVB_SYS_DVBT2));
  htsmsg_add_str(list, NULL, dvb_delsys2str(DVB_SYS_TURBO));
  return list;
}

static int
dvb_mux_dvbt_class_frequency_set ( void *o, const void *v )
{
  dvb_mux_t *lm = o;
  uint32_t val = *(uint32_t *)v;

  if (val < 1000)
    val *= 1000000;
  else if (val < 1000000)
    val *= 1000;

  if (val != lm->lm_tuning.dmc_fe_freq) {
    lm->lm_tuning.dmc_fe_freq = val;
    return 1;
  }
  return 0;
}

const idclass_t dvb_mux_dvbt_class =
{
  .ic_super      = &dvb_mux_class,
  .ic_class      = "dvb_mux_dvbt",
  .ic_caption    = "Linux DVB-T Multiplex",
  .ic_properties = (const property_t[]){
    {
      MUX_PROP_STR("delsys", "Delivery System", dvbt, delsys, "DVBT"),
    },
    {
      .type     = PT_U32,
      .id       = "frequency",
      .name     = "Frequency (Hz)",
      .off      = offsetof(dvb_mux_t, lm_tuning.dmc_fe_freq),
      .set      = dvb_mux_dvbt_class_frequency_set,
    },
    {
      MUX_PROP_STR("bandwidth", "Bandwidth", dvbt, bw, "AUTO")
    },
    {
      MUX_PROP_STR("constellation", "Constellation", dvbt, qam, "AUTO")
    },
    {
      MUX_PROP_STR("transmission_mode", "Transmission Mode", dvbt, mode, "AUTO")
    },
    {
      MUX_PROP_STR("guard_interval", "Guard Interval", dvbt, guard, "AUTO")
    },
    {
      MUX_PROP_STR("hierarchy", "Hierarchy", dvbt, hier, "AUTO"),
    },
    {
      MUX_PROP_STR("fec_hi", "FEC High", dvbt, fechi, "AUTO"),
    },
    {
      MUX_PROP_STR("fec_lo", "FEC Low", dvbt, feclo, "AUTO"),
    },
    {
      .type     = PT_INT,
      .id       = "plp_id",
      .name     = "PLP ID",
      .off      = offsetof(dvb_mux_t, lm_tuning.dmc_fe_stream_id),
      .def.i	= DVB_NO_STREAM_ID_FILTER,
    },
    {}
  }
};

/*
 * DVB-C
 */

dvb_mux_class_R(dvbc, modulation,                 qam,
                     DVB_MOD_QAM_AUTO, DVB_MOD_QAM_16, DVB_MOD_QAM_32,
                     DVB_MOD_QAM_64, DVB_MOD_QAM_128, DVB_MOD_QAM_256);
dvb_mux_class_X(dvbc, qam, fec_inner,             fec,
                     DVB_FEC_AUTO, DVB_FEC_NONE,
                     DVB_FEC_1_2, DVB_FEC_2_3, DVB_FEC_3_4, DVB_FEC_4_5,
                     DVB_FEC_5_6, DVB_FEC_8_9, DVB_FEC_9_10);

#define dvb_mux_dvbc_class_delsys_get dvb_mux_class_delsys_get
#define dvb_mux_dvbc_class_delsys_set dvb_mux_class_delsys_set
static htsmsg_t *
dvb_mux_dvbc_class_delsys_enum (void *o)
{
  htsmsg_t *list = htsmsg_create_list();
  htsmsg_add_str(list, NULL, dvb_delsys2str(DVB_SYS_DVBC_ANNEX_A));
  htsmsg_add_str(list, NULL, dvb_delsys2str(DVB_SYS_DVBC_ANNEX_B));
  htsmsg_add_str(list, NULL, dvb_delsys2str(DVB_SYS_DVBC_ANNEX_C));
  return list;
}

const idclass_t dvb_mux_dvbc_class =
{
  .ic_super      = &dvb_mux_class,
  .ic_class      = "dvb_mux_dvbc",
  .ic_caption    = "Linux DVB-C Multiplex",
  .ic_properties = (const property_t[]){
    {
      MUX_PROP_STR("delsys", "Delivery System", dvbc, delsys, "DVBC_ANNEX_AC"),
    },
    {
      .type     = PT_U32,
      .id       = "frequency",
      .name     = "Frequency (Hz)",
      .off      = offsetof(dvb_mux_t, lm_tuning.dmc_fe_freq),
      .set      = dvb_mux_dvbt_class_frequency_set,
    },
    {
      .type     = PT_U32,
      .id       = "symbolrate",
      .name     = "Symbol Rate (Sym/s)",
      .off      = offsetof(dvb_mux_t, lm_tuning.u.dmc_fe_qam.symbol_rate),
    },
    {
      MUX_PROP_STR("constellation", "Constellation", dvbc, qam, "AUTO")
    },
    {
      MUX_PROP_STR("fec", "FEC", dvbc, fec, "AUTO")
    },
    {}
  }
};

dvb_mux_class_X(dvbs, qpsk, fec_inner,             fec,
                     DVB_FEC_AUTO, DVB_FEC_NONE,
                     DVB_FEC_1_2, DVB_FEC_2_3, DVB_FEC_3_4, DVB_FEC_3_5,
                     DVB_FEC_4_5, DVB_FEC_5_6, DVB_FEC_7_8, DVB_FEC_8_9,
                     DVB_FEC_9_10);

static int
dvb_mux_dvbs_class_frequency_set ( void *o, const void *v )
{
  dvb_mux_t *lm = o;
  uint32_t val = *(uint32_t *)v;

  if (val < 100000)
    val *= 1000;

  if (val != lm->lm_tuning.dmc_fe_freq) {
    lm->lm_tuning.dmc_fe_freq = val;
    return 1;
  }
  return 0;
}

static int
dvb_mux_dvbs_class_symbol_rate_set ( void *o, const void *v )
{
  dvb_mux_t *lm = o;
  uint32_t val = *(uint32_t *)v;

  if (val < 100000)
    val *= 1000;

  if (val != lm->lm_tuning.u.dmc_fe_qpsk.symbol_rate) {
    lm->lm_tuning.u.dmc_fe_qpsk.symbol_rate = val;
    return 1;
  }
  return 0;
}

static const void *
dvb_mux_dvbs_class_polarity_get (void *o)
{
  static const char *s;
  dvb_mux_t *lm = o;
  s = dvb_pol2str(lm->lm_tuning.u.dmc_fe_qpsk.polarisation);
  return &s;
}

static int
dvb_mux_dvbs_class_polarity_set (void *o, const void *s)
{
  dvb_mux_t *lm = o;
  lm->lm_tuning.u.dmc_fe_qpsk.polarisation = dvb_str2pol((const char*)s);
  return 1;
}

static htsmsg_t *
dvb_mux_dvbs_class_polarity_enum (void *o)
{
  htsmsg_t *list = htsmsg_create_list();
  htsmsg_add_str(list, NULL, dvb_pol2str(DVB_POLARISATION_VERTICAL));
  htsmsg_add_str(list, NULL, dvb_pol2str(DVB_POLARISATION_HORIZONTAL));
  htsmsg_add_str(list, NULL, dvb_pol2str(DVB_POLARISATION_CIRCULAR_LEFT));
  htsmsg_add_str(list, NULL, dvb_pol2str(DVB_POLARISATION_CIRCULAR_RIGHT));
  return list;
}

static const void *
dvb_mux_dvbs_class_modulation_get ( void *o )
{
  static const char *s;
  dvb_mux_t *lm = o;
  s = dvb_qam2str(lm->lm_tuning.dmc_fe_modulation);
  return &s;
}

static int
dvb_mux_dvbs_class_modulation_set (void *o, const void *s)
{
  int mod = dvb_str2qam(s);
  dvb_mux_t *lm = o;
  if (mod != lm->lm_tuning.dmc_fe_modulation) {
    lm->lm_tuning.dmc_fe_modulation = mod;
    return 1;
  }
  return 0;
}

static htsmsg_t *
dvb_mux_dvbs_class_modulation_list ( void *o )
{
  htsmsg_t *list = htsmsg_create_list();
  htsmsg_add_str(list, NULL, dvb_qam2str(DVB_MOD_AUTO));
  htsmsg_add_str(list, NULL, dvb_qam2str(DVB_MOD_QPSK));
  htsmsg_add_str(list, NULL, dvb_qam2str(DVB_MOD_QAM_16));
  htsmsg_add_str(list, NULL, dvb_qam2str(DVB_MOD_PSK_8));
  htsmsg_add_str(list, NULL, dvb_qam2str(DVB_MOD_APSK_16));
  htsmsg_add_str(list, NULL, dvb_qam2str(DVB_MOD_APSK_32));
  return list;
}

static const void *
dvb_mux_dvbs_class_rolloff_get ( void *o )
{
  static const char *s;
  dvb_mux_t *lm = o;
  s = dvb_rolloff2str(lm->lm_tuning.dmc_fe_rolloff);
  return &s;
}

static int
dvb_mux_dvbs_class_rolloff_set ( void *o, const void *s )
{
  dvb_mux_t *lm = o;
  lm->lm_tuning.dmc_fe_rolloff = dvb_str2rolloff(s);
  return 1;
}

static htsmsg_t *
dvb_mux_dvbs_class_rolloff_list ( void *o )
{
  htsmsg_t *list = htsmsg_create_list();
  htsmsg_add_str(list, NULL, dvb_rolloff2str(DVB_ROLLOFF_35));
  // Note: this is a bit naff, as the below values are only relevant
  //       to S2 muxes, but currently have no way to model that
  htsmsg_add_str(list, NULL, dvb_rolloff2str(DVB_ROLLOFF_20));
  htsmsg_add_str(list, NULL, dvb_rolloff2str(DVB_ROLLOFF_25));
  htsmsg_add_str(list, NULL, dvb_rolloff2str(DVB_ROLLOFF_AUTO));
  return list;
}

static const void *
dvb_mux_dvbs_class_pilot_get ( void *o )
{
  static const char *s;
  dvb_mux_t *lm = o;
  s = dvb_pilot2str(lm->lm_tuning.dmc_fe_pilot);
  return &s;
}

static int
dvb_mux_dvbs_class_pilot_set ( void *o, const void *s )
{
  dvb_mux_t *lm = o;
  lm->lm_tuning.dmc_fe_pilot = dvb_str2pilot(s);
  return 1;
}
static htsmsg_t *
dvb_mux_dvbs_class_pilot_list ( void *o )
{
  htsmsg_t *list = htsmsg_create_list();
  htsmsg_add_str(list, NULL, dvb_pilot2str(DVB_PILOT_AUTO));
  htsmsg_add_str(list, NULL, dvb_pilot2str(DVB_PILOT_ON));
  htsmsg_add_str(list, NULL, dvb_pilot2str(DVB_PILOT_OFF));
  return list;
}

static const void *
dvb_mux_dvbs_class_pls_mode_get ( void *o )
{
  static const char *s;
  dvb_mux_t *lm = o;
  s = dvb_plsmode2str(lm->lm_tuning.dmc_fe_pls_mode);
  return &s;
}

static int
dvb_mux_dvbs_class_pls_mode_set ( void *o, const void *s )
{
  dvb_mux_t *lm = o;
  lm->lm_tuning.dmc_fe_pls_mode = dvb_str2plsmode(s);
  return 1;
}

static htsmsg_t *
dvb_mux_dvbs_class_pls_mode_list ( void *o )
{
  htsmsg_t *list = htsmsg_create_list();
  htsmsg_add_str(list, NULL, dvb_plsmode2str(DVB_PLS_ROOT));
  htsmsg_add_str(list, NULL, dvb_plsmode2str(DVB_PLS_GOLD));
  htsmsg_add_str(list, NULL, dvb_plsmode2str(DVB_PLS_COMBO));
  return list;
}

#define dvb_mux_dvbs_class_delsys_get dvb_mux_class_delsys_get
#define dvb_mux_dvbs_class_delsys_set dvb_mux_class_delsys_set

static htsmsg_t *
dvb_mux_dvbs_class_delsys_enum (void *o)
{
  htsmsg_t *list = htsmsg_create_list();
  htsmsg_add_str(list, NULL, dvb_delsys2str(DVB_SYS_DVBS));
  htsmsg_add_str(list, NULL, dvb_delsys2str(DVB_SYS_DVBS2));
  return list;
}

static const void *
dvb_mux_dvbs_class_orbital_get ( void *o )
{
  static char buf[16], *s = buf;
  dvb_mux_t *lm = o;
  if (lm->lm_tuning.u.dmc_fe_qpsk.orbital_pos == INT_MAX)
    buf[0] = '\0';
  else
    dvb_sat_position_to_str(lm->lm_tuning.u.dmc_fe_qpsk.orbital_pos, buf, sizeof(buf));
  return &s;
}

static int
dvb_mux_dvbs_class_orbital_set ( void *o, const void *s )
{
  dvb_mux_t *lm = o;
  int pos;

  pos = dvb_sat_position_from_str((const char *)s);

  if (pos != lm->lm_tuning.u.dmc_fe_qpsk.orbital_pos) {
    lm->lm_tuning.u.dmc_fe_qpsk.orbital_pos = pos;
    return 1;
  }
  return 0;
}

const idclass_t dvb_mux_dvbs_class =
{
  .ic_super      = &dvb_mux_class,
  .ic_class      = "dvb_mux_dvbs",
  .ic_caption    = "Linux DVB-S Multiplex",
  .ic_properties = (const property_t[]){
    {
      MUX_PROP_STR("delsys", "Delivery System", dvbs, delsys, "DVBS"),
    },
    {
      .type     = PT_U32,
      .id       = "frequency",
      .name     = "Frequency (kHz)",
      .off      = offsetof(dvb_mux_t, lm_tuning.dmc_fe_freq),
      .set      = dvb_mux_dvbs_class_frequency_set,
    },
    {
      .type     = PT_U32,
      .id       = "symbolrate",
      .name     = "Symbol Rate (Sym/s)",
      .off      = offsetof(dvb_mux_t, lm_tuning.u.dmc_fe_qpsk.symbol_rate),
      .set      = dvb_mux_dvbs_class_symbol_rate_set,
    },
    {
      MUX_PROP_STR("polarisation", "Polarisation", dvbs, polarity, NULL)
    },
    {
      .type     = PT_STR,
      .id       = "modulation",
      .name     = "Modulation",
      .set      = dvb_mux_dvbs_class_modulation_set,
      .get      = dvb_mux_dvbs_class_modulation_get,
      .list     = dvb_mux_dvbs_class_modulation_list,
      .def.s    = "AUTO",
    },
    {
      MUX_PROP_STR("fec", "FEC", dvbs, fec, "AUTO")
    },
    {
      .type     = PT_STR,
      .id       = "rolloff",
      .name     = "Rolloff",
      .set      = dvb_mux_dvbs_class_rolloff_set,
      .get      = dvb_mux_dvbs_class_rolloff_get,
      .list     = dvb_mux_dvbs_class_rolloff_list,
      .def.s    = "AUTO"
    },
    {
      .type     = PT_STR,
      .id       = "pilot",
      .name     = "Pilot",
      .opts     = PO_ADVANCED,
      .set      = dvb_mux_dvbs_class_pilot_set,
      .get      = dvb_mux_dvbs_class_pilot_get,
      .list     = dvb_mux_dvbs_class_pilot_list,
    },
    {
      .type     = PT_INT,
      .id       = "stream_id",
      .name     = "ISI (Stream ID)",
      .off      = offsetof(dvb_mux_t, lm_tuning.dmc_fe_stream_id),
      .def.i	= DVB_NO_STREAM_ID_FILTER,
    },
    {
      .type     = PT_STR,
      .id       = "pls_mode",
      .name     = "PLS Mode",
      .set      = dvb_mux_dvbs_class_pls_mode_set,
      .get      = dvb_mux_dvbs_class_pls_mode_get,
      .list     = dvb_mux_dvbs_class_pls_mode_list,
      .def.s    = "ROOT",
    },
    {
      .type     = PT_U32,
      .id       = "pls_code",
      .name     = "PLS Code",
      .off      = offsetof(dvb_mux_t, lm_tuning.dmc_fe_pls_code),
      .def.u32	= 1,
    },
    {
      .type     = PT_STR,
      .id       = "orbital",
      .name     = "Orbital Pos.",
      .set      = dvb_mux_dvbs_class_orbital_set,
      .get      = dvb_mux_dvbs_class_orbital_get,
      .opts     = PO_ADVANCED | PO_RDONLY
    },
    {}
  }
};

#define dvb_mux_atsc_class_delsys_get dvb_mux_class_delsys_get
#define dvb_mux_atsc_class_delsys_set dvb_mux_class_delsys_set

static htsmsg_t *
dvb_mux_atsc_class_delsys_enum (void *o)
{
  htsmsg_t *list = htsmsg_create_list();
  htsmsg_add_str(list, NULL, dvb_delsys2str(DVB_SYS_ATSC));
  htsmsg_add_str(list, NULL, dvb_delsys2str(DVB_SYS_ATSCMH));
  return list;
}

dvb_mux_class_R(atsc, modulation, qam,
                     DVB_MOD_QAM_AUTO, DVB_MOD_QAM_256, DVB_MOD_VSB_8);

const idclass_t dvb_mux_atsc_class =
{
  .ic_super      = &dvb_mux_class,
  .ic_class      = "dvb_mux_atsc",
  .ic_caption    = "Linux ATSC Multiplex",
  .ic_properties = (const property_t[]){
    {
      MUX_PROP_STR("delsys", "Delivery System", atsc, delsys, "ATSC"),
    },
    {
      .type     = PT_U32,
      .id       = "frequency",
      .name     = "Frequency (Hz)",
      .off      = offsetof(dvb_mux_t, lm_tuning.dmc_fe_freq),
      .set      = dvb_mux_dvbt_class_frequency_set,
    },
    {
      MUX_PROP_STR("modulation", "Modulation", atsc, qam, "AUTO")
    },
    {}
  }
};

/* **************************************************************************
 * Class methods
 * *************************************************************************/

static void
dvb_mux_config_save ( mpegts_mux_t *mm )
{
  htsmsg_t *c = htsmsg_create_map();
  mpegts_mux_save(mm, c);
  hts_settings_save(c, "input/dvb/networks/%s/muxes/%s/config",
                    idnode_uuid_as_str(&mm->mm_network->mn_id),
                    idnode_uuid_as_str(&mm->mm_id));
  htsmsg_destroy(c);
}

static void
dvb_mux_display_name ( mpegts_mux_t *mm, char *buf, size_t len )
{
  dvb_mux_t *lm = (dvb_mux_t*)mm;
  dvb_network_t *ln = (dvb_network_t*)mm->mm_network;
  uint32_t freq = lm->lm_tuning.dmc_fe_freq, freq2;
  char extra[8], buf2[5], *p;
  if (ln->ln_type == DVB_TYPE_S) {
    const char *s = dvb_pol2str(lm->lm_tuning.u.dmc_fe_qpsk.polarisation);
    if (s) extra[0] = *s;
    extra[1] = '\0';
  } else {
    freq /= 1000;
    strcpy(extra, "MHz");
  }
  freq2 = freq % 1000;
  freq /= 1000;
  snprintf(buf2, sizeof(buf2), "%03d", freq2);
  p = buf2 + 2;
  while (freq2 && (freq2 % 10) == 0) {
    freq2 /= 10;
    *(p--) = '\0';
  }
  if (freq2)
    snprintf(buf, len, "%d.%s%s", freq, buf2, extra);
  else
    snprintf(buf, len, "%d%s", freq, extra);
}

static void
dvb_mux_create_instances ( mpegts_mux_t *mm )
{
  mpegts_network_link_t *mnl;
  LIST_FOREACH(mnl, &mm->mm_network->mn_inputs, mnl_mn_link) {
    mpegts_input_t *mi = mnl->mnl_input;
    if (mi->mi_is_enabled(mi, mm, 0))
      mi->mi_create_mux_instance(mi, mm);
  }
}

static void
dvb_mux_delete ( mpegts_mux_t *mm, int delconf )
{
  /* Remove config */
  if (delconf)
    hts_settings_remove("input/dvb/networks/%s/muxes/%s",
                      idnode_uuid_as_str(&mm->mm_network->mn_id),
                      idnode_uuid_as_str(&mm->mm_id));

  /* Delete the mux */
  mpegts_mux_delete(mm, delconf);
}

/* **************************************************************************
 * Creation/Config
 * *************************************************************************/

dvb_mux_t *
dvb_mux_create0
  ( dvb_network_t *ln,
    uint16_t onid, uint16_t tsid, const dvb_mux_conf_t *dmc,
    const char *uuid, htsmsg_t *conf )
{
  const idclass_t *idc;
  mpegts_mux_t *mm;
  dvb_mux_t *lm;
  htsmsg_t *c, *e;
  htsmsg_field_t *f;
  dvb_fe_delivery_system_t delsys;

  /* Class */
  if (ln->ln_type == DVB_TYPE_S) {
    idc = &dvb_mux_dvbs_class;
    delsys = DVB_SYS_DVBS;
  } else if (ln->ln_type == DVB_TYPE_C) {
    idc = &dvb_mux_dvbc_class;
    delsys = DVB_SYS_DVBC_ANNEX_A;
  } else if (ln->ln_type == DVB_TYPE_T) {
    idc = &dvb_mux_dvbt_class;
    delsys = DVB_SYS_DVBT;
  } else if (ln->ln_type == DVB_TYPE_ATSC) {
    idc = &dvb_mux_atsc_class;
    delsys = DVB_SYS_ATSC;
  } else {
    tvherror("dvb", "unknown FE type %d", ln->ln_type);
    return NULL;
  }

  /* Create */
  mm = calloc(1, sizeof(dvb_mux_t));
  lm = (dvb_mux_t*)mm;

  /* Defaults */
  dvb_mux_conf_init(&lm->lm_tuning, delsys);

  /* Parent init and load config */
  if (!(mm = mpegts_mux_create0(mm, idc, uuid,
                                (mpegts_network_t*)ln, onid, tsid, conf))) {
    free(mm);
    return NULL;
  }

  /* Tuning */
  if (dmc)
    memcpy(&lm->lm_tuning, dmc, sizeof(dvb_mux_conf_t));
  lm->lm_tuning.dmc_fe_type = ln->ln_type;

  /* Callbacks */
  lm->mm_delete           = dvb_mux_delete;
  lm->mm_display_name     = dvb_mux_display_name;
  lm->mm_config_save      = dvb_mux_config_save;
  lm->mm_create_instances = dvb_mux_create_instances;

  /* No config */
  if (!conf) return lm;

  /* Services */
  c = hts_settings_load_r(1, "input/dvb/networks/%s/muxes/%s/services",
                         idnode_uuid_as_str(&ln->mn_id),
                         idnode_uuid_as_str(&mm->mm_id));
  if (c) {
    HTSMSG_FOREACH(f, c) {
      if (!(e = htsmsg_get_map_by_field(f))) continue;
      mpegts_service_create1(f->hmf_name, (mpegts_mux_t *)lm, 0, 0, e);
    }
    htsmsg_destroy(c);
  }

  if (ln->ln_type == DVB_TYPE_S) {
    if (ln->mn_satpos == INT_MAX) {
      /* Update the satellite position for the network settings */
      if (lm->lm_tuning.u.dmc_fe_qpsk.orbital_pos != INT_MAX)
        ln->mn_satpos = lm->lm_tuning.u.dmc_fe_qpsk.orbital_pos;
    }
    else {
      /* Update the satellite position for the mux setting */
      lm->lm_tuning.u.dmc_fe_qpsk.orbital_pos = ln->mn_satpos;
    }
  }

  return lm;
}
