// Font Renderer Using Native API version 1.20
//
// Copylight (C) 2006,2007 mocchi
//
//  This software is provided 'as-is', without any express or implied
//  warranty.  In no event will the authors be held liable for any damages
//  arising from the use of this software.
//
//  Permission is granted to anyone to use this software for any purpose,
//  including commercial applications, and to alter it and redistribute it
//  freely, subject to the following restrictions:
//
//  1. The origin of this software must not be misrepresented; you must not
//     claim that you wrote the original software. If you use this software
//     in a product, an acknowledgment in the product documentation would be
//     appreciated but is not required.
//  2. Altered source versions must be plainly marked as such, and must not be
//     misrepresented as being the original software.
//  3. This notice may not be removed or altered from any source distribution.
//
// mocchi mocchi_2003@yahoo.co.jp
//

#ifndef IG_FONT_RENDERER_H_
#define IG_FONT_RENDERER_H_

#include <map>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <locale.h>

#ifdef _WIN32
#include <windows.h>
#include <mbstring.h>
#else
#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xlocale.h>
#endif

class font_renderer{
public:
#ifdef _WIN32
	typedef BYTE byte;
#else
	typedef uint8_t byte;
#endif
	struct rgb_type{
		byte r, g, b;
		rgb_type() : r(0), g(0), b(0){};
		rgb_type(byte r_, byte g_, byte b_) : r(r_), g(g_), b(b_){};
	};
private:
	byte *image;
	int width, height;
	void destroy(){
		width = height = 0;
		delete image;
		image = 0;
	}
	struct vals_internal{
		int width, height;
#ifdef _WIN32
		COLORREF rgb2RGB(rgb_type &rgb){
			return RGB(rgb.r, rgb.g, rgb.b);
		}
		HBITMAP hBmp, hBmpOld;
		HDC hDCBmp;
		BYTE *pixelint;
		HFONT hFont, hFontOld;
		HBRUSH hBrush;
		int pix_width_prev, pix_height_prev;
		bool first;
		int ascender, descender;
		int height_prev;
		char attribute_prev;
		void create(char *str, int height_want, bool italic, bool bold, bool fixed, rgb_type &background, rgb_type &foreground){
			int len = strlen(str);
			if (hDCBmp == NULL){
				HWND hWnd = GetDesktopWindow();
				HDC hDC = GetDC(hWnd);
				hDCBmp = CreateCompatibleDC(hDC);
				ReleaseDC(hWnd, hDC);
			}
			SetTextColor(hDCBmp, rgb2RGB(foreground));
			SetBkColor(hDCBmp, rgb2RGB(background));

			char attribute = (italic ? 1 : 0) | (bold ? 2 : 0) | (fixed ? 4 : 0);
			if (!hFont || height_prev != height || attribute != attribute_prev){
				attribute_prev = attribute;
				height_prev = height_want;
				if (hFont){
					SelectObject(hDCBmp, hFontOld);
					DeleteObject(hFont);
				}
				hFont = CreateFont(height_want, 0, 0, 0, bold ? FW_BOLD : FW_DONTCARE, italic ? TRUE : FALSE, 
					FALSE, FALSE, DEFAULT_CHARSET, OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY,
					fixed ? (FIXED_PITCH | FF_DONTCARE) : (VARIABLE_PITCH | FF_DONTCARE), NULL);
				hFontOld = (HFONT)SelectObject(hDCBmp, hFont);
			}

			{
				SIZE sz;
				GetTextExtentPoint32A(hDCBmp, str, len, &sz);
				width = ((sz.cx + 3) / 4) * 4;
				height = sz.cy;
			}

			if (pix_width_prev < width || pix_height_prev < height){
				if (hBmp){
					SelectObject(hDCBmp, hBmpOld);
					DeleteObject(hBmp);
				}
				pix_width_prev = width * 2;
				pix_height_prev = height * 2;
				BITMAPINFO bi;
				ZeroMemory(&bi, sizeof(bi));
				bi.bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
				bi.bmiHeader.biBitCount = 24;
				bi.bmiHeader.biPlanes = 1;
				bi.bmiHeader.biWidth = pix_width_prev;
				bi.bmiHeader.biHeight = -pix_height_prev;
				hBmp = CreateDIBSection(NULL, &bi, DIB_RGB_COLORS, (void **)&pixelint, NULL, 0);
				hBmpOld = (HBITMAP)SelectObject(hDCBmp, hBmp);
			}

			{
				HBRUSH hBrush = CreateSolidBrush(rgb2RGB(background));
				RECT rc;
				rc.left = rc.top = 0;
				rc.right = pix_width_prev;
				rc.bottom = pix_height_prev;
				FillRect(hDCBmp, &rc, hBrush);
			}

			TextOutA(hDCBmp, 0, 0, str, len);
			TEXTMETRIC tm;
			GetTextMetricsA(hDCBmp,&tm);
			ascender = tm.tmAscent;
			descender = tm.tmDescent;
		}

		vals_internal(char *str, int height_want, bool italic = false,
			bool bold = false, bool fixed = false, rgb_type background = rgb_type(), rgb_type foreground = rgb_type()){
			first = true;
			hFont = NULL;
			hDCBmp = 0;
			hBmpOld = 0;
			hBmp = 0;
			hDCBmp = 0;
			pixelint = 0;
			height_prev = -1;
			attribute_prev = 0;
			create(str, height_want, italic, bold, fixed, background, foreground);
			first = false;
		}
				
		inline int get_ascender(){
			return ascender;
		}

		inline int get_descender(){
			return descender;
		}

		inline void get_pixel(int x, int y, byte &r, byte &g, byte &b){
			byte *p = pixelint + (y * pix_width_prev + x) * 3;
			r = *(p+2), g = *(p+1), b = *p;
		}

		void destroy(){
			SelectObject(hDCBmp, hBmpOld);
			DeleteObject(hBmp);
			SelectObject(hDCBmp, hFontOld);
			DeleteObject(hFont);
			DeleteDC(hDCBmp);
			hFont = NULL;
			hDCBmp = 0;
			hBmpOld = 0;
			hBmp = 0;
			hDCBmp = 0;
			pixelint = 0;
		}
		~vals_internal(){
			destroy();
		}
#else
		XImage *ximg;
		Display *d;
		GC gc;
		XFontSet fs;
		Pixmap pix;
		Colormap cmap;
		int ascender, descender;
		int pix_width_prev, pix_height_prev;
		char fontset_prev[256];
		unsigned long rgb2color(rgb_type col, Display *d, Colormap &cmap){
			XColor xcol;
			xcol.red = col.r * 257;
			xcol.green = col.g * 257;
			xcol.blue = col.b * 257;
			XAllocColor(d, cmap, &xcol);
			return xcol.pixel;
		}
		void create(char *str, int height_want, bool italic, bool bold, bool fixed, rgb_type background, rgb_type foreground){
			int len = strlen(str);
			setlocale(LC_CTYPE, "");
			if (d == NULL){
				d = XOpenDisplay(NULL);
				cmap = DefaultColormap(d, DefaultScreen(d));
			}
			char fontset[256];
			height_want = height_want*4/5;
			sprintf(fontset, "-*-*-%s-%c-normal--%d-*-*-*-%c",
				bold ? "bold" : "medium", italic ? 'i' : 'r', height_want, fixed ? 'm' : 'p');
			bool equal_font;
			if (strcmp(fontset, fontset_prev) == 0){
				equal_font = true;
			}else{
				equal_font = false;
				strcpy(fontset_prev, fontset);
			}

			char **mlist;
			int mcount;
			char *def_str;
			if (!equal_font){
				if (fs){
					XFreeFontSet(d, fs);
				}
				fs = XCreateFontSet(d, fontset, &mlist, &mcount, &def_str);

				XFontSetExtents *extent;
				extent = XExtentsOfFontSet(fs);
				ascender = -extent->max_logical_extent.y;
				descender = extent->max_logical_extent.height - ascender;
				XFreeStringList(mlist);
			}
			XRectangle ink, logical;
			XmbTextExtents (fs, str, len, &ink, &logical);
			width = logical.width;
			height = height_want;

			if (pix == None || pix_width_prev < width || pix_height_prev < height){
				if (pix != None){
					XFreeGC(d, gc);
					XFreePixmap(d, pix);
				}
				pix_width_prev = width * 2;
				pix_height_prev = height * 2;
				pix = XCreatePixmap(d, DefaultRootWindow(d), pix_width_prev, pix_height_prev, XDefaultDepth(d, DefaultScreen(d)));
				gc = XCreateGC(d, pix, 0, NULL);
			}

			unsigned long backcolor = rgb2color(background, d, cmap);
			XSetForeground(d, gc, backcolor);
			XSetBackground(d, gc, backcolor);
			XFillRectangle(d, pix, gc, 0, 0, width, height);
			XSetForeground(d, gc, rgb2color(foreground, d, cmap));
			XmbDrawString(d, pix, fs, gc, 0, ascender, str, len);

			if (ximg) XDestroyImage(ximg);
			ximg = XGetImage(d, pix, 0, 0, width, height, AllPlanes, ZPixmap );
		}

		vals_internal(char *str, int height_want, bool italic = false,
			bool bold = false, bool fixed = false, rgb_type background = rgb_type(), rgb_type foreground = rgb_type()){
			fontset_prev[0] = '\0';
			ximg = NULL;
			d = NULL;
			pix = None;
			fs = NULL;
			ascender = descender = -1;
			pix_width_prev = pix_height_prev = -1;
			create(str, height_want, italic, bold, fixed, background, foreground);
		}

		inline int get_ascender(){
			return ascender;
		}

		inline int get_descender(){
			return descender;
		}

		std::map<unsigned long,rgb_type> col2rgb;
		rgb_type color2rgb(unsigned long color, Display *d, Colormap &cmap){
			if (col2rgb.count(color)){
				return col2rgb[color];
			}else{
				XColor xcol;
				xcol.pixel = color;
				XQueryColor(d, cmap, &xcol);
				rgb_type rgb_((byte)(xcol.red/257), (byte)(xcol.green/257), (byte)(xcol.blue/257));
				col2rgb[color] = rgb_;
				return rgb_;
			}
		}
		inline void get_pixel(int x, int y, byte &r, byte &g, byte &b){
			rgb_type c = color2rgb(XGetPixel(ximg,x,y), d, cmap);
			r = c.r, g = c.g, b = c.b;
		}

		~vals_internal(){
			XDestroyImage(ximg);

			XFreeGC(d, gc);
			XFreeFontSet(d, fs);
			XFreePixmap(d, pix);
			XCloseDisplay(d);
		}
#endif
	};

	struct image_size_setter{
		void operator()(int &width, int &height){
		}
	};

	int ascender, descender;
	vals_internal *vi;
public:
	font_renderer() : image(0), width(0), height(0){
		ascender = descender = 0;
		vi = NULL;
	}

	font_renderer(char *str, int height_want, bool italic = false, bool bold = false, bool fixed = false,			rgb_type background = rgb_type(0,0,0), rgb_type foreground = rgb_type(255,255,255)){
		render(str, height_want, italic, bold, fixed, background, foreground);
	}

	void render(char *str, int height_want,
			bool italic = false, bool bold = false, bool fixed = false,
			rgb_type background = rgb_type(0,0,0), rgb_type foreground = rgb_type(255,255,255)){
		if (vi == NULL){
			vi = new vals_internal(str, height_want, italic, bold, fixed, background, foreground);
		}else{
			vi->create(str, height_want, italic, bold, fixed, background, foreground);
		}
		width = vi->width, height = vi->height;
		image = new byte[width * height * 3];
		ascender = vi->get_ascender();
		descender = vi->get_descender();

		int h = height, w = width;
		for (int j = 0, i3 = 0; j < h; ++j){
			for (int i = 0; i < w; ++i, i3 += 3){
				vi->get_pixel(i, j, image[i3], image[i3+1], image[i3+2]);
			}
		}
	}

	~font_renderer(){
		if (vi) delete vi;
		destroy();
	}
	int get_width(){
		return width;
	}
	int get_height(){
		return height;
	}
	inline int get_ascender(){
		return ascender;
	}
	inline int get_descender(){
		return descender;
	}

	const byte *get_image(){
		return image;
	}
};

// history
// 2006/07/10 ver 1.10 Win32  X11 ŋʂĂ镔̏d𖳂
// 2007/08/16 ver 1.20
//		create\bh쐬D̃\bhŕ`sƁC
//		̕`̏OƓƂ̓EBhE̍ĐȗłD

#endif // IG_FONT_RENDERER_H_
