/***************************************************************************
           CocoaPainter.mm - Cocoa (Mac OS X) class for painting
                             -------------------
    begin                : Thu Oct 17 21:59:18 CEST 2002
    copyright            : (C) 2002 by Pier Philipsen
    email                : pier@localhost.localdomain
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#import "CocoaPainter.h"

CocoaPainter::CocoaPainter(NSView *view):
    mInteractionContainer(mPPlot)
{
    mView = view; // Do not retain. Could cause a retain cycle.
    mFillColor = [[NSColor blackColor] retain];
    mLineColor = [[NSColor blackColor] retain];
    mFont = [[NSFont userFontOfSize:12.0] retain];
}

CocoaPainter::~CocoaPainter()
{
    [mFillColor release];
    [mLineColor release];
    [mFont release];
}

void CocoaPainter::paint(void) {
    mPPlot.Draw(*this);
}

long CocoaPainter::GetWidth () const {
    return (long)[mView bounds].size.width;
}

long CocoaPainter::GetHeight () const {
    return (long)[mView bounds].size.height;
}

long CocoaPainter::GetFontHeight () const {
    return (long) ceil( [mFont ascender] - [mFont descender] );
}

void CocoaPainter::DrawLine (float inX1, float inY1, float inX2, float inY2){
    [mLineColor set];
    [NSBezierPath strokeLineFromPoint:NSMakePoint( inX1, inY1) toPoint:NSMakePoint( inX2, inY2 )];
}

void CocoaPainter::FillRect (int inX, int inY, int inW, int inH) {
    [mFillColor set];
    [NSBezierPath fillRect:NSMakeRect( inX + 0.5, inY + 0.5, inW + 0.5, inH + 0.5 )];
}

void CocoaPainter::InvertRect (int inX, int inY, int inW, int inH) {
    [[[NSColor lightGrayColor] colorWithAlphaComponent:0.2] set];
    [NSBezierPath fillRect:NSMakeRect( inX + 0.5, inY + 0.5, inW + 0.5, inH + 0.5 )];
} 

void CocoaPainter::SetClipRect (int inX, int inY, int inW, int inH) {
    [[NSBezierPath bezierPathWithRect:NSMakeRect( inX + 0.5, inY + 0.5, inW + 0.5, inH + 0.5 )] setClip];
}

void CocoaPainter::SetLineColor (int inR, int inG, int inB) {
    [mLineColor release];
    mLineColor = [[NSColor colorWithCalibratedRed:( inR / 256.0) 
    green:( inG / 256.0 )
    blue: ( inB / 256.0 )
    alpha: 1.0 ] retain];
}

void CocoaPainter::SetFillColor (int inR, int inG, int inB) {
    [mFillColor release];
    mFillColor = [[NSColor colorWithCalibratedRed:( inR / 256.0) 
        green:( inG / 256.0 )
        blue: ( inB / 256.0 )
        alpha: 1.0 ] retain];
}

long CocoaPainter::CalculateTextDrawSize (const char *inString) {
    NSSize stringSize = [[NSString stringWithCString:inString] sizeWithAttributes:
        [NSDictionary dictionaryWithObjectsAndKeys:mFont, NSFontAttributeName, nil]];
    return (long)stringSize.width;
}

void CocoaPainter::DrawText (int inX, int inY, const char *inString) {
    // The signs of the 0.5 offsets used here may seem odd, but this is due to the 
    // way strings are drawn in cocoa. The NSString class automatically flips the
    // string for you, so the y offset is negative.
    NSPoint point = NSMakePoint( inX + 0.5, inY - GetFontHeight() - 0.5 );
    NSDictionary *fontAttribsDict = 
        [NSDictionary dictionaryWithObjectsAndKeys:mFont, NSFontAttributeName, nil];
    NSString *cocoaString = [NSString stringWithCString:inString];
    
    // Draw the cocoa string
    [cocoaString drawAtPoint:point withAttributes:fontAttribsDict];
}

void CocoaPainter::DrawRotatedText (int inX, int inY, float inDegrees, const char *inString) {
    NSAffineTransform *trans = [NSAffineTransform transform];
    float x = inX + 0.5, y = inY - 0.5;
    [trans translateXBy:x yBy:y];
    [trans rotateByDegrees:inDegrees];
    [trans concat];
    this->DrawText( 0, 0, inString );
    [trans invert];
    [trans concat];
}

void CocoaPainter::SetStyle (const PStyle &inStyle) {
    NSString *fontName = [NSString stringWithCString:inStyle.mFont.c_str()];
    [NSBezierPath setDefaultLineWidth:(float)inStyle.mPenWidth];
    [mFont release];
    if ( nil == fontName || [fontName isEqual:@""] ) {
        mFont = [[NSFont userFontOfSize:(float)inStyle.mFontSize] retain];
    }
    else {
        mFont = [[NSFont fontWithName:fontName size:(float)inStyle.mFontSize] retain];
    }
}

