
#include "QP3DPlot.h"

static int timer_interval = 10;			// timer interval (millisec)


#if defined(Q_CC_MSVC)
#pragma warning(disable:4305) // init: truncation from const double to float
#endif

/*!
  Create a QP3DPlot widget
*/


QP3DPlot::QP3DPlot( QWidget* parent, const char* name )
    : QGLWidget( parent, name )
{
  mZRot = 0;
  mScale = 1;			// default object scale
  mObject = 0;
  mMode = kMode_None;
  mAutoRotate = false;
    
    startTimer( timer_interval );
}


/*!
  Release allocated resources
*/

QP3DPlot::~QP3DPlot()
{
    makeCurrent();
    glDeleteLists( mObject, 1 );
}


/*!
  Paint the box. The actual openGL commands for drawing the box are
  performed here.
*/

void QP3DPlot::paintGL()
{
    GLenum errCode = glGetError ();
    const GLubyte *errString;

    if (errCode != GL_NO_ERROR) {
      errString = gluErrorString (errCode);
      fprintf (stderr, "Open gl error %s\n", errString);
    }
    glClear( GL_COLOR_BUFFER_BIT  | GL_DEPTH_BUFFER_BIT);

    glLoadIdentity();
    gluLookAt (4.0,4.0,2.0,  0.0,0.0,0.0,  0.0,0.0,1.0);

    glRotatef(mZRot, 0.0, 0.0, 1.0 );

    glCallList( mObject );
}


/*!
  Set up the OpenGL rendering state, and define display list
*/

void QP3DPlot::initializeGL()
{
    GLfloat mat_specular[] = { 1.0, 1.0, 1.0, 1.0 };
    GLfloat mat_shininess[] = { 50.0 };
    GLfloat light_position[] = { 1.0, 1.0, 1.0, 0.0 };

    glLightfv(GL_LIGHT0, GL_POSITION, light_position);

    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);

    glEnable(GL_DEPTH_TEST);
    qglClearColor( white ); 		// Let OpenGL clear to black
    mObject = makeObject();		// Generate an OpenGL display list
    glShadeModel( GL_FLAT );
}



/*!
  Set up the OpenGL view port, matrix mode, etc.
*/

void QP3DPlot::resizeGL( int w, int h )
{
    glViewport( 0, 0, (GLint)w, (GLint)h );
    glMatrixMode( GL_PROJECTION );
    glLoadIdentity();
    glFrustum( -1.0, 1.0, -1.0, 1.0, 5.0, 15.0 );
    glMatrixMode( GL_MODELVIEW );
}


/*!
  Generate an OpenGL display list for the object to be shown, i.e. the box
*/

GLuint QP3DPlot::makeObject()
{	
    GLuint list;

    list = glGenLists( 1 );

    glNewList( list, GL_COMPILE );

    qglColor( white );		      // Shorthand for glColor3f or glIndex

    glLineWidth( 2.0 );

    mPPlot.MakeModel (*this);


    glEndList();


    return list;
}

void QP3DPlot::setZRotation( int degrees )
{
    mZRot = (GLfloat)(degrees % 360);
    updateGL();
}
void QP3DPlot::MakeDemoPlot (int inIndex) {
  //  glFinish ();
  glDeleteLists (mObject, 1);

  if (inIndex == 0) {
    MakeExamplePlot1 (mPPlot);
  }
  if (inIndex == 1) {
    MakeExamplePlot2 (mPPlot);
  }
  mObject = makeObject();
  updateGL ();
}

void QP3DPlot::timerEvent(QTimerEvent*)
{
  if (!mAutoRotate) {
    return;
  }
    updateGL();
    mZRot += 2;
}


void QP3DPlot::MakeLine (const PVertex &inVertex1, const PVertex &inVertex2) {
  GLfloat theAmbLine [] = {1,1,1,1};
  GLfloat theAmbDefault [] = {0.2,0.2,0.2,1};
  glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT, theAmbLine);
  SetMode (kMode_Line);
  glBegin( GL_LINES );
  AddVertex (inVertex1);
  AddVertex (inVertex2);
  glEnd();
  glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT, theAmbDefault);
}

void QP3DPlot::MakePolygon (const vector<PVertex> &inVertices) {

  if (inVertices.size ()!= 4) {
    return;
  }
    SetMode (kMode_Rect);
    glBegin( GL_QUADS );
    for (int theI=0;theI<inVertices.size ();theI++) {
      const PVertex & theVertex = inVertices[theI];
      AddVertex (theVertex);
    }
    glEnd();
}

void QP3DPlot::SetColor (int inR, int inG, int inB) {
  float theFac = 1.0/255;
  glColor3f (inR*theFac, inG*theFac, inB*theFac);
}

void QP3DPlot::SetSurfaceNormal (const PVertex &inVertex) {
  glNormal3f (inVertex.mX, inVertex.mY, inVertex.mZ );
}

void QP3DPlot::Finish () {
  SetMode (kMode_None);
}

void QP3DPlot::AddVertex (const PVertex &inVertex) {
    glVertex3f(  inVertex.mX, inVertex.mY, inVertex.mZ );
}


void QP3DPlot::SetMode (EMode inMode) {
  return;
    if (inMode == mMode) {
    return;
    }
  if (mMode != kMode_None) {
    glEnd ();
  }

  switch (inMode) {
  case kMode_Line:
    glBegin (GL_LINES);
    break;
  case kMode_Rect:
    glBegin (GL_QUADS);
    break;
  case kMode_None:
    glEnd ();
    break;
  }
  mMode = inMode;
}
