/*
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *    Tertius.java
 *    Copyright (C) 2003 Peter A. Flach, Nicolas Lachiche
 *
 *    Thanks to Amelie Deltour for porting the original C code to Java
 *    and integrating it into Weka.
 */

package weka.associations;

import weka.associations.tertius.*;
import weka.core.Instances;
import weka.core.Attribute;
import weka.core.AttributeStats;
import weka.core.SelectedTag;
import weka.core.OptionHandler;
import weka.core.Option;
import weka.core.Tag;
import weka.core.Utils;
import weka.associations.Associator;
import java.util.ArrayList;
import java.util.Vector;
import java.util.Enumeration;
import java.util.Date;
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.Reader;
import java.io.File;
import java.text.SimpleDateFormat;
import java.awt.Frame;
import java.awt.TextField;
import java.awt.Font;
import java.awt.Button;
import java.awt.Label;
import java.awt.BorderLayout;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;

/**
 * Class implementing a Tertius-type algorithm. <p>
 * 
 * References: P. A. Flach, N. Lachiche (1999). <i>Confirmation-Guided 
 * Discovery of first-order rules with Tertius</i>. 
 * Machine Learning, 42, 61-95. <p>
 * 
 * Valid options are:<p>
 *
 * -K number of values in result <br>
 * Set maximum number of confirmation  values in the result. (default: 10) <p>
 *
 * -F frequency threshold <br>
 * Set frequency threshold for pruning. (default: 0) <p>
 *
 * -C confirmation threshold <br>
 * Set confirmation threshold. (default: 0) <p>
 *
 * -N noise threshold <br>
 * Set noise threshold : maximum frequency of counter-examples.
 * 0 gives only satisfied rules. (default: 1) <p>
 *
 * -R <br>
 * Allow attributes to be repeated in a same rule. <p>
 *
 * -L number of literals <br>
 * Set maximum number of literals in a rule. (default: 4) <p>
 *
 * -G 0=no negation | 1=body | 2=head | 3=body and head <br>
 * Set the negations in the rule. (default: 0) <p>
 *
 * -S <br>
 * Consider only classification rules. <p>
 *
 * -c class index <br>
 * Set index of class attribute. (default: last). <p>
 *
 * -H <br>
 * Consider only horn clauses. <p>
 *
 * -E <br>
 * Keep equivalent rules. <p>
 *
 * -M <br>
 * Keep same clauses. <p>
 *
 * -T <br>
 * Keep subsumed rules. <p>
 *
 * -I 0=always match | 1=never match | 2=significant <br>
 * Set the way to handle missing values. (default: 0) <p>
 *
 * -O <br>
 * Use ROC analysis. <p>
 *
 * -p name of file <br>
 * Set the file containing the parts of the individual for individual-based 
 * learning. <p>
 *
 * -P 0=no output | 1=on stdout | 2=in separate window <br>
 * Set output of current values. (default: 0) <p>
 *
 * @author <a href="mailto:adeltour@netcourrier.com">Amelie Deltour</a>
 * @version $Revision: 1.3 $
 */

public class Tertius extends Associator implements OptionHandler, Runnable {

  /** The results. */
  private SimpleLinkedList m_results;

  /** Number of hypotheses considered. */
  private int m_hypotheses;

  /** Number of hypotheses explored. */
  private int m_explored;

  /** Time needed for the search. */
  private Date m_time;

  /** Field to output the current values. */ 
  private TextField m_valuesText;

  /** Instances used for the search. */
  private Instances m_instances;

  /** Predicates used in the rules. */
  private ArrayList m_predicates;

  /** Status of the search. */
  private int m_status;
  private static final int NORMAL = 0;
  private static final int MEMORY = 1; // memory problem
  private static final int STOP = 2; // user interruption
  
  /* Pruning options. */

  /** Number of best confirmation values to search. */
  private int m_best;

  /** Frequency threshold for the body and the negation of the head. */
  private double m_frequencyThreshold;

  /** Confirmation threshold for the rules. */
  private double m_confirmationThreshold;

  /** Maximal number of counter-instances. */
  private double m_noiseThreshold;

  /* Search space & language bias options. */

  /** Repeat attributes ? */
  private boolean m_repeat;

  /** Number of literals in a rule. */
  private int m_numLiterals;

  /** Types of negation. */
  private static final int NONE = 0;
  private static final int BODY = 1;
  private static final int HEAD = 2;
  private static final int ALL = 3;
  private static final Tag [] TAGS_NEGATION = {
    new Tag(NONE, weka.LocalString.get("None")),
    new Tag(BODY, weka.LocalString.get("Body")),
    new Tag(HEAD, weka.LocalString.get("Head")),
    new Tag(ALL, weka.LocalString.get("Both"))
      };

  /** Type of negation used in the rules. */
  private int m_negation;

  /** Classification bias. */
  private boolean m_classification;

  /** Index of class attribute. */
  private int m_classIndex;

  /** Horn clauses bias. */
  private boolean m_horn;

  /* Subsumption tests options. */

  /** Perform test on equivalent rules ? */
  private boolean m_equivalent;

  /** Perform test on same clauses ? */
  private boolean m_sameClause;
  
  /** Perform subsumption test ? */
  private boolean m_subsumption;

  /** Ways of handling missing values.  */
  public static final int EXPLICIT = 0; // min counterinstances
  public static final int IMPLICIT = 1; // max counterinstances
  public static final int SIGNIFICANT = 2; // missing as a particular value
  private static final Tag [] TAGS_MISSING = {
    new Tag(EXPLICIT, weka.LocalString.get("Matches all")),
    new Tag(IMPLICIT, weka.LocalString.get("Matches none")),
    new Tag(SIGNIFICANT, weka.LocalString.get("Significant"))
      };

  /** Way of handling missing values in the search. */
  private int m_missing;

  /** Perform ROC analysis ? */
  private boolean m_roc;

  /** Name of the file containing the parts for individual-based learning. */
  private String m_partsString;
  
  /** Part instances for individual-based learning. */
  private Instances m_parts;

  /* Types of values output. */ 
  private static final int NO = 0;
  private static final int OUT = 1;
  private static final int WINDOW = 2;
  private static final Tag [] TAGS_VALUES = {
    new Tag(NO, "No"),
    new Tag(OUT, weka.LocalString.get("stdout")),
    new Tag(WINDOW, weka.LocalString.get("Window"))
      };

  /** Type of values output. */
  private int m_printValues;

  /**
   * Constructor that sets the options to the default values.
   */
  public Tertius() {

    resetOptions();
  }

  /**
   * Returns a string describing this associator.
   *
   * @return A description of the evaluator suitable for
   * displaying in the explorer/experimenter gui.
   */
  public String globalInfo() {

    return weka.LocalString.get("Finds rules according to confirmation measure.");
  }


  /**
   * Resets the options to the default values.
   */
  public void resetOptions() {

    /* Pruning options. */
    m_best = 10;
    m_frequencyThreshold = 0;
    m_confirmationThreshold = 0;
    m_noiseThreshold = 1;

    /* Search space & language bias options. */
    m_repeat = false;
    m_numLiterals = 4;
    m_negation = NONE;
    m_classification = false;
    m_classIndex = 0;
    m_horn = false;

    /* Subsumption tests options. */
    m_equivalent = true;
    m_sameClause = true;
    m_subsumption = true;

    /* Missing values. */
    m_missing = EXPLICIT;

    /* ROC analysis. */
    m_roc = false;

    /* Individual-based learning. */
    m_partsString = "";
    m_parts = null;

    /* Values output. */
    m_printValues = NO;
  }

  /**
   * Returns an enumeration describing the available options.
   *
   * @return An enumeration of all the available options.
   */
  public Enumeration listOptions() {
    
    Vector newVector = new Vector(17);

    /* Pruning options. */
    newVector.addElement(new Option(weka.LocalString.get("\tSet maximum number of confirmation  ")
				    + weka.LocalString.get("values in the result. (default: 10)"),
				    "K", 1, weka.LocalString.get("-K <number of values in result>")));
    newVector.addElement(new Option(weka.LocalString.get("\tSet frequency threshold for pruning. ")
				    + weka.LocalString.get("(default: 0)"),
				    "F", 1, weka.LocalString.get("-F <frequency threshold>")));
    newVector.addElement(new Option(weka.LocalString.get("\tSet confirmation threshold. ")
				    + weka.LocalString.get("(default: 0)"),
				    "C", 1, weka.LocalString.get("-C <confirmation threshold>")));
    newVector.addElement(new Option(weka.LocalString.get("\tSet noise threshold : maximum frequency ")
				    + weka.LocalString.get("of counter-examples.\n\t0 gives only ")
				    + weka.LocalString.get("satisfied rules. (default: 1)"),
				    "N", 1, weka.LocalString.get("-N <noise threshold>")));

    /* Search space & language bias options. */
    newVector.addElement(new Option(weka.LocalString.get("\tAllow attributes to be repeated in a ")
				    + weka.LocalString.get("same rule."),
				    "R", 0, "-R"));
    newVector.addElement(new Option(weka.LocalString.get("\tSet maximum number of literals in a ")
				    + weka.LocalString.get("rule. (default: 4)"),
				    "L", 1, weka.LocalString.get("-L <number of literals>")));
    newVector.addElement(new Option(weka.LocalString.get("\tSet the negations in the rule. ")
				    + weka.LocalString.get("(default: 0)"),
				    "G", 1, weka.LocalString.get("-G <0=no negation | ")
				    + weka.LocalString.get("1=body | ")
				    + weka.LocalString.get("2=head | ")
				    + weka.LocalString.get("3=body and head>")));
    newVector.addElement(new Option(weka.LocalString.get("\tConsider only classification rules."),
				    "S", 0, "-S"));
    newVector.addElement(new Option(weka.LocalString.get("\tSet index of class attribute. ")
				    + weka.LocalString.get("(default: last)."),
				    "c", 1, weka.LocalString.get("-c <class index>")));
    newVector.addElement(new Option(weka.LocalString.get("\tConsider only horn clauses."),
				    "H", 0, "-H"));

    /* Subsumption tests options. */
    newVector.addElement(new Option(weka.LocalString.get("\tKeep equivalent rules."),
				    "E", 0, "-E"));
    newVector.addElement(new Option(weka.LocalString.get("\tKeep same clauses."),
				    "M", 0, "-M"));
    newVector.addElement(new Option(weka.LocalString.get("\tKeep subsumed rules."),
				    "T", 0, "-T"));

    /* Missing values options. */
    newVector.addElement(new Option(weka.LocalString.get("\tSet the way to handle missing values. ") 
				    + weka.LocalString.get("(default: 0)"),
				    "I", 1, weka.LocalString.get("-I <0=always match | ")
				    + weka.LocalString.get("1=never match | ")
				    + weka.LocalString.get("2=significant>")));

    /* ROC analysis. */
    newVector.addElement(new Option(weka.LocalString.get("\tUse ROC analysis. "),
				    "O", 0, "-O"));

    /* Individual-based learning. */
    newVector.addElement(new Option(weka.LocalString.get("\tSet the file containing the parts of ")
				    + weka.LocalString.get("the individual for individual-based ")
				    + weka.LocalString.get("learning."),
				    "p", 1, weka.LocalString.get("-p <name of file>")));

    /* Values output. */
    newVector.addElement(new Option(weka.LocalString.get("\tSet output of current values. ")
				    + weka.LocalString.get("(default: 0)"),
				    "P", 1, weka.LocalString.get("-P <0=no output | ")
				    + weka.LocalString.get("1=on stdout | ")
				    + weka.LocalString.get("2=in separate window>")));
    
    return newVector.elements();
  }
  
  /**
   * Parses a given list of options.
   *
   * @param options The list of options as an array of strings.
   * @exception Exception if an option is not supported.
   */
  public void setOptions(String [] options) throws Exception {
    
    resetOptions();
    
    /* Pruning options. */
    String bestString = Utils.getOption('K', options);
    if (bestString.length() != 0) {
      try {
	m_best = Integer.parseInt(bestString);
      } catch (Exception e) {
	throw new Exception(weka.LocalString.get("Invalid value for -K option: ")
			    + e.getMessage() + ".");
      }
      if (m_best < 1) {
	throw new Exception(weka.LocalString.get("Number of confirmation values has to be ")
			    + weka.LocalString.get("greater than one!"));
      }
    }
    String frequencyThresholdString = Utils.getOption('F', options);
    if (frequencyThresholdString.length() != 0) {
      try {	
	m_frequencyThreshold 
	  = (new Double(frequencyThresholdString)).doubleValue();
      } catch (Exception e) {
	throw new Exception(weka.LocalString.get("Invalid value for -F option: ")
			    + e.getMessage() + ".");
      }
      if (m_frequencyThreshold < 0 || m_frequencyThreshold > 1) {
	throw new Exception(weka.LocalString.get("Frequency threshold has to be between ")
			    + weka.LocalString.get("zero and one!"));
      }
    }
    String confirmationThresholdString = Utils.getOption('C', options);
    if (confirmationThresholdString.length() != 0) {
      try {
	m_confirmationThreshold 
	  = (new Double(confirmationThresholdString)).doubleValue();
      } catch (Exception e) {
	throw new Exception(weka.LocalString.get("Invalid value for -C option: ")
			    + e.getMessage() + ".");
      }
      if (m_confirmationThreshold < 0 || m_confirmationThreshold > 1) {
	throw new Exception(weka.LocalString.get("Confirmation threshold has to be between ")
			    + weka.LocalString.get("zero and one!"));
      }
      if (bestString.length() != 0) {
	throw new Exception(weka.LocalString.get("Specifying both a number of confirmation ")
			    + weka.LocalString.get("values and a confirmation threshold ")
			    + weka.LocalString.get("doesn't make sense!"));
      }
      if (m_confirmationThreshold != 0) {
	m_best = 0;
      }
    }
    String noiseThresholdString = Utils.getOption('N', options);
    if (noiseThresholdString.length() != 0) {
      try {
	m_noiseThreshold = (new Double(noiseThresholdString)).doubleValue();
      } catch (Exception e) {
	throw new Exception(weka.LocalString.get("Invalid value for -N option: ")
			    + e.getMessage() + ".");
      }
      if (m_noiseThreshold < 0 || m_noiseThreshold > 1) {
	throw new Exception(weka.LocalString.get("Noise threshold has to be between ")
			    + weka.LocalString.get("zero and one!"));
      }
    }

    /* Search space and language bias options. */
    m_repeat = Utils.getFlag('R', options);
    String numLiteralsString = Utils.getOption('L', options);
    if (numLiteralsString.length() != 0) {
      try {
	m_numLiterals = Integer.parseInt(numLiteralsString);
      } catch (Exception e) {
	throw new Exception(weka.LocalString.get("Invalid value for -L option: ")
			    + e.getMessage() + ".");
      }
      if (m_numLiterals < 1) {
	throw new Exception(weka.LocalString.get("Number of literals has to be ")
			    + weka.LocalString.get("greater than one!"));
      }
    }
    String negationString = Utils.getOption('G', options);
    if (negationString.length() != 0) {
      SelectedTag selected;
      int tag;
      try {
	tag = Integer.parseInt(negationString);
      } catch (Exception e) {
	throw new Exception(weka.LocalString.get("Invalid value for -G option: ")
			    + e.getMessage() + ".");
      }
      try {
	selected = new SelectedTag(tag, TAGS_NEGATION);
      } catch (Exception e) {
	throw new Exception(weka.LocalString.get("Value for -G option has to be ")
			    + weka.LocalString.get("between zero and three!"));
      }
      setNegation(selected);
    }
    m_classification = Utils.getFlag('S', options);
    String classIndexString = Utils.getOption('c', options);
    if (classIndexString.length() != 0) {
      try {
	m_classIndex = Integer.parseInt(classIndexString);
      } catch (Exception e) {
	throw new Exception(weka.LocalString.get("Invalid value for -c option: ")
			    + e.getMessage() + ".");
      }
    }
    m_horn = Utils.getFlag('H', options);
    if (m_horn && (m_negation != NONE)) {
      throw new Exception(weka.LocalString.get("Considering horn clauses doesn't make sense ")
			  + weka.LocalString.get("if negation allowed!"));
    }
    
    /* Subsumption tests options. */
    m_equivalent = !(Utils.getFlag('E', options));
    m_sameClause = !(Utils.getFlag('M', options));
    m_subsumption = !(Utils.getFlag('T', options));

    /* Missing values options. */
    String missingString = Utils.getOption('I', options);
    if (missingString.length() != 0) {
      SelectedTag selected;
      int tag;
      try {
	tag = Integer.parseInt(missingString);
      } catch (Exception e) {
	throw new Exception(weka.LocalString.get("Invalid value for -I option: ")
			    + e.getMessage() + ".");
      }
      try {
	selected = new SelectedTag(tag, TAGS_MISSING);
      } catch (Exception e) {
	throw new Exception(weka.LocalString.get("Value for -I option has to be ")
			    + weka.LocalString.get("between zero and two!"));
      }
      setMissingValues(selected);
    }

    /* ROC analysis. */
    m_roc = Utils.getFlag('O', options);


    /* Individual-based learning. */
    m_partsString = Utils.getOption('p', options);
    if (m_partsString.length() != 0) {
      Reader reader;
      try {
	reader = new BufferedReader(new FileReader(m_partsString));
      } catch (Exception e) {
	throw new Exception(weka.LocalString.get("Can't open file ") + e.getMessage() + ".");
      }
      m_parts = new Instances(reader);	
    }

    /* Values output. */
    String printValuesString = Utils.getOption('P', options);
    if (printValuesString.length() != 0) {
      SelectedTag selected;
      int tag;
      try {
	tag = Integer.parseInt(printValuesString);
      } catch (Exception e) {
	throw new Exception(weka.LocalString.get("Invalid value for -P option: ")
			    + e.getMessage() + ".");
      }
      try {
	selected = new SelectedTag(tag, TAGS_VALUES);
      } catch (Exception e) {
	throw new Exception(weka.LocalString.get("Value for -P option has to be ")
			    + weka.LocalString.get("between zero and two!"));
      }
      setValuesOutput(selected);
    }
  }

  /**
   * Gets the current settings of the Tertius object.
   *
   * @return An array of strings suitable for passing to setOptions.
   */
  public String [] getOptions() {
    
    String [] options = new String [24];
    int current = 0;
    
    /* Pruning options. */
    options[current++] = "-K"; options[current++] = "" + m_best;
    options[current++] = "-F"; options[current++] = "" + m_frequencyThreshold;
    options[current++] = "-C"; options[current++] = "" + m_confirmationThreshold;
    options[current++] = "-N"; options[current++] = "" + m_noiseThreshold;
    /* Search space and language bias options. */
    if (m_repeat) {
      options[current++] = "-R";
    }
    options[current++] = "-L"; options[current++] = "" + m_numLiterals;
    options[current++] = "-G"; options[current++] = "" + m_negation;
    if (m_classification) {
      options[current++] = "-S";
    }
    options[current++] = "-c"; options[current++] = "" + m_classIndex;
    if (m_horn) {
      options[current++] = "-H";
    }
    /* Subsumption tests options. */
    if (!m_equivalent) {
      options[current++] = "-E";
    }
    if (!m_sameClause) {
      options[current++] = "-M";
    }
    if (!m_subsumption) {
      options[current++] = "-T";
    }

    /* Missing values options. */
    options[current++] = "-I"; options[current++] = "" + m_missing;

    /* ROC analysis. */
    if (m_roc) {
      options[current++] = "-O";
    }

    /* Individual-based learning. */
    options[current++] = "-p"; options[current++] = "" + m_partsString;

    /* Values output. */
    options[current++] = "-P"; options[current++] = "" + m_printValues;

    while (current < options.length) {
      options[current++] = "";
    }
    return options;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return Tip text for this property suitable for
   * displaying in the explorer/experimenter GUI.
   */
  public String confirmationValuesTipText() {

    return weka.LocalString.get("Number of best confirmation values to find.");
  }

  /**
   * Get the value of confirmationValues.
   *
   * @return Value of confirmationValues.
   */
  public int getConfirmationValues() {

    return m_best;
  }

  /**
   * Set the value of confirmationValues.
   *
   * @param v  Value to assign to confirmationValues.
   */
  public void setConfirmationValues(int v) throws Exception {

    m_best = v;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return Tip text for this property suitable for
   * displaying in the explorer/experimenter GUI.
   */
  public String frequencyThresholdTipText() {
    
    return weka.LocalString.get("Minimum proportion of instances satisfying head and body of rules");
  }

  /**
   * Get the value of frequencyThreshold.
   *
   * @return Value of frequencyThreshold.
   */
  public double getFrequencyThreshold() {
    
    return m_frequencyThreshold;
  }

  /**
   * Set the value of frequencyThreshold.
   *
   * @param v  Value to assign to frequencyThreshold.
   */
  public void setFrequencyThreshold(double v) {
    
    m_frequencyThreshold = v;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return Tip text for this property suitable for
   * displaying in the explorer/experimenter GUI.
   */
  public String confirmationThresholdTipText() {
    
    return weka.LocalString.get("Minimum confirmation of the rules.");
  }

  /**
   * Get the value of confirmationThreshold.
   *
   * @return Value of confirmationThreshold.
   */
  public double getConfirmationThreshold() {
    
    return m_confirmationThreshold;
  }

  /**
   * Set the value of confirmationThreshold.
   *
   * @param v  Value to assign to confirmationThreshold.
   */
  public void setConfirmationThreshold(double v) {
    
    m_confirmationThreshold = v;
    if (v != 0) {
      m_best = 0;
    }
  }

  /**
   * Returns the tip text for this property.
   *
   * @return Tip text for this property suitable for
   * displaying in the explorer/experimenter GUI.
   */
  public String noiseThresholdTipText() {
    
    return weka.LocalString.get("Maximum proportion of counter-instances of rules. ")
      + weka.LocalString.get("If set to 0, only satisfied rules will be given.");
  }

  /**
   * Get the value of noiseThreshold.
   *
   * @return Value of noiseThreshold.
   */
  public double getNoiseThreshold() {
    
    return m_noiseThreshold;
  }

  /**
   * Set the value of noiseThreshold.
   *
   * @param v  Value to assign to noiseThreshold.
   */
  public void setNoiseThreshold(double v) {
    
    m_noiseThreshold = v;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return Tip text for this property suitable for
   * displaying in the explorer/experimenter GUI.
   */
  public String repeatLiteralsTipText() {

    return weka.LocalString.get("Repeated attributes allowed.");
  }

  /**
   * Get the value of repeatLiterals.
   *
   * @return Value of repeatLiterals.
   */
  public boolean getRepeatLiterals() {
    
    return m_repeat;
  }

  /**
   * Set the value of repeatLiterals.
   *
   * @param v  Value to assign to repeatLiterals.
   */
  public void setRepeatLiterals(boolean v) {
    
    m_repeat = v;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return Tip text for this property suitable for
   * displaying in the explorer/experimenter GUI.
   */
  public String numberLiteralsTipText() {
    
    return weka.LocalString.get("Maximum number of literals in a rule.");
  }

  /**
   * Get the value of numberLiterals.
   *
   * @return Value of numberLiterals.
   */
  public int getNumberLiterals() {

    return m_numLiterals;
  }

  /**
   * Set the value of numberLiterals.
   *
   * @param v  Value to assign to numberLiterals.
   */
  public void setNumberLiterals(int v) {
    
    m_numLiterals = v;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return Tip text for this property suitable for
   * displaying in the explorer/experimenter GUI.
   */
  public String negationTipText() {
    
    return weka.LocalString.get("Set the type of negation allowed in the rule. ")
      + weka.LocalString.get("Negation can be allowed in the body, in the head, in both ")
      + weka.LocalString.get("or in none.");
  }

  /**
   * Get the value of negation.
   *
   * @return Value of negation.
   */
  public SelectedTag getNegation() {
    
    return new SelectedTag(m_negation, TAGS_NEGATION);
  }

  /**
   * Set the value of negation.
   *
   * @param v  Value to assign to negation.
   */
  public void setNegation(SelectedTag v) {
    
    if (v.getTags() == TAGS_NEGATION) {
      m_negation = v.getSelectedTag().getID();
    }
  }

  /**
   * Returns the tip text for this property.
   *
   * @return Tip text for this property suitable for
   * displaying in the explorer/experimenter GUI.
   */
  public String classificationTipText() {
    
    return weka.LocalString.get("Find only rules with the class in the head.");
  }

  /**
   * Get the value of classification.
   *
   * @return Value of classification.
   */
  public boolean getClassification() {

    return m_classification;
  }

  /**
   * Set the value of classification.
   *
   * @param v  Value to assign to classification.
   */
  public void setClassification(boolean v) {

    m_classification = v;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return Tip text for this property suitable for
   * displaying in the explorer/experimenter GUI.
   */
  public String classIndexTipText() {

    return weka.LocalString.get("Index of the class attribute. If set to 0, the class will be the last attribute.");
  }

  /**
   * Get the value of classIndex.
   *
   * @return Value of classIndex.
   */
  public int getClassIndex() {

    return m_classIndex;
  }

  /**
   * Set the value of classIndex.
   *
   * @param v  Value to assign to classIndex.
   */
  public void setClassIndex(int v) {

    m_classIndex = v;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return Tip text for this property suitable for
   * displaying in the explorer/experimenter GUI.
   */
  public String hornClausesTipText() {
    
    return weka.LocalString.get("Find rules with a single conclusion literal only.");
  }

  /**
   * Get the value of hornClauses.
   *
   * @return Value of hornClauses.
   */
  public boolean getHornClauses() {

    return m_horn;
  }

  /**
   * Set the value of hornClauses.
   *
   * @param v  Value to assign to hornClauses.
   */
  public void setHornClauses(boolean v) {

    m_horn = v;
  }
  
  /**
   * Returns the tip text for this property.
   *
   * @return Tip text for this property suitable for
   * displaying in the explorer/experimenter GUI.
   */
  public String equivalentTipText() {
    
    return weka.LocalString.get("Keep equivalent rules. ")
      + weka.LocalString.get("A rule r2 is equivalent to a rule r1 if the body of r2 is the ")
      + weka.LocalString.get("negation of the head of r1, and the head of r2 is the ")
      + weka.LocalString.get("negation of the body of r1.");
  }

  /**
   * Get the value of equivalent.
   *
   * @return Value of equivalent.
   */
  public boolean disabled_getEquivalent() {
    
    return !m_equivalent;
  }

  /**
   * Set the value of equivalent.
   *
   * @param v  Value to assign to equivalent.
   */
  public void disabled_setEquivalent(boolean v) {
    
    m_equivalent = !v;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return Tip text for this property suitable for
   * displaying in the explorer/experimenter GUI.
   */
  public String sameClauseTipText() {

    return weka.LocalString.get("Keep rules corresponding to the same clauses. ")
      + weka.LocalString.get("If set to false, only the rule with the best confirmation ")
      + weka.LocalString.get("value and rules with a lower number of counter-instances ")
      + weka.LocalString.get("will be kept.");
  }

  /**
   * Get the value of sameClause.
   *
   * @return Value of sameClause.
   */
  public boolean disabled_getSameClause() {

    return !m_sameClause;
  }

  /**
   * Set the value of sameClause.
   *
   * @param v  Value to assign to sameClause.
   */
  public void disabled_setSameClause(boolean v) {

    m_sameClause = !v;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return Tip text for this property suitable for
   * displaying in the explorer/experimenter GUI.
   */
  public String subsumptionTipText() {

    return weka.LocalString.get("Keep subsumed rules. ")
      + weka.LocalString.get("If set to false, subsumed rules will only be kept if they ")
      + weka.LocalString.get("have a better confirmation or a lower number of counter-instances.");
  }

  /**
   * Get the value of subsumption.
   *
   * @return Value of subsumption.
   */
  public boolean disabled_getSubsumption() {

    return !m_subsumption;
  }

  /**
   * Set the value of subsumption.
   *
   * @param v  Value to assign to subsumption.
   */
  public void disabled_setSubsumption(boolean v) {

    m_subsumption = !v;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return Tip text for this property suitable for
   * displaying in the explorer/experimenter GUI.
   */
  public String missingValuesTipText() {
    
    return weka.LocalString.get("Set the way to handle missing values. ")
      + weka.LocalString.get("Missing values can be set to match any value, or never match values ")
      + weka.LocalString.get("or to be significant and possibly appear in rules.");
  }

  /**
   * Get the value of missingValues.
   *
   * @return Value of missingValues.
   */
  public SelectedTag getMissingValues() {
    
    return new SelectedTag(m_missing, TAGS_MISSING);
  }

  /**
   * Set the value of missingValues.
   *
   * @param v  Value to assign to missingValues.
   */
  public void setMissingValues(SelectedTag v) {
    
    if (v.getTags() == TAGS_MISSING) {
      m_missing = v.getSelectedTag().getID();
    }
  }

  /**
   * Returns the tip text for this property.
   *
   * @return Tip text for this property suitable for
   * displaying in the explorer/experimenter GUI.
   */
  public String rocAnalysisTipText() {
    
    return weka.LocalString.get("Return TP-rate and FP-rate for each rule found.");
  }

  /**
   * Get the value of rocAnalysis.
   *
   * @return Value of rocAnalysis.
   */
  public boolean getRocAnalysis() {

    return m_roc;
  }

  /**
   * Set the value of rocAnalysis.
   *
   * @param v  Value to assign to rocAnalysis.
   */
  public void setRocAnalysis(boolean v) {

    m_roc = v;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return Tip text for this property suitable for
   * displaying in the explorer/experimenter GUI.
   */
  public String partFileTipText() {
    
    return weka.LocalString.get("Set file containing the parts of the individual ")
      + weka.LocalString.get("for individual-based learning.");
  }

  /**
   * Get the value of partFile.
   *
   * @return Value of partFile.
   */
  public File disabled_getPartFile() {

    return new File(m_partsString);
  }

  /**
   * Set the value of partFile.
   *
   * @param v  Value to assign to partFile.
   */
  public void disabled_setPartFile(File v) throws Exception {

    m_partsString = v.getAbsolutePath();
    if (m_partsString.length() != 0) {
      Reader reader;
      try {
	reader = new BufferedReader(new FileReader(m_partsString));
      } catch (Exception e) {
	throw new Exception(weka.LocalString.get("Can't open file ") + e.getMessage() + ".");
      }
      m_parts = new Instances(reader);	
    }
  }

  /**
   * Returns the tip text for this property.
   *
   * @return Tip text for this property suitable for
   * displaying in the explorer/experimenter GUI.
   */
  public String valuesOutputTipText() {
    
    return weka.LocalString.get("Give visual feedback during the search. ")
      + weka.LocalString.get("The current best and worst values can be output either to stdout or to a separate window.");
  }

  /**
   * Get the value of valuesOutput.
   *
   * @return Value of valuesOutput.
   */
  public SelectedTag getValuesOutput() {
    
    return new SelectedTag(m_printValues, TAGS_VALUES);
  }

  /**
   * Set the value of valuesOutput.
   *
   * @param v  Value to assign to valuesOutput.
   */
  public void setValuesOutput(SelectedTag v) {
    
    if (v.getTags() == TAGS_VALUES) {
      m_printValues = v.getSelectedTag().getID();
    }
  }

  /**
   * Build the predicate corresponding to an attribute.
   *
   * @param instances The instances this predicates comes from.
   * @param attr The attribute this predicate corresponds to.
   * @param isClass Saying if the attribute is the class attribute.
   * @return The corresponding Predicate.
   * @exception Exception if the predicate could not be build 
   * (the attribute is numeric).
   */
  private Predicate buildPredicate(Instances instances,
				   Attribute attr, boolean isClass) 
    throws Exception {

    Predicate predicate; /* The result. */
    Literal lit;
    Literal negation;
    boolean missingValues; /* Missing values for this attribute ? */
    boolean individual = (m_parts != null); /* Individual-based learning ? */
    int type = (instances == m_parts)
      ? IndividualLiteral.PART_PROPERTY
      : IndividualLiteral.INDIVIDUAL_PROPERTY; /* Type of property. */

    if (attr.isNumeric()) {
      throw new Exception(weka.LocalString.get("Can't handle numeric attributes!"));
    }
	
    missingValues = instances.attributeStats(attr.index()).missingCount > 0;

    /* Build predicate. */
    if (individual) {
      predicate = new Predicate(instances.relationName() + "." + attr.name(), 
				attr.index(), isClass);
    } else {
      predicate = new Predicate(attr.name(), attr.index(), isClass);
    }
	
    if (attr.numValues() == 2
	&& (!missingValues || m_missing == EXPLICIT)) {
      /* Case of two values.
       * If there are missing values, this case is treated like other cases.
       */
      if (individual) {
	lit = new IndividualLiteral(predicate, attr.value(0), 0, 
				    Literal.POS, m_missing, type);
	negation = new IndividualLiteral(predicate, attr.value(1), 1, 
					 Literal.POS, m_missing, type);
      } else {
	lit = new AttributeValueLiteral(predicate, attr.value(0), 0, 
					Literal.POS, m_missing);
	negation = new AttributeValueLiteral(predicate, attr.value(1), 1, 
					     Literal.POS, m_missing);
      }
      lit.setNegation(negation);
      negation.setNegation(lit);
      predicate.addLiteral(lit);      
    } else {
      /* Case of several values. */
      for (int i = 0; i < attr.numValues(); i++) {
	if (individual) {
	  lit = new IndividualLiteral(predicate, attr.value(i), i,
				      Literal.POS, m_missing, type);
	} else {
	  lit = new AttributeValueLiteral(predicate, attr.value(i), i, 
					  Literal.POS, m_missing);
	}
	if (m_negation != NONE) {
	  if (individual) {
	    negation = new IndividualLiteral(predicate, attr.value(i), i, 
					     Literal.NEG, m_missing, type);
	  } else {
	    negation = new AttributeValueLiteral(predicate, attr.value(i), i, 
						 Literal.NEG, m_missing);
	  }
	  lit.setNegation(negation);
	  negation.setNegation(lit);
	}
	predicate.addLiteral(lit);
      }

      /* One more value if missing is significant. */
      if (missingValues && m_missing == SIGNIFICANT) {
	if (individual) {
	  lit = new IndividualLiteral(predicate, "?", -1, 
				      Literal.POS, m_missing, type);
	} else {
	  lit = new AttributeValueLiteral(predicate, "?", -1, 
					  Literal.POS, m_missing);
	}
	if (m_negation != NONE) {
	  if (individual) {
	    negation = new IndividualLiteral(predicate, "?", -1, 
					     Literal.NEG, m_missing, type);
	  } else {
	    negation = new AttributeValueLiteral(predicate, "?", -1, 
						 Literal.NEG, m_missing);
	  }
	  lit.setNegation(negation);
	  negation.setNegation(lit);
	}
	predicate.addLiteral(lit);
      }
    }
    return predicate;
  }
   
  /*
   * Build the predicates to use in the rules.
   *
   * @exception If the predicates could not be built 
   * (numeric attribute).
   */
  private ArrayList buildPredicates() throws Exception {

    ArrayList predicates = new ArrayList(); /* The result. */
    Predicate predicate;
    Attribute attr;
    Enumeration attributes = m_instances.enumerateAttributes();
    boolean individual = (m_parts != null); /* Individual-based learning ? */

    /* Attributes. */
    while (attributes.hasMoreElements()) {
      attr = (Attribute) attributes.nextElement();
      /* Identifiers do not appear in rules in individual-based learning. */
      if (!(individual && attr.name().equals("id"))) {
	predicate = buildPredicate(m_instances, attr, false);
	predicates.add(predicate);
      }
    }
    /* Class attribute. */
    attr = m_instances.classAttribute();
    /* Identifiers do not appear in rules. */
    if (!(individual && attr.name().equals("id"))) {
      predicate = buildPredicate(m_instances, attr, true);
      predicates.add(predicate);
    }

    /* Attributes of the parts in individual-based learning. */
    if (individual) {
      attributes = m_parts.enumerateAttributes();
      while (attributes.hasMoreElements()) {
	attr = (Attribute) attributes.nextElement();
	/* Identifiers do not appear in rules. */
	if (!attr.name().equals("id")) {
	  predicate = buildPredicate(m_parts, attr, false);
	  predicates.add(predicate);
	}
      }
    }
	
    return predicates;
  }

  /**
   * Count the number of distinct confirmation values in the results.
   *
   * @return Number of confirmation values in the results.
   */
  private int numValuesInResult() {

    int result = 0;
    SimpleLinkedList.LinkedListIterator iter = m_results.iterator();
    Rule current;
    Rule next;
    if (!iter.hasNext()) {
      return result;
    } else {
      current = (Rule) iter.next();
      while (iter.hasNext()) {
	next = (Rule) iter.next();
	if (current.getConfirmation() > next.getConfirmation()) {
	  result++;
	}
	current = next;
      }
      return result + 1;
    }
  }

  /**
   * Test if it is worth refining a rule.
   *
   * @param rule The rule to consider.
   * @return True if the rule can be refined, false otherwise.
   */
  private boolean canRefine(Rule rule) {
    if (rule.isEmpty()) {
      return true;
    }
    if (m_best != 0) {
      if (numValuesInResult() < m_best) {
	return true;
      }
      Rule worstResult = (Rule) m_results.getLast();
      if (rule.getOptimistic() >= worstResult.getConfirmation()) {
	return true;
      }
      return false;
    } else {
      return true;
    }
  }

  /**
   * Test if it is worth calculating the optimistic estimate of a rule.
   *
   * @param rule The rule to consider.
   * @return True if the optimistic estimate can be calculated, false otherwise.
   */
  private boolean canCalculateOptimistic(Rule rule) {
    if (rule.hasTrueBody() || rule.hasFalseHead()) {
      return false;
    }
    if (!rule.overFrequencyThreshold(m_frequencyThreshold)) {
      return false;
    }
    return true;
  }

  /**
   * Test if a rule can be explored (if it is interesting for the results 
   * or for refining).
   *
   * @param rule The rule to consider.
   * @return True if the rule can be explored, false otherwise.
   */
  private boolean canExplore(Rule rule) {
    if (rule.getOptimistic() < m_confirmationThreshold) {
      return false;
    }
    if (m_best != 0) {
      if (numValuesInResult() < m_best) {
	return true;
      }	
      Rule worstResult = (Rule) m_results.getLast();
      if (rule.getOptimistic() >= worstResult.getConfirmation()) {
	return true;
      }
      return false;      
    } else {
      return true;
    }
  }    

  /**
   * Test if a rule can be stored in the agenda.
   *
   * @param rule The rule to consider.
   * @return True if the rule can be stored, false otherwise.
   */
  private boolean canStoreInNodes(Rule rule) {
    if (rule.getObservedNumber() == 0) {
      return false;
    }
    return true;
  }

  /**
   * Test if it is worth calculating the confirmation of a rule.
   *
   * @param rule The rule to consider.
   * @return True if the confirmation can be calculated, false otherwise.
   */
  private boolean canCalculateConfirmation(Rule rule) {
    if (rule.getObservedFrequency() > m_noiseThreshold) {
      return false;
    }
    return true;
  }

  /**
   * Test if a rule can be added to the results.
   *
   * @param rule The rule to consider.
   * @return True if the rule can be stored, false otherwise.
   */
  private boolean canStoreInResults(Rule rule) {
    if (rule.getConfirmation() < m_confirmationThreshold) {
      return false;
    }
    if (m_best != 0) {
      if (numValuesInResult() < m_best) {
	return true;
      }
      Rule worstResult = (Rule) m_results.getLast();
      if (rule.getConfirmation() >= worstResult.getConfirmation()) {
	return true;
      }
      return false;    
    } else {
      return true;
    }
  }

  /**
   * Add a rule in the appropriate place in the list of the results, 
   * according to the confirmation and 
   * number of counter-instances of the rule. <p>
   * Subsumption tests are performed and the rule may not be added. <p>
   * Previous results may also be removed because of sumption.
   */
  private void addResult(Rule rule) {

    Rule current;
    boolean added = false;

    /* Iterate the list until we find the right place. */
    SimpleLinkedList.LinkedListIterator iter = m_results.iterator();
    while (iter.hasNext()) {
      current = (Rule) iter.next();
      if (Rule.confirmationThenObservedComparator.compare(current, rule) > 0) {
	iter.addBefore(rule);
	added = true;
	break;
      }
      /* Subsumption tests to see if the rule can be added. */
      if ((m_subsumption || m_sameClause || m_equivalent)
	  && current.subsumes(rule)) {
	if (current.numLiterals() == rule.numLiterals()) {
	  if (current.equivalentTo(rule)) {
	    /* Equivalent rules. */
	    if (m_equivalent) {
	      return;
	    }
	  } else {
	    /* Same clauses. */
	    if (m_sameClause
		&& Rule.confirmationComparator.compare(current, rule) < 0) {
	      return;
	    }
	  }
	} else {
	  /* Subsumption. */
	  if (m_subsumption
	      && Rule.observedComparator.compare(current, rule) <= 0) {	
	    return;
	  }
	}
      }
    }

    if (added == false) {
      /* The rule must be added in the end of the results. */
      m_results.add(rule);
    }

    /* Iterate the results with a lower confirmation 
     *  to see if some of them must be removed. */
    SimpleLinkedList.LinkedListInverseIterator inverse 
      = m_results.inverseIterator();
    while (inverse.hasPrevious()) {
      current = (Rule) inverse.previous();
      if (Rule.confirmationThenObservedComparator.compare(current, rule) < 0) {
	break;
      }
      if (current != rule && rule.subsumes(current)) {
	if (current.numLiterals() == rule.numLiterals()) {
	  if (!current.equivalentTo(rule)) {
	    /* Same clauses. */
	    if (m_sameClause
		&& Rule.confirmationComparator.compare(current, rule) > 0) {
	      inverse.remove();
	    }
	  }
	} else {
	  /* Subsumption. */
	  if (m_subsumption
	      && Rule.observedComparator.compare(rule, current) <= 0) {	
	    inverse.remove();
	  }
	}	
      }
    }

    /* Remove the rules with the worst confirmation value 
     * if there are too many results. */
    if (m_best != 0 && numValuesInResult() > m_best) {
      Rule worstRule = (Rule) m_results.getLast();
      inverse = m_results.inverseIterator();
      while (inverse.hasPrevious()) {
	current = (Rule) inverse.previous();
	if (Rule.confirmationComparator.compare(current, worstRule) < 0) {
	  break;
	}
	inverse.remove();
      }
    }

    /* Print the new current values. */
    printValues();
  }
  
  /**
   * Method that launches the search to find the rules with the highest 
   * confirmation.
   *
   * @param instances The instances to be used for generating the rules.
   * @exception Exception if rules can't be built successfully.
   */
  public void buildAssociations(Instances instances) throws Exception {

    Frame valuesFrame = null; /* Frame to display the current values. */

    /* Initialization of the search. */
    if (m_parts == null) {
      m_instances = instances;
    } else {
      m_instances = new IndividualInstances(instances, m_parts);
    }    
    m_results = new SimpleLinkedList();
    m_hypotheses = 0;
    m_explored = 0;
    m_status = NORMAL;

    /* Set class index. */
    if (m_classIndex == 0) {
      m_instances.setClassIndex(instances.numAttributes() - 1);
    } else if ((m_classIndex > instances.numAttributes())
	       || (m_classIndex < 0)) {
      throw new Exception(weka.LocalString.get("Class index has to be between zero ")
			  + weka.LocalString.get("and the number of attributes!"));
    } else {
      m_instances.setClassIndex(m_classIndex - 1);
    }

    /* Initialization of the window for current values. */
    if (m_printValues == WINDOW) {
      m_valuesText = new TextField(37);
      m_valuesText.setEditable(false);
      m_valuesText.setFont(new Font(weka.LocalString.get("Monospaced"), Font.PLAIN, 12));    
      Label valuesLabel = new Label(weka.LocalString.get("Best and worst current values:"));
      Button stop = new Button(weka.LocalString.get("Stop search"));
      stop.addActionListener(new ActionListener() {
	  public void actionPerformed(ActionEvent e) {
	    /* Signal the interruption to the search. */
	    m_status = STOP;
	  }
	});
      valuesFrame = new Frame(weka.LocalString.get("Tertius status"));
      valuesFrame.setResizable(false);
      valuesFrame.add(m_valuesText, BorderLayout.CENTER);
      valuesFrame.add(stop, BorderLayout.SOUTH);
      valuesFrame.add(valuesLabel, BorderLayout.NORTH);
      valuesFrame.pack();
      valuesFrame.setVisible(true);
    } else if (m_printValues == OUT) {
      System.out.println(weka.LocalString.get("Best and worst current values:"));
    }
      
    Date start = new Date();

    /* Build the predicates and launch the search. */
    m_predicates = buildPredicates();
    beginSearch();    

    Date end = new Date();

    if (m_printValues == WINDOW) {
      valuesFrame.dispose();
    }

    m_time = new Date(end.getTime() - start.getTime());
  }

  /**
   * Run the search.
   */
  public void run() {
    try {
      search();
    } catch (OutOfMemoryError e) {
      /* Garbage collect what can be collected to be able to continue. */
      System.gc();
      m_status = MEMORY;
    }
    endSearch();
  }

  /**
   * Begin the search by starting a new thread.
   */
  private synchronized void beginSearch() throws Exception {
    /* This method must be synchronized to be able to 
     * call the wait() method. */
    Thread search = new Thread(this);
    search.start();
    try {
      /* Wait for the end of the thread. */
      wait();
    } catch (InterruptedException e) {
      /* Signal the interruption to the search. */
      m_status = STOP;
    }
  }

  /**
   * End the search by notifying to the waiting thread that it is finished.
   */
  private synchronized void endSearch() {
    /* This method must be synchronized to be able to
     * call the notify() method. */
    notify();
  }

  /**
   * Search in the space of hypotheses the rules that have the highest 
   * confirmation.
   * The search starts with the empty rule, other rules are generated by 
   * refinement.
   */
  public void search() {

    SimpleLinkedList nodes = new SimpleLinkedList(); /* The agenda. */
    Rule currentNode;
    SimpleLinkedList children;
    SimpleLinkedList.LinkedListIterator iter;
    Rule child;
    boolean negBody = (m_negation == BODY || m_negation == ALL);
    boolean negHead = (m_negation == HEAD || m_negation == ALL);

    /* Start with the empty rule. */
      nodes.add(new Rule(m_repeat, m_numLiterals, negBody, negHead,
			 m_classification, m_horn));
    
    /* Print the current values. */
    printValues();

    /* Explore the rules in the agenda. */
    while (m_status != STOP && !nodes.isEmpty()) {
      currentNode = (Rule) nodes.removeFirst();
      if (canRefine(currentNode)) {
	children = currentNode.refine(m_predicates);
	iter = children.iterator();
	/* Calculate the optimistic estimate of the children and 
	 * consider them for adding to the agenda and to the results. */
	while (iter.hasNext()) {
	  m_hypotheses++;
	  child = (Rule) iter.next();
	  child.upDate(m_instances);
	  if (canCalculateOptimistic(child)) {
	    child.calculateOptimistic();
	    if (canExplore(child)) {
	      m_explored++;
	      if (canStoreInNodes(child)) {
	      } else {
		iter.remove();
	      }
	      if (canCalculateConfirmation(child)) {
		child.calculateConfirmation();
		if (canStoreInResults(child)) {
		  addResult(child);
		}	  
	      }
	    } else {
	      iter.remove();
	    }
	  } else {
	    iter.remove();
	  }
	}
	/* Since the agenda is already sorted it is more efficient
	 * to sort the children only and then merge. */
	children.sort(Rule.optimisticThenObservedComparator);
	nodes.merge(children, Rule.optimisticThenObservedComparator);
      } else {
	/* The agenda being sorted, it is not worth considering the following 
	 * nodes. */
	break;
      }
    }
  }


  /**
   * Print the current best and worst values. 
   */
  private void printValues() {

    if (m_printValues == NO) {
      return;
    } else {
      if (m_results.isEmpty()) {
	if (m_printValues == OUT) {
	  System.out.print(weka.LocalString.get("0.000000 0.000000 - 0.000000 0.000000"));
	} else { //m_printValues == WINDOW
	  m_valuesText.setText(weka.LocalString.get("0.000000 0.000000 - 0.000000 0.000000"));
	}
      } else {
	Rule best = (Rule) m_results.getFirst();
	Rule worst = (Rule) m_results.getLast();
	String values = best.valuesToString() + " - " + worst.valuesToString();
	if (m_printValues == OUT) {
	  System.out.print(weka.LocalString.get("\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b")
			   + weka.LocalString.get("\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b"));
	  System.out.print(values);
	} else { //m_printValues == WINDOW
	  m_valuesText.setText(values);
	}
      }
    }
  }

  /**
   * Outputs the best rules found with their confirmation value and number 
   * of counter-instances.
   * Also gives the number of hypotheses considered and explored, and the 
   * time needed. 
   */
  public String toString() {

    StringBuffer text = new StringBuffer();
    SimpleDateFormat dateFormat 
      = new SimpleDateFormat(weka.LocalString.get("mm 'min' ss 's' SSS 'ms'"));
    SimpleLinkedList.LinkedListIterator iter = m_results.iterator();
    int size = m_results.size();
    int i = 0;

    text.append(weka.LocalString.get("\nTertius\n=======\n\n"));

    while (iter.hasNext()) {
      Rule current = (Rule) iter.next();
      text.append(Utils.doubleToString((double) i + 1,
				       (int) (Math.log(size) 
					      / Math.log(10) + 1),
				       0)
		  + ". ");
      text.append("/* ");
      if (m_roc) {
	text.append(current.rocToString());
      } else {
	text.append(current.valuesToString());
      }
      text.append(" */ ");
      text.append(current.toString());
      text.append("\n");
      i++;
    }
 
    text.append(weka.LocalString.get("\nNumber of hypotheses considered: ") + m_hypotheses);
    text.append(weka.LocalString.get("\nNumber of hypotheses explored: ") + m_explored);
    text.append(weka.LocalString.get("\nTime: ") + dateFormat.format(m_time));

    if (m_status == MEMORY) {
      text.append(weka.LocalString.get("\n\nNot enough memory to continue the search"));
    } else if (m_status == STOP) {
      text.append(weka.LocalString.get("\n\nSearch interrupted"));
    }

    return text.toString();
  }

  /**
   * Main method for testing this class.
   */
  public static void main(String [] options) {

    String trainFileString;
    Reader reader;
    Instances instances;
    Tertius tertius = new Tertius();
    StringBuffer text = new StringBuffer();

    try {
      /* Help string giving all the command line options. */
      text.append(weka.LocalString.get("\n\nTertius options:\n\n"));
      text.append(weka.LocalString.get("-t <name of training file>\n"));
      text.append(weka.LocalString.get("\tSet training file.\n"));
      Enumeration enu = tertius.listOptions();
      while (enu.hasMoreElements()) {
	Option option = (Option) enu.nextElement();
	text.append(option.synopsis() + "\n");
	text.append(option.description() + "\n");
      }

      /* Training file. */
      trainFileString = Utils.getOption('t', options);
      if (trainFileString.length() == 0) {
	throw new Exception(weka.LocalString.get("No training file given!"));
      }
      try {
	reader = new BufferedReader(new FileReader(trainFileString));
      } catch (Exception e) {
	throw new Exception(weka.LocalString.get("Can't open file ") + e.getMessage() + ".");
      }

      instances = new Instances(reader);

      /* Tertius options. */
      tertius.setOptions(options);
      Utils.checkForRemainingOptions(options);

      /* Build the rules and output the results. */
      tertius.buildAssociations(instances);
      System.out.println(tertius);

    } catch (Exception e) {
      System.err.println(weka.LocalString.get("\nWeka exception: ") + e.getMessage() + text);
    }
  }
}





