package config

import (
	"strings"
	"testing"
)

func TestResolvePassword(t *testing.T) {
	tests := []struct {
		username    string
		passwordCmd []string
		errContains string // Empty for success cases.
		wantPass    string // Only used for success cases
	}{
		{
			username:    "testuser",
			passwordCmd: []string{"echo", "mypassword"},
			wantPass:    "mypassword",
		},
		{
			username:    "testuser",
			passwordCmd: []string{"sh", "-c", "echo secret123"},
			wantPass:    "secret123",
		},
		{
			username:    "testuser",
			passwordCmd: []string{"sh", "-c", "echo '  password  '"},
			wantPass:    "password",
		},
		{
			username:    "testuser",
			passwordCmd: []string{"sh", "-c", "printf 'pass\\n'"},
			wantPass:    "pass",
		},
		{
			username:    "testuser",
			passwordCmd: []string{"false"},
			errContains: "exit code 1",
		},
		{
			username:    "testuser",
			passwordCmd: []string{"true"},
			errContains: "empty password",
		},
		{
			username:    "testuser",
			passwordCmd: []string{"nonexistent-command-12345"},
			errContains: "password-cmd failed",
		},
		{
			username:    "testuser",
			passwordCmd: []string{},
			errContains: "password-cmd not configured",
		},
		{
			username:    "",
			passwordCmd: nil,
			wantPass:    "",
		},
	}

	for _, tt := range tests {
		name := strings.Join(tt.passwordCmd, " ")
		if tt.errContains != "" {
			name = tt.errContains
		}
		if tt.username == "" {
			name = "pre-authenticated session"
		}

		t.Run(name, func(t *testing.T) {
			account := &Account{
				Name:        "test",
				Username:    tt.username,
				PasswordCmd: tt.passwordCmd,
			}

			err := account.ResolvePassword()

			if tt.errContains != "" {
				if err == nil {
					t.Errorf("ResolvePassword() expected error but got nil")
					return
				}
				if !strings.Contains(err.Error(), tt.errContains) {
					t.Errorf("ResolvePassword() error = %q, want error containing %q", err.Error(), tt.errContains)
				}
			} else {
				if err != nil {
					t.Errorf("ResolvePassword() unexpected error: %v", err)
					return
				}
				if account.Password != tt.wantPass {
					t.Errorf("ResolvePassword() password = %q, want %q", account.Password, tt.wantPass)
				}
			}
		})
	}
}
