package daemon

import (
	"context"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net"
	"os/exec"
	"strings"
	"syscall"
	"time"

	"git.sr.ht/~whynothugo/ImapGoose/internal/config"
	"git.sr.ht/~whynothugo/ImapGoose/internal/imap"
	"git.sr.ht/~whynothugo/ImapGoose/internal/maildir"
	"git.sr.ht/~whynothugo/ImapGoose/internal/status"
	imapsync "git.sr.ht/~whynothugo/ImapGoose/internal/sync"
)

// Worker represents a persistent IMAP connection worker.
type Worker struct {
	id         int
	account    config.Account
	localPath  string
	statusRepo *status.Repository
	logger     *slog.Logger
	client     *imap.Client
}

// NewWorker creates a new worker.
func NewWorker(id int, account config.Account, localPath string, statusRepo *status.Repository, logger *slog.Logger) *Worker {
	return &Worker{
		id:         id,
		account:    account,
		localPath:  localPath,
		statusRepo: statusRepo,
		logger:     logger.With("worker", id),
	}
}

// Run starts the worker processing tasks from the queue.
func (w *Worker) Run(ctx context.Context, taskQueue <-chan WorkerTask) error {
	w.logger.Info("Worker starting")

	// Connections are lazy-initialised when needed. Defer closing if necessary.
	defer func() {
		if w.client != nil {
			_ = w.client.Close() // Best effort close
		}
	}()

	// Close client on context cancellation.
	// This is somewhat of a hack around go-imap's lack of context support.
	go func() {
		<-ctx.Done()
		if w.client != nil {
			_ = w.client.Close()
		}
	}()

	// Process tasks from queue.
	for {
		select {
		case <-ctx.Done():
			w.logger.Info("Worker shutting down")
			return ctx.Err()

		case task, ok := <-taskQueue:
			if !ok {
				w.logger.Info("Task queue closed, worker exiting")
				return nil
			}

			// Process task and ensure completion is signaled.
			w.processTaskWithCompletion(ctx, task)
		}
	}
}

// processTaskWithCompletion processes a single task and signals completion.
func (w *Worker) processTaskWithCompletion(ctx context.Context, task WorkerTask) {
	defer func() {
		select {
		case task.Done <- task.Mailbox:
		case <-ctx.Done():
			w.logger.Warn("Workers cancelled before it could signal completion.", "mailbox", task.Mailbox)
		}
	}()

	w.logger.Info("Processing task", "mailbox", task.Mailbox, "absPaths", task.AbsPaths, "scanLocal", task.ScanLocal, "scanRemote", task.ScanRemote)

	// Retry loop for this task until success or non-connection error.
	for {
		// Establish connection (on first task or if previously disconnected).
		if w.client == nil {
			var err error
			// connect() retries indefinitely until success or context cancellation.
			w.client, err = w.connect(ctx)
			if err != nil {
				// Only returns error if context was cancelled.
				return
			}
		}

		// Execute task with reconnection on failure.
		if err := w.processTask(ctx, w.client, task); err != nil {
			w.logger.Error("Task failed", "mailbox", task.Mailbox, "error", err)

			if err := ctx.Err(); err != nil {
				return
			}

			if isConnectionError(err) {
				w.logger.Info("Connection error detected, will reconnect and retry task")
				_ = w.client.Close()
				w.client = nil // Clear client so next iteration reconnects
				continue       // Retry this task
			}

			// Non-connection error - log and move to next task.
			w.logger.Error("Non-connection error, skipping task", "mailbox", task.Mailbox, "error", err)
			return
		}

		// Task succeeded.
		w.runPostSyncCmd(ctx, task.Mailbox)
		return
	}
}

// connect establishes IMAP connection with infinite retry logic.
// Retries with exponential back-off.
// Only returns an error in case of context cancellation.
func (w *Worker) connect(ctx context.Context) (*imap.Client, error) {
	attempt := 0
	for {
		// Workers don't need UnilateralDataHandler (no NOTIFY), pass nil.
		client, err := imap.Connect(ctx, w.account.Server, w.account.Username, w.account.Password, w.account.Plaintext, nil, w.logger)
		if err == nil {
			if attempt > 0 {
				w.logger.Info("Connection established", "after_attempts", attempt+1)
			}
			return client, nil
		}

		attempt++
		w.logger.Warn("Connect failed, retrying", "attempt", attempt, "error", err)

		// Exponential back-off: 1s, 2s, 4s, 8s, 16s, 32s, 64s, 128s, 256s, 512s, 1024s
		// Cap at 2^10 seconds (1024s = ~17 minutes) for retry 11 and beyond.
		backoff := 1 << min(attempt-1, 10)
		w.logger.Debug("Backing off before retry", "seconds", backoff)

		select {
		case <-time.After(time.Duration(backoff) * time.Second):
		case <-ctx.Done():
			return nil, ctx.Err()
		}
	}
}

// min returns the minimum of two integers.
func min(a, b int) int {
	if a < b {
		return a
	}
	return b
}

// processTask executes a single sync task.
func (w *Worker) processTask(ctx context.Context, client *imap.Client, task WorkerTask) error {
	taskLogger := w.logger.With("mailbox", task.Mailbox)

	// Create Maildir.
	md := maildir.New(w.localPath, task.Mailbox, taskLogger)
	if err := md.Init(); err != nil {
		return fmt.Errorf("failed to initialize maildir: %w", err)
	}

	syncer := imapsync.New(client, md, w.statusRepo, task.Mailbox, taskLogger)

	if err := syncer.Sync(ctx, task.AbsPaths, task.ScanLocal, task.ScanRemote); err != nil {
		return fmt.Errorf("sync failed: %w", err)
	}

	return nil
}

// runPostSyncCmd executes the post-sync command, if any.
func (w *Worker) runPostSyncCmd(ctx context.Context, mailbox string) {
	if len(w.account.PostSyncCmd) == 0 {
		return // No post-sync command configured.
	}

	w.logger.Info("Running post-sync command", "mailbox", mailbox, "cmd", w.account.PostSyncCmd)

	cmd := exec.CommandContext(ctx, w.account.PostSyncCmd[0], w.account.PostSyncCmd[1:]...)
	cmd.Dir = w.localPath // Set working directory to account's local path.
	output, err := cmd.CombinedOutput()
	if err != nil {
		if exitErr, ok := err.(*exec.ExitError); ok {
			w.logger.Error(
				"post-sync-cmd returned an error",
				"mailbox", mailbox,
				"exit_code", exitErr.ExitCode(),
				"output", string(output),
			)
		} else {
			w.logger.Error("Failed to run post-sync-cmd", "mailbox", mailbox, "error", err)
		}
		return
	}

	w.logger.Info("Post-sync command completed", "mailbox", mailbox)
	w.logger.Debug("Post-sync command output", "mailbox", mailbox, "output", string(output))
}

// isConnectionError checks if error indicates connection failure.
// Returns true for network errors, I/O errors, and connection-related syscall errors.
// Returns false for protocol errors, authentication failures, and other non-connection issues.
func isConnectionError(err error) bool {
	if err == nil {
		return false
	}

	// Common connection-related errors.
	var netErr *net.OpError
	if errors.As(err, &netErr) {
		return true
	}

	// EOF and unexpected EOF (connection closed).
	if errors.Is(err, io.EOF) || errors.Is(err, io.ErrUnexpectedEOF) {
		return true
	}

	// Connection reset, broken pipe, and other syscall errors.
	if errors.Is(err, syscall.ECONNRESET) ||
		errors.Is(err, syscall.ECONNABORTED) ||
		errors.Is(err, syscall.EPIPE) ||
		errors.Is(err, syscall.ETIMEDOUT) {
		return true
	}

	// FIXME: conditions below are definitive. Conditions above are guesswork.

	// Context cancellation (not a connection error, but handle gracefully).
	if errors.Is(err, context.Canceled) || errors.Is(err, context.DeadlineExceeded) {
		return false
	}

	if strings.Contains(err.Error(), "use of closed network connection") {
		return true
	}

	// TODO: we might be missing some other errors here.

	// Default: assume non-connection error (protocol errors, auth failures, etc.)
	// This conservative approach prevents infinite retry loops on persistent errors.
	return false
}
