package status

import (
	"context"
	"crypto/sha256"
	"database/sql"
	"encoding/hex"
	"fmt"
	"os"
	"path/filepath"
	"strings"

	"github.com/emersion/go-imap/v2"
	_ "github.com/mattn/go-sqlite3"
)

// Repository represents the status repository (S in the sync algorithm).
// It tracks which messages have been synchronised using SQLite for crash safety.
type Repository struct {
	db          *sql.DB
	accountName string
	stateDir    string
}

// MessageStatus represents a synchronised message in the status repository.
type MessageStatus struct {
	UID      imap.UID
	Filename string // Maildir filename (basename only, not full path)
	Flags    []imap.Flag
}

// New creates a new status repository for a specific account.
// Each account gets its own SQLite database in XDG_STATE_HOME/imapgoose/.
func New(accountName string) *Repository {
	// Get XDG_STATE_HOME, default to ~/.local/state
	stateDir := os.Getenv("XDG_STATE_HOME")
	if stateDir == "" {
		home, _ := os.UserHomeDir()
		stateDir = filepath.Join(home, ".local", "state")
	}

	return &Repository{
		accountName: accountName,
		stateDir:    stateDir,
	}
}

// NewWithStateDir creates a new status repository with a custom state directory.
func NewWithStateDir(accountName, stateDir string) *Repository {
	return &Repository{
		accountName: accountName,
		stateDir:    stateDir,
	}
}

// Init initializes the status repository database.
// Opens (or creates) a SQLite database for this account.
func (r *Repository) Init() error {
	// Create imapgoose state directory.
	statusDir := filepath.Join(r.stateDir, "imapgoose")
	if err := os.MkdirAll(statusDir, 0700); err != nil {
		return fmt.Errorf("failed to create status directory: %w", err)
	}

	// Database filename: hash of account name to handle special characters.
	// This ensures each account has a separate database.
	dbFilename := r.getDatabaseFilename()
	dbPath := filepath.Join(statusDir, dbFilename)

	// Open database with appropriate flags for crash safety.
	// WAL mode + FULL sync ensures durability on crash.
	db, err := sql.Open("sqlite3", dbPath+"?_journal_mode=WAL&_sync=FULL&_foreign_keys=on")
	if err != nil {
		return fmt.Errorf("failed to open database: %w", err)
	}

	// The default creates an unlimited amount of connections, and when one
	// blocks the database, other fail with a "database is locked" error.
	db.SetMaxOpenConns(1)
	r.db = db

	// Create table if not exists.
	// TODO: remove unused column messages.modseq.
	schema := `
		CREATE TABLE IF NOT EXISTS messages (
			mailbox TEXT NOT NULL,
			uid INTEGER NOT NULL,
			filename TEXT NOT NULL,
			flags TEXT NOT NULL,
			modseq INTEGER NOT NULL DEFAULT 0,
			PRIMARY KEY (mailbox, uid)
		);
		CREATE INDEX IF NOT EXISTS idx_mailbox ON messages(mailbox);
		CREATE INDEX IF NOT EXISTS idx_filename ON messages(mailbox, filename);
		CREATE TABLE IF NOT EXISTS mailbox_state (
			mailbox TEXT PRIMARY KEY,
			uidvalidity INTEGER NOT NULL,
			highestmodseq INTEGER NOT NULL DEFAULT 0
		);
		CREATE TABLE IF NOT EXISTS metadata (
			key TEXT PRIMARY KEY,
			value TEXT NOT NULL
		);
	`

	if _, err := r.db.Exec(schema); err != nil {
		defer func() {
			_ = r.db.Close() // Best effort close
		}()
		return fmt.Errorf("failed to initialise schema: %w", err)
	}

	// Store account name in metadata for reference.
	if _, err := r.db.Exec("INSERT OR REPLACE INTO metadata (key, value) VALUES ('account_name', ?)", r.accountName); err != nil {
		_ = r.db.Close()
		return fmt.Errorf("failed to store account metadata: %w", err)
	}

	var count int
	if err := r.db.QueryRow("SELECT COUNT(*) FROM messages").Scan(&count); err != nil {
		_ = r.db.Close()
		return fmt.Errorf("failed to check messages table: %w", err)
	}

	if count == 0 {
		// Fresh install: start with v2.
		if _, err := r.db.Exec("INSERT OR IGNORE INTO metadata (key, value) VALUES ('schema_version', '2')"); err != nil {
			_ = r.db.Close()
			return fmt.Errorf("failed to initialize schema version: %w", err)
		}
	} else {
		// Existing database: default to v1 if not set.
		if _, err := r.db.Exec("INSERT OR IGNORE INTO metadata (key, value) VALUES ('schema_version', '1')"); err != nil {
			_ = r.db.Close()
			return fmt.Errorf("failed to initialize schema version: %w", err)
		}
	}

	return nil
}

// getDatabaseFilename returns a sanitized filename for this account's database.
// Uses SHA256 hash of account name to handle special characters safely.
func (r *Repository) getDatabaseFilename() string {
	// Hash the account name to get a safe filename.
	hash := sha256.Sum256([]byte(r.accountName))
	hashStr := hex.EncodeToString(hash[:])[:16] // Use first 16 chars of hash

	// Also include a sanitized version of the account name for readability.
	safeName := r.sanitizeFilename(r.accountName)
	if len(safeName) > 32 {
		safeName = safeName[:32]
	}

	return fmt.Sprintf("%s-%s.db", safeName, hashStr)
}

// sanitizeFilename removes unsafe characters from account name.
func (r *Repository) sanitizeFilename(name string) string {
	// Replace unsafe characters with underscores.
	safe := strings.Map(func(r rune) rune {
		switch {
		case r >= 'a' && r <= 'z':
			return r
		case r >= 'A' && r <= 'Z':
			return r
		case r >= '0' && r <= '9':
			return r
		case r == '-' || r == '_' || r == '.':
			return r
		default:
			return '_'
		}
	}, name)

	// If empty or only underscores, use default.
	if safe == "" || strings.Trim(safe, "_") == "" {
		safe = "account"
	}

	return safe
}

// Close closes the database connection.
func (r *Repository) Close() error {
	if r.db != nil {
		return r.db.Close()
	}
	return nil
}

// GetUIDs returns all UIDs for the specified mailbox.
// Used for determining which messages exist in status repository.
func (r *Repository) GetUIDs(ctx context.Context, mailbox string) (map[imap.UID]bool, error) {
	uids := make(map[imap.UID]bool)

	rows, err := r.db.QueryContext(ctx, "SELECT uid FROM messages WHERE mailbox = ?", mailbox)
	if err != nil {
		return nil, fmt.Errorf("failed to query UIDs: %w", err)
	}
	defer func() {
		_ = rows.Close() // Best effort close
	}()

	for rows.Next() {
		var uid uint32
		if err := rows.Scan(&uid); err != nil {
			return nil, fmt.Errorf("failed to scan UID: %w", err)
		}
		uids[imap.UID(uid)] = true
	}

	if err := rows.Err(); err != nil {
		return nil, fmt.Errorf("error iterating rows: %w", err)
	}

	// XXX: missing value are simply missing from the return dict.
	return uids, nil
}

// GetByUID retrieves a single message status by UID for the specified mailbox.
// Returns nil if the message does not exist.
func (r *Repository) GetByUID(ctx context.Context, mailbox string, uid imap.UID) (*MessageStatus, error) {
	var filename, flagsStr string
	err := r.db.QueryRowContext(ctx, "SELECT filename, flags FROM messages WHERE mailbox = ? AND uid = ?", mailbox, uint32(uid)).Scan(&filename, &flagsStr)
	if err == sql.ErrNoRows {
		return nil, nil
	}
	if err != nil {
		return nil, fmt.Errorf("failed to query message %d: %w", uid, err)
	}

	var flags []imap.Flag
	if flagsStr != "" {
		for _, f := range strings.Split(flagsStr, ",") {
			flags = append(flags, imap.Flag(f))
		}
	}

	return &MessageStatus{
		UID:      uid,
		Filename: filename,
		Flags:    flags,
	}, nil
}

// GetByFilename retrieves a single message status by filename for the specified mailbox.
// Returns nil if the message does not exist.
//
// This method matches on the unique portion of the Maildir filename (before ":2,"), so it
// correctly finds messages even when the flags portion has changed due to local flag updates.
// For example, both "1234.abc,U=42:2,S" and "1234.abc,U=42:2,SF" match the same message.
//
// TODO: Store the unique portion (separately? instead?) in the schema to avoid LIKE queries.
func (r *Repository) GetByFilename(ctx context.Context, mailbox string, filename string) (*MessageStatus, error) {
	uniquePart := filename
	if idx := strings.Index(filename, ":2,"); idx != -1 {
		uniquePart = filename[:idx]
	}
	pattern := uniquePart + "%"

	var uid uint32
	var storedFilename string
	var flagsStr string
	err := r.db.QueryRowContext(ctx, "SELECT uid, filename, flags FROM messages WHERE mailbox = ? AND filename LIKE ?", mailbox, pattern).Scan(&uid, &storedFilename, &flagsStr)
	if err == sql.ErrNoRows {
		return nil, nil
	}
	if err != nil {
		return nil, fmt.Errorf("failed to query message by filename %s: %w", filename, err)
	}

	var flags []imap.Flag
	if flagsStr != "" {
		for _, f := range strings.Split(flagsStr, ",") {
			flags = append(flags, imap.Flag(f))
		}
	}

	return &MessageStatus{
		UID:      imap.UID(uid),
		Filename: storedFilename, // Return filename from DB, not input.
		Flags:    flags,
	}, nil
}

// Add adds a message to the status repository for the specified mailbox.
// Atomically inserts or replaces the message.
func (r *Repository) Add(mailbox string, uid imap.UID, filename string, flags []imap.Flag) error {
	flagsStr := joinFlags(flags)
	_, err := r.db.Exec("INSERT OR REPLACE INTO messages (mailbox, uid, filename, flags) VALUES (?, ?, ?, ?)",
		mailbox, uint32(uid), filename, flagsStr)
	if err != nil {
		return fmt.Errorf("failed to add message %d: %w", uid, err)
	}
	return nil
}

// Remove removes a message from the status repository for the specified mailbox.
func (r *Repository) Remove(mailbox string, uid imap.UID) error {
	_, err := r.db.Exec("DELETE FROM messages WHERE mailbox = ? AND uid = ?", mailbox, uint32(uid))
	if err != nil {
		return fmt.Errorf("failed to remove message %d: %w", uid, err)
	}
	return nil
}

// UpdateFlags updates the flags and filename for a message in the specified mailbox.
// Changes to flags imply that the Maildir filename has change, so the new filename is required.
func (r *Repository) UpdateFlags(mailbox string, uid imap.UID, newFilename string, flags []imap.Flag) error {
	flagsStr := joinFlags(flags)
	result, err := r.db.Exec("UPDATE messages SET flags = ?, filename = ? WHERE mailbox = ? AND uid = ?",
		flagsStr, newFilename, mailbox, uint32(uid))
	if err != nil {
		return fmt.Errorf("failed to update flags for message %d: %w", uid, err)
	}

	rowsAffected, err := result.RowsAffected()
	if err != nil {
		return fmt.Errorf("failed to check rows affected: %w", err)
	}

	if rowsAffected == 0 {
		return fmt.Errorf("message %d not found in status repository", uid)
	}

	return nil
}

// GetUIDValidity retrieves the stored UIDValidity for the specified mailbox.
// Returns (value, true, nil) if stored, (0, false, nil) if not stored, (0, false, err) on error.
func (r *Repository) GetUIDValidity(mailbox string) (uint32, bool, error) {
	var uidvalidity uint32
	err := r.db.QueryRow("SELECT uidvalidity FROM mailbox_state WHERE mailbox = ?", mailbox).Scan(&uidvalidity)
	if err == sql.ErrNoRows {
		return 0, false, nil
	}
	if err != nil {
		return 0, false, fmt.Errorf("failed to query UIDValidity: %w", err)
	}
	return uidvalidity, true, nil
}

// SetUIDValidity stores the UIDValidity for the specified mailbox.
// If a value already exists and matches, this is a no-op.
// If a value already exists and differs, returns an error.
// TODO: rename to EnsureUIDValidity
func (r *Repository) SetUIDValidity(mailbox string, uidvalidity uint32) error {
	// First, try to insert the new value.
	_, err := r.db.Exec("INSERT INTO mailbox_state (mailbox, uidvalidity, highestmodseq) VALUES (?, ?, 0)",
		mailbox, uidvalidity)
	if err == nil {
		// Successfully inserted.
		return nil
	}

	// Insert failed, likely due to PRIMARY KEY constraint.
	// Check if existing value matches.
	var existingUIDValidity uint32
	err = r.db.QueryRow("SELECT uidvalidity FROM mailbox_state WHERE mailbox = ?", mailbox).Scan(&existingUIDValidity)
	if err != nil {
		return fmt.Errorf("failed to query existing UIDValidity: %w", err)
	}

	if existingUIDValidity == uidvalidity {
		// Same value, no-op.
		return nil
	}

	// Different value.
	return fmt.Errorf("UIDValidity mismatch: attempting to store %d but database has %d", uidvalidity, existingUIDValidity)
}

// GetHighestModSeq retrieves the stored HIGHESTMODSEQ for the specified mailbox.
// Returns 0 if not set or mailbox doesn't exist.
func (r *Repository) GetHighestModSeq(mailbox string) (uint64, error) {
	var highestModSeq uint64
	err := r.db.QueryRow("SELECT highestmodseq FROM mailbox_state WHERE mailbox = ?", mailbox).Scan(&highestModSeq)
	if err == sql.ErrNoRows {
		return 0, nil
	}
	if err != nil {
		return 0, fmt.Errorf("failed to query HIGHESTMODSEQ: %w", err)
	}
	return highestModSeq, nil
}

// SetHighestModSeq stores the HIGHESTMODSEQ for the specified mailbox.
// The mailbox_state row must already exist (created by SetUIDValidity).
func (r *Repository) SetHighestModSeq(mailbox string, highestModSeq uint64) error {
	result, err := r.db.Exec(
		"UPDATE mailbox_state SET highestmodseq = ? WHERE mailbox = ? AND highestmodseq <= ?",
		highestModSeq,
		mailbox,
		highestModSeq,
	)
	if err != nil {
		return fmt.Errorf("failed to update HIGHESTMODSEQ: %w", err)
	}

	rowsAffected, err := result.RowsAffected()
	if err != nil {
		return fmt.Errorf("failed to check rows affected: %w", err)
	}

	if rowsAffected == 0 {
		var exists int
		err := r.db.QueryRow("SELECT 1 FROM mailbox_state WHERE mailbox = ?", mailbox).Scan(&exists)
		if err == sql.ErrNoRows {
			return fmt.Errorf("mailbox %s not found in mailbox_state table", mailbox)
		}
		if err != nil {
			return fmt.Errorf("failed to check mailbox existence: %w", err)
		}
		// else: mailbox exists, so the update was skipped because new value < current value.
	}

	return nil
}

// GetVersion retrieves the schema version from this status repository.
// Returns 1 if not set (i.e. legacy repositories).
func (r *Repository) GetVersion() (int, error) {
	var versionStr string
	err := r.db.QueryRow("SELECT value FROM metadata WHERE key = 'schema_version'").Scan(&versionStr)
	if err == sql.ErrNoRows {
		return 1, nil // Default to v1 for legacy databases
	}
	if err != nil {
		return 0, fmt.Errorf("failed to query schema version: %w", err)
	}

	var version int
	if _, err := fmt.Sscanf(versionStr, "%d", &version); err != nil {
		return 0, fmt.Errorf("invalid schema version %q: %w", versionStr, err)
	}

	return version, nil
}

// SetSchemaVersion updates the schema version in metadata.
func (r *Repository) SetSchemaVersion(version int) error {
	_, err := r.db.Exec("INSERT OR REPLACE INTO metadata (key, value) VALUES ('schema_version', ?)",
		fmt.Sprintf("%d", version))
	if err != nil {
		return fmt.Errorf("failed to set schema version: %w", err)
	}
	return nil
}

// joinFlags converts flag slice to comma-separated string.
// Flags are sorted alphabetically to ensure consistent representation.
func joinFlags(flags []imap.Flag) string {
	if len(flags) == 0 {
		return ""
	}

	var parts []string
	for _, f := range flags {
		parts = append(parts, string(f))
	}

	// Sort flags alphabetically for consistent comparison.
	for i := 0; i < len(parts); i++ {
		for j := i + 1; j < len(parts); j++ {
			if parts[i] > parts[j] {
				parts[i], parts[j] = parts[j], parts[i]
			}
		}
	}

	return strings.Join(parts, ",")
}
