#' Create and Verify Signatures
#'
#' Cryptographic signatures can be used to verify the integrity of a message using
#' the author's public key.
#'
#' A signature is an authenticated checksum that can be used to check that a message
#' (any data) was created by a particular author and was not tampered with. The signature is
#' created using a private key and can be verified from the corresponding public key.
#'
#' Signatures are used when the message itself is not confidential but integrity is
#' important. A common use is for software repositories where maintainers include
#' a signature of the package index. This allows client package managers to verify
#' that the binaries were not modified by intermediate parties in the distribution
#' process.
#'
#' For confidential data, use authenticated encryption (\link{auth_encrypt})
#' which allows for sending signed and encrypted messages in a single method.
#'
#' Currently sodium requires a different type of key pairfor signatures  (ed25519)
#' than for encryption (curve25519).
#'
#' @rdname sig
#' @name Signatures
#' @aliases sig
#' @export
#' @useDynLib sodium R_sig_sign
#' @param msg message to sign
#' @param key private key to sign message with
#' @param sig a signature generated by \code{signature_sign}
#' @param pubkey a public key of the keypair used by the signature
#' @references \url{https://doc.libsodium.org/public-key_cryptography/public-key_signatures.html}
#' @examples # Generate keypair
#' key <- sig_keygen()
#' pubkey <- sig_pubkey(key)
#'
#' # Create signature
#' msg <- serialize(iris, NULL)
#' sig <- sig_sign(msg, key)
#' sig_verify(msg, sig, pubkey)
sig_sign <- function(msg, key){
  stopifnot(is.raw(msg))
  stopifnot(is.raw(key))
  .Call(R_sig_sign, msg, key)
}

#' @export
#' @rdname sig
#' @useDynLib sodium R_sig_verify
sig_verify <- function(msg, sig, pubkey){
  stopifnot(is.raw(msg))
  stopifnot(is.raw(sig))
  stopifnot(is.raw(pubkey))
  .Call(R_sig_verify, msg, sig, pubkey)
}

#' @export
#' @rdname sig
#' @useDynLib sodium R_sig_keygen
#' @param seed random data to seed the keygen
sig_keygen <- function(seed = random(32)){
  stopifnot(is.raw(seed))
  .Call(R_sig_keygen, seed)
}

#' @export
#' @rdname sig
#' @useDynLib sodium R_sig_pubkey
sig_pubkey <- function(key){
  stopifnot(is.raw(key))
  .Call(R_sig_pubkey, key)
}
