// Copyright 2020-2024 Buf Technologies, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package protosourcepath

import (
	"context"
	"testing"

	"github.com/bufbuild/protocompile"
	"github.com/stretchr/testify/require"
	"google.golang.org/protobuf/reflect/protoreflect"
)

func TestGetAssociatedSourcePathsProto2(t *testing.T) {
	t.Parallel()
	testGetAssociatedSourcePaths(
		t,
		"testdata/proto2/test.proto",
		map[string][]protoreflect.SourcePath{
			".syntax":                                                                             {[]int32{12}},
			".package":                                                                            {[]int32{2}},
			".dependency[0]":                                                                      {[]int32{3, 0}},
			".service[0]":                                                                         {[]int32{6, 0}},
			".service[0].name":                                                                    {[]int32{6, 0}},
			".service[0].method[0]":                                                               {[]int32{6, 0}, []int32{6, 0, 2, 0}},
			".service[0].method[0].name":                                                          {[]int32{6, 0}, []int32{6, 0, 2, 0}},
			".service[0].method[0].input_type":                                                    {[]int32{6, 0}, []int32{6, 0, 2, 0}},
			".service[0].method[0].output_type":                                                   {[]int32{6, 0}, []int32{6, 0, 2, 0}},
			".service[0].method[1]":                                                               {[]int32{6, 0}, []int32{6, 0, 2, 1}},
			".service[0].method[1].name":                                                          {[]int32{6, 0}, []int32{6, 0, 2, 1}},
			".service[0].method[1].client_streaming":                                              {[]int32{6, 0}, []int32{6, 0, 2, 1}},
			".service[0].method[1].input_type":                                                    {[]int32{6, 0}, []int32{6, 0, 2, 1}},
			".service[0].method[1].server_streaming":                                              {[]int32{6, 0}, []int32{6, 0, 2, 1}},
			".service[0].method[1].output_type":                                                   {[]int32{6, 0}, []int32{6, 0, 2, 1}},
			".message_type[0]":                                                                    {[]int32{4, 0}},
			".message_type[0].name":                                                               {[]int32{4, 0}},
			".message_type[0].field[0]":                                                           {[]int32{4, 0}, []int32{4, 0, 2, 0}},
			".message_type[0].field[0].label":                                                     {[]int32{4, 0}, []int32{4, 0, 2, 0}},
			".message_type[0].field[0].type":                                                      {[]int32{4, 0}, []int32{4, 0, 2, 0}},
			".message_type[0].field[0].name":                                                      {[]int32{4, 0}, []int32{4, 0, 2, 0}},
			".message_type[0].field[0].number":                                                    {[]int32{4, 0}, []int32{4, 0, 2, 0}},
			".message_type[1]":                                                                    {[]int32{4, 1}},
			".message_type[1].name":                                                               {[]int32{4, 1}},
			".message_type[1].nested_type[0]":                                                     {[]int32{4, 1}, []int32{4, 1, 3, 0}},
			".message_type[1].nested_type[0].name":                                                {[]int32{4, 1}, []int32{4, 1, 3, 0}},
			".message_type[1].nested_type[0].field[0]":                                            {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 2, 0}},
			".message_type[1].nested_type[0].field[0].label":                                      {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 2, 0}},
			".message_type[1].nested_type[0].field[0].type":                                       {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 2, 0}},
			".message_type[1].nested_type[0].field[0].name":                                       {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 2, 0}},
			".message_type[1].nested_type[0].field[0].number":                                     {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 2, 0}},
			".message_type[1].nested_type[0].field[1]":                                            {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 2, 1}},
			".message_type[1].nested_type[0].field[1].label":                                      {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 2, 1}},
			".message_type[1].nested_type[0].field[1].type_name":                                  {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 2, 1}},
			".message_type[1].nested_type[0].field[1].name":                                       {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 2, 1}},
			".message_type[1].nested_type[0].field[1].number":                                     {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 2, 1}},
			".message_type[1].nested_type[0].extension_range":                                     {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 5}},
			".message_type[1].nested_type[0].extension_range[0]":                                  {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}},
			".message_type[1].nested_type[0].extension_range[0].start":                            {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}},
			".message_type[1].nested_type[0].extension_range[0].end":                              {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}},
			".message_type[1].nested_type[0].extension_range[0].options":                          {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 3}},
			".message_type[1].nested_type[0].extension_range[0].options.verification":             {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 3, 3}},
			".message_type[1].nested_type[0].extension_range[0].options.declaration[0]":           {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 3, 2, 0}},
			".message_type[1].nested_type[0].extension_range[0].options.declaration[0].number":    {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 3, 2, 0, 1}},
			".message_type[1].nested_type[0].extension_range[0].options.declaration[0].full_name": {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 3, 2, 0, 2}},
			".message_type[1].nested_type[0].extension_range[0].options.declaration[0].repeated":  {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 3, 2, 0, 6}},
			".message_type[1].nested_type[0].extension_range[0].options.declaration[0].type":      {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 3, 2, 0, 3}},
			".message_type[1].nested_type[0].extension_range[0].options.declaration[1]":           {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 3, 2, 1}},
			".message_type[1].nested_type[0].extension_range[0].options.declaration[1].number":    {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 3, 2, 1, 1}},
			".message_type[1].nested_type[0].extension_range[0].options.declaration[1].full_name": {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 3, 2, 1, 2}},
			".message_type[1].nested_type[0].extension_range[0].options.declaration[1].type":      {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 3, 2, 1, 3}},
			".message_type[1].field[0]":                                                           {[]int32{4, 1}, []int32{4, 1, 2, 0}},
			".message_type[1].field[0].label":                                                     {[]int32{4, 1}, []int32{4, 1, 2, 0}},
			".message_type[1].field[0].type_name":                                                 {[]int32{4, 1}, []int32{4, 1, 2, 0}},
			".message_type[1].field[0].name":                                                      {[]int32{4, 1}, []int32{4, 1, 2, 0}},
			".message_type[1].field[0].number":                                                    {[]int32{4, 1}, []int32{4, 1, 2, 0}},
			".message_type[2]":                                                                    {[]int32{4, 2}},
			".message_type[2].name":                                                               {[]int32{4, 2}},
			".message_type[2].field[0]":                                                           {[]int32{4, 2}, []int32{4, 2, 2, 0}},
			".message_type[2].field[0].type_name":                                                 {[]int32{4, 2}, []int32{4, 2, 2, 0}},
			".message_type[2].field[0].name":                                                      {[]int32{4, 2}, []int32{4, 2, 2, 0}},
			".message_type[2].field[0].number":                                                    {[]int32{4, 2}, []int32{4, 2, 2, 0}},
			".message_type[3]":                                                                    {[]int32{4, 3}},
			".message_type[3].name":                                                               {[]int32{4, 3}},
			".message_type[3].field[0]":                                                           {[]int32{4, 3}, []int32{4, 3, 2, 0}},
			".message_type[3].field[0].label":                                                     {[]int32{4, 3}, []int32{4, 3, 2, 0}},
			".message_type[3].field[0].type":                                                      {[]int32{4, 3}, []int32{4, 3, 2, 0}},
			".message_type[3].field[0].name":                                                      {[]int32{4, 3}, []int32{4, 3, 2, 0}},
			".message_type[3].field[0].number":                                                    {[]int32{4, 3}, []int32{4, 3, 2, 0}},
			".message_type[3].reserved_name":                                                      {[]int32{4, 3}, []int32{4, 3, 10}},
			".message_type[3].reserved_name[0]":                                                   {[]int32{4, 3}, []int32{4, 3, 10}, []int32{4, 3, 10, 0}},
			".message_type[3].reserved_name[1]":                                                   {[]int32{4, 3}, []int32{4, 3, 10}, []int32{4, 3, 10, 1}},
			".message_type[4]":                                                                    {[]int32{4, 4}},
			".message_type[4].name":                                                               {[]int32{4, 4}},
			".message_type[4].extension":                                                          {[]int32{4, 4}, []int32{4, 4, 6}},
			".message_type[4].extension[0]":                                                       {[]int32{4, 4}, []int32{4, 4, 6}, []int32{4, 4, 6, 0}},
			".message_type[4].extension[0].extendee":                                              {[]int32{4, 4}, []int32{4, 4, 6}, []int32{4, 4, 6, 0}},
			".message_type[4].extension[0].label":                                                 {[]int32{4, 4}, []int32{4, 4, 6}, []int32{4, 4, 6, 0}},
			".message_type[4].extension[0].type":                                                  {[]int32{4, 4}, []int32{4, 4, 6}, []int32{4, 4, 6, 0}},
			".message_type[4].extension[0].name":                                                  {[]int32{4, 4}, []int32{4, 4, 6}, []int32{4, 4, 6, 0}},
			".message_type[4].extension[0].number":                                                {[]int32{4, 4}, []int32{4, 4, 6}, []int32{4, 4, 6, 0}},
			".message_type[4].extension[1]":                                                       {[]int32{4, 4}, []int32{4, 4, 6}, []int32{4, 4, 6, 1}},
			".message_type[4].extension[1].extendee":                                              {[]int32{4, 4}, []int32{4, 4, 6}, []int32{4, 4, 6, 1}},
			".message_type[4].extension[1].label":                                                 {[]int32{4, 4}, []int32{4, 4, 6}, []int32{4, 4, 6, 1}},
			".message_type[4].extension[1].type":                                                  {[]int32{4, 4}, []int32{4, 4, 6}, []int32{4, 4, 6, 1}},
			".message_type[4].extension[1].name":                                                  {[]int32{4, 4}, []int32{4, 4, 6}, []int32{4, 4, 6, 1}},
			".message_type[4].extension[1].number":                                                {[]int32{4, 4}, []int32{4, 4, 6}, []int32{4, 4, 6, 1}},
			".message_type[4].extension[1].options":                                               {[]int32{4, 4}, []int32{4, 4, 6}, []int32{4, 4, 6, 1}, []int32{4, 4, 6, 1, 8}},
			".message_type[4].extension[1].default_value":                                         {[]int32{4, 4}, []int32{4, 4, 6}, []int32{4, 4, 6, 1}, []int32{4, 4, 6, 1, 7}},
		},
		true,
	)
	testGetAssociatedSourcePaths(
		t,
		"testdata/proto2/test.proto",
		map[string][]protoreflect.SourcePath{
			".syntax":                                                                             {[]int32{12}},
			".package":                                                                            {[]int32{2}},
			".dependency[0]":                                                                      {[]int32{3, 0}},
			".service[0]":                                                                         {[]int32{6, 0}, []int32{6, 0, 1}},
			".service[0].name":                                                                    {[]int32{6, 0}, []int32{6, 0, 1}},
			".service[0].method[0]":                                                               {[]int32{6, 0}, []int32{6, 0, 1}, []int32{6, 0, 2, 0}, []int32{6, 0, 2, 0, 1}, []int32{6, 0, 2, 0, 2}, []int32{6, 0, 2, 0, 3}, []int32{6, 0, 2, 0, 5}, []int32{6, 0, 2, 0, 6}},
			".service[0].method[0].name":                                                          {[]int32{6, 0}, []int32{6, 0, 1}, []int32{6, 0, 2, 0}, []int32{6, 0, 2, 0, 1}, []int32{6, 0, 2, 0, 2}, []int32{6, 0, 2, 0, 3}, []int32{6, 0, 2, 0, 5}, []int32{6, 0, 2, 0, 6}},
			".service[0].method[0].input_type":                                                    {[]int32{6, 0}, []int32{6, 0, 1}, []int32{6, 0, 2, 0}, []int32{6, 0, 2, 0, 1}, []int32{6, 0, 2, 0, 2}, []int32{6, 0, 2, 0, 3}, []int32{6, 0, 2, 0, 5}, []int32{6, 0, 2, 0, 6}},
			".service[0].method[0].output_type":                                                   {[]int32{6, 0}, []int32{6, 0, 1}, []int32{6, 0, 2, 0}, []int32{6, 0, 2, 0, 1}, []int32{6, 0, 2, 0, 2}, []int32{6, 0, 2, 0, 3}, []int32{6, 0, 2, 0, 5}, []int32{6, 0, 2, 0, 6}},
			".service[0].method[1]":                                                               {[]int32{6, 0}, []int32{6, 0, 1}, []int32{6, 0, 2, 1}, []int32{6, 0, 2, 1, 1}, []int32{6, 0, 2, 1, 2}, []int32{6, 0, 2, 1, 3}, []int32{6, 0, 2, 1, 5}, []int32{6, 0, 2, 1, 6}},
			".service[0].method[1].name":                                                          {[]int32{6, 0}, []int32{6, 0, 1}, []int32{6, 0, 2, 1}, []int32{6, 0, 2, 1, 1}, []int32{6, 0, 2, 1, 2}, []int32{6, 0, 2, 1, 3}, []int32{6, 0, 2, 1, 5}, []int32{6, 0, 2, 1, 6}},
			".service[0].method[1].client_streaming":                                              {[]int32{6, 0}, []int32{6, 0, 1}, []int32{6, 0, 2, 1}, []int32{6, 0, 2, 1, 1}, []int32{6, 0, 2, 1, 2}, []int32{6, 0, 2, 1, 3}, []int32{6, 0, 2, 1, 5}, []int32{6, 0, 2, 1, 6}},
			".service[0].method[1].input_type":                                                    {[]int32{6, 0}, []int32{6, 0, 1}, []int32{6, 0, 2, 1}, []int32{6, 0, 2, 1, 1}, []int32{6, 0, 2, 1, 2}, []int32{6, 0, 2, 1, 3}, []int32{6, 0, 2, 1, 5}, []int32{6, 0, 2, 1, 6}},
			".service[0].method[1].server_streaming":                                              {[]int32{6, 0}, []int32{6, 0, 1}, []int32{6, 0, 2, 1}, []int32{6, 0, 2, 1, 1}, []int32{6, 0, 2, 1, 2}, []int32{6, 0, 2, 1, 3}, []int32{6, 0, 2, 1, 5}, []int32{6, 0, 2, 1, 6}},
			".service[0].method[1].output_type":                                                   {[]int32{6, 0}, []int32{6, 0, 1}, []int32{6, 0, 2, 1}, []int32{6, 0, 2, 1, 1}, []int32{6, 0, 2, 1, 2}, []int32{6, 0, 2, 1, 3}, []int32{6, 0, 2, 1, 5}, []int32{6, 0, 2, 1, 6}},
			".message_type[0]":                                                                    {[]int32{4, 0}, []int32{4, 0, 1}},
			".message_type[0].name":                                                               {[]int32{4, 0}, []int32{4, 0, 1}},
			".message_type[0].field[0]":                                                           {[]int32{4, 0}, []int32{4, 0, 1}, []int32{4, 0, 2, 0}, []int32{4, 0, 2, 0, 1}, []int32{4, 0, 2, 0, 3}, []int32{4, 0, 2, 0, 4}, []int32{4, 0, 2, 0, 5}, []int32{4, 0, 2, 0, 6}},
			".message_type[0].field[0].label":                                                     {[]int32{4, 0}, []int32{4, 0, 1}, []int32{4, 0, 2, 0}, []int32{4, 0, 2, 0, 1}, []int32{4, 0, 2, 0, 3}, []int32{4, 0, 2, 0, 4}, []int32{4, 0, 2, 0, 5}, []int32{4, 0, 2, 0, 6}},
			".message_type[0].field[0].type":                                                      {[]int32{4, 0}, []int32{4, 0, 1}, []int32{4, 0, 2, 0}, []int32{4, 0, 2, 0, 1}, []int32{4, 0, 2, 0, 3}, []int32{4, 0, 2, 0, 4}, []int32{4, 0, 2, 0, 5}, []int32{4, 0, 2, 0, 6}},
			".message_type[0].field[0].name":                                                      {[]int32{4, 0}, []int32{4, 0, 1}, []int32{4, 0, 2, 0}, []int32{4, 0, 2, 0, 1}, []int32{4, 0, 2, 0, 3}, []int32{4, 0, 2, 0, 4}, []int32{4, 0, 2, 0, 5}, []int32{4, 0, 2, 0, 6}},
			".message_type[0].field[0].number":                                                    {[]int32{4, 0}, []int32{4, 0, 1}, []int32{4, 0, 2, 0}, []int32{4, 0, 2, 0, 1}, []int32{4, 0, 2, 0, 3}, []int32{4, 0, 2, 0, 4}, []int32{4, 0, 2, 0, 5}, []int32{4, 0, 2, 0, 6}},
			".message_type[1]":                                                                    {[]int32{4, 1}, []int32{4, 1, 1}},
			".message_type[1].name":                                                               {[]int32{4, 1}, []int32{4, 1, 1}},
			".message_type[1].nested_type[0]":                                                     {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}},
			".message_type[1].nested_type[0].name":                                                {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}},
			".message_type[1].nested_type[0].field[0]":                                            {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 2, 0}, []int32{4, 1, 3, 0, 2, 0, 1}, []int32{4, 1, 3, 0, 2, 0, 3}, []int32{4, 1, 3, 0, 2, 0, 4}, []int32{4, 1, 3, 0, 2, 0, 5}, []int32{4, 1, 3, 0, 2, 0, 6}},
			".message_type[1].nested_type[0].field[0].label":                                      {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 2, 0}, []int32{4, 1, 3, 0, 2, 0, 1}, []int32{4, 1, 3, 0, 2, 0, 3}, []int32{4, 1, 3, 0, 2, 0, 4}, []int32{4, 1, 3, 0, 2, 0, 5}, []int32{4, 1, 3, 0, 2, 0, 6}},
			".message_type[1].nested_type[0].field[0].type":                                       {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 2, 0}, []int32{4, 1, 3, 0, 2, 0, 1}, []int32{4, 1, 3, 0, 2, 0, 3}, []int32{4, 1, 3, 0, 2, 0, 4}, []int32{4, 1, 3, 0, 2, 0, 5}, []int32{4, 1, 3, 0, 2, 0, 6}},
			".message_type[1].nested_type[0].field[0].name":                                       {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 2, 0}, []int32{4, 1, 3, 0, 2, 0, 1}, []int32{4, 1, 3, 0, 2, 0, 3}, []int32{4, 1, 3, 0, 2, 0, 4}, []int32{4, 1, 3, 0, 2, 0, 5}, []int32{4, 1, 3, 0, 2, 0, 6}},
			".message_type[1].nested_type[0].field[0].number":                                     {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 2, 0}, []int32{4, 1, 3, 0, 2, 0, 1}, []int32{4, 1, 3, 0, 2, 0, 3}, []int32{4, 1, 3, 0, 2, 0, 4}, []int32{4, 1, 3, 0, 2, 0, 5}, []int32{4, 1, 3, 0, 2, 0, 6}},
			".message_type[1].nested_type[0].field[1]":                                            {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 2, 1}, []int32{4, 1, 3, 0, 2, 1, 1}, []int32{4, 1, 3, 0, 2, 1, 3}, []int32{4, 1, 3, 0, 2, 1, 4}, []int32{4, 1, 3, 0, 2, 1, 5}, []int32{4, 1, 3, 0, 2, 1, 6}},
			".message_type[1].nested_type[0].field[1].label":                                      {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 2, 1}, []int32{4, 1, 3, 0, 2, 1, 1}, []int32{4, 1, 3, 0, 2, 1, 3}, []int32{4, 1, 3, 0, 2, 1, 4}, []int32{4, 1, 3, 0, 2, 1, 5}, []int32{4, 1, 3, 0, 2, 1, 6}},
			".message_type[1].nested_type[0].field[1].type_name":                                  {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 2, 1}, []int32{4, 1, 3, 0, 2, 1, 1}, []int32{4, 1, 3, 0, 2, 1, 3}, []int32{4, 1, 3, 0, 2, 1, 4}, []int32{4, 1, 3, 0, 2, 1, 5}, []int32{4, 1, 3, 0, 2, 1, 6}},
			".message_type[1].nested_type[0].field[1].name":                                       {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 2, 1}, []int32{4, 1, 3, 0, 2, 1, 1}, []int32{4, 1, 3, 0, 2, 1, 3}, []int32{4, 1, 3, 0, 2, 1, 4}, []int32{4, 1, 3, 0, 2, 1, 5}, []int32{4, 1, 3, 0, 2, 1, 6}},
			".message_type[1].nested_type[0].field[1].number":                                     {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 2, 1}, []int32{4, 1, 3, 0, 2, 1, 1}, []int32{4, 1, 3, 0, 2, 1, 3}, []int32{4, 1, 3, 0, 2, 1, 4}, []int32{4, 1, 3, 0, 2, 1, 5}, []int32{4, 1, 3, 0, 2, 1, 6}},
			".message_type[1].nested_type[0].extension_range":                                     {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 5}},
			".message_type[1].nested_type[0].extension_range[0]":                                  {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 1}, []int32{4, 1, 3, 0, 5, 0, 2}},
			".message_type[1].nested_type[0].extension_range[0].start":                            {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 1}, []int32{4, 1, 3, 0, 5, 0, 2}},
			".message_type[1].nested_type[0].extension_range[0].end":                              {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 1}, []int32{4, 1, 3, 0, 5, 0, 2}},
			".message_type[1].nested_type[0].extension_range[0].options":                          {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 1}, []int32{4, 1, 3, 0, 5, 0, 2}, []int32{4, 1, 3, 0, 5, 0, 3}},
			".message_type[1].nested_type[0].extension_range[0].options.verification":             {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 1}, []int32{4, 1, 3, 0, 5, 0, 2}, []int32{4, 1, 3, 0, 5, 0, 3, 3}},
			".message_type[1].nested_type[0].extension_range[0].options.declaration[0]":           {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 1}, []int32{4, 1, 3, 0, 5, 0, 2}, []int32{4, 1, 3, 0, 5, 0, 3, 2, 0}},
			".message_type[1].nested_type[0].extension_range[0].options.declaration[0].number":    {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 1}, []int32{4, 1, 3, 0, 5, 0, 2}, []int32{4, 1, 3, 0, 5, 0, 3, 2, 0, 1}},
			".message_type[1].nested_type[0].extension_range[0].options.declaration[0].full_name": {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 1}, []int32{4, 1, 3, 0, 5, 0, 2}, []int32{4, 1, 3, 0, 5, 0, 3, 2, 0, 2}},
			".message_type[1].nested_type[0].extension_range[0].options.declaration[0].repeated":  {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 1}, []int32{4, 1, 3, 0, 5, 0, 2}, []int32{4, 1, 3, 0, 5, 0, 3, 2, 0, 6}},
			".message_type[1].nested_type[0].extension_range[0].options.declaration[0].type":      {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 1}, []int32{4, 1, 3, 0, 5, 0, 2}, []int32{4, 1, 3, 0, 5, 0, 3, 2, 0, 3}},
			".message_type[1].nested_type[0].extension_range[0].options.declaration[1]":           {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 1}, []int32{4, 1, 3, 0, 5, 0, 2}, []int32{4, 1, 3, 0, 5, 0, 3, 2, 1}},
			".message_type[1].nested_type[0].extension_range[0].options.declaration[1].number":    {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 1}, []int32{4, 1, 3, 0, 5, 0, 2}, []int32{4, 1, 3, 0, 5, 0, 3, 2, 1, 1}},
			".message_type[1].nested_type[0].extension_range[0].options.declaration[1].full_name": {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 1}, []int32{4, 1, 3, 0, 5, 0, 2}, []int32{4, 1, 3, 0, 5, 0, 3, 2, 1, 2}},
			".message_type[1].nested_type[0].extension_range[0].options.declaration[1].type":      {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 5}, []int32{4, 1, 3, 0, 5, 0}, []int32{4, 1, 3, 0, 5, 0, 1}, []int32{4, 1, 3, 0, 5, 0, 2}, []int32{4, 1, 3, 0, 5, 0, 3, 2, 1, 3}},
			".message_type[1].field[0]":                                                           {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 2, 0}, []int32{4, 1, 2, 0, 1}, []int32{4, 1, 2, 0, 3}, []int32{4, 1, 2, 0, 4}, []int32{4, 1, 2, 0, 5}, []int32{4, 1, 2, 0, 6}},
			".message_type[1].field[0].label":                                                     {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 2, 0}, []int32{4, 1, 2, 0, 1}, []int32{4, 1, 2, 0, 3}, []int32{4, 1, 2, 0, 4}, []int32{4, 1, 2, 0, 5}, []int32{4, 1, 2, 0, 6}},
			".message_type[1].field[0].type_name":                                                 {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 2, 0}, []int32{4, 1, 2, 0, 1}, []int32{4, 1, 2, 0, 3}, []int32{4, 1, 2, 0, 4}, []int32{4, 1, 2, 0, 5}, []int32{4, 1, 2, 0, 6}},
			".message_type[1].field[0].name":                                                      {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 2, 0}, []int32{4, 1, 2, 0, 1}, []int32{4, 1, 2, 0, 3}, []int32{4, 1, 2, 0, 4}, []int32{4, 1, 2, 0, 5}, []int32{4, 1, 2, 0, 6}},
			".message_type[1].field[0].number":                                                    {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 2, 0}, []int32{4, 1, 2, 0, 1}, []int32{4, 1, 2, 0, 3}, []int32{4, 1, 2, 0, 4}, []int32{4, 1, 2, 0, 5}, []int32{4, 1, 2, 0, 6}},
			".message_type[2]":                                                                    {[]int32{4, 2}, []int32{4, 2, 1}},
			".message_type[2].name":                                                               {[]int32{4, 2}, []int32{4, 2, 1}},
			".message_type[2].field[0]":                                                           {[]int32{4, 2}, []int32{4, 2, 1}, []int32{4, 2, 2, 0}, []int32{4, 2, 2, 0, 1}, []int32{4, 2, 2, 0, 3}, []int32{4, 2, 2, 0, 4}, []int32{4, 2, 2, 0, 5}, []int32{4, 2, 2, 0, 6}},
			".message_type[2].field[0].type_name":                                                 {[]int32{4, 2}, []int32{4, 2, 1}, []int32{4, 2, 2, 0}, []int32{4, 2, 2, 0, 1}, []int32{4, 2, 2, 0, 3}, []int32{4, 2, 2, 0, 4}, []int32{4, 2, 2, 0, 5}, []int32{4, 2, 2, 0, 6}},
			".message_type[2].field[0].name":                                                      {[]int32{4, 2}, []int32{4, 2, 1}, []int32{4, 2, 2, 0}, []int32{4, 2, 2, 0, 1}, []int32{4, 2, 2, 0, 3}, []int32{4, 2, 2, 0, 4}, []int32{4, 2, 2, 0, 5}, []int32{4, 2, 2, 0, 6}},
			".message_type[2].field[0].number":                                                    {[]int32{4, 2}, []int32{4, 2, 1}, []int32{4, 2, 2, 0}, []int32{4, 2, 2, 0, 1}, []int32{4, 2, 2, 0, 3}, []int32{4, 2, 2, 0, 4}, []int32{4, 2, 2, 0, 5}, []int32{4, 2, 2, 0, 6}},
			".message_type[3]":                                                                    {[]int32{4, 3}, []int32{4, 3, 1}},
			".message_type[3].name":                                                               {[]int32{4, 3}, []int32{4, 3, 1}},
			".message_type[3].field[0]":                                                           {[]int32{4, 3}, []int32{4, 3, 1}, []int32{4, 3, 2, 0}, []int32{4, 3, 2, 0, 1}, []int32{4, 3, 2, 0, 3}, []int32{4, 3, 2, 0, 4}, []int32{4, 3, 2, 0, 5}, []int32{4, 3, 2, 0, 6}},
			".message_type[3].field[0].label":                                                     {[]int32{4, 3}, []int32{4, 3, 1}, []int32{4, 3, 2, 0}, []int32{4, 3, 2, 0, 1}, []int32{4, 3, 2, 0, 3}, []int32{4, 3, 2, 0, 4}, []int32{4, 3, 2, 0, 5}, []int32{4, 3, 2, 0, 6}},
			".message_type[3].field[0].type":                                                      {[]int32{4, 3}, []int32{4, 3, 1}, []int32{4, 3, 2, 0}, []int32{4, 3, 2, 0, 1}, []int32{4, 3, 2, 0, 3}, []int32{4, 3, 2, 0, 4}, []int32{4, 3, 2, 0, 5}, []int32{4, 3, 2, 0, 6}},
			".message_type[3].field[0].name":                                                      {[]int32{4, 3}, []int32{4, 3, 1}, []int32{4, 3, 2, 0}, []int32{4, 3, 2, 0, 1}, []int32{4, 3, 2, 0, 3}, []int32{4, 3, 2, 0, 4}, []int32{4, 3, 2, 0, 5}, []int32{4, 3, 2, 0, 6}},
			".message_type[3].field[0].number":                                                    {[]int32{4, 3}, []int32{4, 3, 1}, []int32{4, 3, 2, 0}, []int32{4, 3, 2, 0, 1}, []int32{4, 3, 2, 0, 3}, []int32{4, 3, 2, 0, 4}, []int32{4, 3, 2, 0, 5}, []int32{4, 3, 2, 0, 6}},
			".message_type[3].reserved_name":                                                      {[]int32{4, 3}, []int32{4, 3, 1}, []int32{4, 3, 10}},
			".message_type[3].reserved_name[0]":                                                   {[]int32{4, 3}, []int32{4, 3, 1}, []int32{4, 3, 10}, []int32{4, 3, 10, 0}},
			".message_type[3].reserved_name[1]":                                                   {[]int32{4, 3}, []int32{4, 3, 1}, []int32{4, 3, 10}, []int32{4, 3, 10, 1}},
			".message_type[4]":                                                                    {[]int32{4, 4}, []int32{4, 4, 1}},
			".message_type[4].name":                                                               {[]int32{4, 4}, []int32{4, 4, 1}},
			".message_type[4].extension":                                                          {[]int32{4, 4}, []int32{4, 4, 1}, []int32{4, 4, 6}},
			".message_type[4].extension[0]":                                                       {[]int32{4, 4}, []int32{4, 4, 1}, []int32{4, 4, 6}, []int32{4, 4, 6, 0}, []int32{4, 4, 6, 0, 1}, []int32{4, 4, 6, 0, 3}, []int32{4, 4, 6, 0, 4}, []int32{4, 4, 6, 0, 5}, []int32{4, 4, 6, 0, 6}, []int32{4, 4, 6, 0, 2}},
			".message_type[4].extension[0].extendee":                                              {[]int32{4, 4}, []int32{4, 4, 1}, []int32{4, 4, 6}, []int32{4, 4, 6, 0}, []int32{4, 4, 6, 0, 1}, []int32{4, 4, 6, 0, 3}, []int32{4, 4, 6, 0, 4}, []int32{4, 4, 6, 0, 5}, []int32{4, 4, 6, 0, 6}, []int32{4, 4, 6, 0, 2}},
			".message_type[4].extension[0].label":                                                 {[]int32{4, 4}, []int32{4, 4, 1}, []int32{4, 4, 6}, []int32{4, 4, 6, 0}, []int32{4, 4, 6, 0, 1}, []int32{4, 4, 6, 0, 3}, []int32{4, 4, 6, 0, 4}, []int32{4, 4, 6, 0, 5}, []int32{4, 4, 6, 0, 6}, []int32{4, 4, 6, 0, 2}},
			".message_type[4].extension[0].type":                                                  {[]int32{4, 4}, []int32{4, 4, 1}, []int32{4, 4, 6}, []int32{4, 4, 6, 0}, []int32{4, 4, 6, 0, 1}, []int32{4, 4, 6, 0, 3}, []int32{4, 4, 6, 0, 4}, []int32{4, 4, 6, 0, 5}, []int32{4, 4, 6, 0, 6}, []int32{4, 4, 6, 0, 2}},
			".message_type[4].extension[0].name":                                                  {[]int32{4, 4}, []int32{4, 4, 1}, []int32{4, 4, 6}, []int32{4, 4, 6, 0}, []int32{4, 4, 6, 0, 1}, []int32{4, 4, 6, 0, 3}, []int32{4, 4, 6, 0, 4}, []int32{4, 4, 6, 0, 5}, []int32{4, 4, 6, 0, 6}, []int32{4, 4, 6, 0, 2}},
			".message_type[4].extension[0].number":                                                {[]int32{4, 4}, []int32{4, 4, 1}, []int32{4, 4, 6}, []int32{4, 4, 6, 0}, []int32{4, 4, 6, 0, 1}, []int32{4, 4, 6, 0, 3}, []int32{4, 4, 6, 0, 4}, []int32{4, 4, 6, 0, 5}, []int32{4, 4, 6, 0, 6}, []int32{4, 4, 6, 0, 2}},
			".message_type[4].extension[1]":                                                       {[]int32{4, 4}, []int32{4, 4, 1}, []int32{4, 4, 6}, []int32{4, 4, 6, 1}, []int32{4, 4, 6, 1, 1}, []int32{4, 4, 6, 1, 3}, []int32{4, 4, 6, 1, 4}, []int32{4, 4, 6, 1, 5}, []int32{4, 4, 6, 1, 6}, []int32{4, 4, 6, 1, 2}},
			".message_type[4].extension[1].extendee":                                              {[]int32{4, 4}, []int32{4, 4, 1}, []int32{4, 4, 6}, []int32{4, 4, 6, 1}, []int32{4, 4, 6, 1, 1}, []int32{4, 4, 6, 1, 3}, []int32{4, 4, 6, 1, 4}, []int32{4, 4, 6, 1, 5}, []int32{4, 4, 6, 1, 6}, []int32{4, 4, 6, 1, 2}},
			".message_type[4].extension[1].label":                                                 {[]int32{4, 4}, []int32{4, 4, 1}, []int32{4, 4, 6}, []int32{4, 4, 6, 1}, []int32{4, 4, 6, 1, 1}, []int32{4, 4, 6, 1, 3}, []int32{4, 4, 6, 1, 4}, []int32{4, 4, 6, 1, 5}, []int32{4, 4, 6, 1, 6}, []int32{4, 4, 6, 1, 2}},
			".message_type[4].extension[1].type":                                                  {[]int32{4, 4}, []int32{4, 4, 1}, []int32{4, 4, 6}, []int32{4, 4, 6, 1}, []int32{4, 4, 6, 1, 1}, []int32{4, 4, 6, 1, 3}, []int32{4, 4, 6, 1, 4}, []int32{4, 4, 6, 1, 5}, []int32{4, 4, 6, 1, 6}, []int32{4, 4, 6, 1, 2}},
			".message_type[4].extension[1].name":                                                  {[]int32{4, 4}, []int32{4, 4, 1}, []int32{4, 4, 6}, []int32{4, 4, 6, 1}, []int32{4, 4, 6, 1, 1}, []int32{4, 4, 6, 1, 3}, []int32{4, 4, 6, 1, 4}, []int32{4, 4, 6, 1, 5}, []int32{4, 4, 6, 1, 6}, []int32{4, 4, 6, 1, 2}},
			".message_type[4].extension[1].number":                                                {[]int32{4, 4}, []int32{4, 4, 1}, []int32{4, 4, 6}, []int32{4, 4, 6, 1}, []int32{4, 4, 6, 1, 1}, []int32{4, 4, 6, 1, 3}, []int32{4, 4, 6, 1, 4}, []int32{4, 4, 6, 1, 5}, []int32{4, 4, 6, 1, 6}, []int32{4, 4, 6, 1, 2}},
			".message_type[4].extension[1].options":                                               {[]int32{4, 4}, []int32{4, 4, 1}, []int32{4, 4, 6}, []int32{4, 4, 6, 1}, []int32{4, 4, 6, 1, 1}, []int32{4, 4, 6, 1, 3}, []int32{4, 4, 6, 1, 4}, []int32{4, 4, 6, 1, 5}, []int32{4, 4, 6, 1, 6}, []int32{4, 4, 6, 1, 2}, []int32{4, 4, 6, 1, 8}},
			".message_type[4].extension[1].default_value":                                         {[]int32{4, 4}, []int32{4, 4, 1}, []int32{4, 4, 6}, []int32{4, 4, 6, 1}, []int32{4, 4, 6, 1, 1}, []int32{4, 4, 6, 1, 3}, []int32{4, 4, 6, 1, 4}, []int32{4, 4, 6, 1, 5}, []int32{4, 4, 6, 1, 6}, []int32{4, 4, 6, 1, 2}, []int32{4, 4, 6, 1, 7}},
		},
		false,
	)
}

func TestGetAssociatedSourcePathsProto3(t *testing.T) {
	t.Parallel()
	testGetAssociatedSourcePaths(
		t,
		"testdata/proto3/test.proto",
		map[string][]protoreflect.SourcePath{
			".syntax":                                            {[]int32{12}},
			".package":                                           {[]int32{2}},
			".dependency[0]":                                     {[]int32{3, 0}},
			".service[0]":                                        {[]int32{6, 0}},
			".service[0].name":                                   {[]int32{6, 0}},
			".service[0].method[0]":                              {[]int32{6, 0}, []int32{6, 0, 2, 0}},
			".service[0].method[0].name":                         {[]int32{6, 0}, []int32{6, 0, 2, 0}},
			".service[0].method[0].input_type":                   {[]int32{6, 0}, []int32{6, 0, 2, 0}},
			".service[0].method[0].output_type":                  {[]int32{6, 0}, []int32{6, 0, 2, 0}},
			".service[0].method[1]":                              {[]int32{6, 0}, []int32{6, 0, 2, 1}},
			".service[0].method[1].name":                         {[]int32{6, 0}, []int32{6, 0, 2, 1}},
			".service[0].method[1].client_streaming":             {[]int32{6, 0}, []int32{6, 0, 2, 1}},
			".service[0].method[1].input_type":                   {[]int32{6, 0}, []int32{6, 0, 2, 1}},
			".service[0].method[1].server_streaming":             {[]int32{6, 0}, []int32{6, 0, 2, 1}},
			".service[0].method[1].output_type":                  {[]int32{6, 0}, []int32{6, 0, 2, 1}},
			".message_type[0]":                                   {[]int32{4, 0}},
			".message_type[0].name":                              {[]int32{4, 0}},
			".message_type[0].field[0]":                          {[]int32{4, 0}, []int32{4, 0, 2, 0}},
			".message_type[0].field[0].type":                     {[]int32{4, 0}, []int32{4, 0, 2, 0}},
			".message_type[0].field[0].name":                     {[]int32{4, 0}, []int32{4, 0, 2, 0}},
			".message_type[0].field[0].number":                   {[]int32{4, 0}, []int32{4, 0, 2, 0}},
			".message_type[1]":                                   {[]int32{4, 1}},
			".message_type[1].name":                              {[]int32{4, 1}},
			".message_type[1].nested_type[0]":                    {[]int32{4, 1}, []int32{4, 1, 3, 0}},
			".message_type[1].nested_type[0].name":               {[]int32{4, 1}, []int32{4, 1, 3, 0}},
			".message_type[1].nested_type[0].field[0]":           {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 2, 0}},
			".message_type[1].nested_type[0].field[0].label":     {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 2, 0}},
			".message_type[1].nested_type[0].field[0].type":      {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 2, 0}},
			".message_type[1].nested_type[0].field[0].name":      {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 2, 0}},
			".message_type[1].nested_type[0].field[0].number":    {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 2, 0}},
			".message_type[1].nested_type[0].field[1]":           {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 2, 1}},
			".message_type[1].nested_type[0].field[1].type_name": {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 2, 1}},
			".message_type[1].nested_type[0].field[1].name":      {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 2, 1}},
			".message_type[1].nested_type[0].field[1].number":    {[]int32{4, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 2, 1}},
			".message_type[1].enum_type[0]":                      {[]int32{4, 1}, []int32{4, 1, 4, 0}},
			".message_type[1].enum_type[0].name":                 {[]int32{4, 1}, []int32{4, 1, 4, 0}},
			".message_type[1].enum_type[0].value[0]":             {[]int32{4, 1}, []int32{4, 1, 4, 0}, []int32{4, 1, 4, 0, 2, 0}},
			".message_type[1].enum_type[0].value[0].name":        {[]int32{4, 1}, []int32{4, 1, 4, 0}, []int32{4, 1, 4, 0, 2, 0}},
			".message_type[1].enum_type[0].value[0].number":      {[]int32{4, 1}, []int32{4, 1, 4, 0}, []int32{4, 1, 4, 0, 2, 0}},
			".message_type[1].enum_type[0].value[1]":             {[]int32{4, 1}, []int32{4, 1, 4, 0}, []int32{4, 1, 4, 0, 2, 1}},
			".message_type[1].enum_type[0].value[1].name":        {[]int32{4, 1}, []int32{4, 1, 4, 0}, []int32{4, 1, 4, 0, 2, 1}},
			".message_type[1].enum_type[0].value[1].number":      {[]int32{4, 1}, []int32{4, 1, 4, 0}, []int32{4, 1, 4, 0, 2, 1}},
			".message_type[1].field[0]":                          {[]int32{4, 1}, []int32{4, 1, 2, 0}},
			".message_type[1].field[0].label":                    {[]int32{4, 1}, []int32{4, 1, 2, 0}},
			".message_type[1].field[0].type_name":                {[]int32{4, 1}, []int32{4, 1, 2, 0}},
			".message_type[1].field[0].name":                     {[]int32{4, 1}, []int32{4, 1, 2, 0}},
			".message_type[1].field[0].number":                   {[]int32{4, 1}, []int32{4, 1, 2, 0}},
			".message_type[1].field[1]":                          {[]int32{4, 1}, []int32{4, 1, 2, 1}},
			".message_type[1].field[1].type_name":                {[]int32{4, 1}, []int32{4, 1, 2, 1}},
			".message_type[1].field[1].name":                     {[]int32{4, 1}, []int32{4, 1, 2, 1}},
			".message_type[1].field[1].number":                   {[]int32{4, 1}, []int32{4, 1, 2, 1}},
			".message_type[2]":                                   {[]int32{4, 2}},
			".message_type[2].name":                              {[]int32{4, 2}},
			".message_type[2].field[0]":                          {[]int32{4, 2}, []int32{4, 2, 2, 0}},
			".message_type[2].field[0].type_name":                {[]int32{4, 2}, []int32{4, 2, 2, 0}},
			".message_type[2].field[0].name":                     {[]int32{4, 2}, []int32{4, 2, 2, 0}},
			".message_type[2].field[0].number":                   {[]int32{4, 2}, []int32{4, 2, 2, 0}},
			".message_type[3]":                                   {[]int32{4, 3}},
			".message_type[3].name":                              {[]int32{4, 3}},
			".message_type[3].field[0]":                          {[]int32{4, 3}, []int32{4, 3, 2, 0}},
			".message_type[3].field[0].label":                    {[]int32{4, 3}, []int32{4, 3, 2, 0}},
			".message_type[3].field[0].type":                     {[]int32{4, 3}, []int32{4, 3, 2, 0}},
			".message_type[3].field[0].name":                     {[]int32{4, 3}, []int32{4, 3, 2, 0}},
			".message_type[3].field[0].number":                   {[]int32{4, 3}, []int32{4, 3, 2, 0}},
			".message_type[3].reserved_name":                     {[]int32{4, 3}, []int32{4, 3, 10}},
			".message_type[3].reserved_name[0]":                  {[]int32{4, 3}, []int32{4, 3, 10}, []int32{4, 3, 10, 0}},
			".message_type[3].reserved_name[1]":                  {[]int32{4, 3}, []int32{4, 3, 10}, []int32{4, 3, 10, 1}},
		},
		true,
	)
	testGetAssociatedSourcePaths(
		t,
		"testdata/proto3/test.proto",
		map[string][]protoreflect.SourcePath{
			".syntax":                                            {[]int32{12}},
			".package":                                           {[]int32{2}},
			".dependency[0]":                                     {[]int32{3, 0}},
			".service[0]":                                        {[]int32{6, 0}, []int32{6, 0, 1}},
			".service[0].name":                                   {[]int32{6, 0}, []int32{6, 0, 1}},
			".service[0].method[0]":                              {[]int32{6, 0}, []int32{6, 0, 1}, []int32{6, 0, 2, 0}, []int32{6, 0, 2, 0, 1}, []int32{6, 0, 2, 0, 2}, []int32{6, 0, 2, 0, 3}, []int32{6, 0, 2, 0, 5}, []int32{6, 0, 2, 0, 6}},
			".service[0].method[0].name":                         {[]int32{6, 0}, []int32{6, 0, 1}, []int32{6, 0, 2, 0}, []int32{6, 0, 2, 0, 1}, []int32{6, 0, 2, 0, 2}, []int32{6, 0, 2, 0, 3}, []int32{6, 0, 2, 0, 5}, []int32{6, 0, 2, 0, 6}},
			".service[0].method[0].input_type":                   {[]int32{6, 0}, []int32{6, 0, 1}, []int32{6, 0, 2, 0}, []int32{6, 0, 2, 0, 1}, []int32{6, 0, 2, 0, 2}, []int32{6, 0, 2, 0, 3}, []int32{6, 0, 2, 0, 5}, []int32{6, 0, 2, 0, 6}},
			".service[0].method[0].output_type":                  {[]int32{6, 0}, []int32{6, 0, 1}, []int32{6, 0, 2, 0}, []int32{6, 0, 2, 0, 1}, []int32{6, 0, 2, 0, 2}, []int32{6, 0, 2, 0, 3}, []int32{6, 0, 2, 0, 5}, []int32{6, 0, 2, 0, 6}},
			".service[0].method[1]":                              {[]int32{6, 0}, []int32{6, 0, 1}, []int32{6, 0, 2, 1}, []int32{6, 0, 2, 1, 1}, []int32{6, 0, 2, 1, 2}, []int32{6, 0, 2, 1, 3}, []int32{6, 0, 2, 1, 5}, []int32{6, 0, 2, 1, 6}},
			".service[0].method[1].name":                         {[]int32{6, 0}, []int32{6, 0, 1}, []int32{6, 0, 2, 1}, []int32{6, 0, 2, 1, 1}, []int32{6, 0, 2, 1, 2}, []int32{6, 0, 2, 1, 3}, []int32{6, 0, 2, 1, 5}, []int32{6, 0, 2, 1, 6}},
			".service[0].method[1].client_streaming":             {[]int32{6, 0}, []int32{6, 0, 1}, []int32{6, 0, 2, 1}, []int32{6, 0, 2, 1, 1}, []int32{6, 0, 2, 1, 2}, []int32{6, 0, 2, 1, 3}, []int32{6, 0, 2, 1, 5}, []int32{6, 0, 2, 1, 6}},
			".service[0].method[1].input_type":                   {[]int32{6, 0}, []int32{6, 0, 1}, []int32{6, 0, 2, 1}, []int32{6, 0, 2, 1, 1}, []int32{6, 0, 2, 1, 2}, []int32{6, 0, 2, 1, 3}, []int32{6, 0, 2, 1, 5}, []int32{6, 0, 2, 1, 6}},
			".service[0].method[1].server_streaming":             {[]int32{6, 0}, []int32{6, 0, 1}, []int32{6, 0, 2, 1}, []int32{6, 0, 2, 1, 1}, []int32{6, 0, 2, 1, 2}, []int32{6, 0, 2, 1, 3}, []int32{6, 0, 2, 1, 5}, []int32{6, 0, 2, 1, 6}},
			".service[0].method[1].output_type":                  {[]int32{6, 0}, []int32{6, 0, 1}, []int32{6, 0, 2, 1}, []int32{6, 0, 2, 1, 1}, []int32{6, 0, 2, 1, 2}, []int32{6, 0, 2, 1, 3}, []int32{6, 0, 2, 1, 5}, []int32{6, 0, 2, 1, 6}},
			".message_type[0]":                                   {[]int32{4, 0}, []int32{4, 0, 1}},
			".message_type[0].name":                              {[]int32{4, 0}, []int32{4, 0, 1}},
			".message_type[0].field[0]":                          {[]int32{4, 0}, []int32{4, 0, 1}, []int32{4, 0, 2, 0}, []int32{4, 0, 2, 0, 1}, []int32{4, 0, 2, 0, 3}, []int32{4, 0, 2, 0, 4}, []int32{4, 0, 2, 0, 5}, []int32{4, 0, 2, 0, 6}},
			".message_type[0].field[0].type":                     {[]int32{4, 0}, []int32{4, 0, 1}, []int32{4, 0, 2, 0}, []int32{4, 0, 2, 0, 1}, []int32{4, 0, 2, 0, 3}, []int32{4, 0, 2, 0, 4}, []int32{4, 0, 2, 0, 5}, []int32{4, 0, 2, 0, 6}},
			".message_type[0].field[0].name":                     {[]int32{4, 0}, []int32{4, 0, 1}, []int32{4, 0, 2, 0}, []int32{4, 0, 2, 0, 1}, []int32{4, 0, 2, 0, 3}, []int32{4, 0, 2, 0, 4}, []int32{4, 0, 2, 0, 5}, []int32{4, 0, 2, 0, 6}},
			".message_type[0].field[0].number":                   {[]int32{4, 0}, []int32{4, 0, 1}, []int32{4, 0, 2, 0}, []int32{4, 0, 2, 0, 1}, []int32{4, 0, 2, 0, 3}, []int32{4, 0, 2, 0, 4}, []int32{4, 0, 2, 0, 5}, []int32{4, 0, 2, 0, 6}},
			".message_type[1]":                                   {[]int32{4, 1}, []int32{4, 1, 1}},
			".message_type[1].name":                              {[]int32{4, 1}, []int32{4, 1, 1}},
			".message_type[1].nested_type[0]":                    {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}},
			".message_type[1].nested_type[0].name":               {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}},
			".message_type[1].nested_type[0].field[0]":           {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 2, 0}, []int32{4, 1, 3, 0, 2, 0, 1}, []int32{4, 1, 3, 0, 2, 0, 3}, []int32{4, 1, 3, 0, 2, 0, 4}, []int32{4, 1, 3, 0, 2, 0, 5}, []int32{4, 1, 3, 0, 2, 0, 6}},
			".message_type[1].nested_type[0].field[0].label":     {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 2, 0}, []int32{4, 1, 3, 0, 2, 0, 1}, []int32{4, 1, 3, 0, 2, 0, 3}, []int32{4, 1, 3, 0, 2, 0, 4}, []int32{4, 1, 3, 0, 2, 0, 5}, []int32{4, 1, 3, 0, 2, 0, 6}},
			".message_type[1].nested_type[0].field[0].type":      {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 2, 0}, []int32{4, 1, 3, 0, 2, 0, 1}, []int32{4, 1, 3, 0, 2, 0, 3}, []int32{4, 1, 3, 0, 2, 0, 4}, []int32{4, 1, 3, 0, 2, 0, 5}, []int32{4, 1, 3, 0, 2, 0, 6}},
			".message_type[1].nested_type[0].field[0].name":      {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 2, 0}, []int32{4, 1, 3, 0, 2, 0, 1}, []int32{4, 1, 3, 0, 2, 0, 3}, []int32{4, 1, 3, 0, 2, 0, 4}, []int32{4, 1, 3, 0, 2, 0, 5}, []int32{4, 1, 3, 0, 2, 0, 6}},
			".message_type[1].nested_type[0].field[0].number":    {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 2, 0}, []int32{4, 1, 3, 0, 2, 0, 1}, []int32{4, 1, 3, 0, 2, 0, 3}, []int32{4, 1, 3, 0, 2, 0, 4}, []int32{4, 1, 3, 0, 2, 0, 5}, []int32{4, 1, 3, 0, 2, 0, 6}},
			".message_type[1].nested_type[0].field[1]":           {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 2, 1}, []int32{4, 1, 3, 0, 2, 1, 1}, []int32{4, 1, 3, 0, 2, 1, 3}, []int32{4, 1, 3, 0, 2, 1, 4}, []int32{4, 1, 3, 0, 2, 1, 5}, []int32{4, 1, 3, 0, 2, 1, 6}},
			".message_type[1].nested_type[0].field[1].type_name": {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 2, 1}, []int32{4, 1, 3, 0, 2, 1, 1}, []int32{4, 1, 3, 0, 2, 1, 3}, []int32{4, 1, 3, 0, 2, 1, 4}, []int32{4, 1, 3, 0, 2, 1, 5}, []int32{4, 1, 3, 0, 2, 1, 6}},
			".message_type[1].nested_type[0].field[1].name":      {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 2, 1}, []int32{4, 1, 3, 0, 2, 1, 1}, []int32{4, 1, 3, 0, 2, 1, 3}, []int32{4, 1, 3, 0, 2, 1, 4}, []int32{4, 1, 3, 0, 2, 1, 5}, []int32{4, 1, 3, 0, 2, 1, 6}},
			".message_type[1].nested_type[0].field[1].number":    {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 3, 0}, []int32{4, 1, 3, 0, 1}, []int32{4, 1, 3, 0, 2, 1}, []int32{4, 1, 3, 0, 2, 1, 1}, []int32{4, 1, 3, 0, 2, 1, 3}, []int32{4, 1, 3, 0, 2, 1, 4}, []int32{4, 1, 3, 0, 2, 1, 5}, []int32{4, 1, 3, 0, 2, 1, 6}},
			".message_type[1].enum_type[0]":                      {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 4, 0}, []int32{4, 1, 4, 0, 1}},
			".message_type[1].enum_type[0].name":                 {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 4, 0}, []int32{4, 1, 4, 0, 1}},
			".message_type[1].enum_type[0].value[0]":             {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 4, 0}, []int32{4, 1, 4, 0, 1}, []int32{4, 1, 4, 0, 2, 0}, []int32{4, 1, 4, 0, 2, 0, 1}, []int32{4, 1, 4, 0, 2, 0, 2}},
			".message_type[1].enum_type[0].value[0].name":        {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 4, 0}, []int32{4, 1, 4, 0, 1}, []int32{4, 1, 4, 0, 2, 0}, []int32{4, 1, 4, 0, 2, 0, 1}, []int32{4, 1, 4, 0, 2, 0, 2}},
			".message_type[1].enum_type[0].value[0].number":      {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 4, 0}, []int32{4, 1, 4, 0, 1}, []int32{4, 1, 4, 0, 2, 0}, []int32{4, 1, 4, 0, 2, 0, 1}, []int32{4, 1, 4, 0, 2, 0, 2}},
			".message_type[1].enum_type[0].value[1]":             {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 4, 0}, []int32{4, 1, 4, 0, 1}, []int32{4, 1, 4, 0, 2, 1}, []int32{4, 1, 4, 0, 2, 1, 1}, []int32{4, 1, 4, 0, 2, 1, 2}},
			".message_type[1].enum_type[0].value[1].name":        {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 4, 0}, []int32{4, 1, 4, 0, 1}, []int32{4, 1, 4, 0, 2, 1}, []int32{4, 1, 4, 0, 2, 1, 1}, []int32{4, 1, 4, 0, 2, 1, 2}},
			".message_type[1].enum_type[0].value[1].number":      {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 4, 0}, []int32{4, 1, 4, 0, 1}, []int32{4, 1, 4, 0, 2, 1}, []int32{4, 1, 4, 0, 2, 1, 1}, []int32{4, 1, 4, 0, 2, 1, 2}},
			".message_type[1].field[0]":                          {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 2, 0}, []int32{4, 1, 2, 0, 1}, []int32{4, 1, 2, 0, 3}, []int32{4, 1, 2, 0, 4}, []int32{4, 1, 2, 0, 5}, []int32{4, 1, 2, 0, 6}},
			".message_type[1].field[0].label":                    {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 2, 0}, []int32{4, 1, 2, 0, 1}, []int32{4, 1, 2, 0, 3}, []int32{4, 1, 2, 0, 4}, []int32{4, 1, 2, 0, 5}, []int32{4, 1, 2, 0, 6}},
			".message_type[1].field[0].type_name":                {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 2, 0}, []int32{4, 1, 2, 0, 1}, []int32{4, 1, 2, 0, 3}, []int32{4, 1, 2, 0, 4}, []int32{4, 1, 2, 0, 5}, []int32{4, 1, 2, 0, 6}},
			".message_type[1].field[0].name":                     {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 2, 0}, []int32{4, 1, 2, 0, 1}, []int32{4, 1, 2, 0, 3}, []int32{4, 1, 2, 0, 4}, []int32{4, 1, 2, 0, 5}, []int32{4, 1, 2, 0, 6}},
			".message_type[1].field[0].number":                   {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 2, 0}, []int32{4, 1, 2, 0, 1}, []int32{4, 1, 2, 0, 3}, []int32{4, 1, 2, 0, 4}, []int32{4, 1, 2, 0, 5}, []int32{4, 1, 2, 0, 6}},
			".message_type[1].field[1]":                          {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 2, 1}, []int32{4, 1, 2, 1, 1}, []int32{4, 1, 2, 1, 3}, []int32{4, 1, 2, 1, 4}, []int32{4, 1, 2, 1, 5}, []int32{4, 1, 2, 1, 6}},
			".message_type[1].field[1].type_name":                {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 2, 1}, []int32{4, 1, 2, 1, 1}, []int32{4, 1, 2, 1, 3}, []int32{4, 1, 2, 1, 4}, []int32{4, 1, 2, 1, 5}, []int32{4, 1, 2, 1, 6}},
			".message_type[1].field[1].name":                     {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 2, 1}, []int32{4, 1, 2, 1, 1}, []int32{4, 1, 2, 1, 3}, []int32{4, 1, 2, 1, 4}, []int32{4, 1, 2, 1, 5}, []int32{4, 1, 2, 1, 6}},
			".message_type[1].field[1].number":                   {[]int32{4, 1}, []int32{4, 1, 1}, []int32{4, 1, 2, 1}, []int32{4, 1, 2, 1, 1}, []int32{4, 1, 2, 1, 3}, []int32{4, 1, 2, 1, 4}, []int32{4, 1, 2, 1, 5}, []int32{4, 1, 2, 1, 6}},
			".message_type[2]":                                   {[]int32{4, 2}, []int32{4, 2, 1}},
			".message_type[2].name":                              {[]int32{4, 2}, []int32{4, 2, 1}},
			".message_type[2].field[0]":                          {[]int32{4, 2}, []int32{4, 2, 1}, []int32{4, 2, 2, 0}, []int32{4, 2, 2, 0, 1}, []int32{4, 2, 2, 0, 3}, []int32{4, 2, 2, 0, 4}, []int32{4, 2, 2, 0, 5}, []int32{4, 2, 2, 0, 6}},
			".message_type[2].field[0].type_name":                {[]int32{4, 2}, []int32{4, 2, 1}, []int32{4, 2, 2, 0}, []int32{4, 2, 2, 0, 1}, []int32{4, 2, 2, 0, 3}, []int32{4, 2, 2, 0, 4}, []int32{4, 2, 2, 0, 5}, []int32{4, 2, 2, 0, 6}},
			".message_type[2].field[0].name":                     {[]int32{4, 2}, []int32{4, 2, 1}, []int32{4, 2, 2, 0}, []int32{4, 2, 2, 0, 1}, []int32{4, 2, 2, 0, 3}, []int32{4, 2, 2, 0, 4}, []int32{4, 2, 2, 0, 5}, []int32{4, 2, 2, 0, 6}},
			".message_type[2].field[0].number":                   {[]int32{4, 2}, []int32{4, 2, 1}, []int32{4, 2, 2, 0}, []int32{4, 2, 2, 0, 1}, []int32{4, 2, 2, 0, 3}, []int32{4, 2, 2, 0, 4}, []int32{4, 2, 2, 0, 5}, []int32{4, 2, 2, 0, 6}},
			".message_type[3]":                                   {[]int32{4, 3}, []int32{4, 3, 1}},
			".message_type[3].name":                              {[]int32{4, 3}, []int32{4, 3, 1}},
			".message_type[3].field[0]":                          {[]int32{4, 3}, []int32{4, 3, 1}, []int32{4, 3, 2, 0}, []int32{4, 3, 2, 0, 1}, []int32{4, 3, 2, 0, 3}, []int32{4, 3, 2, 0, 4}, []int32{4, 3, 2, 0, 5}, []int32{4, 3, 2, 0, 6}},
			".message_type[3].field[0].label":                    {[]int32{4, 3}, []int32{4, 3, 1}, []int32{4, 3, 2, 0}, []int32{4, 3, 2, 0, 1}, []int32{4, 3, 2, 0, 3}, []int32{4, 3, 2, 0, 4}, []int32{4, 3, 2, 0, 5}, []int32{4, 3, 2, 0, 6}},
			".message_type[3].field[0].type":                     {[]int32{4, 3}, []int32{4, 3, 1}, []int32{4, 3, 2, 0}, []int32{4, 3, 2, 0, 1}, []int32{4, 3, 2, 0, 3}, []int32{4, 3, 2, 0, 4}, []int32{4, 3, 2, 0, 5}, []int32{4, 3, 2, 0, 6}},
			".message_type[3].field[0].name":                     {[]int32{4, 3}, []int32{4, 3, 1}, []int32{4, 3, 2, 0}, []int32{4, 3, 2, 0, 1}, []int32{4, 3, 2, 0, 3}, []int32{4, 3, 2, 0, 4}, []int32{4, 3, 2, 0, 5}, []int32{4, 3, 2, 0, 6}},
			".message_type[3].field[0].number":                   {[]int32{4, 3}, []int32{4, 3, 1}, []int32{4, 3, 2, 0}, []int32{4, 3, 2, 0, 1}, []int32{4, 3, 2, 0, 3}, []int32{4, 3, 2, 0, 4}, []int32{4, 3, 2, 0, 5}, []int32{4, 3, 2, 0, 6}},
			".message_type[3].reserved_name":                     {[]int32{4, 3}, []int32{4, 3, 1}, []int32{4, 3, 10}},
			".message_type[3].reserved_name[0]":                  {[]int32{4, 3}, []int32{4, 3, 1}, []int32{4, 3, 10}, []int32{4, 3, 10, 0}},
			".message_type[3].reserved_name[1]":                  {[]int32{4, 3}, []int32{4, 3, 1}, []int32{4, 3, 10}, []int32{4, 3, 10, 1}},
		},
		false,
	)
}

func testGetAssociatedSourcePaths(
	t *testing.T,
	testFilePath string,
	sourcePathToExpectedAssociatedPaths map[string][]protoreflect.SourcePath,
	excludeChildAssociatedPaths bool,
) {
	compiler := protocompile.Compiler{
		Resolver:       &protocompile.SourceResolver{},
		SourceInfoMode: protocompile.SourceInfoStandard,
	}
	files, err := compiler.Compile(context.Background(), testFilePath)
	require.NoError(t, err)
	require.Len(t, files, 1, "expect only one test file")
	sourceLocations := files[0].SourceLocations()
	// SourceLocations are indexed starting from 1
	for i := 1; i < sourceLocations.Len(); i++ {
		sourceLocation := sourceLocations.Get(i)
		associatedSourcePaths, err := getAssociatedSourcePaths(sourceLocation.Path, excludeChildAssociatedPaths)
		require.NoError(t, err)
		expectedAssociatedPaths, ok := sourcePathToExpectedAssociatedPaths[sourceLocation.Path.String()]
		require.True(t, ok, sourceLocation.Path)
		require.Equal(t, expectedAssociatedPaths, associatedSourcePaths, i)
	}
}
