import './style.scss';

import clsx from 'clsx';
import { type PropsWithChildren, type ReactNode, useId, useMemo } from 'react';

import { useI18nContext } from '../../../../../../i18n/i18n-react';
import { isPresent } from '../../../../../../shared/defguard-ui/utils/isPresent';
import { DeviceSetupMethod } from '../../types';

type StandaloneConfig = {
  icon: ReactNode;
  title: string;
  description: string;
  testId: string;
  extras?: ReactNode;
};

type Props = {
  active: boolean;
  onClick: () => void;
  methodType?: DeviceSetupMethod;
  custom?: StandaloneConfig;
  disabled?: boolean;
};

type ContentConfiguration = {
  title: string;
  description: string;
  testId: string;
} & Pick<Props, 'onClick' | 'active' | 'disabled'> &
  PropsWithChildren;

export const DeviceSetupMethodCard = ({
  methodType,
  active,
  onClick,
  custom,
  disabled = false,
}: Props) => {
  const { LL } = useI18nContext();
  const localLL = LL.addDevicePage.steps.setupMethod.methods;

  const [title, description, testId] = useMemo(() => {
    if (!isPresent(custom) && methodType) {
      const testId = `add-device-method-${methodType.valueOf()}`;
      switch (methodType) {
        case DeviceSetupMethod.CLIENT:
          return [localLL.client.title(), localLL.client.description(), testId];
        case DeviceSetupMethod.NATIVE_WG:
          return [localLL.wg.title(), localLL.wg.description(), testId];
        default:
          throw Error('Unimplemented setup method supplied to method card.');
      }
    }
    if (isPresent(custom)) {
      return [custom.title, custom.description, custom.testId];
    }
    throw Error('Bad props for DeviceSetupMethodCard');
  }, [custom, localLL.client, localLL.wg, methodType]);

  return (
    <Content
      active={active}
      description={description}
      onClick={onClick}
      testId={testId}
      title={title}
      disabled={disabled}
    >
      {methodType === DeviceSetupMethod.CLIENT && (
        <>
          <div className="icon">
            <PhoneSvg />
            <DesktopSvg />
          </div>
          <div className="icon">
            <DefguardLogo />
          </div>
        </>
      )}
      {methodType === DeviceSetupMethod.NATIVE_WG && (
        <div className="wg-icon">
          <WireguardLogo />
        </div>
      )}
      {isPresent(custom) && (
        <>
          <div className="icon">{custom.icon}</div>
          {custom.extras}
        </>
      )}
    </Content>
  );
};

const Content = ({
  active,
  description,
  onClick,
  testId,
  title,
  children,
  disabled = false,
}: ContentConfiguration) => {
  return (
    <div
      className={clsx('device-method-card', {
        active,
        disabled,
      })}
      data-testid={testId}
      data-active={active}
      onClick={() => {
        if (!disabled) {
          onClick?.();
        }
      }}
    >
      <p className="title">{title}</p>
      <p className="description">{description}</p>
      {children}
    </div>
  );
};

const PhoneSvg = () => {
  return (
    <svg
      xmlns="http://www.w3.org/2000/svg"
      width="41"
      height="64"
      viewBox="0 0 41 64"
      fill="none"
    >
      <path
        d="M40.8332 54.4228L40.8333 9.57727C40.8333 4.62831 36.8213 0.616333 31.8724 0.616333L9.44274 0.616332C4.49377 0.616331 0.481838 4.62831 0.481838 9.57727L0.481836 54.4228C0.481836 59.3718 4.49377 63.3837 9.44273 63.3837L31.8723 63.3837C36.8213 63.3837 40.8332 59.3718 40.8332 54.4228Z"
        fill="#899CA8"
      />
      <circle cx="20.788" cy="54.0529" r="4.1105" fill="#D9D9D9" />
    </svg>
  );
};

const DesktopSvg = () => {
  return (
    <svg xmlns="http://www.w3.org/2000/svg" width={77} height={64} fill="none">
      <path
        fill="#DFDFDD"
        d="M8.09 44.438a8.93 8.93 0 0 1 8.342-5.74H61.27a8.93 8.93 0 0 1 8.42 5.958l2.627 7.442c2.05 5.81-2.26 11.902-8.42 11.902h-50.31c-6.26 0-10.577-6.273-8.342-12.12l2.845-7.442Z"
      />
      <rect width={58.047} height={38.698} x={8.797} fill="#899CA8" rx={8.93} />
      <rect
        width={14.884}
        height={5.953}
        x={31.123}
        y={55.07}
        fill="#CECECC"
        rx={2.977}
      />
      <rect
        width={35.721}
        height={2.977}
        x={16.239}
        y={50.605}
        fill="#CECECC"
        rx={1.488}
      />
      <rect width={8.93} height={2.977} x={53.448} y={50.605} fill="#CECECC" rx={1.488} />
      <rect
        width={4.465}
        height={2.977}
        x={16.239}
        y={41.674}
        fill="#CECECC"
        rx={1.488}
      />
      <rect
        width={4.465}
        height={2.977}
        x={19.216}
        y={46.139}
        fill="#CECECC"
        rx={1.488}
      />
      <rect
        width={4.465}
        height={2.977}
        x={13.262}
        y={46.139}
        fill="#CECECC"
        rx={1.488}
      />
      <rect
        width={4.465}
        height={2.977}
        x={10.285}
        y={50.605}
        fill="#CECECC"
        rx={1.488}
      />
      <rect
        width={4.465}
        height={2.977}
        x={22.192}
        y={41.674}
        fill="#CECECC"
        rx={1.488}
      />
      <rect
        width={4.465}
        height={2.977}
        x={25.169}
        y={46.139}
        fill="#CECECC"
        rx={1.488}
      />
      <rect
        width={4.465}
        height={2.977}
        x={28.146}
        y={41.674}
        fill="#CECECC"
        rx={1.488}
      />
      <rect
        width={4.465}
        height={2.977}
        x={31.123}
        y={46.139}
        fill="#CECECC"
        rx={1.488}
      />
      <rect
        width={4.465}
        height={2.977}
        x={34.099}
        y={41.674}
        fill="#CECECC"
        rx={1.488}
      />
      <rect
        width={4.465}
        height={2.977}
        x={37.076}
        y={46.139}
        fill="#CECECC"
        rx={1.488}
      />
      <rect
        width={4.465}
        height={2.977}
        x={40.053}
        y={41.674}
        fill="#CECECC"
        rx={1.488}
      />
      <rect width={4.465} height={2.977} x={43.03} y={46.139} fill="#CECECC" rx={1.488} />
      <rect
        width={4.465}
        height={2.977}
        x={46.006}
        y={41.674}
        fill="#CECECC"
        rx={1.488}
      />
      <rect
        width={4.465}
        height={2.977}
        x={48.983}
        y={46.139}
        fill="#CECECC"
        rx={1.488}
      />
      <rect width={4.465} height={2.977} x={51.96} y={41.674} fill="#CECECC" rx={1.488} />
      <rect
        width={4.465}
        height={2.977}
        x={54.937}
        y={46.139}
        fill="#CECECC"
        rx={1.488}
      />
      <rect
        width={4.465}
        height={2.977}
        x={57.913}
        y={41.674}
        fill="#CECECC"
        rx={1.488}
      />
      <rect width={4.465} height={2.977} x={60.89} y={46.139} fill="#CECECC" rx={1.488} />
      <rect
        width={4.465}
        height={2.977}
        x={63.867}
        y={50.605}
        fill="#CECECC"
        rx={1.488}
      />
    </svg>
  );
};

const DefguardLogo = () => {
  const id = useId();
  return (
    <svg xmlns="http://www.w3.org/2000/svg" width={211} height={53} fill="none">
      <g fill="#222" clipPath={`url(#${id})`}>
        <path d="M206.588 7.884v12.779h-.289c-1.294-2.452-3.861-4.15-6.814-4.15h-2.857a7.719 7.719 0 0 0-7.721 7.72v7.741a7.719 7.719 0 0 0 7.721 7.721h2.857c2.915 0 5.443-1.64 6.756-4.034h.212l.155 3.61h2.779V7.883h-2.799ZM200.489 36.8h-2.857a5.783 5.783 0 0 1-5.791-5.791v-5.79a5.783 5.783 0 0 1 5.791-5.792h2.857a5.782 5.782 0 0 1 5.79 5.791v5.79a5.783 5.783 0 0 1-5.79 5.792ZM177.982 16.937h-1.255c-1.062 0-1.93.869-1.93 1.93V39.29h3.185V19.466h8.686v-2.529h-8.686ZM167.423 39.29l-.039-15.867c0-3.803-3.088-6.91-6.91-6.91h-3.764c-3.803 0-6.911 3.088-6.911 6.91h2.877a4.941 4.941 0 0 1 4.922-4.42h1.969a4.98 4.98 0 0 1 4.98 4.98v1.95h-8.571c-3.802 0-6.91 3.088-6.91 6.91a6.903 6.903 0 0 0 6.91 6.91h2.78c2.297 0 4.304-1.139 5.559-2.857h.212v2.432h2.857l.039-.038Zm-8.59-2.066h-2.316a4.379 4.379 0 0 1-4.382-4.381 4.391 4.391 0 0 1 4.382-4.382h8.049v3.05a5.733 5.733 0 0 1-5.733 5.733v-.02ZM140.843 31.009a5.783 5.783 0 0 1-5.79 5.79h-1.467a5.783 5.783 0 0 1-5.791-5.79V16.937h-2.934v15.056a7.719 7.719 0 0 0 7.721 7.721h1.486c2.915 0 5.444-1.64 6.756-4.034h.212l.155 3.61h2.78V16.937h-3.07l-.038 14.072h-.02ZM115.152 35.12h-9.922c-.946 0-1.699-.675-1.699-1.621a1.69 1.69 0 0 1 1.699-1.699h6.08a6.763 6.763 0 0 0 6.756-6.756v-1.776c0-2.142-1.023-4.034-2.567-5.27v-1.563h4.671v-2.528h-4.903c-1.061 0-1.93.868-1.93 1.93v1.023a6.379 6.379 0 0 0-2.027-.347h-3.32a6.764 6.764 0 0 0-6.756 6.755v1.776a6.736 6.736 0 0 0 2.857 5.502c-1.756.077-3.165 1.505-3.165 3.281 0 1.467.984 2.664 2.316 3.05-2.239.695-3.86 2.567-3.86 4.787v.637c0 2.818 2.586 5.096 5.79 5.096h9.941a5.783 5.783 0 0 0 5.791-5.79v-.734a5.783 5.783 0 0 0-5.791-5.791l.039.038Zm-10.984-12.392c0-2.046 1.738-3.706 3.861-3.706h3.301c2.123 0 3.86 1.66 3.86 3.706v2.857c0 2.046-1.737 3.706-3.86 3.706h-3.301c-2.123 0-3.861-1.66-3.861-3.706v-2.857Zm10.849 21.928h-9.671a3.402 3.402 0 0 1-3.397-3.397 3.402 3.402 0 0 1 3.397-3.398h9.671a3.402 3.402 0 0 1 3.397 3.398 3.402 3.402 0 0 1-3.397 3.397ZM96.582 10.432V7.884h-6.485c-1.062 0-1.93.869-1.93 1.93v7.123h-3.88v2.606h3.88V39.29h2.837V19.543h5.076v-2.606h-5.076v-6.505h5.578ZM78.65 28.75h2.722v-2.586c0-5.328-4.324-9.651-9.652-9.651h-.888c-5.327 0-9.65 4.323-9.65 9.65v3.379c0 5.598 4.574 10.153 10.152 10.153h.888c4.401 0 8.069-2.953 9.227-6.968h-3.32c-1.255 3.166-4.13 5.25-7.895 4.382-3.764-.87-6.022-4.247-6.022-7.934v-.425H78.65Zm-7.22-9.844c3.977 0 6.834 3.224 6.834 7.22v.173H64.212v-.174a7.22 7.22 0 0 1 7.219-7.219ZM52.726 7.884v12.779h-.29c-1.292-2.452-3.86-4.15-6.813-4.15h-2.857a7.719 7.719 0 0 0-7.72 7.72v7.741a7.719 7.719 0 0 0 7.72 7.721h2.857c2.915 0 5.443-1.64 6.756-4.034h.212l.155 3.61h2.78V7.883h-2.8Zm-6.1 28.916H43.77a5.783 5.783 0 0 1-5.79-5.791v-5.79a5.783 5.783 0 0 1 5.79-5.792h2.857a5.783 5.783 0 0 1 5.79 5.791v5.79a5.783 5.783 0 0 1-5.79 5.792ZM22.653.163v11.29L12.88 5.808.3 13.07v27.422l12.57 7.262 9.773-5.645v5.818l-4.729 2.736 2.797 1.618 4.729-2.737V25.968l-12.57-7.263-9.773 5.645v-9.673l9.773-5.644 9.773 5.645v3.214l2.797 1.617V1.78L22.643.163h.01ZM3.097 38.875v-9.673l9.773 5.645 9.773-5.645v9.673L12.87 44.52l-9.773-5.645ZM21.25 26.78l-8.38 4.842-8.38-4.842 8.38-4.841 8.38 4.841Z" />
      </g>
      <defs>
        <clipPath id={id}>
          <path fill="#fff" d="M.3.163h210.4V52.28H.3z" />
        </clipPath>
      </defs>
    </svg>
  );
};

const WireguardLogo = () => {
  return (
    <svg xmlns="http://www.w3.org/2000/svg" width={219} height={36} fill="none">
      <path
        fill="#485964"
        d="M21.073 3.967a.168.168 0 0 0-.067.207c.01.024.026.046.045.063a.254.254 0 0 0 .349.094l.22-.116.117-.063-.095-.081a10.782 10.782 0 0 0-.175-.15c-.15-.124-.274-.046-.394.046Z"
      />
      <path
        fill="#485964"
        fillRule="evenodd"
        d="M16.869.9C35.007.9 34.22 17.494 34.22 17.494S35.722 35.1 17.24 35.1C-2.032 35.1.327 16.818.327 16.818S.828.9 16.87.9Zm4.306 13.13c-1.63-3.148-5.978-4.428-9.38-2.335-4.376 2.692-4.119 8.874-.112 11.338.308.19.447.167.65-.107a7.26 7.26 0 0 1 2.328-2.037c.15-.084.301-.164.474-.254l.087-.046.191-.101c-1.993-.348-3.058-1.262-3.181-2.674-.133-1.52.604-2.755 1.951-3.27a3.018 3.018 0 0 1 3.494 1.07 3.017 3.017 0 0 1 .515 2.39c-.164.87-.677 1.5-1.322 2.054 2.188-.514 3.782-1.726 4.457-3.887.196-.626.156-1.545-.152-2.141ZM7.469 27.337c.444-.154.881-.306 1.326-.42.452-.115.913-.193 1.391-.274.213-.036.43-.073.651-.114a5.592 5.592 0 0 1 1.048-2.951c-2.15-.23-4.693 1.78-5.015 3.964.202-.067.402-.136.6-.205ZM23.582 5.621a36.57 36.57 0 0 1-1.092-.019c-.077-.003-.152-.058-.228-.114a1.223 1.223 0 0 0-.105-.072c.037-.015.074-.034.111-.053.08-.04.16-.08.24-.081.851-.008 1.703-.008 2.554-.007h.74c.002-.574-.762-1.36-1.44-1.573l-.015.227c-.674.016-1.335.004-1.936-.318-.109-.058-.191-.164-.274-.27-.038-.05-.076-.099-.117-.143a3.216 3.216 0 0 1-.145-.175c-.106-.133-.212-.265-.348-.345-.263-.153-.54-.28-.82-.405a16.425 16.425 0 0 1-.451-.21c-1.503-.736-3.09-.71-4.796-.554l2.86.67-.031.169c-.841.113-1.651-.063-2.464-.24a15.444 15.444 0 0 0-1.137-.217c.433.258.878.493 1.336.702.322.146.651.276.983.406.146.058.292.115.438.175-.608.522-1.218.637-1.982.461a4.773 4.773 0 0 0-1.287-.126c-.457.012-.901.15-1.285.4.126.064.248.125.369.184.29.144.564.28.827.435.154.09.33.244.373.403.086.387.15.778.191 1.172-.701.08-1.935.798-2.185 1.264.353.069.716.088 1.08.107.751.039 1.51.078 2.202.548-.176.134-.474.285-.76.43-.264.135-.519.263-.658.37.358.094 1.187.047 1.512.025.273-.018.399-.025.51.068l3.175 2.499c.334.27 1.682 1.553 2.034 2.36.2.445.314.925.336 1.413.001.565-.103 1.124-.308 1.65-.11.28-.432.901-1.097 1.626-1.03 1.122-2.357 1.73-3.806 2.03-3.371.699-6.172 4.318-5.381 8.309.922 4.658 6.036 7.18 10.215 4.964 2.702-1.432 4.134-4.226 3.75-7.268-.232-1.838-1.059-3.337-2.445-4.542-.18-.157-.296-.157-.508-.02-.723.463-1.459.902-2.209 1.317-.291.161-.599.293-.935.436h-.001c-.16.069-.325.14-.498.217l.169.044.194.05c2.015.54 3.091 2.318 2.615 4.31-.425 1.773-2.214 2.906-3.947 2.607-1.446-.25-2.707-1.456-2.918-2.902-.23-1.576.55-3.092 1.94-3.727.37-.169.744-.327 1.118-.486.405-.171.812-.343 1.211-.528.231-.107.467-.208.703-.31.657-.28 1.317-.563 1.879-.978 1.893-1.398 3.062-3.323 3.518-5.646.273-1.391.255-2.777-.378-4.098-.486-1.014-1.284-1.75-2.14-2.423a44.818 44.818 0 0 0-1.3-.97c-.469-.34-.939-.683-1.393-1.043-.238-.189-.398-.514-.508-.808-.046-.125.105-.464.206-.482.541-.094 1.088-.15 1.636-.168.497-.02.996-.01 1.495-.002l.404.006h.072c.125-.001.264-.002.33.062.328.324.585.116.812-.097.178-.193.338-.4.48-.62a3.28 3.28 0 0 0-.472-.066c-.272-.006-.545-.008-.818-.01Z"
        clipRule="evenodd"
      />
      <path
        fill="#485964"
        d="M132.261 23.507v-3.345h-6.182v-2.623h8.995v6.856c-1.095 1.9-2.523 3.356-4.282 4.368-1.759 1.012-3.755 1.518-5.987 1.518-3.396 0-6.199-1.126-8.41-3.377-2.212-2.25-3.318-5.104-3.317-8.56 0-3.469 1.109-6.329 3.327-8.58 2.218-2.25 5.018-3.376 8.4-3.376 2.095 0 4.005.46 5.731 1.383a11.148 11.148 0 0 1 4.21 3.945l-2.362 1.693a7.778 7.778 0 0 0-3.081-3.201 8.866 8.866 0 0 0-4.498-1.157c-2.547 0-4.655.878-6.326 2.633-1.671 1.756-2.506 3.976-2.506 6.66 0 2.685.836 4.902 2.506 6.65 1.671 1.749 3.779 2.623 6.326 2.623 1.588 0 3.002-.345 4.241-1.033 1.239-.688 2.311-1.714 3.215-3.077ZM38.498 6.883l7.395 22.798h2.074l5.915-18.358L59.86 29.68h2.054l7.373-22.798h-2.69l-5.649 17.553-5.75-17.553H52.67L46.9 24.477 41.27 6.883h-2.773ZM68.78 12.004V29.68h2.794V12.004H68.78Z"
      />
      <path
        fill="#485964"
        fillRule="evenodd"
        d="M78.049 21.979v7.703h-2.752V12.087h12.077c1.793 0 3.197.434 4.21 1.301s1.52 2.072 1.52 3.614a4.777 4.777 0 0 1-1.202 3.253c-.8.915-1.845 1.476-3.132 1.683l4.868 7.744H90.66l-4.97-7.703h-7.641Zm0-2.582h9.324c.945 0 1.667-.206 2.167-.62.5-.412.75-1.004.75-1.775 0-.771-.25-1.36-.75-1.766-.5-.406-1.222-.609-2.167-.61H78.05v4.771Z"
        clipRule="evenodd"
      />
      <path
        fill="#485964"
        d="M95.698 12.046V29.68h17.211v-2.54H98.45v-5.575h8.996v-2.54H98.45v-4.399h13.7v-2.581H95.698ZM140.897 12.087h-2.752v12.019c0 2.189.671 3.758 2.013 4.708 1.341.95 3.546 1.425 6.613 1.425 3.081 0 5.302-.482 6.665-1.445 1.362-.964 2.044-2.527 2.044-4.688V12.087h-2.732v11.296c0 1.694-.397 2.836-1.191 3.428-.795.592-2.376.888-4.745.888-2.355 0-3.93-.296-4.724-.888-.794-.592-1.191-1.735-1.191-3.428V12.087Z"
      />
      <path
        fill="#485964"
        fillRule="evenodd"
        d="m155.05 29.681 8.913-17.635h1.726l8.996 17.635h-2.937l-2.28-4.481h-9.263l-2.259 4.481h-2.896Zm6.346-6.815h6.86l-3.409-6.753-3.451 6.753ZM177.737 29.681V21.98h7.64l4.971 7.703h2.978l-4.868-7.744c1.287-.207 2.331-.768 3.132-1.683a4.78 4.78 0 0 0 1.202-3.253c0-1.542-.507-2.747-1.52-3.614s-2.417-1.301-4.211-1.301h-12.077v17.595h2.753Zm9.324-10.284h-9.324v-4.77h9.324c.945 0 1.667.203 2.167.61.5.405.75.994.75 1.765 0 .77-.25 1.363-.75 1.776-.5.413-1.222.62-2.167.62ZM203.213 12.046c2.861 0 5.169.822 6.921 2.467 1.753 1.646 2.629 3.783 2.629 6.412 0 2.658-.859 4.781-2.577 6.371-1.719 1.59-4.043 2.386-6.973 2.386h-7.785V12.046h7.785Zm.041 2.54h-5.073v12.556h5.073c2.081 0 3.717-.561 4.908-1.684 1.192-1.121 1.787-2.646 1.787-4.574 0-1.858-.616-3.373-1.848-4.543-1.233-1.17-2.848-1.755-4.847-1.755Z"
        clipRule="evenodd"
      />
      <path
        fill="#485964"
        d="M214.806 16.305v-3.422h1.175c.802 0 1.165.426 1.165.843 0 .446-.382.9-.955 1.005l1.175 1.574h-.563l-1.3-1.849c.105.02.22.038.306.038.583 0 .908-.379.908-.71 0-.276-.22-.55-.669-.55h-.812v3.07h-.43Z"
      />
      <path
        fill="#485964"
        fillRule="evenodd"
        d="M215.828 11.66h.296c1.72 0 2.809 1.422 2.876 2.929v.17c0 1.441-.898 2.844-2.952 2.901h-.134c-1.978 0-2.914-1.46-2.914-2.92 0-1.507 1.022-3.061 2.828-3.08Zm.124.408c-.735 0-1.366.237-1.968.881-.411.493-.478 1.232-.478 1.232l-.01.12c-.009.093-.018.19-.018.269 0 1.346.955 2.692 2.455 2.692a.595.595 0 0 0 .057-.005c.022-.002.043-.005.058-.005 1.185 0 2.207-.815 2.407-1.98.039-.171.058-.38.058-.56 0-1.27-.946-2.588-2.255-2.644h-.306Z"
        clipRule="evenodd"
      />
    </svg>
  );
};
