/*
 * SPDX-FileType: SOURCE
 * SPDX-FileCopyrightText: Michael Bäuerle
 * SPDX-License-Identifier: BSD-2-Clause
 */

#ifndef SSIC0_ICONV_H
#define SSIC0_ICONV_H  1

#include <stddef.h>


/*! Library header of libssiconv
 *
 * \file      iconv.h
 * \author    Michael Baeuerle
 * \date      2022-2024
 * \copyright BSD-2-Clause
 *
 * Public API (compatible between all versions with same major number).
 *
 * API documentation: \ref SSICONV
 */


/*! \mainpage Documentation
 *
 * \verbinclude "README"
 */


/*! \defgroup SSICONV libssiconv: Simple and small iconv implementation
 *
 * \note
 * The API is derived from the \c iconvstr() function of Solaris 11:
 * <br>
 * https://docs.oracle.com/cd/E88353_01/html/E37843/iconvstr-3c.html
 *
 * This library occupies the namespaces with "sic0_" and "SIC0_" prefix.
 * <br>
 * The namespaces with "ssic0_i_" and "SSIC0_I_" prefix is reserved for
 * internal, never use it outside of the library (the shared library may
 * not even export such symbols).
 */
/*! \{ */


/*! \anchor SSICONV_OPTIONS \name SSICONV_OPTIONS Conversion options */
/*! \{ */

/*! Do not stop on \c NUL control characters in input data */
#define SSIC0_ICONV_IGNORE_NULL      0x01

/*! Do not stop if an input character cannot be converted.
 *
 * For non-identical conversions this implementation uses:
 * - U+FFFD as replacement for Unicode
 * - U+001A as replacement otherwise
 */
#define SSIC0_ICONV_REPLACE_INVALID  0x02

/*! \} */


/*! Get release version of library
 *
 * The release version is a constant string compiled into the library.
 * <br>
 * The content does not change at runtime and this function always
 * returns the same address.
 *
 * \return
 * Pointer to NUL-terminated string with format "MAJOR.MINOR.PATCH",
 * encoded with the execution character set.
 * <br>
 * A suffix may be appended at the end for snapshots and release candidates.
 */
const char *ssic0_version(void);


/*! Convert encoding
 *
 * \param[in]      tocode    Target codeset name (NUL-terminated string)
 * \param[in]      fromcode  Source codeset name (NUL-terminated string)
 * \param[in]      inarray   Pointer to a byte array
 * \param[in,out]  inlen     Number of input bytes (left)
 * \param[out]     outarray  Pointer to a byte array
 * \param[in,out]  outlen    Number of output bytes (converted)
 * \param[in]      flag      Conversion options
 *
 * \e inarray must contain a sequence of character bytes with \e fromcode
 * encoding to be converted.
 * <br>
 * \e inlen , as input argument, contains the number of bytes to be converted in
 * \e inarray.
 * <br>
 * \e inlen , as output argument after the conversion, contains the number of
 * bytes in inarray still not converted after the conversion.
 *
 * \e outarray must provide enough space for the output data (stored with
 * \e tocode encoding).
 * <br>
 * \e outlen , as intput argument, contains the number of bytes at \e outarray
 * where converted character bytes can be saved.
 * <br>
 * \e outlen , as output argument after the conversion, contains the number
 * of bytes still available at \e outarray .
 *
 * The parameter \e flag must be constructed by a bitwise-inclusive-OR of the
 * values from \ref SSICONV_OPTIONS .
 *
 * \return
 * - Number of non-identical conversions performed
 * - (size_t)-1, if an error occurs (errno is set to nonzero value in this case)
 */

size_t ssic0_iconvstr(const char *tocode, const char *fromcode,
                      char *inarray, size_t *inlen,
                      char *outarray, size_t *outlen,
                      int flag);


/*! Print supported codeset descriptions to standard output
 *
 * \note Used by the CLI frontend for the "-l" option.
 */
void ssic0_print_codesets(void);


/*! \} */


#endif  /* SSIC0_ICONV_H */
