/*
 * Copyright @ 2021 - present 8x8, Inc.
 * Copyright @ 2021 - Vowel, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
@file:Suppress("NAME_SHADOWING")

package org.jitsi.videobridge.cc.allocation

import io.kotest.core.spec.style.ShouldSpec
import io.kotest.matchers.shouldBe
import org.jitsi.nlj.MediaSourceDesc
import org.jitsi.nlj.RtpEncodingDesc
import org.jitsi.nlj.VideoType
import org.jitsi.nlj.util.bps
import org.jitsi.nlj.util.kbps
import org.jitsi.utils.logging.DiagnosticContext
import org.jitsi.utils.time.FakeClock

/**
 * Test the logic for selecting the layers to be considered for a source and the "preferred" layer.
 */
class SingleSourceAllocationTest : ShouldSpec() {
    private val clock = FakeClock()
    private val diagnosticContext = DiagnosticContext()

    private val ld7point5 =
        MockRtpLayerDesc(tid = 0, eid = 0, height = 180, frameRate = 7.5, bitrate = bitrateLd * 0.33)
    private val ld15 = MockRtpLayerDesc(tid = 1, eid = 0, height = 180, frameRate = 15.0, bitrate = bitrateLd * 0.66)
    private val ld30 = MockRtpLayerDesc(tid = 2, eid = 0, height = 180, frameRate = 30.0, bitrate = bitrateLd)

    private val sd7point5 =
        MockRtpLayerDesc(tid = 0, eid = 1, height = 360, frameRate = 7.5, bitrate = bitrateSd * 0.33)
    private val sd15 = MockRtpLayerDesc(tid = 1, eid = 1, height = 360, frameRate = 15.0, bitrate = bitrateSd * 0.66)
    private val sd30 = MockRtpLayerDesc(tid = 2, eid = 1, height = 360, frameRate = 30.0, bitrate = bitrateSd)

    private val hd7point5 =
        MockRtpLayerDesc(tid = 0, eid = 2, height = 720, frameRate = 7.5, bitrate = bitrateHd * 0.33)
    private val hd15 = MockRtpLayerDesc(tid = 1, eid = 2, height = 720, frameRate = 15.0, bitrate = bitrateHd * 0.66)
    private val hd30 = MockRtpLayerDesc(tid = 2, eid = 2, height = 720, frameRate = 30.0, bitrate = bitrateHd)

    init {
        context("Camera") {
            context("When all layers are active") {
                val endpointId = "A"
                val mediaSource = MediaSourceDesc(
                    arrayOf(
                        RtpEncodingDesc(1L, arrayOf(ld7point5, ld15, ld30)),
                        RtpEncodingDesc(1L, arrayOf(sd7point5, sd15, sd30)),
                        RtpEncodingDesc(1L, arrayOf(hd7point5, hd15, hd30))
                    ),
                    sourceName = SOURCE_NAME,
                    owner = OWNER,
                    videoType = VideoType.CAMERA
                )

                context("Without constraints") {
                    val allocation =
                        SingleSourceAllocation(
                            endpointId,
                            mediaSource,
                            VideoConstraints(720),
                            false,
                            diagnosticContext,
                            clock
                        )

                    // We include all resolutions up to the preferred resolution, and only high-FPS (at least
                    // "preferred FPS") layers for higher resolutions.
                    allocation.preferredLayer shouldBe sd30
                    allocation.oversendLayer shouldBe null
                    allocation.layers.map { it.layer } shouldBe listOf(ld7point5, ld15, ld30, sd30, hd30)
                }
                context("With constraints") {
                    val allocation =
                        SingleSourceAllocation(
                            endpointId,
                            mediaSource,
                            VideoConstraints(360),
                            false,
                            diagnosticContext,
                            clock
                        )

                    // We include all resolutions up to the preferred resolution, and only high-FPS (at least
                    // "preferred FPS") layers for higher resolutions.
                    allocation.preferredLayer shouldBe sd30
                    allocation.oversendLayer shouldBe null
                    allocation.layers.map { it.layer } shouldBe listOf(ld7point5, ld15, ld30, sd30)
                }
                context("With constraints unmet by any layer") {
                    // Single high-res stream with 3 temporal layers.
                    val endpointId = "A"
                    val mediaSource = MediaSourceDesc(
                        // No simulcast.
                        arrayOf(RtpEncodingDesc(1L, arrayOf(hd7point5, hd15, hd30))),
                        sourceName = SOURCE_NAME,
                        owner = OWNER,
                        videoType = VideoType.CAMERA
                    )

                    context("Non-zero constraints") {
                        val allocation =
                            SingleSourceAllocation(
                                endpointId,
                                mediaSource,
                                VideoConstraints(360),
                                false,
                                diagnosticContext,
                                clock
                            )

                        // The receiver set 360p constraints, but we only have a 720p stream.
                        allocation.preferredLayer shouldBe hd30
                        allocation.oversendLayer shouldBe null
                        allocation.layers.map { it.layer } shouldBe listOf(hd7point5, hd15, hd30)
                    }
                    context("Zero constraints") {
                        val allocation =
                            SingleSourceAllocation(
                                endpointId,
                                mediaSource,
                                VideoConstraints(0),
                                false,
                                diagnosticContext,
                                clock
                            )

                        // The receiver set a maxHeight=0 constraint.
                        allocation.preferredLayer shouldBe null
                        allocation.oversendLayer shouldBe null
                        allocation.layers.map { it.layer } shouldBe emptyList()
                    }
                }
            }
            context("When some layers are inactive") {
                // Override layers with bitrate=0. Simulate only up to 360p/15 being active.
                val sd30 = MockRtpLayerDesc(tid = 2, eid = 1, height = 360, frameRate = 30.0, bitrate = 0.bps)
                val hd7point5 = MockRtpLayerDesc(tid = 0, eid = 2, height = 720, frameRate = 7.5, bitrate = 0.bps)
                val hd15 = MockRtpLayerDesc(tid = 1, eid = 2, height = 720, frameRate = 15.0, bitrate = 0.bps)
                val hd30 = MockRtpLayerDesc(tid = 2, eid = 2, height = 720, frameRate = 30.0, bitrate = 0.bps)
                val endpointId = "A"
                val mediaSource = MediaSourceDesc(
                    arrayOf(
                        RtpEncodingDesc(1L, arrayOf(ld7point5, ld15, ld30)),
                        RtpEncodingDesc(1L, arrayOf(sd7point5, sd15, sd30)),
                        RtpEncodingDesc(1L, arrayOf(hd7point5, hd15, hd30))
                    ),
                    sourceName = SOURCE_NAME,
                    owner = OWNER,
                    videoType = VideoType.CAMERA
                )

                val allocation =
                    SingleSourceAllocation(
                        endpointId,
                        mediaSource,
                        VideoConstraints(720),
                        false,
                        diagnosticContext,
                        clock
                    )

                // We include all resolutions up to the preferred resolution, and only high-FPS (at least
                // "preferred FPS") layers for higher resolutions.
                allocation.preferredLayer shouldBe ld30
                allocation.oversendLayer shouldBe null
                allocation.layers.map { it.layer } shouldBe listOf(ld7point5, ld15, ld30)
            }
        }
        context("Screensharing") {
            context("When all layers are active") {
                val endpointId = "A"
                val mediaSource = MediaSourceDesc(
                    arrayOf(
                        RtpEncodingDesc(1L, arrayOf(ld7point5, ld15, ld30)),
                        RtpEncodingDesc(1L, arrayOf(sd7point5, sd15, sd30)),
                        RtpEncodingDesc(1L, arrayOf(hd7point5, hd15, hd30))
                    ),
                    sourceName = SOURCE_NAME,
                    owner = OWNER,
                    videoType = VideoType.DESKTOP
                )

                context("With no constraints") {
                    val allocation =
                        SingleSourceAllocation(
                            endpointId,
                            mediaSource,
                            VideoConstraints(720),
                            true,
                            diagnosticContext,
                            clock
                        )

                    // For screensharing the "preferred" layer should be the highest -- always prioritized over other
                    // endpoints.
                    allocation.preferredLayer shouldBe hd30
                    allocation.oversendLayer shouldBe hd7point5
                    allocation.layers.map { it.layer } shouldBe
                        listOf(ld7point5, ld15, ld30, sd7point5, sd15, sd30, hd7point5, hd15, hd30)
                }
                context("With 360p constraints") {
                    val allocation =
                        SingleSourceAllocation(
                            endpointId,
                            mediaSource,
                            VideoConstraints(360),
                            true,
                            diagnosticContext,
                            clock
                        )

                    allocation.preferredLayer shouldBe sd30
                    allocation.oversendLayer shouldBe sd7point5
                    allocation.layers.map { it.layer } shouldBe listOf(ld7point5, ld15, ld30, sd7point5, sd15, sd30)
                }
            }
            context("The high layers are inactive (send-side bwe restrictions)") {
                // Override layers with bitrate=0. Simulate only up to 360p/30 being active.
                val hd7point5 = MockRtpLayerDesc(tid = 0, eid = 2, height = 720, frameRate = 7.5, bitrate = 0.bps)
                val hd15 = MockRtpLayerDesc(tid = 1, eid = 2, height = 720, frameRate = 15.0, bitrate = 0.bps)
                val hd30 = MockRtpLayerDesc(tid = 2, eid = 2, height = 720, frameRate = 30.0, bitrate = 0.bps)
                val mediaSource = MediaSourceDesc(
                    arrayOf(
                        RtpEncodingDesc(1L, arrayOf(ld7point5, ld15, ld30)),
                        RtpEncodingDesc(1L, arrayOf(sd7point5, sd15, sd30)),
                        RtpEncodingDesc(1L, arrayOf(hd7point5, hd15, hd30))
                    ),
                    sourceName = SOURCE_NAME,
                    owner = OWNER,
                    videoType = VideoType.DESKTOP
                )

                val allocation =
                    SingleSourceAllocation(
                        "A",
                        mediaSource,
                        VideoConstraints(720),
                        true,
                        diagnosticContext,
                        clock
                    )

                // For screensharing the "preferred" layer should be the highest -- always prioritized over other
                // endpoints.
                allocation.preferredLayer shouldBe sd30
                allocation.oversendLayer shouldBe sd7point5
                allocation.layers.map { it.layer } shouldBe listOf(ld7point5, ld15, ld30, sd7point5, sd15, sd30)
            }
            context("The low layers are inactive (simulcast signaled but not used)") {
                // Override layers with bitrate=0. Simulate simulcast being signaled but effectively disabled.
                val ld7point5 = MockRtpLayerDesc(tid = 0, eid = 2, height = 720, frameRate = 7.5, bitrate = 0.bps)
                val ld15 = MockRtpLayerDesc(tid = 1, eid = 2, height = 720, frameRate = 15.0, bitrate = 0.bps)
                val ld30 = MockRtpLayerDesc(tid = 2, eid = 2, height = 720, frameRate = 30.0, bitrate = 0.bps)
                val sd7point5 = MockRtpLayerDesc(tid = 0, eid = 1, height = 360, frameRate = 7.5, bitrate = 0.bps)
                val sd15 = MockRtpLayerDesc(tid = 1, eid = 1, height = 360, frameRate = 15.0, bitrate = 0.bps)
                val sd30 = MockRtpLayerDesc(tid = 2, eid = 1, height = 360, frameRate = 30.0, bitrate = 0.bps)
                val mediaSource = MediaSourceDesc(
                    arrayOf(
                        RtpEncodingDesc(1L, arrayOf(ld7point5, ld15, ld30)),
                        RtpEncodingDesc(1L, arrayOf(sd7point5, sd15, sd30)),
                        RtpEncodingDesc(1L, arrayOf(hd7point5, hd15, hd30))
                    ),
                    sourceName = SOURCE_NAME,
                    owner = OWNER,
                    videoType = VideoType.DESKTOP
                )

                context("With no constraints") {
                    val allocation =
                        SingleSourceAllocation(
                            "A",
                            mediaSource,
                            VideoConstraints(720),
                            true,
                            diagnosticContext,
                            clock
                        )

                    // For screensharing the "preferred" layer should be the highest -- always prioritized over other
                    // endpoints.
                    allocation.preferredLayer shouldBe hd30
                    allocation.oversendLayer shouldBe hd7point5
                    allocation.layers.map { it.layer } shouldBe listOf(hd7point5, hd15, hd30)
                }
                context("With 180p constraints") {
                    val allocation =
                        SingleSourceAllocation(
                            "A",
                            mediaSource,
                            VideoConstraints(180),
                            true,
                            diagnosticContext,
                            clock
                        )

                    // For screensharing the "preferred" layer should be the highest -- always prioritized over other
                    // endpoints. Since no layers satisfy the resolution constraints, we consider layers from the
                    // lowest available resolution (which is high).
                    allocation.preferredLayer shouldBe hd30
                    allocation.oversendLayer shouldBe hd7point5
                    allocation.layers.map { it.layer } shouldBe listOf(hd7point5, hd15, hd30)
                }
            }
            context("VP9") {
                val l1 = MockRtpLayerDesc(tid = 0, eid = 0, sid = 0, height = 720, frameRate = -1.0, bitrate = 150.kbps)
                val l2 = MockRtpLayerDesc(tid = 0, eid = 0, sid = 1, height = 720, frameRate = -1.0, bitrate = 370.kbps)
                val l3 = MockRtpLayerDesc(tid = 0, eid = 0, sid = 2, height = 720, frameRate = -1.0, bitrate = 750.kbps)

                val mediaSource = MediaSourceDesc(
                    arrayOf(
                        RtpEncodingDesc(1L, arrayOf(l1)),
                        RtpEncodingDesc(1L, arrayOf(l2)),
                        RtpEncodingDesc(1L, arrayOf(l3))
                    ),
                    sourceName = SOURCE_NAME,
                    owner = OWNER,
                    videoType = VideoType.DESKTOP
                )

                context("With no constraints") {
                    val allocation =
                        SingleSourceAllocation(
                            "A",
                            mediaSource,
                            VideoConstraints(720),
                            true,
                            diagnosticContext,
                            clock
                        )

                    allocation.preferredLayer shouldBe l3
                    allocation.oversendLayer shouldBe l1
                    allocation.layers.map { it.layer } shouldBe listOf(l1, l2, l3)
                }
                context("With 180p constraints") {
                    // For screensharing the "preferred" layer should be the highest -- always prioritized over other
                    // endpoints. Since no layers satisfy the resolution constraints, we consider layers from the
                    // lowest available resolution (which is high). If we are off-stage we only consider the first of
                    // these layers.
                    context("On stage") {
                        val allocation = SingleSourceAllocation(
                            "A",
                            mediaSource,
                            VideoConstraints(180),
                            true,
                            diagnosticContext,
                            clock
                        )

                        allocation.preferredLayer shouldBe l3
                        allocation.oversendLayer shouldBe l1
                        allocation.layers.map { it.layer } shouldBe listOf(l1, l2, l3)
                    }
                    context("Off stage") {
                        val allocation = SingleSourceAllocation(
                            "A",
                            mediaSource,
                            VideoConstraints(180),
                            false,
                            diagnosticContext,
                            clock
                        )

                        allocation.preferredLayer shouldBe l1
                        allocation.oversendLayer shouldBe null
                        allocation.layers.map { it.layer } shouldBe listOf(l1)
                    }
                }
            }
        }
    }
}

private const val SOURCE_NAME = "sourceName"
private const val OWNER = "owner"
