// Copyright 2020 The Jujutsu Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

use std::slice;

use jj_lib::matchers::EverythingMatcher;
use jj_lib::object_id::ObjectId;
use tracing::instrument;

use crate::cli_util::CommandHelper;
use crate::cli_util::RevisionArg;
use crate::command_error::user_error;
use crate::command_error::CommandError;
use crate::description_util::combine_messages;
use crate::ui::Ui;

/// Move changes from a revision's parent into the revision
///
/// After moving the changes out of the parent, the child revision will have the
/// same content state as before. If moving the change out of the parent change
/// made it empty compared to its parent, it will be abandoned. Without
/// `--interactive`, the parent change will always become empty.
///
/// If the source became empty and both the source and destination had a
/// non-empty description, you will be asked for the combined description. If
/// either was empty, then the other one will be used.
///
/// If a working-copy commit gets abandoned, it will be given a new, empty
/// commit. This is true in general; it is not specific to this command.
#[derive(clap::Args, Clone, Debug)]
pub(crate) struct UnsquashArgs {
    #[arg(long, short, default_value = "@", value_name = "REVSET")]
    revision: RevisionArg,
    /// Interactively choose which parts to unsquash
    // TODO: It doesn't make much sense to run this without -i. We should make that
    // the default.
    #[arg(long, short)]
    interactive: bool,
    /// Specify diff editor to be used (implies --interactive)
    #[arg(long, value_name = "NAME")]
    tool: Option<String>,
}

#[instrument(skip_all)]
pub(crate) fn cmd_unsquash(
    ui: &mut Ui,
    command: &CommandHelper,
    args: &UnsquashArgs,
) -> Result<(), CommandError> {
    writeln!(
        ui.warning_default(),
        "`jj unsquash` is deprecated; use `jj diffedit --restore-descendants` or `jj squash` \
         instead"
    )?;
    writeln!(
        ui.warning_default(),
        "`jj unsquash` will be removed in a future version, and this will be a hard error"
    )?;
    let mut workspace_command = command.workspace_helper(ui)?;
    let commit = workspace_command.resolve_single_rev(ui, &args.revision)?;
    workspace_command.check_rewritable([commit.id()])?;
    if commit.parent_ids().len() > 1 {
        return Err(user_error("Cannot unsquash merge commits"));
    }
    let parent = commit.parents().next().unwrap()?;
    workspace_command.check_rewritable([parent.id()])?;
    let interactive_editor = if args.tool.is_some() || args.interactive {
        Some(workspace_command.diff_editor(ui, args.tool.as_deref())?)
    } else {
        None
    };
    let text_editor = workspace_command.text_editor()?;
    let mut tx = workspace_command.start_transaction();
    let parent_base_tree = parent.parent_tree(tx.repo())?;
    let new_parent_tree_id;
    if let Some(diff_editor) = &interactive_editor {
        let format_instructions = || {
            format!(
                "\
You are moving changes from: {}
into its child: {}

The diff initially shows the parent commit's changes.

Adjust the right side until it shows the contents you want to keep in
the parent commit. The changes you edited out will be moved into the
child commit. If you don't make any changes, then the operation will be
aborted.
",
                tx.format_commit_summary(&parent),
                tx.format_commit_summary(&commit)
            )
        };
        let parent_tree = parent.tree()?;
        new_parent_tree_id = diff_editor.edit(
            &parent_base_tree,
            &parent_tree,
            &EverythingMatcher,
            format_instructions,
        )?;
        if new_parent_tree_id == parent_base_tree.id() {
            return Err(user_error("No changes selected"));
        }
    } else {
        new_parent_tree_id = parent_base_tree.id().clone();
    }
    // Abandon the parent if it is now empty (always the case in the non-interactive
    // case).
    if new_parent_tree_id == parent_base_tree.id() {
        tx.repo_mut().record_abandoned_commit(&parent);
        let description = combine_messages(&text_editor, slice::from_ref(&parent), &commit)?;
        // Commit the new child on top of the parent's parents.
        tx.repo_mut()
            .rewrite_commit(&commit)
            .set_parents(parent.parent_ids().to_vec())
            .set_description(description)
            .write()?;
    } else {
        let new_parent = tx
            .repo_mut()
            .rewrite_commit(&parent)
            .set_tree_id(new_parent_tree_id)
            .set_predecessors(vec![parent.id().clone(), commit.id().clone()])
            .write()?;
        // Commit the new child on top of the new parent.
        tx.repo_mut()
            .rewrite_commit(&commit)
            .set_parents(vec![new_parent.id().clone()])
            .write()?;
    }
    tx.finish(ui, format!("unsquash commit {}", commit.id().hex()))?;
    Ok(())
}
