--
--  Copyright (C) 2014-2022, AdaCore
--  SPDX-License-Identifier: Apache-2.0
--

with Ada.Containers.Hashed_Maps;
with Ada.Containers.Vectors;
with Ada.Strings.Unbounded; use Ada.Strings.Unbounded;
with Ada.Strings.Unbounded.Hash;
with Ada.Strings.Wide_Wide_Unbounded; use Ada.Strings.Wide_Wide_Unbounded;

with Interfaces.C;

with System;
with System.Address_To_Access_Conversions;

with Langkit_Support.Bump_Ptr; use Langkit_Support.Bump_Ptr;
with Langkit_Support.Bump_Ptr_Vectors;
with Langkit_Support.Generic_API.Introspection;
use Langkit_Support.Generic_API.Introspection;

with Libadalang.Common;   use Libadalang.Common;
with Libadalang.Implementation;   use Libadalang.Implementation;
with Libadalang.Implementation.C; use Libadalang.Implementation.C;

--  Internal package: low-level primitives to implement syntax-based source
--  rewriting.

private package Libadalang.Rewriting_Implementation is

   use Support.Diagnostics, Support.Text;

   type Rewriting_Handle_Type;
   type Unit_Rewriting_Handle_Type;
   type Node_Rewriting_Handle_Type;

   type Rewriting_Handle is access Rewriting_Handle_Type
      with Convention => C;
   --  Internal handle for an analysis context rewriting session

   type Unit_Rewriting_Handle is access Unit_Rewriting_Handle_Type
      with Convention => C;
   --  Internal handle for the process of rewriting an analysis unit

   type Node_Rewriting_Handle is access Node_Rewriting_Handle_Type
      with Convention => C;
   --  Internal handle for the process of rewriting an analysis unit

   pragma No_Strict_Aliasing (Rewriting_Handle);
   pragma No_Strict_Aliasing (Unit_Rewriting_Handle);
   pragma No_Strict_Aliasing (Node_Rewriting_Handle);

   package Unit_Maps is new Ada.Containers.Hashed_Maps
     (Key_Type        => Unbounded_String,
      Element_Type    => Unit_Rewriting_Handle,
      Hash            => Ada.Strings.Unbounded.Hash,
      Equivalent_Keys => "=");

   package Node_Maps is new Ada.Containers.Hashed_Maps
     (Key_Type        => Bare_Ada_Node,
      Element_Type    => Node_Rewriting_Handle,
      Hash            => Named_Hash,
      Equivalent_Keys => "=");

   package Nodes_Pools is new Langkit_Support.Bump_Ptr_Vectors
     (Node_Rewriting_Handle);

   type Rewriting_Handle_Type is record
      Context : Internal_Context;
      --  Analysis context this rewriting handle relates to

      Units : Unit_Maps.Map;
      --  Keep track of rewriting handles we create for the units that Context
      --  owns.

      Pool      : Bump_Ptr_Pool;
      New_Nodes : Nodes_Pools.Vector;
      --  Keep track of all node rewriting handles that don't map to original
      --  nodes, i.e. all nodes that were created during this rewriting
      --  session.

      Stubs : Nodes_Pools.Vector;
      --  Keep track of all allocated stub rewriting nodes. These are used in
      --  ``Rotate`` as stubs for rotated ones, and are re-used each time
      --  ``Rotate`` is called.
   end record;

   type Unit_Rewriting_Handle_Type is record
      Context_Handle : Rewriting_Handle;
      --  Rewriting handle for the analysis context this relates to

      Unit : Internal_Unit;
      --  Analysis unit this relates to

      Root : Node_Rewriting_Handle;
      --  Handle for the node that will become the root node of this analysis
      --  unit.

      Nodes : Node_Maps.Map;
      --  Keep track of rewriting handles we create for base AST nodes that
      --  Unit owns.
   end record;

   package Node_Vectors is new Ada.Containers.Vectors
     (Positive, Node_Rewriting_Handle);

   type Node_Children_Kind is (
      Unexpanded,
      --  Dummy node rewriting handle: children don't have their own handle yet

      Expanded_Regular,
      --  Expanded node rewriting handle: children have their own handle. Note
      --  that this is for all but token nodes.

      Expanded_List,
      --  Expanded node rewriting handle, specific for list nodes: element
      --  nodes are stored as a doubly linked list.

      Expanded_Token_Node
      --  Expanded node rewriting handle, specific for token nodes: there is no
      --  children, only some associated text.
   );

   type Node_Children (Kind : Node_Children_Kind := Unexpanded) is record
      case Kind is
         when Unexpanded =>
            null;

         when Expanded_Regular =>
            Vector : Node_Vectors.Vector;
            --  Vector of children for all non-null syntax fields

         when Expanded_List =>
            First, Last : Node_Rewriting_Handle;
            --  Doubly linked list of children

            Count : Natural;
            --  Number of children

         when Expanded_Token_Node =>
            Text : Unbounded_Wide_Wide_String;
            --  Text for this token node
      end case;
   end record;
   --  Lazily evaluated vector of children for a Node_Rewriting_Handle.
   --
   --  In order to avoid constructing the whole tree of Node_Rewriting_Handle
   --  for some analysis unit at once, we build them in a lazy fashion.

   Unexpanded_Children : constant Node_Children := (Kind => Unexpanded);

   type Node_Rewriting_Handle_Type is record
      Context_Handle : Rewriting_Handle;
      --  Rewriting handle for the analysis context that owns Node

      Node : Bare_Ada_Node;
      --  Bare AST node which this rewriting handle relates to

      Parent : Node_Rewriting_Handle;
      --  Rewriting handle for Node's parent, or No_Node_Rewriting_Handle if
      --  Node is a root node.

      Previous, Next : Node_Rewriting_Handle;
      --  If ``Parent`` is a list node, ``Previous`` is the previous subling
      --  for this node in that list (``No_Node_Rewriting_Handle`` for the
      --  first sibling), and ``Next`` is the next sibling
      --  (``No_Node_Rewriting_Handle`` for the last sibling).
      --
      --  If ``Parent`` is not a list node, both are set to
      --  ``No_Node_Rewriting_Handle``).

      Kind : Ada_Node_Kind_Type;
      --  Kind for the node this handle represents. When Node is not null (i.e.
      --  when this represents an already existing node, rather than a new
      --  one), this must be equal to Node.Kind.

      Tied : Boolean;
      --  Whether this node is tied to an analysis unit tree. It can be
      --  assigned as a child to another node iff it is not tied.

      Root_Of : Unit_Rewriting_Handle;
      --  If the node this handle represents is the root of a rewritten unit,
      --  this references this unit. No_Unit_Rewriting_Handle in all other
      --  cases.

      Children : Node_Children;
      --  Lazily evaluated vector of children for the rewritten node
   end record;

   type Unit_Rewriting_Handle_Array is
      array (Positive range <>) of Unit_Rewriting_Handle;

   type Node_Rewriting_Handle_Array is
      array (Positive range <>) of Node_Rewriting_Handle;

   No_Rewriting_Handle      : constant Rewriting_Handle      := null;
   No_Unit_Rewriting_Handle : constant Unit_Rewriting_Handle := null;
   No_Node_Rewriting_Handle : constant Node_Rewriting_Handle := null;

   --------------------------------------------------
   -- Implementation of context rewriting routines --
   --------------------------------------------------

   function Handle (Context : Internal_Context) return Rewriting_Handle;
   --  Implementation for Rewriting.Handle

   function C_Context_To_Handle
     (Context : Internal_Context) return Rewriting_Handle
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_context_to_handle";

   function Context (Handle : Rewriting_Handle) return Internal_Context;
   --  Implementation for Rewriting.Context

   function C_Handle_To_Context
     (Handle : Rewriting_Handle) return Internal_Context
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_handle_to_context";

   function Start_Rewriting
     (Context : Internal_Context) return Rewriting_Handle
      with Post => Handle (Context) /= No_Rewriting_Handle
                   and then Has_With_Trivia (Context)
                   and then Start_Rewriting'Result = Handle (Context)
                   and then Libadalang.Rewriting_Implementation.Context
                             (Start_Rewriting'Result) = Context;

   function C_Start_Rewriting
     (Context : Internal_Context) return Rewriting_Handle
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_start_rewriting";
   --  Implementation for Rewriting.Start_Rewriting

   procedure Abort_Rewriting (Handle : in out Rewriting_Handle)
      with Post => Handle = No_Rewriting_Handle;
   --  Implementation for Rewriting.Abort_Rewriting

   procedure C_Abort_Rewriting (Handle : Rewriting_Handle)
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_abort_rewriting";

   type Apply_Result (Success : Boolean := True) is record
      case Success is
         when False =>
            Unit : Internal_Unit;
            --  Reference to the analysis unit that was being processed when
            --  the error occurred.

            Diagnostics : Diagnostics_Vectors.Vector;
            --  Corresponding list of error messages
         when True => null;
      end case;
   end record;

   function Apply (Handle : in out Rewriting_Handle) return Apply_Result
      with Post => (if Apply'Result.Success
                    then Handle = No_Rewriting_Handle
                    else Handle = Handle'Old);
   --  Implementation for Rewriting.Apply

   package C_Diagnostic_Array is new
     System.Address_To_Access_Conversions (ada_diagnostic);

   type C_Apply_Result is record
      Success           : Interfaces.C.int;
      Unit              : Internal_Unit;
      Diagnostics_Count : Interfaces.C.int;
      Diagnostics       : C_Diagnostic_Array.Object_Pointer;
   end record
      with Convention => C;

   procedure C_Apply
     (Handle : Rewriting_Handle;
      Result : access C_Apply_Result)
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_apply";

   procedure Free_Apply_Result (Result : access C_Apply_Result)
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_free_apply_result";

   package C_Unit_Array is new
     System.Address_To_Access_Conversions (Unit_Rewriting_Handle);

   function Unit_Handles
     (Handle : Rewriting_Handle) return Unit_Rewriting_Handle_Array;
   --  Implementation for Rewriting.Unit_Handles

   function C_Unit_Handles
     (Handle : Rewriting_Handle) return C_Unit_Array.Object_Pointer
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_unit_handles";

   ---------------------------------------
   -- Implementation for unit rewriting --
   ---------------------------------------

   function Handle (Unit : Internal_Unit) return Unit_Rewriting_Handle;
   --  Implementation for Rewriting.Handle

   function C_Unit_To_Handle
     (Unit : Internal_Unit) return Unit_Rewriting_Handle
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_unit_to_handle";

   function Unit (Handle : Unit_Rewriting_Handle) return Internal_Unit;
   --  Implementation for Rewriting.Unit

   function C_Handle_To_Unit
     (Handle : Unit_Rewriting_Handle) return Internal_Unit
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_handle_to_unit";

   function Root (Handle : Unit_Rewriting_Handle) return Node_Rewriting_Handle;
   --  Implementation for Rewriting.Root

   function C_Root
     (Handle : Unit_Rewriting_Handle) return Node_Rewriting_Handle
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_unit_root";

   procedure Set_Root
     (Handle : Unit_Rewriting_Handle;
      Root   : Node_Rewriting_Handle);
   --  Implementation for Rewriting.Set_Root

   procedure C_Set_Root
     (Handle : Unit_Rewriting_Handle;
      Root   : Node_Rewriting_Handle)
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_unit_set_root";

   function Unparse (Handle : Unit_Rewriting_Handle) return Unbounded_Text_Type;
   --  Implementation for Rewriting.Unparse

   procedure C_Unparse
     (Handle : Unit_Rewriting_Handle; Result : access ada_text)
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_unit_unparse";

   ---------------------------------------
   -- Implementation for node rewriting --
   ---------------------------------------

   function Handle
     (Node : Bare_Ada_Node) return Node_Rewriting_Handle;
   --  Implementation for Rewriting.Handle

   function C_Node_To_Handle
     (Node : ada_base_node) return Node_Rewriting_Handle
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_node_to_handle";

   function Node
     (Handle : Node_Rewriting_Handle) return Bare_Ada_Node;
   --  Implementation for Rewriting.Node

   function C_Handle_To_Node
     (Handle : Node_Rewriting_Handle) return ada_base_node
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_handle_to_node";

   function Context (Handle : Node_Rewriting_Handle) return Rewriting_Handle;
   --  Implementation for Rewriting.Context

   function C_Node_To_Context
     (Node : Node_Rewriting_Handle) return Rewriting_Handle
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_node_to_context";

   function Unparse (Handle : Node_Rewriting_Handle) return Text_Type;
   --  Implementation for Rewriting.Unparse

   procedure C_Unparse
     (Handle : Node_Rewriting_Handle; Result : access ada_text)
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_node_unparse";

   function Kind (Handle : Node_Rewriting_Handle) return Ada_Node_Kind_Type;
   --  Implementation for Rewriting.Kind

   function C_Kind (Handle : Node_Rewriting_Handle) return ada_node_kind_enum
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_kind";

   function Image (Handle : Node_Rewriting_Handle) return String;
   --  Implementation for Rewriting.Image

   procedure C_Image
     (Handle : Node_Rewriting_Handle; Result : access ada_text)
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_node_image";

   function Tied (Handle : Node_Rewriting_Handle) return Boolean;
   --  Implementation for Rewriting.Tied

   function C_Tied (Handle : Node_Rewriting_Handle) return Interfaces.C.int
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_tied";

   function Parent
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle;
   --  Implementation for Rewriting.Parent

   function C_Parent
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_parent";

   function Children_Count (Handle : Node_Rewriting_Handle) return Natural;
   --  Implementation for Rewriting.Children_Count

   function C_Children_Count
     (Handle : Node_Rewriting_Handle) return Interfaces.C.int
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_children_count";

   function Child
     (Handle : Node_Rewriting_Handle;
      Field  : Struct_Member_Ref) return Node_Rewriting_Handle;
   --  Implementation for Rewriting.Child

   function C_Child
     (Handle : Node_Rewriting_Handle;
      Field  : Interfaces.C.int) return Node_Rewriting_Handle
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_child";

   function Children
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle_Array;
   --  Implementation for Rewriting.Children

   package C_Node_Array is new
     System.Address_To_Access_Conversions (Node_Rewriting_Handle);

   procedure C_Children
     (Handle   : Node_Rewriting_Handle;
      Children : access C_Node_Array.Object_Pointer;
      Count    : access Interfaces.C.int)
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_children";

   procedure Set_Child
     (Handle : Node_Rewriting_Handle;
      Field  : Struct_Member_Ref;
      Child  : Node_Rewriting_Handle);
   --  Implementation for Rewriting.Set_Child

   procedure C_Set_Child
     (Handle : Node_Rewriting_Handle;
      Field  : Interfaces.C.int;
      Child  : Node_Rewriting_Handle)
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_set_child";

   function Text (Handle : Node_Rewriting_Handle) return Text_Type;
   --  Implementation for Rewriting.Text

   procedure C_Text
     (Handle : Node_Rewriting_Handle; Result : access ada_text)
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_text";

   procedure Set_Text (Handle : Node_Rewriting_Handle; Text : Text_Type);
   --  Implementation for Rewriting.Set_Text

   procedure C_Set_Text
     (Handle : Node_Rewriting_Handle; Text : access ada_text)
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_set_text";

   procedure Replace (Handle, New_Node : Node_Rewriting_Handle);
   --  Implementation for Rewriting.Replace

   procedure C_Replace (Handle, New_Node : Node_Rewriting_Handle)
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_replace";

   procedure Rotate (Handles : Node_Rewriting_Handle_Array);
   --  Implementation for Rewriting.Rotate

   procedure C_Rotate
     (Handles : C_Node_Array.Object_Pointer;
      Count   : Interfaces.C.int)
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_rotate";

   function Is_List_Node (Handle : Node_Rewriting_Handle) return Boolean;
   --  Implementation for Rewriting.Is_List_Node

   --------------------------------------------
   -- Implementation for list node rewriting --
   --------------------------------------------

   function First_Child
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle;
   --  Implementation for Rewriting.First_Child

   function C_First_Child
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_first_child";

   function Last_Child
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle;
   --  Implementation for Rewriting.Last_Child

   function C_Last_Child
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_last_child";

   function Next_Child
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle;
   --  Implementation for Rewriting.Next_Child

   function C_Next_Child
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_next_child";

   function Previous_Child
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle;
   --  Implementation for Rewriting.Previous_Child

   function C_Previous_Child
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_previous_child";

   procedure Insert_Before (Handle, New_Sibling : Node_Rewriting_Handle);
   --  Implementation for Rewriting.Insert_Before

   procedure C_Insert_Before (Handle, New_Sibling : Node_Rewriting_Handle)
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_insert_before";

   procedure Insert_After (Handle, New_Sibling : Node_Rewriting_Handle);
   --  Implementation for Rewriting.Insert_After

   procedure C_Insert_After (Handle, New_Sibling : Node_Rewriting_Handle)
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_insert_after";

   procedure Insert_First (Handle, New_Child : Node_Rewriting_Handle);
   --  Implementation for Rewriting.Insert_First

   procedure C_Insert_First (Handle, New_Sibling : Node_Rewriting_Handle)
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_insert_first";

   procedure Insert_Last (Handle, New_Child : Node_Rewriting_Handle);
   --  Implementation for Rewriting.Insert_Last

   procedure C_Insert_Last (Handle, New_Sibling : Node_Rewriting_Handle)
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_insert_last";

   procedure Remove_Child (Handle : Node_Rewriting_Handle);
   --  Implementation for Rewriting.Remove_Child

   procedure C_Remove_Child (Handle : Node_Rewriting_Handle)
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_remove_child";

   --------------------------------------
   -- Implementation for node creation --
   --------------------------------------

   function Clone
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle;
   --  Implementation for Rewriting.Clone

   function C_Clone
     (Handle : Node_Rewriting_Handle) return Node_Rewriting_Handle
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_clone";

   function Create_Node
     (Handle : Rewriting_Handle;
      Kind   : Ada_Node_Kind_Type) return Node_Rewriting_Handle;
   --  Implementation for Rewriting.Create_Node

   function C_Create_Node
     (Handle : Rewriting_Handle;
      Kind   : ada_node_kind_enum) return Node_Rewriting_Handle
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_create_node";

   function Create_Token_Node
     (Handle : Rewriting_Handle;
      Kind   : Ada_Node_Kind_Type;
      Text   : Text_Type) return Node_Rewriting_Handle;
   --  Implementation for Rewriting.Create_Token_Node

   function C_Create_Token_Node
     (Handle : Rewriting_Handle;
      Kind   : ada_node_kind_enum;
      Text   : access ada_text) return Node_Rewriting_Handle
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_create_token_node";

   function Create_Regular_Node
     (Handle   : Rewriting_Handle;
      Kind     : Ada_Node_Kind_Type;
      Children : Node_Rewriting_Handle_Array) return Node_Rewriting_Handle;
   --  Implementation for Rewriting.Create_Regular_Node

   function C_Create_Regular_Node
     (Handle   : Rewriting_Handle;
      Kind     : ada_node_kind_enum;
      Children : C_Node_Array.Object_Pointer;
      Count    : Interfaces.C.int) return Node_Rewriting_Handle
      with Export        => True,
           Convention    => C,
           External_Name => "ada_rewriting_create_regular_node";

   ----------------------------------
   -- Implementation for templates --
   ----------------------------------

   function Create_From_Template
     (Handle    : Rewriting_Handle;
      Template  : Text_Type;
      Arguments : Node_Rewriting_Handle_Array;
      Rule      : Grammar_Rule) return Node_Rewriting_Handle;
   --  Implementation for Rewriting.Create_From_Template

   function C_Create_From_Template
     (Handle    : Rewriting_Handle;
      Template  : access ada_text;
      Arguments : C_Node_Array.Object_Pointer;
      Count     : Interfaces.C.int;
      Rule      : ada_grammar_rule) return Node_Rewriting_Handle
      with Export        => True,
           Convention    => C,
           External_Name =>
             "ada_rewriting_create_from_template";

   -----------------------------
   -- Node creation shortcuts --
   -----------------------------



         function Create_Constrained_Array_Indices
           (Handle : Rewriting_Handle
               ; Constrained_Array_Indices_F_List : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Unconstrained_Array_Indices
           (Handle : Rewriting_Handle
               ; Unconstrained_Array_Indices_F_Types : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Aspect_Assoc
           (Handle : Rewriting_Handle
               ; Aspect_Assoc_F_Id : Node_Rewriting_Handle
               ; Aspect_Assoc_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_At_Clause
           (Handle : Rewriting_Handle
               ; At_Clause_F_Name : Node_Rewriting_Handle
               ; At_Clause_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Attribute_Def_Clause
           (Handle : Rewriting_Handle
               ; Attribute_Def_Clause_F_Attribute_Expr : Node_Rewriting_Handle
               ; Attribute_Def_Clause_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Enum_Rep_Clause
           (Handle : Rewriting_Handle
               ; Enum_Rep_Clause_F_Type_Name : Node_Rewriting_Handle
               ; Enum_Rep_Clause_F_Aggregate : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Record_Rep_Clause
           (Handle : Rewriting_Handle
               ; Record_Rep_Clause_F_Name : Node_Rewriting_Handle
               ; Record_Rep_Clause_F_At_Expr : Node_Rewriting_Handle
               ; Record_Rep_Clause_F_Components : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Aspect_Spec
           (Handle : Rewriting_Handle
               ; Aspect_Spec_F_Aspect_Assocs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Contract_Case_Assoc
           (Handle : Rewriting_Handle
               ; Contract_Case_Assoc_F_Guard : Node_Rewriting_Handle
               ; Contract_Case_Assoc_F_Consequence : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Pragma_Argument_Assoc
           (Handle : Rewriting_Handle
               ; Pragma_Argument_Assoc_F_Name : Node_Rewriting_Handle
               ; Pragma_Argument_Assoc_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Entry_Spec
           (Handle : Rewriting_Handle
               ; Entry_Spec_F_Entry_Name : Node_Rewriting_Handle
               ; Entry_Spec_F_Family_Type : Node_Rewriting_Handle
               ; Entry_Spec_F_Entry_Params : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Subp_Spec
           (Handle : Rewriting_Handle
               ; Subp_Spec_F_Subp_Kind : Node_Rewriting_Handle
               ; Subp_Spec_F_Subp_Name : Node_Rewriting_Handle
               ; Subp_Spec_F_Subp_Params : Node_Rewriting_Handle
               ; Subp_Spec_F_Subp_Returns : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Synthetic_Binary_Spec
           (Handle : Rewriting_Handle
               ; Synthetic_Binary_Spec_F_Left_Param : Node_Rewriting_Handle
               ; Synthetic_Binary_Spec_F_Right_Param : Node_Rewriting_Handle
               ; Synthetic_Binary_Spec_F_Return_Type_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Synthetic_Unary_Spec
           (Handle : Rewriting_Handle
               ; Synthetic_Unary_Spec_F_Right_Param : Node_Rewriting_Handle
               ; Synthetic_Unary_Spec_F_Return_Type_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Component_List
           (Handle : Rewriting_Handle
               ; Component_List_F_Components : Node_Rewriting_Handle
               ; Component_List_F_Variant_Part : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Known_Discriminant_Part
           (Handle : Rewriting_Handle
               ; Known_Discriminant_Part_F_Discr_Specs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Entry_Completion_Formal_Params
           (Handle : Rewriting_Handle
               ; Entry_Completion_Formal_Params_F_Params : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Generic_Formal_Part
           (Handle : Rewriting_Handle
               ; Generic_Formal_Part_F_Decls : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Null_Record_Def
           (Handle : Rewriting_Handle
               ; Base_Record_Def_F_Components : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Record_Def
           (Handle : Rewriting_Handle
               ; Base_Record_Def_F_Components : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Aggregate_Assoc
           (Handle : Rewriting_Handle
               ; Aggregate_Assoc_F_Designators : Node_Rewriting_Handle
               ; Aggregate_Assoc_F_R_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Multi_Dim_Array_Assoc
           (Handle : Rewriting_Handle
               ; Aggregate_Assoc_F_Designators : Node_Rewriting_Handle
               ; Aggregate_Assoc_F_R_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Composite_Constraint_Assoc
           (Handle : Rewriting_Handle
               ; Composite_Constraint_Assoc_F_Ids : Node_Rewriting_Handle
               ; Composite_Constraint_Assoc_F_Constraint_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Iterated_Assoc
           (Handle : Rewriting_Handle
               ; Iterated_Assoc_F_Spec : Node_Rewriting_Handle
               ; Iterated_Assoc_F_Key_Expr : Node_Rewriting_Handle
               ; Iterated_Assoc_F_R_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Param_Assoc
           (Handle : Rewriting_Handle
               ; Param_Assoc_F_Designator : Node_Rewriting_Handle
               ; Param_Assoc_F_R_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Abstract_State_Decl
           (Handle : Rewriting_Handle
               ; Abstract_State_Decl_F_Name : Node_Rewriting_Handle
               ; Abstract_State_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Anonymous_Expr_Decl
           (Handle : Rewriting_Handle
               ; Anonymous_Expr_Decl_F_Expr : Node_Rewriting_Handle
               ; Anonymous_Expr_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Component_Decl
           (Handle : Rewriting_Handle
               ; Component_Decl_F_Ids : Node_Rewriting_Handle
               ; Component_Decl_F_Component_Def : Node_Rewriting_Handle
               ; Component_Decl_F_Default_Expr : Node_Rewriting_Handle
               ; Component_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Discriminant_Spec
           (Handle : Rewriting_Handle
               ; Discriminant_Spec_F_Ids : Node_Rewriting_Handle
               ; Discriminant_Spec_F_Type_Expr : Node_Rewriting_Handle
               ; Discriminant_Spec_F_Default_Expr : Node_Rewriting_Handle
               ; Discriminant_Spec_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Generic_Formal_Obj_Decl
           (Handle : Rewriting_Handle
               ; Generic_Formal_F_Decl : Node_Rewriting_Handle
               ; Generic_Formal_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Generic_Formal_Package
           (Handle : Rewriting_Handle
               ; Generic_Formal_F_Decl : Node_Rewriting_Handle
               ; Generic_Formal_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Generic_Formal_Subp_Decl
           (Handle : Rewriting_Handle
               ; Generic_Formal_F_Decl : Node_Rewriting_Handle
               ; Generic_Formal_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Generic_Formal_Type_Decl
           (Handle : Rewriting_Handle
               ; Generic_Formal_F_Decl : Node_Rewriting_Handle
               ; Generic_Formal_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Param_Spec
           (Handle : Rewriting_Handle
               ; Param_Spec_F_Ids : Node_Rewriting_Handle
               ; Param_Spec_F_Has_Aliased : Node_Rewriting_Handle
               ; Param_Spec_F_Mode : Node_Rewriting_Handle
               ; Param_Spec_F_Type_Expr : Node_Rewriting_Handle
               ; Param_Spec_F_Default_Expr : Node_Rewriting_Handle
               ; Param_Spec_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Synthetic_Formal_Param_Decl
           (Handle : Rewriting_Handle
               ; Synthetic_Formal_Param_Decl_F_Param_Type : Node_Rewriting_Handle
               ; Synthetic_Formal_Param_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Generic_Package_Internal
           (Handle : Rewriting_Handle
               ; Base_Package_Decl_F_Package_Name : Node_Rewriting_Handle
               ; Base_Package_Decl_F_Aspects : Node_Rewriting_Handle
               ; Base_Package_Decl_F_Public_Part : Node_Rewriting_Handle
               ; Base_Package_Decl_F_Private_Part : Node_Rewriting_Handle
               ; Base_Package_Decl_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Package_Decl
           (Handle : Rewriting_Handle
               ; Base_Package_Decl_F_Package_Name : Node_Rewriting_Handle
               ; Base_Package_Decl_F_Aspects : Node_Rewriting_Handle
               ; Base_Package_Decl_F_Public_Part : Node_Rewriting_Handle
               ; Base_Package_Decl_F_Private_Part : Node_Rewriting_Handle
               ; Base_Package_Decl_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Discrete_Base_Subtype_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Discrete_Base_Subtype_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Subtype_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Subtype_Decl_F_Subtype : Node_Rewriting_Handle
               ; Subtype_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Classwide_Type_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Classwide_Type_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Incomplete_Type_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Incomplete_Type_Decl_F_Discriminants : Node_Rewriting_Handle
               ; Incomplete_Type_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Incomplete_Formal_Type_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Incomplete_Type_Decl_F_Discriminants : Node_Rewriting_Handle
               ; Incomplete_Type_Decl_F_Aspects : Node_Rewriting_Handle
               ; Incomplete_Formal_Type_Decl_F_Is_Tagged : Node_Rewriting_Handle
               ; Incomplete_Formal_Type_Decl_F_Default_Type : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Incomplete_Tagged_Type_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Incomplete_Type_Decl_F_Discriminants : Node_Rewriting_Handle
               ; Incomplete_Type_Decl_F_Aspects : Node_Rewriting_Handle
               ; Incomplete_Tagged_Type_Decl_F_Has_Abstract : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Protected_Type_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Protected_Type_Decl_F_Discriminants : Node_Rewriting_Handle
               ; Protected_Type_Decl_F_Aspects : Node_Rewriting_Handle
               ; Protected_Type_Decl_F_Interfaces : Node_Rewriting_Handle
               ; Protected_Type_Decl_F_Definition : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Task_Type_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Task_Type_Decl_F_Discriminants : Node_Rewriting_Handle
               ; Task_Type_Decl_F_Aspects : Node_Rewriting_Handle
               ; Task_Type_Decl_F_Definition : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Single_Task_Type_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Task_Type_Decl_F_Discriminants : Node_Rewriting_Handle
               ; Task_Type_Decl_F_Aspects : Node_Rewriting_Handle
               ; Task_Type_Decl_F_Definition : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Anonymous_Type_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Type_Decl_F_Discriminants : Node_Rewriting_Handle
               ; Type_Decl_F_Type_Def : Node_Rewriting_Handle
               ; Anonymous_Type_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Synth_Anonymous_Type_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Type_Decl_F_Discriminants : Node_Rewriting_Handle
               ; Type_Decl_F_Type_Def : Node_Rewriting_Handle
               ; Anonymous_Type_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Concrete_Type_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Type_Decl_F_Discriminants : Node_Rewriting_Handle
               ; Type_Decl_F_Type_Def : Node_Rewriting_Handle
               ; Concrete_Type_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Formal_Type_Decl
           (Handle : Rewriting_Handle
               ; Base_Type_Decl_F_Name : Node_Rewriting_Handle
               ; Type_Decl_F_Discriminants : Node_Rewriting_Handle
               ; Type_Decl_F_Type_Def : Node_Rewriting_Handle
               ; Formal_Type_Decl_F_Default_Type : Node_Rewriting_Handle
               ; Formal_Type_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Abstract_Subp_Decl
           (Handle : Rewriting_Handle
               ; Classic_Subp_Decl_F_Overriding : Node_Rewriting_Handle
               ; Classic_Subp_Decl_F_Subp_Spec : Node_Rewriting_Handle
               ; Abstract_Subp_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Abstract_Formal_Subp_Decl
           (Handle : Rewriting_Handle
               ; Classic_Subp_Decl_F_Overriding : Node_Rewriting_Handle
               ; Classic_Subp_Decl_F_Subp_Spec : Node_Rewriting_Handle
               ; Formal_Subp_Decl_F_Default_Expr : Node_Rewriting_Handle
               ; Formal_Subp_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Concrete_Formal_Subp_Decl
           (Handle : Rewriting_Handle
               ; Classic_Subp_Decl_F_Overriding : Node_Rewriting_Handle
               ; Classic_Subp_Decl_F_Subp_Spec : Node_Rewriting_Handle
               ; Formal_Subp_Decl_F_Default_Expr : Node_Rewriting_Handle
               ; Formal_Subp_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Subp_Decl
           (Handle : Rewriting_Handle
               ; Classic_Subp_Decl_F_Overriding : Node_Rewriting_Handle
               ; Classic_Subp_Decl_F_Subp_Spec : Node_Rewriting_Handle
               ; Subp_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Entry_Decl
           (Handle : Rewriting_Handle
               ; Entry_Decl_F_Overriding : Node_Rewriting_Handle
               ; Entry_Decl_F_Spec : Node_Rewriting_Handle
               ; Entry_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Enum_Literal_Decl
           (Handle : Rewriting_Handle
               ; Enum_Literal_Decl_F_Name : Node_Rewriting_Handle
               ; Enum_Literal_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Synthetic_Char_Enum_Lit
           (Handle : Rewriting_Handle
               ; Enum_Literal_Decl_F_Name : Node_Rewriting_Handle
               ; Enum_Literal_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Generic_Subp_Internal
           (Handle : Rewriting_Handle
               ; Generic_Subp_Internal_F_Subp_Spec : Node_Rewriting_Handle
               ; Generic_Subp_Internal_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Synthetic_Subp_Decl
           (Handle : Rewriting_Handle
               ; Synthetic_Subp_Decl_F_Aspects : Node_Rewriting_Handle
               ; Synthetic_Subp_Decl_F_Spec : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Accept_Stmt_Body
           (Handle : Rewriting_Handle
               ; Accept_Stmt_Body_F_Name : Node_Rewriting_Handle
               ; Accept_Stmt_Body_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Expr_Function
           (Handle : Rewriting_Handle
               ; Base_Subp_Body_F_Overriding : Node_Rewriting_Handle
               ; Base_Subp_Body_F_Subp_Spec : Node_Rewriting_Handle
               ; Expr_Function_F_Expr : Node_Rewriting_Handle
               ; Expr_Function_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Null_Subp_Decl
           (Handle : Rewriting_Handle
               ; Base_Subp_Body_F_Overriding : Node_Rewriting_Handle
               ; Base_Subp_Body_F_Subp_Spec : Node_Rewriting_Handle
               ; Null_Subp_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Subp_Body
           (Handle : Rewriting_Handle
               ; Base_Subp_Body_F_Overriding : Node_Rewriting_Handle
               ; Base_Subp_Body_F_Subp_Spec : Node_Rewriting_Handle
               ; Subp_Body_F_Aspects : Node_Rewriting_Handle
               ; Subp_Body_F_Decls : Node_Rewriting_Handle
               ; Subp_Body_F_Stmts : Node_Rewriting_Handle
               ; Subp_Body_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Subp_Renaming_Decl
           (Handle : Rewriting_Handle
               ; Base_Subp_Body_F_Overriding : Node_Rewriting_Handle
               ; Base_Subp_Body_F_Subp_Spec : Node_Rewriting_Handle
               ; Subp_Renaming_Decl_F_Renames : Node_Rewriting_Handle
               ; Subp_Renaming_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Package_Body_Stub
           (Handle : Rewriting_Handle
               ; Package_Body_Stub_F_Name : Node_Rewriting_Handle
               ; Package_Body_Stub_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Protected_Body_Stub
           (Handle : Rewriting_Handle
               ; Protected_Body_Stub_F_Name : Node_Rewriting_Handle
               ; Protected_Body_Stub_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Subp_Body_Stub
           (Handle : Rewriting_Handle
               ; Subp_Body_Stub_F_Overriding : Node_Rewriting_Handle
               ; Subp_Body_Stub_F_Subp_Spec : Node_Rewriting_Handle
               ; Subp_Body_Stub_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Task_Body_Stub
           (Handle : Rewriting_Handle
               ; Task_Body_Stub_F_Name : Node_Rewriting_Handle
               ; Task_Body_Stub_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Entry_Body
           (Handle : Rewriting_Handle
               ; Entry_Body_F_Entry_Name : Node_Rewriting_Handle
               ; Entry_Body_F_Index_Spec : Node_Rewriting_Handle
               ; Entry_Body_F_Params : Node_Rewriting_Handle
               ; Entry_Body_F_Aspects : Node_Rewriting_Handle
               ; Entry_Body_F_Barrier : Node_Rewriting_Handle
               ; Entry_Body_F_Decls : Node_Rewriting_Handle
               ; Entry_Body_F_Stmts : Node_Rewriting_Handle
               ; Entry_Body_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Package_Body
           (Handle : Rewriting_Handle
               ; Package_Body_F_Package_Name : Node_Rewriting_Handle
               ; Package_Body_F_Aspects : Node_Rewriting_Handle
               ; Package_Body_F_Decls : Node_Rewriting_Handle
               ; Package_Body_F_Stmts : Node_Rewriting_Handle
               ; Package_Body_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Protected_Body
           (Handle : Rewriting_Handle
               ; Protected_Body_F_Name : Node_Rewriting_Handle
               ; Protected_Body_F_Aspects : Node_Rewriting_Handle
               ; Protected_Body_F_Decls : Node_Rewriting_Handle
               ; Protected_Body_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Task_Body
           (Handle : Rewriting_Handle
               ; Task_Body_F_Name : Node_Rewriting_Handle
               ; Task_Body_F_Aspects : Node_Rewriting_Handle
               ; Task_Body_F_Decls : Node_Rewriting_Handle
               ; Task_Body_F_Stmts : Node_Rewriting_Handle
               ; Task_Body_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Entry_Index_Spec
           (Handle : Rewriting_Handle
               ; Entry_Index_Spec_F_Id : Node_Rewriting_Handle
               ; Entry_Index_Spec_F_Subtype : Node_Rewriting_Handle
               ; Entry_Index_Spec_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Error_Decl
           (Handle : Rewriting_Handle
               ; Error_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Exception_Decl
           (Handle : Rewriting_Handle
               ; Exception_Decl_F_Ids : Node_Rewriting_Handle
               ; Exception_Decl_F_Renames : Node_Rewriting_Handle
               ; Exception_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Exception_Handler
           (Handle : Rewriting_Handle
               ; Exception_Handler_F_Exception_Name : Node_Rewriting_Handle
               ; Exception_Handler_F_Handled_Exceptions : Node_Rewriting_Handle
               ; Exception_Handler_F_Stmts : Node_Rewriting_Handle
               ; Exception_Handler_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_For_Loop_Var_Decl
           (Handle : Rewriting_Handle
               ; For_Loop_Var_Decl_F_Id : Node_Rewriting_Handle
               ; For_Loop_Var_Decl_F_Id_Type : Node_Rewriting_Handle
               ; For_Loop_Var_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Generic_Package_Decl
           (Handle : Rewriting_Handle
               ; Generic_Decl_F_Formal_Part : Node_Rewriting_Handle
               ; Generic_Package_Decl_F_Package_Decl : Node_Rewriting_Handle
               ; Generic_Package_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Generic_Subp_Decl
           (Handle : Rewriting_Handle
               ; Generic_Decl_F_Formal_Part : Node_Rewriting_Handle
               ; Generic_Subp_Decl_F_Subp_Decl : Node_Rewriting_Handle
               ; Generic_Subp_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Generic_Package_Instantiation
           (Handle : Rewriting_Handle
               ; Generic_Package_Instantiation_F_Name : Node_Rewriting_Handle
               ; Generic_Package_Instantiation_F_Generic_Pkg_Name : Node_Rewriting_Handle
               ; Generic_Package_Instantiation_F_Params : Node_Rewriting_Handle
               ; Generic_Package_Instantiation_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Generic_Subp_Instantiation
           (Handle : Rewriting_Handle
               ; Generic_Subp_Instantiation_F_Overriding : Node_Rewriting_Handle
               ; Generic_Subp_Instantiation_F_Kind : Node_Rewriting_Handle
               ; Generic_Subp_Instantiation_F_Subp_Name : Node_Rewriting_Handle
               ; Generic_Subp_Instantiation_F_Generic_Subp_Name : Node_Rewriting_Handle
               ; Generic_Subp_Instantiation_F_Params : Node_Rewriting_Handle
               ; Generic_Subp_Instantiation_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Generic_Package_Renaming_Decl
           (Handle : Rewriting_Handle
               ; Generic_Package_Renaming_Decl_F_Name : Node_Rewriting_Handle
               ; Generic_Package_Renaming_Decl_F_Renames : Node_Rewriting_Handle
               ; Generic_Package_Renaming_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Generic_Subp_Renaming_Decl
           (Handle : Rewriting_Handle
               ; Generic_Subp_Renaming_Decl_F_Kind : Node_Rewriting_Handle
               ; Generic_Subp_Renaming_Decl_F_Name : Node_Rewriting_Handle
               ; Generic_Subp_Renaming_Decl_F_Renames : Node_Rewriting_Handle
               ; Generic_Subp_Renaming_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Label_Decl
           (Handle : Rewriting_Handle
               ; Label_Decl_F_Name : Node_Rewriting_Handle
               ; Label_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Named_Stmt_Decl
           (Handle : Rewriting_Handle
               ; Named_Stmt_Decl_F_Name : Node_Rewriting_Handle
               ; Named_Stmt_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Number_Decl
           (Handle : Rewriting_Handle
               ; Number_Decl_F_Ids : Node_Rewriting_Handle
               ; Number_Decl_F_Expr : Node_Rewriting_Handle
               ; Number_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Object_Decl
           (Handle : Rewriting_Handle
               ; Object_Decl_F_Ids : Node_Rewriting_Handle
               ; Object_Decl_F_Has_Aliased : Node_Rewriting_Handle
               ; Object_Decl_F_Has_Constant : Node_Rewriting_Handle
               ; Object_Decl_F_Mode : Node_Rewriting_Handle
               ; Object_Decl_F_Type_Expr : Node_Rewriting_Handle
               ; Object_Decl_F_Default_Expr : Node_Rewriting_Handle
               ; Object_Decl_F_Renaming_Clause : Node_Rewriting_Handle
               ; Object_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Extended_Return_Stmt_Object_Decl
           (Handle : Rewriting_Handle
               ; Object_Decl_F_Ids : Node_Rewriting_Handle
               ; Object_Decl_F_Has_Aliased : Node_Rewriting_Handle
               ; Object_Decl_F_Has_Constant : Node_Rewriting_Handle
               ; Object_Decl_F_Mode : Node_Rewriting_Handle
               ; Object_Decl_F_Type_Expr : Node_Rewriting_Handle
               ; Object_Decl_F_Default_Expr : Node_Rewriting_Handle
               ; Object_Decl_F_Renaming_Clause : Node_Rewriting_Handle
               ; Object_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_No_Type_Object_Renaming_Decl
           (Handle : Rewriting_Handle
               ; Object_Decl_F_Ids : Node_Rewriting_Handle
               ; Object_Decl_F_Has_Aliased : Node_Rewriting_Handle
               ; Object_Decl_F_Has_Constant : Node_Rewriting_Handle
               ; Object_Decl_F_Mode : Node_Rewriting_Handle
               ; Object_Decl_F_Type_Expr : Node_Rewriting_Handle
               ; Object_Decl_F_Default_Expr : Node_Rewriting_Handle
               ; Object_Decl_F_Renaming_Clause : Node_Rewriting_Handle
               ; Object_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Package_Renaming_Decl
           (Handle : Rewriting_Handle
               ; Package_Renaming_Decl_F_Name : Node_Rewriting_Handle
               ; Package_Renaming_Decl_F_Renames : Node_Rewriting_Handle
               ; Package_Renaming_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Single_Protected_Decl
           (Handle : Rewriting_Handle
               ; Single_Protected_Decl_F_Name : Node_Rewriting_Handle
               ; Single_Protected_Decl_F_Aspects : Node_Rewriting_Handle
               ; Single_Protected_Decl_F_Interfaces : Node_Rewriting_Handle
               ; Single_Protected_Decl_F_Definition : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Single_Task_Decl
           (Handle : Rewriting_Handle
               ; Single_Task_Decl_F_Task_Type : Node_Rewriting_Handle
               ; Single_Task_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Synthetic_Object_Decl
           (Handle : Rewriting_Handle
               ; Synthetic_Object_Decl_F_Aspects : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Case_Stmt_Alternative
           (Handle : Rewriting_Handle
               ; Case_Stmt_Alternative_F_Choices : Node_Rewriting_Handle
               ; Case_Stmt_Alternative_F_Stmts : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Compilation_Unit
           (Handle : Rewriting_Handle
               ; Compilation_Unit_F_Prelude : Node_Rewriting_Handle
               ; Compilation_Unit_F_Body : Node_Rewriting_Handle
               ; Compilation_Unit_F_Pragmas : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Component_Clause
           (Handle : Rewriting_Handle
               ; Component_Clause_F_Id : Node_Rewriting_Handle
               ; Component_Clause_F_Position : Node_Rewriting_Handle
               ; Component_Clause_F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Component_Def
           (Handle : Rewriting_Handle
               ; Component_Def_F_Has_Aliased : Node_Rewriting_Handle
               ; Component_Def_F_Has_Constant : Node_Rewriting_Handle
               ; Component_Def_F_Type_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Composite_Constraint
           (Handle : Rewriting_Handle
               ; Composite_Constraint_F_Constraints : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Delta_Constraint
           (Handle : Rewriting_Handle
               ; Delta_Constraint_F_Delta : Node_Rewriting_Handle
               ; Delta_Constraint_F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Digits_Constraint
           (Handle : Rewriting_Handle
               ; Digits_Constraint_F_Digits : Node_Rewriting_Handle
               ; Digits_Constraint_F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Range_Constraint
           (Handle : Rewriting_Handle
               ; Range_Constraint_F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Declarative_Part
           (Handle : Rewriting_Handle
               ; Declarative_Part_F_Decls : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Private_Part
           (Handle : Rewriting_Handle
               ; Declarative_Part_F_Decls : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Public_Part
           (Handle : Rewriting_Handle
               ; Declarative_Part_F_Decls : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Elsif_Expr_Part
           (Handle : Rewriting_Handle
               ; Elsif_Expr_Part_F_Cond_Expr : Node_Rewriting_Handle
               ; Elsif_Expr_Part_F_Then_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Elsif_Stmt_Part
           (Handle : Rewriting_Handle
               ; Elsif_Stmt_Part_F_Cond_Expr : Node_Rewriting_Handle
               ; Elsif_Stmt_Part_F_Stmts : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Abstract_State_Decl_Expr
           (Handle : Rewriting_Handle
               ; Abstract_State_Decl_Expr_F_State_Decl : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Allocator
           (Handle : Rewriting_Handle
               ; Allocator_F_Subpool : Node_Rewriting_Handle
               ; Allocator_F_Type_Or_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Aggregate
           (Handle : Rewriting_Handle
               ; Base_Aggregate_F_Ancestor_Expr : Node_Rewriting_Handle
               ; Base_Aggregate_F_Assocs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Bracket_Aggregate
           (Handle : Rewriting_Handle
               ; Base_Aggregate_F_Ancestor_Expr : Node_Rewriting_Handle
               ; Base_Aggregate_F_Assocs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Delta_Aggregate
           (Handle : Rewriting_Handle
               ; Base_Aggregate_F_Ancestor_Expr : Node_Rewriting_Handle
               ; Base_Aggregate_F_Assocs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Bracket_Delta_Aggregate
           (Handle : Rewriting_Handle
               ; Base_Aggregate_F_Ancestor_Expr : Node_Rewriting_Handle
               ; Base_Aggregate_F_Assocs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Null_Record_Aggregate
           (Handle : Rewriting_Handle
               ; Base_Aggregate_F_Ancestor_Expr : Node_Rewriting_Handle
               ; Base_Aggregate_F_Assocs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Bin_Op
           (Handle : Rewriting_Handle
               ; Bin_Op_F_Left : Node_Rewriting_Handle
               ; Bin_Op_F_Op : Node_Rewriting_Handle
               ; Bin_Op_F_Right : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Relation_Op
           (Handle : Rewriting_Handle
               ; Bin_Op_F_Left : Node_Rewriting_Handle
               ; Bin_Op_F_Op : Node_Rewriting_Handle
               ; Bin_Op_F_Right : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Case_Expr_Alternative
           (Handle : Rewriting_Handle
               ; Case_Expr_Alternative_F_Choices : Node_Rewriting_Handle
               ; Case_Expr_Alternative_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Concat_Op
           (Handle : Rewriting_Handle
               ; Concat_Op_F_First_Operand : Node_Rewriting_Handle
               ; Concat_Op_F_Other_Operands : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Concat_Operand
           (Handle : Rewriting_Handle
               ; Concat_Operand_F_Operator : Node_Rewriting_Handle
               ; Concat_Operand_F_Operand : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Case_Expr
           (Handle : Rewriting_Handle
               ; Case_Expr_F_Expr : Node_Rewriting_Handle
               ; Case_Expr_F_Cases : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_If_Expr
           (Handle : Rewriting_Handle
               ; If_Expr_F_Cond_Expr : Node_Rewriting_Handle
               ; If_Expr_F_Then_Expr : Node_Rewriting_Handle
               ; If_Expr_F_Alternatives : Node_Rewriting_Handle
               ; If_Expr_F_Else_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Contract_Cases
           (Handle : Rewriting_Handle
               ; Contract_Cases_F_Contract_Cases : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Decl_Expr
           (Handle : Rewriting_Handle
               ; Decl_Expr_F_Decls : Node_Rewriting_Handle
               ; Decl_Expr_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Format_String_Literal
           (Handle : Rewriting_Handle
               ; Format_String_Literal_F_Opening_Chunk : Node_Rewriting_Handle
               ; Format_String_Literal_F_Mid_Exprs : Node_Rewriting_Handle
               ; Format_String_Literal_F_Trailing_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Membership_Expr
           (Handle : Rewriting_Handle
               ; Membership_Expr_F_Expr : Node_Rewriting_Handle
               ; Membership_Expr_F_Op : Node_Rewriting_Handle
               ; Membership_Expr_F_Membership_Exprs : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Array_Subcomponent_Choice_Name
           (Handle : Rewriting_Handle
               ; Array_Subcomponent_Choice_Name_F_Name : Node_Rewriting_Handle
               ; Array_Subcomponent_Choice_Name_F_Suffix : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Attribute_Ref
           (Handle : Rewriting_Handle
               ; Attribute_Ref_F_Prefix : Node_Rewriting_Handle
               ; Attribute_Ref_F_Attribute : Node_Rewriting_Handle
               ; Attribute_Ref_F_Args : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Call_Expr
           (Handle : Rewriting_Handle
               ; Call_Expr_F_Name : Node_Rewriting_Handle
               ; Call_Expr_F_Suffix : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Defining_Name
           (Handle : Rewriting_Handle
               ; Defining_Name_F_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Synthetic_Defining_Name
           (Handle : Rewriting_Handle
               ; Defining_Name_F_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Discrete_Subtype_Name
           (Handle : Rewriting_Handle
               ; Discrete_Subtype_Name_F_Subtype : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Dotted_Name
           (Handle : Rewriting_Handle
               ; Dotted_Name_F_Prefix : Node_Rewriting_Handle
               ; Dotted_Name_F_Suffix : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_End_Name
           (Handle : Rewriting_Handle
               ; End_Name_F_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Explicit_Deref
           (Handle : Rewriting_Handle
               ; Explicit_Deref_F_Prefix : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Qual_Expr
           (Handle : Rewriting_Handle
               ; Qual_Expr_F_Prefix : Node_Rewriting_Handle
               ; Qual_Expr_F_Suffix : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Reduce_Attribute_Ref
           (Handle : Rewriting_Handle
               ; Reduce_Attribute_Ref_F_Prefix : Node_Rewriting_Handle
               ; Reduce_Attribute_Ref_F_Attribute : Node_Rewriting_Handle
               ; Reduce_Attribute_Ref_F_Args : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Update_Attribute_Ref
           (Handle : Rewriting_Handle
               ; Update_Attribute_Ref_F_Prefix : Node_Rewriting_Handle
               ; Update_Attribute_Ref_F_Attribute : Node_Rewriting_Handle
               ; Update_Attribute_Ref_F_Values : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Paren_Expr
           (Handle : Rewriting_Handle
               ; Paren_Expr_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Quantified_Expr
           (Handle : Rewriting_Handle
               ; Quantified_Expr_F_Quantifier : Node_Rewriting_Handle
               ; Quantified_Expr_F_Loop_Spec : Node_Rewriting_Handle
               ; Quantified_Expr_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Raise_Expr
           (Handle : Rewriting_Handle
               ; Raise_Expr_F_Exception_Name : Node_Rewriting_Handle
               ; Raise_Expr_F_Error_Message : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Un_Op
           (Handle : Rewriting_Handle
               ; Un_Op_F_Op : Node_Rewriting_Handle
               ; Un_Op_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_For_Loop_Iter_Filter
           (Handle : Rewriting_Handle
               ; For_Loop_Iter_Filter_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Format_String_Chunk
           (Handle : Rewriting_Handle
               ; Format_String_Chunk_F_Expr : Node_Rewriting_Handle
               ; Format_String_Chunk_F_String_Tok : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Handled_Stmts
           (Handle : Rewriting_Handle
               ; Handled_Stmts_F_Stmts : Node_Rewriting_Handle
               ; Handled_Stmts_F_Exceptions : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Library_Item
           (Handle : Rewriting_Handle
               ; Library_Item_F_Has_Private : Node_Rewriting_Handle
               ; Library_Item_F_Item : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_For_Loop_Spec
           (Handle : Rewriting_Handle
               ; For_Loop_Spec_F_Var_Decl : Node_Rewriting_Handle
               ; For_Loop_Spec_F_Loop_Type : Node_Rewriting_Handle
               ; For_Loop_Spec_F_Has_Reverse : Node_Rewriting_Handle
               ; For_Loop_Spec_F_Iter_Expr : Node_Rewriting_Handle
               ; For_Loop_Spec_F_Iter_Filter : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_While_Loop_Spec
           (Handle : Rewriting_Handle
               ; While_Loop_Spec_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Multi_Abstract_State_Decl
           (Handle : Rewriting_Handle
               ; Multi_Abstract_State_Decl_F_Decls : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Params
           (Handle : Rewriting_Handle
               ; Params_F_Params : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Paren_Abstract_State_Decl
           (Handle : Rewriting_Handle
               ; Paren_Abstract_State_Decl_F_Decl : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Pp_Elsif_Directive
           (Handle : Rewriting_Handle
               ; Pp_Elsif_Directive_F_Expr : Node_Rewriting_Handle
               ; Pp_Elsif_Directive_F_Then_Kw : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Pp_If_Directive
           (Handle : Rewriting_Handle
               ; Pp_If_Directive_F_Expr : Node_Rewriting_Handle
               ; Pp_If_Directive_F_Then_Kw : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Pragma_Node
           (Handle : Rewriting_Handle
               ; Pragma_Node_F_Id : Node_Rewriting_Handle
               ; Pragma_Node_F_Args : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Protected_Def
           (Handle : Rewriting_Handle
               ; Protected_Def_F_Public_Part : Node_Rewriting_Handle
               ; Protected_Def_F_Private_Part : Node_Rewriting_Handle
               ; Protected_Def_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Range_Spec
           (Handle : Rewriting_Handle
               ; Range_Spec_F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Renaming_Clause
           (Handle : Rewriting_Handle
               ; Renaming_Clause_F_Renamed_Object : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Synthetic_Renaming_Clause
           (Handle : Rewriting_Handle
               ; Renaming_Clause_F_Renamed_Object : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Select_When_Part
           (Handle : Rewriting_Handle
               ; Select_When_Part_F_Cond_Expr : Node_Rewriting_Handle
               ; Select_When_Part_F_Stmts : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Accept_Stmt
           (Handle : Rewriting_Handle
               ; Accept_Stmt_F_Body_Decl : Node_Rewriting_Handle
               ; Accept_Stmt_F_Entry_Index_Expr : Node_Rewriting_Handle
               ; Accept_Stmt_F_Params : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Accept_Stmt_With_Stmts
           (Handle : Rewriting_Handle
               ; Accept_Stmt_F_Body_Decl : Node_Rewriting_Handle
               ; Accept_Stmt_F_Entry_Index_Expr : Node_Rewriting_Handle
               ; Accept_Stmt_F_Params : Node_Rewriting_Handle
               ; Accept_Stmt_With_Stmts_F_Stmts : Node_Rewriting_Handle
               ; Accept_Stmt_With_Stmts_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_For_Loop_Stmt
           (Handle : Rewriting_Handle
               ; Base_Loop_Stmt_F_Spec : Node_Rewriting_Handle
               ; Base_Loop_Stmt_F_Stmts : Node_Rewriting_Handle
               ; Base_Loop_Stmt_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Loop_Stmt
           (Handle : Rewriting_Handle
               ; Base_Loop_Stmt_F_Spec : Node_Rewriting_Handle
               ; Base_Loop_Stmt_F_Stmts : Node_Rewriting_Handle
               ; Base_Loop_Stmt_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_While_Loop_Stmt
           (Handle : Rewriting_Handle
               ; Base_Loop_Stmt_F_Spec : Node_Rewriting_Handle
               ; Base_Loop_Stmt_F_Stmts : Node_Rewriting_Handle
               ; Base_Loop_Stmt_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Begin_Block
           (Handle : Rewriting_Handle
               ; Begin_Block_F_Stmts : Node_Rewriting_Handle
               ; Begin_Block_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Decl_Block
           (Handle : Rewriting_Handle
               ; Decl_Block_F_Decls : Node_Rewriting_Handle
               ; Decl_Block_F_Stmts : Node_Rewriting_Handle
               ; Decl_Block_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Case_Stmt
           (Handle : Rewriting_Handle
               ; Case_Stmt_F_Expr : Node_Rewriting_Handle
               ; Case_Stmt_F_Pragmas : Node_Rewriting_Handle
               ; Case_Stmt_F_Alternatives : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Extended_Return_Stmt
           (Handle : Rewriting_Handle
               ; Extended_Return_Stmt_F_Decl : Node_Rewriting_Handle
               ; Extended_Return_Stmt_F_Stmts : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_If_Stmt
           (Handle : Rewriting_Handle
               ; If_Stmt_F_Cond_Expr : Node_Rewriting_Handle
               ; If_Stmt_F_Then_Stmts : Node_Rewriting_Handle
               ; If_Stmt_F_Alternatives : Node_Rewriting_Handle
               ; If_Stmt_F_Else_Stmts : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Named_Stmt
           (Handle : Rewriting_Handle
               ; Named_Stmt_F_Decl : Node_Rewriting_Handle
               ; Named_Stmt_F_Stmt : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Select_Stmt
           (Handle : Rewriting_Handle
               ; Select_Stmt_F_Guards : Node_Rewriting_Handle
               ; Select_Stmt_F_Else_Stmts : Node_Rewriting_Handle
               ; Select_Stmt_F_Abort_Stmts : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Abort_Stmt
           (Handle : Rewriting_Handle
               ; Abort_Stmt_F_Names : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Assign_Stmt
           (Handle : Rewriting_Handle
               ; Assign_Stmt_F_Dest : Node_Rewriting_Handle
               ; Assign_Stmt_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Call_Stmt
           (Handle : Rewriting_Handle
               ; Call_Stmt_F_Call : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Delay_Stmt
           (Handle : Rewriting_Handle
               ; Delay_Stmt_F_Has_Until : Node_Rewriting_Handle
               ; Delay_Stmt_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Exit_Stmt
           (Handle : Rewriting_Handle
               ; Exit_Stmt_F_Loop_Name : Node_Rewriting_Handle
               ; Exit_Stmt_F_Cond_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Goto_Stmt
           (Handle : Rewriting_Handle
               ; Goto_Stmt_F_Label_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Label
           (Handle : Rewriting_Handle
               ; Label_F_Decl : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Raise_Stmt
           (Handle : Rewriting_Handle
               ; Raise_Stmt_F_Exception_Name : Node_Rewriting_Handle
               ; Raise_Stmt_F_Error_Message : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Requeue_Stmt
           (Handle : Rewriting_Handle
               ; Requeue_Stmt_F_Call_Name : Node_Rewriting_Handle
               ; Requeue_Stmt_F_Has_Abort : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Return_Stmt
           (Handle : Rewriting_Handle
               ; Return_Stmt_F_Return_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Simple_Decl_Stmt
           (Handle : Rewriting_Handle
               ; Simple_Decl_Stmt_F_Decl : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Subunit
           (Handle : Rewriting_Handle
               ; Subunit_F_Name : Node_Rewriting_Handle
               ; Subunit_F_Body : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Task_Def
           (Handle : Rewriting_Handle
               ; Task_Def_F_Interfaces : Node_Rewriting_Handle
               ; Task_Def_F_Public_Part : Node_Rewriting_Handle
               ; Task_Def_F_Private_Part : Node_Rewriting_Handle
               ; Task_Def_F_End_Name : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Access_To_Subp_Def
           (Handle : Rewriting_Handle
               ; Access_Def_F_Has_Not_Null : Node_Rewriting_Handle
               ; Access_To_Subp_Def_F_Has_Protected : Node_Rewriting_Handle
               ; Access_To_Subp_Def_F_Subp_Spec : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Anonymous_Type_Access_Def
           (Handle : Rewriting_Handle
               ; Access_Def_F_Has_Not_Null : Node_Rewriting_Handle
               ; Anonymous_Type_Access_Def_F_Type_Decl : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Type_Access_Def
           (Handle : Rewriting_Handle
               ; Access_Def_F_Has_Not_Null : Node_Rewriting_Handle
               ; Type_Access_Def_F_Has_All : Node_Rewriting_Handle
               ; Type_Access_Def_F_Has_Constant : Node_Rewriting_Handle
               ; Type_Access_Def_F_Subtype_Indication : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Array_Type_Def
           (Handle : Rewriting_Handle
               ; Array_Type_Def_F_Indices : Node_Rewriting_Handle
               ; Array_Type_Def_F_Component_Type : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Derived_Type_Def
           (Handle : Rewriting_Handle
               ; Derived_Type_Def_F_Has_Abstract : Node_Rewriting_Handle
               ; Derived_Type_Def_F_Has_Limited : Node_Rewriting_Handle
               ; Derived_Type_Def_F_Has_Synchronized : Node_Rewriting_Handle
               ; Derived_Type_Def_F_Subtype_Indication : Node_Rewriting_Handle
               ; Derived_Type_Def_F_Interfaces : Node_Rewriting_Handle
               ; Derived_Type_Def_F_Record_Extension : Node_Rewriting_Handle
               ; Derived_Type_Def_F_Has_With_Private : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Enum_Type_Def
           (Handle : Rewriting_Handle
               ; Enum_Type_Def_F_Enum_Literals : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Interface_Type_Def
           (Handle : Rewriting_Handle
               ; Interface_Type_Def_F_Interface_Kind : Node_Rewriting_Handle
               ; Interface_Type_Def_F_Interfaces : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Mod_Int_Type_Def
           (Handle : Rewriting_Handle
               ; Mod_Int_Type_Def_F_Expr : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Private_Type_Def
           (Handle : Rewriting_Handle
               ; Private_Type_Def_F_Has_Abstract : Node_Rewriting_Handle
               ; Private_Type_Def_F_Has_Tagged : Node_Rewriting_Handle
               ; Private_Type_Def_F_Has_Limited : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Decimal_Fixed_Point_Def
           (Handle : Rewriting_Handle
               ; Decimal_Fixed_Point_Def_F_Delta : Node_Rewriting_Handle
               ; Decimal_Fixed_Point_Def_F_Digits : Node_Rewriting_Handle
               ; Decimal_Fixed_Point_Def_F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Floating_Point_Def
           (Handle : Rewriting_Handle
               ; Floating_Point_Def_F_Num_Digits : Node_Rewriting_Handle
               ; Floating_Point_Def_F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Ordinary_Fixed_Point_Def
           (Handle : Rewriting_Handle
               ; Ordinary_Fixed_Point_Def_F_Delta : Node_Rewriting_Handle
               ; Ordinary_Fixed_Point_Def_F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Record_Type_Def
           (Handle : Rewriting_Handle
               ; Record_Type_Def_F_Has_Abstract : Node_Rewriting_Handle
               ; Record_Type_Def_F_Has_Tagged : Node_Rewriting_Handle
               ; Record_Type_Def_F_Has_Limited : Node_Rewriting_Handle
               ; Record_Type_Def_F_Record_Def : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Signed_Int_Type_Def
           (Handle : Rewriting_Handle
               ; Signed_Int_Type_Def_F_Range : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Anonymous_Type
           (Handle : Rewriting_Handle
               ; Anonymous_Type_F_Type_Decl : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Subtype_Indication
           (Handle : Rewriting_Handle
               ; Subtype_Indication_F_Has_Not_Null : Node_Rewriting_Handle
               ; Subtype_Indication_F_Name : Node_Rewriting_Handle
               ; Subtype_Indication_F_Constraint : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Constrained_Subtype_Indication
           (Handle : Rewriting_Handle
               ; Subtype_Indication_F_Has_Not_Null : Node_Rewriting_Handle
               ; Subtype_Indication_F_Name : Node_Rewriting_Handle
               ; Subtype_Indication_F_Constraint : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Discrete_Subtype_Indication
           (Handle : Rewriting_Handle
               ; Subtype_Indication_F_Has_Not_Null : Node_Rewriting_Handle
               ; Subtype_Indication_F_Name : Node_Rewriting_Handle
               ; Subtype_Indication_F_Constraint : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Synthetic_Type_Expr
           (Handle : Rewriting_Handle
               ; Synthetic_Type_Expr_F_Target_Type : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Unconstrained_Array_Index
           (Handle : Rewriting_Handle
               ; Unconstrained_Array_Index_F_Subtype_Name : Node_Rewriting_Handle
               ; Unconstrained_Array_Index_F_Lower_Bound : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Use_Package_Clause
           (Handle : Rewriting_Handle
               ; Use_Package_Clause_F_Packages : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Use_Type_Clause
           (Handle : Rewriting_Handle
               ; Use_Type_Clause_F_Has_All : Node_Rewriting_Handle
               ; Use_Type_Clause_F_Types : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Value_Sequence
           (Handle : Rewriting_Handle
               ; Value_Sequence_F_Iter_Assoc : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Variant
           (Handle : Rewriting_Handle
               ; Variant_F_Choices : Node_Rewriting_Handle
               ; Variant_F_Components : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_Variant_Part
           (Handle : Rewriting_Handle
               ; Variant_Part_F_Discr_Name : Node_Rewriting_Handle
               ; Variant_Part_F_Variant : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


         function Create_With_Clause
           (Handle : Rewriting_Handle
               ; With_Clause_F_Has_Limited : Node_Rewriting_Handle
               ; With_Clause_F_Has_Private : Node_Rewriting_Handle
               ; With_Clause_F_Packages : Node_Rewriting_Handle
            ) return Node_Rewriting_Handle;


end Libadalang.Rewriting_Implementation;
