//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// デバッガ (内部用)
//

#pragma once

#include "debugger_defs.h"
#include "bus.h"

class IODevice;

enum class CPUState
{
	Normal = 0,
	Stop,		// STOP 命令など割り込み待ち状態
	Halt,		// ホールト、リセットなど非動作状態
};

// デバッガの CPU 依存部分
class DebuggerMD
{
 public:
	DebuggerMD(Debugger *parent_, CPUArch arch_);
	virtual ~DebuggerMD();

	// CPU の実行状態を返す。
	virtual CPUState GetCPUState() const = 0;
	// 特権モードなら true を返す
	virtual bool IsSuper() const = 0;
	// MMU が有効なら true を返す
	virtual bool MMUEnabled() const = 0;

	// アドレス変換をする。
	// MMUEnabled() が true であることは呼び出し側が確認すること。
	// 変換出来なければ BusAddr::BusErr を返す。
	// テーブルサーチを行ったら BusAddr::TableSearched を立てる。
	virtual busaddr TranslateAddr(busaddr laddr) const = 0;

	// 現在の命令先頭アドレスを返す
	virtual uint32 GetPC() const = 0;

	// ステップアウトを設定する
	virtual void SetStepOut() = 0;
	// ステップアウトなら true を返す
	virtual bool IsStepOut() const = 0;

	// この命令がステップイン出来るなら true を返す (cmd_n 用)
	virtual bool IsOpStepIn(DebuggerMemoryStream& mem) = 0;

	// name で示されるのレジスタの値を返す。
	// メモリダンプで指定するケースを想定しているので、明らかにアドレスでは
	// ないレジスタは含まなくてよい (m680x0 の SR など)。ただし Dn は含めて
	// よい。また m680x0 の SRP/CRP は下位32ビットでよい。
	// name が正しくない場合は (uint64)-1 を返す。
	virtual uint64 GetRegAddr(const char *name) const = 0;

	// 現在のレジスタセットを内部にバックアップする
	virtual void BackupRegs() = 0;

	// レジスタ表示系のコマンドを実行。処理すれば true を返す。
	virtual bool ShowRegister(FILE *, const std::vector<std::string>& args) = 0;
	// レジスタ表示系コマンドのヘルプ一覧を取得
	virtual const HelpMessages& GetHelpListReg() const = 0;
	// レジスタ表示系コマンドのヘルプメッセージを取得
	virtual const HelpMessages& GetHelpReg() const = 0;

	// 直近の命令でアクセスした laddr が一致したら true を返す
	// 1 命令で複数にアクセスする場合を考慮して、md 側で判断する
	virtual bool CheckLEA(uint32 laddr) = 0;

	// mem.laddr の位置の命令を一つ逆アセンブルする。
	virtual std::string Disassemble(DebuggerMemoryStream& mem) = 0;

	// 機種名を取得する
	const std::string& GetName() const { return name; }

	// この CPU のバスを返す
	IODevice *GetBus() const { return bus; }
	// このバスの論理アドレス幅 (ビット数) を返す
	int GetLASize() const { return lasize; }
	// このバスの物理アドレス幅 (ビット数) を返す
	int GetPASize() const { return pasize; }

	// 機種
	CPUArch arch {};

	// 最短の1命令のバイト数
	uint inst_bytes {};
	// 命令が固定長ならそのバイト数、可変長なら 0
	uint inst_bytes_fixed {};

	// 例外ブレーク。
	// 機種固有ではないが main と xp でそれぞれに必要なので、ここに持っておく。
	int bv_vector {};

 protected:
	Debugger *parent {};	// 親クラス

	std::string name {};	// 機種名

	IODevice *bus {};		// 対応する外部バス
	int lasize {};			// バスの論理アドレス幅
	int pasize {};			// バスの物理アドレス幅
};

#define NORM		"\x1b[0m"
#define BOLD		"\x1b[1m"
#define BOLDIF(n)	((n) ? BOLD : "")
