//
// nono
// Copyright (C) 2022 nono project
// Licensed under nono-license.txt
//

//
// ベクタテーブル (モニタ)
//

#include "vectortable.h"
#include "mainapp.h"
#include "mainbus.h"
#include "monitor.h"
#include "mpu.h"

// コンストラクタ
VectorTable::VectorTable(VMType vmtype)
	: inherited(OBJ_VECTOR_TABLE)
{
	// ログは不要
	ClearAlias();

	// 表示名を作成。それぞれ微妙に処理が違う…
	switch (vmtype) {
	 case VMType::X68030:
	 case VMType::LUNA1:
	 case VMType::NEWS:
	 case VMType::VIRT68K:
		InitTableM680x0(vmtype);
		break;
	 case VMType::LUNA88K:
		InitTableLuna88k();
		break;
	 default:
		PANIC("invalid vmtype %d", (int)vmtype);
	}

	monitor = gMonitorManager->Regist(ID_SUBWIN_VECTOR, this);
	monitor->func = ToMonitorCallback(&VectorTable::MonitorUpdate);
	monitor->SetSize(75, 1 + 32);	// ヘッダの1行と全エントリ数
	monitor->SetMaxHeight(1 + Size());
}

// m68k 機種の表示名テーブル初期化。コンストラクタの一部。
void
VectorTable::InitTableM680x0(VMType vmtype)
{
	nametable.clear();
	nametable.resize(256);

	// まず m680x0 標準の名称で埋める
	for (int v = 0; v < 64; v++) {
		if (name_m680x0[v]) {
			nametable[v] = name_m680x0[v];
		}
	}

	// 機種ごとに必要なところだけ上書き
	const std::map<int, const char * const> *names;
	switch (vmtype) {
	 case VMType::X68030:
		names = &name_x68030;
		break;
	 case VMType::LUNA1:
		names = &name_luna1;
		break;
	 case VMType::NEWS:
		names = &name_news;
		break;
	 case VMType::VIRT68K:
		names = &name_virt68k;
		break;
	 default:
		names = NULL;
		break;
	}
	if (names) {
		for (const auto& p : *names) {
			int v = p.first;
			const char *name = p.second;
			nametable[v] = name;
		}
	}
}

// LUNA-88K の表示名テーブルを初期化。コンストラクタの一部。
void
VectorTable::InitTableLuna88k()
{
	nametable.clear();
	nametable.resize(512);

	// 先頭から10個くらいだけ連続しているのでテーブルからコピー
	for (int i = 0; i < countof(name_luna88k); i++) {
		nametable[i] = name_luna88k[i];
	}

	// 以降は不連続で、間が空きすぎているので、ここで代入

	// SFU
	// (SFU2〜7 はいいか…)
	//                01234567890123456789012 <- 例外履歴欄の横幅
	nametable[114] = "SFU1 Precise Exception";
	nametable[115] = "SFU1 ImpreciseException";

	// NetBSD/luna88k
	nametable[128] = "NetBSD System Call";

	// OpenBSD
	// XXX 本来は OpenBSD 稼働時に限定すべきだろうけど、そうする意味もない
	//                01234567890123456789012
	nametable[450] = "OpenBSD System Call";
	nametable[451] = "OpenBSD Cache Flush";
	nametable[503] = "Division by Zero(GCC)";
	nametable[504] = "OpenBSD stepbpt";
	nametable[511] = "OpenBSD userbpt";
}

// デストラクタ
VectorTable::~VectorTable()
{
}

// 初期化
bool
VectorTable::Init()
{
	if (inherited::Init() == false) {
		return false;
	}

	mainbus = GetMainbusDevice();
	mpu = GetMPUDevice();

	return true;
}

// 例外名を返す (ベクタテーブル用)。
// 名前がなければ "" を返す。
const char *
VectorTable::GetTableName(int vector) const
{
	assertmsg(0 <= vector && vector < nametable.size(), "vector=%d", vector);

	return nametable[vector] ?: "";
}

// 例外名を返す (例外履歴用)。
// 名前がなければ NULL を返す。
const char *
VectorTable::GetExceptionName(int vector) const
{
	assertmsg(0 <= vector && vector < nametable.size(), "vector=%d", vector);

	if (__predict_false(vector < 2)) {
		if (gMainApp.Has(VMCap::M68K)) {
			// ベクタテーブル的には 0:"Reset(SP)"、1:"Reset(PC)" のほうが
			// 分かりやすいが、例外の名前は 0 が "Reset Exception" で
			// 1 は存在しない。
			if (vector == 0) {
				return "Reset Exception";
			} else {
				return "(vector 1?)";
			}
		}
	}

	if (__predict_true(nametable[vector] != NULL)) {
		return nametable[vector];
	} else {
		return NULL;
	}
}

// モニタ更新
void
VectorTable::MonitorUpdate(Monitor *, TextScreen& screen)
{
// 0         1         2         3         4         5         6         7
// 012345678901234567890123456789012345678901234567890123456789012345678901234
// No.      Offset Address   Name                  ><                    Count
// $02(  2) $0008  $12345678 0123456789012345678901201234567890123456789012345

	uint32 vbr = mpu->GetVBR();

	// userdata が表示開始位置
	int v = (int)screen.userdata;
	int row = screen.GetRow();
	int vend = v + row - 1;

	screen.Clear();

	// 最初の1行は常にヘッダ
	screen.Print(0, 0, "No.      Offset Address   Name");
	screen.Print(70, 0, "Count");

	for (int y = 1; v < vend; v++, y++) {
		screen.Print(0, y, "$%02x(%3u) $%04x", v, v, v * 4);
		busdata addr = mainbus->Peek4(vbr + v * 4);
		if (__predict_false(addr.IsBusErr())) {
			screen.Print(16, y, "BusErr");
		} else {
			screen.Print(16, y, "$%08x", addr.Data());
		}
		screen.Print(26, y, "%-23s", GetTableName(v));
		screen.Print(49, y, "%26s",
			format_number(mpu->excep_counter[v]).c_str());
	}
}

// ベクタ名 (MC680x0)。
// とりあえず CPU の区別はないことにする。
/*static*/ std::array<const char * const, 64> VectorTable::name_m680x0 = {
			//   0123456789012345678 <- 例外履歴欄の横幅
	/*  0 */	"Reset (SP)",
	/*  1 */	"Reset (PC)",
	/*  2 */	"Bus Error",
	/*  3 */	"Address Error",
	/*  4 */	"Illegal Instruction",
	/*  5 */	"Zero Divide",
	/*  6 */	"CHK/CHK2 Insn",
	/*  7 */	"TRAPV/*TRAPcc Insn",
	/*  8 */	"Privilege Violation",
	/*  9 */	"Trace",
	/* 10 */	"Line A emulator",
	/* 11 */	"Line F emulator",
	/* 12 */	NULL,
	/* 13 */	"CoproProtoViolation",	// 68030 only
	/* 14 */	"Format Error",
	/* 15 */	"Uninitialized Intr",
	/* 16 */	NULL, NULL, NULL, NULL,
	/* 20 */	NULL, NULL, NULL, NULL,
	/* 24 */	"Spurious Interrupt",
	/* 25 */	"Level 1 Interrupt",
	/* 26 */	"Level 2 Interrupt",
	/* 27 */	"Level 3 Interrupt",
	/* 28 */	"Level 4 Interrupt",
	/* 29 */	"Level 5 Interrupt",
	/* 30 */	"Level 6 Interrupt",
	/* 31 */	"Level 7 Interrupt",
	/* 32 */	"Trap #0",
	/* 33 */	"Trap #1",
	/* 34 */	"Trap #2",
	/* 35 */	"Trap #3",
	/* 36 */	"Trap #4",
	/* 37 */	"Trap #5",
	/* 38 */	"Trap #6",
	/* 39 */	"Trap #7",
	/* 40 */	"Trap #8",
	/* 41 */	"Trap #9",
	/* 42 */	"Trap #10",
	/* 43 */	"Trap #11",
	/* 44 */	"Trap #12",
	/* 45 */	"Trap #13",
	/* 46 */	"Trap #14",
	/* 47 */	"Trap #15",
			//   0123456789012345678
	/* 48 */	"FPCP Branch",
	/* 49 */	"FPCP InexcactResult",
	/* 50 */	"FPCP Divide by Zero",
	/* 51 */	"FPCP UnderFlow",
	/* 52 */	"FPCP Operand Error",
	/* 53 */	"FPCP OverFlow",
	/* 54 */	"FPCP Signaling NAN",
	/* 55 */	"FPCP UnsuppDataType",	// 68040 only
	/* 56 */	"MMU Config Error",		// 68030 only
	/* 57 */	NULL,
	/* 58 */	NULL,
	/* 59 */	NULL,
	/* 60 */	NULL,
	/* 61 */	NULL,
	/* 62 */	NULL,
	/* 63 */	NULL,
};

// ベクタ名 (X68030)
/*static*/ std::map<int, const char * const> VectorTable::name_x68030 = {
	//           0123456789012345678
	{ 0x40,		"MFP Alarm" },
	{ 0x41,		"MFP EXPON" },
	{ 0x42,		"MFP POWSW" },
	{ 0x43,		"MFP FM IRQ" },
	{ 0x44,		"MFP Timer-D" },
	{ 0x45,		"MFP Timer-C" },
	{ 0x46,		"MFP V-Disp" },
	{ 0x47,		"MFP GPIP5" },
	{ 0x48,		"MFP Timer-B" },
	{ 0x49,		"MFP TX Error" },
	{ 0x4a,		"MFP TX Empty" },
	{ 0x4b,		"MFP RX Error" },
	{ 0x4c,		"MFP RX Full" },
	{ 0x4d,		"MFP Timer-A" },
	{ 0x4e,		"MFP CRTC IRQ" },
	{ 0x4f,		"MFP H-Sync" },

	{ 0x50,		"SCC#B TX Empty" },
	{ 0x52,		"SCC#B E/S" },
	{ 0x54,		"SCC#B RX Intr" },
	{ 0x56,		"SCC#B SpCond" },
	{ 0x58,		"SCC#A TX Empty" },
	{ 0x5a,		"SCC#A E/S" },
	{ 0x5c,		"SCC#A RX Intr" },
	{ 0x5e,		"SCC#A SpCond" },

	//           0123456789012345678
	{ 0x60,		"I/O FDC Intr" },
	{ 0x61,		"I/O FDD Intr" },
	{ 0x62,		"I/O HDC Intr" },
	{ 0x63,		"I/O PRN Intr" },
	{ 0x64,		"DMAC#0 Complete" },
	{ 0x65,		"DMAC#0 Error" },
	{ 0x66,		"DMAC#1 Complete" },
	{ 0x67,		"DMAC#1 Error" },
	{ 0x68,		"DMAC#2 Complete" },
	{ 0x69,		"DMAC#2 Error" },
	{ 0x6a,		"DMAC#3 Complete" },
	{ 0x6b,		"DMAC#3 Error" },
	{ 0x6c,		"SPC Interrupt" },

	//           0123456789012345678
	{ 0xf0,		"PSX16x50 Interrupt" },
	{ 0xf6,		"ExSPC Interrupt" },
	//0xf7,		TS-6BSI
	{ 0xf8,		"Nereid#1 Interrupt" },
	{ 0xf9,		"Neptune-X Interrupt" },
	{ 0xfa,		"Nereid#1 USB" },
	{ 0xfb,		"Nereid#0 USB" },
};

// LUNA-I
/*static*/ std::map<int, const char * const> VectorTable::name_luna1 = {
	//      	 0123456789012345678 <- 例外履歴欄の横幅
	{ 0x19,		"Lv1 Intr (XP Low)" },
	{ 0x1a,		"Lv2 Intr (SPC)" },
	{ 0x1b,		"Lv3 Intr (Lance)" },
	{ 0x1c,		"Lv4 Intr (XP High)" },
	{ 0x1d,		"Lv5 Intr (SysClk)" },
	{ 0x1e,		"Lv6 Intr (SIO)" },
	{ 0x1f,		"Lv7 Intr (NMI)" },
};

// NEWS はレベル割り込みと、一部が vectored */
/*static*/ std::map<int, const char * const> VectorTable::name_news = {
	//      	 0123456789012345678 <- 例外履歴欄の横幅
	{ 0x19,		"Lv1 Intr (AST)" },
	{ 0x1a,		"Lv2 Intr" },
	{ 0x1b,		"Lv3 Intr" },
	{ 0x1c,		"Lv4 Intr (SIC/LAN)" },
	{ 0x1d,		"Lv5 Intr (SCC)" },
	{ 0x1e,		"Lv6 Intr (SysTimer)" },
	{ 0x1f,		"Lv7 Intr" },

	{ 0x40,		"SCC" },
};

// virt-m68k
/*static*/ std::map<int, const char * const> VectorTable::name_virt68k = {
	//      	 0123456789012345678 <- 例外履歴欄の横幅
	{ 0x19,		"Lv1 Intr (GFPIC1)" },
	{ 0x1a,		"Lv2 Intr (GFPIC2)" },
	{ 0x1b,		"Lv3 Intr (GFPIC3)" },
	{ 0x1c,		"Lv4 Intr (GFPIC4)" },
	{ 0x1d,		"Lv5 Intr (GFPIC5)" },
	{ 0x1e,		"Lv6 Intr (GFPIC6)" },
	{ 0x1f,		"Lv7 Intr" },
};

// LUNA-88K は先頭11個のみ (残りはコードで処理)
/*static*/ std::array<const char * const, 11> VectorTable::name_luna88k = {
	//           01234567890123456789012 <- 例外履歴欄の横幅
	/*  0 */	"Reset Exception",
	/*  1 */	"Interrupt Exception",
	/*  2 */	"Inst Access Exception",
	/*  3 */	"Data Access Exception",
	/*  4 */	"Misaligned Access Excep",
	/*  5 */	"Unimplemented Opcode",
	/*  6 */	"Priv. Violation Excep.",
	/*  7 */	"Bounds Check Violation",
	/*  8 */	"Illegal Integer Divide",
	/*  9 */	"Int Overflow Exception",
	/* 10 */	"Error Exception",
	//           01234567890123456789012
};
