//
// nono
// Copyright (C) 2021 nono project
// Licensed under nono-license.txt
//

//
// シリアルポートの標準入出力ドライバ
//

#include "comdriver_stdio.h"
#include "hostcom.h"

// コンストラクタ
COMDriverStdio::COMDriverStdio(HostDevice *hostdev_)
	: inherited(hostdev_, "Stdio")
{
	// 現在の端末属性を取得
	tcgetattr(STDIN_FILENO, &oldtc);
}

// デストラクタ
COMDriverStdio::~COMDriverStdio()
{
	// 端末属性を元に戻す
	tcsetattr(STDIN_FILENO, TCSANOW, &oldtc);
}

// ドライバ初期化
bool
COMDriverStdio::InitDriver()
{
	struct termios tc;

	// 端末の設定を変更
	tc = oldtc;

	// 入力時の変換は無し
	tc.c_iflag = 0;

	tc.c_lflag &= ~ECHO;	// エコーオフ
	tc.c_lflag &= ~ICANON;	// 非カノニカルモード
	tc.c_lflag &= ~IEXTEN;	// DISCARD(^O)と LNEXT をオフ?
	tc.c_lflag &= ~ISIG;	// シグナルを生成しない
	tc.c_cc[VTIME] = 0;
	tc.c_cc[VMIN] = 1;

	if (tcsetattr(STDIN_FILENO, TCSANOW, &tc) < 0) {
		putmsg(0, "tcsetattr: %s", strerror(errno));
		return false;
	}

	if (hostdev->AddOuter(STDIN_FILENO) < 0) {
		putmsg(0, "AddOuter: %s", strerror(errno));
		return false;
	}

	return true;
}

// 外部への1バイト書き出し
void
COMDriverStdio::Write(uint32 data)
{
	uint8 buf[1];
	int r;

	buf[0] = data;
	r = write(STDOUT_FILENO, buf, sizeof(buf));
	if (r < 0) {
		putmsg(0, "write: %s", strerror(errno));
		return;
	}
	if (r < sizeof(buf)) {
		putmsg(0, "write: short");
		return;
	}
}

// 外部からの1バイト読み込み
int
COMDriverStdio::Read()
{
	uint8 buf[1];
	int r;

	r = read(STDIN_FILENO, buf, sizeof(buf));
	if (r < 0) {
		putmsg(0, "read: %s", strerror(errno));
		return NODATA;
	}
	if (r == 0) {
		return NODATA;
	}
	return buf[0];
}
