//
// nono
// Copyright (C) 2021 nono project
// Licensed under nono-license.txt
//

//
// シリアルポートのホストデバイス
//

// 送信のフロー
//
// VM thread                :     Host thread
//
// 各デバイス::Tx()         :
//     |
// HostCOMDevice::Tx()      :   HostDevice::ThreadRun
//     |
//     +-------------->| queue |----+         … 送信キューに追加し
//     +-------------->| pipe  |----+         … パイプでホストスレッドに通知
//     |                            |
//   <-+                    :       v
//                                 --- kevent
//                          :       |
//                              HostCOMDevice::Write()
//                          :       |
//                              COMDriver*::Write()
//                          :       |
//                                  +-----> Send to the real world

// 受信のフロー
//
// VM thread                :     Host thread
//
//                          :   HostDevice::ThreadRun
//
//                          :       +-----< Recv from the real world
//                                  |
//                          :       v
//                                 --- kevent
//                          :       |
//                              HostCOMDevice::Read()
//                                  |
//                     | queue |<---+         … 受信キューに追加
//                         ‖
//                         ‖   HostCOMDevice::*(rx_func)()
//                                  |
//     +-------------| Message |<---+         … メッセージで VM スレッドに通知
//     v
// 各デバイス::RxMessage() ‖                 … メッセージコールバック
//     |                   ‖
// 各デバイス::Rx()        ‖                 … イベントコールバック
//     |                   ‖
// HostCOMDevice::Rx()  <==++                 … ここで queue から読み出す

// ログ名
//        MPSCC        HostDevice       COMDriver
//         |            |                |
//         v            v                v
//        SIODevice -> HostCOMDevice -> COMDriver***
// 表示名 (sio)        (HostCOMx)      (HostCOMx.***)
// 識別名 "sio"        "hostcomx"      "hostcomx"
//
// HostCOM は 1VM あたり 1個とは限らないので、親デバイスがそれぞれ対応する
// HostCOM に名前を付ける。


#include "hostcom.h"
#include "comdriver_cons.h"
#include "comdriver_none.h"
#include "comdriver_stdio.h"
#include "comdriver_tcp.h"
#include "config.h"
#include "mainapp.h"
#include "monitor.h"

// コンストラクタ
//
// objname_ は "HostCOM" とか (スレッド名も同じになる)。
// 設定ファイルキーワードプレフィックスはこれを小文字にしたもの("hostcom")。
HostCOMDevice::HostCOMDevice(Device *parent_, const std::string& objname_)
	: inherited(parent_, OBJ_NONE)
{
	SetName(objname_);

	// 親が hostcom* か debugger かによっていろいろ動作が違う。
	if (GetName() == "Debugger") {
		// デバッガコンソールならログレベルは親に準じるため、こっちは不要。
		ClearAlias();

		// 送受信バイト数がメインのモニタはこっちにはなくていいだろう。
	} else {
		// hostcom* なら親(mpscc などのシリアルデバイス)とこっち(HostCOM*)
		// のログレベルは独立。

		// モニタは hostcom* のみ必要。
		monitor = gMonitorManager->Regist(ID_MONITOR_HOSTCOM, this);
		monitor->func = ToMonitorCallback(&HostCOMDevice::MonitorUpdate);
		monitor->SetSize(30, 17);
	}
}

// デストラクタ
HostCOMDevice::~HostCOMDevice()
{
}

// ログレベル設定
void
HostCOMDevice::SetLogLevel(int loglevel_)
{
	inherited::SetLogLevel(loglevel_);

	if ((bool)driver) {
		driver->SetLogLevel(loglevel_);
	}
}

// 動的コンストラクションその2
bool
HostCOMDevice::Create2()
{
	if (inherited::Create2() == false) {
		return false;
	}

	if (SelectDriver() == false) {
		return false;
	}

	return true;
}

// ドライバ(再)選択
bool
HostCOMDevice::SelectDriver()
{
	// 設定のキー名は親デバイスによって異なる
	std::string key = GetConfigKey();
	const ConfigItem& item = gConfig->Find(key + "-driver");
	const std::string& type = item.AsString();

	driver.reset();

	if (type != "none") {
		if (type == "stdio") {
			// debugger-driver=stdio (または -D オプション) と
			// hostcom*-driver=stdio とは共存できない。
			// 歴史的経緯でとりあえず debugger-driver=stdio 側を優先とする。
			// XXX これも後指定優先にするかどうか
			// XXX hostcom が増えたらたぶんこうじゃなくなる
			if (key != "debugger") {
				const ConfigItem& dditem = gConfig->Find("debugger-driver");
				const std::string& ddtype = dditem.AsString();
				if (ddtype == "stdio") {
					auto where = dditem.GetWhere();
					if (where == "-V") {
						where += " debugger-driver=stdio";
					}
					item.Err("stdio driver conflicts with " + where);
					return false;
				}
			}

			try {
				driver.reset(new COMDriverStdio(this));
			} catch (...) { }
			if ((bool)driver && driver->InitDriver() == false) {
				driver.reset();
			}

		} else if (type == "tcp") {
			try {
				driver.reset(new COMDriverTCP(this));
			} catch (...) { }
			if ((bool)driver && driver->InitDriver() == false) {
				driver.reset();
			}

		} else if (type == "cons") {
			// console デバイスのいない機種では指定出来ない。
			if (gMainApp.FindObject(OBJ_CONSOLE) == NULL) {
				item.Err("cannot be specified on vmtype=%s",
					gMainApp.GetVMTypeStr().c_str());
				return false;
			}

			try {
				driver.reset(new COMDriverCons(this));
			} catch (...) { }
			if ((bool)driver && driver->InitDriver() == false) {
				driver.reset();
			}

		} else {
			// 知らないドライバ種別
			item.Err();
			return false;
		}

		if ((bool)driver == false) {
			bool fallback = false;

			// 指定のドライバが使えなかった場合、
			// hostcom*-driver なら fallback するかどうかは設定による。
			// debugger-driver なら常に fallback せず終了する。
			//
			// 元々 hostnet が tap, bpf のように順に試してという流れだったので
			// fallback の選択肢が用意されていたが、hostcom は今の所どれか1つ
			// (stdio か tcp) か、そうでなければ none しかないので、fallback
			// するかどうかを選ばせる意味があまりような気もするけど、
			// とりあえず形式だけ真似してみる…。
			// 一方、debugger-driver なら常に fallback せず、設定自体も用意
			// しない。デバッガを使いたいと言ってる時点でデバッガがなくても
			// とりあえず起動してほしいとはならないだろうと思うので。
			if (key != "debugger") {
				fallback = gConfig->Find(key + "-fallback").AsInt();
				putmsg(1, "%s-fallback=%d", key.c_str(), fallback ? 1 : 0);
			}

			if (fallback == false) {
				// フォールバックしないならエラー終了
				std::string errmsg;
				errmsg = string_format("No %s driver found", key.c_str());
				putmsg(1, "%s", errmsg.c_str());
				warnx("%s (See details with option -C -L%s=1)",
					errmsg.c_str(), key.c_str());
				return false;
			}
		}
	}

	if ((bool)driver == false) {
		try {
			driver.reset(new COMDriverNone(this));
		} catch (...) { }
		if ((bool)driver && driver->InitDriver() == false) {
			// 失敗したら出来ることはあまりなさげ
			assert(false);
		}
	}
	assert((bool)driver);

	// ドライバ名は Capitalize だがログはほぼ小文字なので雰囲気を揃える…
	putmsg(1, "selected host driver: %s",
		string_tolower(driver->GetDriverName()).c_str());

	return true;
}

void
HostCOMDevice::Dispatch(int udata)
{
	// driver の Dispatch は処理し終わったら DONE を返す
	udata = driver->Dispatch(udata);

	// ...のだが、Dispatch(LISTEN_SOCKET) は着信を受け付けた時には
	// LISTEN_SOCKET を返し、それを受けてここで通知処理を行う。
	if (udata == LISTEN_SOCKET) {
		if (accept_func) {
			(parent->*accept_func)();
		}
		udata = DONE;
	}

	inherited::Dispatch(udata);
}

// VM からの送信 (VM スレッドで呼ばれる)
bool
HostCOMDevice::Tx(uint32 data)
{
	if (loglevel >= 2) {
		char buf[8];

		buf[0] = '\0';
		if (0x20 <= data && data < 0x7f) {
			snprintf(buf, sizeof(buf), " '%c'", data);
		}
		putlog(2, "Send $%02x%s", data, buf);
	}

	// 送信キューに入れて..
	if (txq.Enqueue(data) == false) {
		putlog(2, "txq exhausted");
		stat.txqfull_bytes++;
		return false;
	}
	stat.tx_bytes++;

	// ざっくりピーク値
	stat.txq_peak = std::max((uint)txq.Length(), stat.txq_peak);

	// パイプに通知 (値はダミー)
	return WritePipe(0);
}

// キューから取り出す (VM スレッドで呼ばれる)
uint32
HostCOMDevice::Rx()
{
	uint8 data;

	if (rxq.Dequeue(&data) == false) {
		// キューが空
		return -1;
	}
	stat.rx_bytes++;

	if (loglevel >= 2) {
		char buf[8];

		buf[0] = '\0';
		if (0x20 <= data && data < 0x7f) {
			snprintf(buf, sizeof(buf), " '%c'", data);
		}
		putlog(2, "Recv $%02x%s", data, buf);
	}

	return data;
}

// ドライバから読み込む。
// 戻り値はキューに投入したデータ数。
int
HostCOMDevice::Read()
{
	assert((bool)driver);

	int data = driver->Read();
	if (data < 0) {
		return 0;
	}
	stat.read_bytes++;

	if (rxq.Enqueue(data) == false) {
		stat.rxqfull_bytes++;
		return 0;
	}

	// ざっくりピーク値
	stat.rxq_peak = std::max((uint)rxq.Length(), stat.rxq_peak);

	return 1;
}

// 外部への書き出し (ホストスレッドで呼ばれる)
void
HostCOMDevice::Write(uint32 dummy)
{
	uint8 data;

	assert(driver);

	// 送信キューを全部吐き出す
	while (txq.Dequeue(&data)) {
		driver->Write(data);
		stat.write_bytes++;
	}
}

// モニタ
void
HostCOMDevice::MonitorUpdate(Monitor *, TextScreen& screen)
{
	screen.Clear();

	screen.Print(0, 0, "Parent Device : %s", parent->GetName().c_str());
	screen.Print(0, 1, "HostCOM Driver: %s", driver->GetDriverName());
	// 次の1行はドライバ依存情報
	driver->MonitorUpdateMD(screen, 2);

	int y = 4;
	screen.Print(0, y++, "%-17s%13s", "<Tx>", "Bytes");
	screen.Print(0, y++, "%-17s%13s", "VM sends",
		format_number(stat.tx_bytes).c_str());
	screen.Print(0, y++, "%-17s%13s", "Write to host",
		format_number(stat.write_bytes).c_str());
	screen.Print(0, y++, "%-17s%13s", "Drop:TxQ Full",
		format_number(stat.txqfull_bytes).c_str());
	y++;

	screen.Print(0, y++, "%-17s%13s", "<Rx>", "Bytes");
	screen.Print(0, y++, "%-17s%13s", "Read from host",
		format_number(stat.read_bytes).c_str());
	screen.Print(0, y++, "%-17s%13s", "VM receives",
		format_number(stat.rx_bytes).c_str());
	screen.Print(0, y++, "%-17s%13s", "Drop:RxQ Full",
		format_number(stat.rxqfull_bytes).c_str());
	y++;

	screen.Print(5, y++, "Capacity Peak");
	screen.Print(0, y++, "TxQ %4u/%4u %4u",
		(uint)txq.Length(), (uint)txq.Capacity(), stat.txq_peak);
	screen.Print(0, y++, "RxQ %4u/%4u %4u",
		(uint)rxq.Length(), (uint)rxq.Capacity(), stat.rxq_peak);
}
