//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// 文字列操作
//

#include "mystring.h"
#include <algorithm>

std::string
string_format(const char *fmt, ...)
{
	va_list ap;
	char *buf;

	va_start(ap, fmt);
	vasprintf(&buf, fmt, ap);
	va_end(ap);
	std::string rv(buf);
	free(buf);

	return rv;
}

// 文字列 str から先頭の連続する空白文字を取り除いた新しい文字列を返す。
std::string
string_ltrim(const std::string& str)
{
	auto it = str.begin();
	for (; it != str.end(); it++) {
		if (!isspace(*it))
			break;
	}
	return std::string(it, str.end());
}

// 文字列 str から末尾の連続する空白文字を取り除く (str を書き換える)。
void
string_rtrim(std::string& str)
{
	while (isspace(*str.rbegin())) {
		str.pop_back();
	}
}

// 文字列 str から末尾の連続する空白文字を取り除く (str を書き換える)。
void
rtrim(char *str)
{
	char *p = strchr(str, '\0');
	while (--p >= str && isspace((int)*p)) {
		*p = '\0';
	}
}

// 文字列 str から先頭と末尾の連続する空白文字を取り除いた新しい文字列を返す。
std::string
string_trim(const std::string& str)
{
	int s = 0;
	int e = str.size();

	for (; s < e; s++) {
		if (!isspace(str[s]))
			break;
	}
	for (e--; e >= s; e--) {
		if (!isspace(str[e]))
			break;
	}

	return str.substr(s, e - s + 1);
}

// 文字列 src 中の ASCII 大文字を小文字にした新しい文字列を返す。
std::string
string_tolower(const std::string& src)
{
	std::string dst(src);
	std::transform(dst.begin(), dst.end(), dst.begin(),
		[](unsigned char c){ return std::tolower(c); });
	return dst;
}

// 文字列 src 中の ASCII 小文字を大文字にした新しい文字列を返す。
std::string
string_toupper(const std::string& src)
{
	std::string dst(src);
	std::transform(dst.begin(), dst.end(), dst.begin(),
		[](unsigned char c){ return std::toupper(c); });
	return dst;
}

// 文字列 lhs の先頭が rhs と大文字小文字の区別なしで一致すれば true を返す。
// ASCII 専用。
// C++20 の starts_with に似せておく。
bool
starts_with_ignorecase(const std::string& lhs, const std::string& rhs)
{
	if (lhs.length() < rhs.length()) {
		return false;
	}
#if 0
	return std::equal(
		lhs.begin(), lhs.begin() + rhs.length(),
		rhs.begin(),
		[](std::string::value_type l, std::string::value_type r) {
			return std::tolower(l) == std::tolower(r);
		}
	);
#else
	// こっちのほうが分かりやすいよな
	return strncasecmp(lhs.c_str(), rhs.c_str(), rhs.length()) == 0;
#endif
}

// 文字列 str (長さ len) を文字 c で分割したリストを返す。
std::vector<std::string>
string_split(const char *str, int len, char c, int nlimit)
{
	std::vector<std::string> list;

	// 空文字列なら空リスト
	if (len == 0) {
		return list;
	}

	int pos = 0;
	int end = 0;

	for (; ; pos = end + 1) {
		// 上限に達するならこれ以降は一要素として返す
		if (nlimit > 0 && list.size() >= nlimit - 1) {
			list.emplace_back(str + pos, len - pos);
			break;
		}

		const char *p = strchr(str + pos, c);
		if (p) {
			end = p - str;
		} else {
			end = len;
		}

		list.emplace_back(str + pos, end - pos);
		if (p == NULL)
			break;
	}

	return list;
}

// val を3桁ずつカンマ区切りした文字列にして返す。最大は 26桁。
// ex) 123   -> "138"
//     12345 -> "12,345"
std::string
format_number(uint64 val)
{
	//                 1   2   3   4   5   6
	// UINT64_MAX = 18,446,744,073,709,551,615
	char part[6][8];
	char buf[32];
	int n;

	n = 0;
	memset(&part, 0, sizeof(part));
	while (val >= 1000) {
		uint32 r = val % 1000;
		val /= 1000;

		snprintf(part[n], sizeof(part[n]), ",%03u", r);
		n++;
	}
	// この時点で
	// part[0] = ",615";
	// part[1] = ",551";
	// :
	// part[5] = ",446";

	// 先頭(val は 1000未満)
	snprintf(buf, sizeof(buf), "%u", (uint32)val);

	// part を連結
	while (--n >= 0) {
		strlcat(buf, part[n], sizeof(buf));
	}
	return std::string(buf);
}

// value を width 桁の16進数文字列にして返す。"%0{width}x" みたいな感じ。
// strhex(0x12345678, 4) -> "5678"
// strhex(0x00000001, 3) -> "001"
std::string
strhex(uint32 value, int width)
{
	std::string s;

	for (width -= 1; width >= 0; width--) {
		uint32 d = (value >> (width * 4)) & 0x0f;
		if (__predict_true(d < 10)) {
			s += '0' + d;
		} else {
			s += 'a' + d - 10;
		}
	}
	return s;
}


#if defined(SELFTEST)
#include <cstdio>
#include "stopwatch.h"
std::string s;
int main()
{
	Stopwatch sw;
	sw.Start();
	for (uint64 i = 0; i < 10000000; i += 3) {
		s = format_number(i);
		if (s.empty())
			return 0;
	}
	sw.Stop();
	uint64 t = sw.Elapsed();
	printf("%ld.%03ld msec\n", t / 1000 / 1000, (t / 1000) % 1000);
	return 0;
}
#endif
