//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

#include "m88100.h"
#include "debugger.h"
#include "mpu88xx0.h"
#include "scheduler.h"

#define OP_DEF(name)	void __CONCAT(MPU88xx0Device::op_,name)()
#define OP_FUNC(name)	__CONCAT(op_,name)()
#include "m88100ops.cpp"

// プリフェッチステージ(?)。
// 命令サイクルは、自然に書けば、
// (1) パイプラインをシフトし、FIP の指すところをフェッチ。
// (2) デバッガはここに入れたい。
// (3) XIP(opX) を解釈、実行。
// の繰り返しになるが、デバッガは Exec() から戻ったところでスケジューラから
// 呼び出される構造のため、順序をずらして 1回の Exec() では (3)->(1) を実行
// してからスケジューラに戻って (2) を実行する、の順にする。
// そのため、リセット例外では1回プリフェッチを実行しておく必要がある。
void
MPU88xx0Device::Prefetch()
{
	// shift pipeline
	reg.opX = reg.opF;
	reg.xip = reg.nip;

	// prefetch next inst
	fetch();
}

// 1命令実行のコールバック。
void
MPU88xx0Device::ExecNormal(Event& ev)
{
	uint64 cycle_start = used_cycle;
	uint32 op12;

	buswait = 0;

	// この時点で実行しようとする命令が opX に入っている

	if (OpIsBusErr(reg.opX)) {
		// プリフェッチしていた命令がバスエラー
		Exception(EXCEP_INST);
		goto exit;
	}

	op12 = op32_to_12(reg.opX);
	switch (op12) {
#include "m88100switch.inc"
	 default:
		OP_FUNC(illegal);
		break;
	}

	if (__predict_false(intr_pending != 0)) {
		if (resetting) {
			// 命令実行の先でリセットが起きたら以降を全部キャンセルして帰る。
			// システムコントローラのリセットポートで起きる。
			// リセットは実際には intr_pending ではないが、
			// ここでフラグ処理をまとめるために立ててある。
			return;
		}
		if (IsIntrEnable()) {
			// 割り込み処理要求が来ていて、
			// 直前の命令が割り込みを禁止していなければ割り込み。
			// XXX 実際は違う気がする
			putlog(4, "INTR take");
			ExceptionCore(EXCEP_INTERRUPT, ExceptionKind::INTR);
			goto exit;
		}
	}

	used_cycle++;

 exit:
	Prefetch();

	int cycle = used_cycle - cycle_start;
	ev.time = cycle * clock_nsec + buswait;
	scheduler->StartEvent(ev);
}

// 通常の例外。
// DataAccessException, エラー、割り込み、TRAP からは使わない。
// vec はベクタ番号。
void
MPU88xx0Device::Exception(int vec)
{
	ExceptionCore(vec, ExceptionKind::NORMAL);
}

// 例外処理コア部分
// vec はベクタ番号。
void
MPU88xx0Device::ExceptionCore(int vec, ExceptionKind cause)
{
	uint syscall = 0;

	// 例外履歴に記録 (例外発生はブランチ履歴にも記録)。
	// ベクタ 450 なら OpenBSD システムコール番号もついでに記録。
	// システムコール番号は tb0 発行時点で r13 にセットされている。
	// ただし 0 番は syscall(2) (間接システムコール) で、実際の番号は r2。
	// 198番 __syscall(2) だと r2:r3 (なので実質 r3)。
	// 非公式 NetBSD/luna88k はベクタ 128 を使っているようだ。
	excep_counter[vec]++;
	if (vec == 128 || vec == 450) {
		syscall = reg.r[13];
		if (syscall == 0) {
			syscall = reg.r[2];
		} else if (syscall == 198) {
			syscall = reg.r[3];
		}
		syscall &= 0xfff;
	}
	uint32 from = reg.xip | (IsSuper() ? 1 : 0);
	uint32 info = 0xfc000000 | (syscall << 12) | vec;
	exhist.AddEntry(from, 0, info);
	brhist.AddEntry(from, 0, info);
	// デバッガに通知 (例外ブレーク用)
	debugger->NotifyExceptionMain(vec);

	// 命令中で起きる例外はとりあえず全部2クロック追加? (Table.7-5)
	if (cause != ExceptionKind::INTR) {
		AddCycle(2);
	}

	if ((reg.psr & PSR_SFRZ)) {
		if (cause != ExceptionKind::TRAP) {
			putlog(2, "SFRZ Error Exception");
			vec = EXCEP_ERROR;
			cause = ExceptionKind::ERROR;
		}
	} else {
		putlog(2, "Exception %x", vec);
		if (vec != 1) {
			putlog(1, "Exception %x x=%x n=%x f=%x",
				vec, reg.xip, reg.nip, reg.fip);
		}
		reg.sxip = reg.xip;
		reg.snip = reg.nip;
		reg.sfip = reg.fip;

		reg.sxip |= SIP_V;
		reg.snip |= SIP_V;
		reg.sfip |= SIP_V;

		if (OpIsBusErr(reg.opX)) {
			reg.sxip |= SIP_E;	// E bit
		}
		if (OpIsBusErr(reg.opF)) {
			reg.snip |= SIP_E;	// E bit
		}

		if (vec == EXCEP_UNIMPL_OP) {
			reg.sxip &= ~SIP_V;
		}
	}

	if (cause != ExceptionKind::DATA) {
		// データアクセス例外以外は、
		// データパイプラインはフラッシュ状態とする。
		// ほんとうは違う。かも。
		reg.dmt0 = 0;
	} else {
		// xip: ld/st/xmem
		// nip: next
		// fip: fetch

		// PROM 1.20 は DAE が起きると、

		// ROM挙動               CPU の RTE 挙動
		// sfip = sxip
		// sxip -> ~V (INVALID)  CPU はそもそも sxip を評価しない
		// snip = 0 (INVALID)    CPU はこれを nop 相当に処理する
		// rte

		// してくる。そのまま読むと命令を再実行しようとしているように
		// 読めるが、再実行すると無限ループになってしまう。
		// DAE の時点で xip が進んでいることを期待して利用している?

		// openbsd は
		// xmem の write バスエラーを検知すると命令を再実行するために
		// fip = nip, nip = xip して rte する。ということは xip = xmem に
		// なっていることを期待している。

		// というわけで
		// ld で DataAccessException のときは、実CPUはすでに次の命令を
		// 実行している?
		// もう少し考えると、スコアボードでレジスタの干渉がなければ、
		// 実行している?

		// 仕方がないのでハックでごまかす。
		// rte のほうも参照のこと。

		if (IsSuper() && reg.xip >= 0x41000000 && reg.xip < 0x42000000) {
			// 遅延例外機構がないので、ROM 合わせのハック。
			reg.sxip = reg.snip;
		}
	}

	reg.epsr = reg.psr;
	SetPSR(reg.psr | PSR_SUPER | PSR_SFD1 | PSR_IND | PSR_SFRZ);

	reg.fip = reg.vbr + (vec << 3);
	fetch();
	// ここでベクタに飛ぶのだが、ベクタ内の2命令でおそらく必ずもう一度
	// 分岐してそっちでも履歴が残るので、ここでブランチ履歴を残すのは
	// ちょっと冗長という気もする。

	if (OpIsBusErr(reg.opF)) {
		if (cause == ExceptionKind::ERROR) {
			// double bus fault
			// 実機は無限ループ状態
			PANIC("Infinite ERROR (double bus fault)");
		} else {
			ExceptionCore(EXCEP_ERROR, ExceptionKind::ERROR);
		}
	}
}

inline uint32
MPU88xx0Device::Calcdmt(uint32 flag)
{
	uint32 dmt;

	dmt = DM_VALID;
	if ((reg.psr & PSR_BO_LE))
		dmt |= DM_BO;
	if (IsSuper() && !(flag & DM_USR))
		dmt |= DM_DAS;

	return dmt;
}

// データアクセス例外(Read)
// ld.b, ld.h, ld.hu, ld 用
void
MPU88xx0Device::ReadDataException32(uint32 addr, uint32 flag)
{
	// ミスアラインドはこの例外より前にチェックされていて、
	// ここにはアラインドしかこない。
	reg.dma0 = addr & ~0x3;
	reg.dmt0  = Calcdmt(flag);
	reg.dmt0 |= FLD_D << 7;
	reg.dmt0 |= flag & DM_SIGNED;
	reg.dmt0 |= (flag & DM_EN_MASK) >> (addr & 0x03);

	reg.dmt1 &= ~DM_VALID;
	reg.dmt2 &= ~DM_VALID;
	putlog(1, "ReadDAE xip=%x addr=%x", reg.xip, addr);
	ExceptionCore(EXCEP_DATA, ExceptionKind::DATA);
}

// データアクセス例外(Read)
// ld.d 用
void
MPU88xx0Device::ReadDataException64(uint32 addr, uint32 flag)
{
	// ミスアラインドはこの例外より前にチェックされていて、
	// ここにはアラインドしかこない。
	reg.dma0 = addr;
	reg.dmt0  = Calcdmt(flag);
	reg.dmt0 |= DM_DOUB1;		// ダブルワードの第1アクセスのときセット
	reg.dmt0 |= FLD_D << 7;
	reg.dmt0 |= DM_EN_MASK;		// 32ビット有効

	reg.dma1 = reg.dma0 + 4;
	reg.dmt1  = Calcdmt(flag);
	reg.dmt1 |= FLD_D2 << 7;
	reg.dmt1 |= DM_EN_MASK;		// 32ビット有効

	reg.dmt2 &= ~DM_VALID;

	if ((flag & DM_DOUB1) == 0) {
		// .d の2ワード目がバスエラー。第1ワードのアクセスは成功した。
		// DMx0 が例外を起こしたアクセスを示す。
		// Manual 6.7.3
		reg.dma0 = reg.dma1;
		reg.dmt0 = reg.dmt1;
		reg.dmt1 &= ~DM_VALID;
	}

	putlog(1, "ReadDAE64 xip=%x addr=%x", reg.xip, addr);
	ExceptionCore(EXCEP_DATA, ExceptionKind::DATA);
}

// データアクセス例外(Write)
// st.b, st.h, st.hu, st 用
void
MPU88xx0Device::WriteDataException32(uint32 addr, uint32 flag)
{
	// ミスアラインドはこの例外より前にチェックされていて、
	// ここにはアラインドしかこない。
	reg.dma0 = addr & ~0x3;
	reg.dmt0  = Calcdmt(flag);
	reg.dmt0 |= FLD_D << 7;
	reg.dmt0 |= (flag & DM_EN_MASK) >> (addr & 0x03);
	reg.dmt0 |= DM_WRITE;
	reg.dmd0 = rD;

	reg.dmt1 &= ~DM_VALID;
	reg.dmt2 &= ~DM_VALID;
	putlog(1, "WriteDAE xip=%x addr=%x", reg.xip, addr);
	ExceptionCore(EXCEP_DATA, ExceptionKind::DATA);
}

// データアクセス例外(Write)
// st.d 用
void
MPU88xx0Device::WriteDataException64(uint32 addr, uint32 flag)
{
	// ミスアラインドはこの例外より前にチェックされていて、
	// ここにはアラインドしかこない。
	reg.dma0 = addr & ~0x3;
	reg.dmt0  = Calcdmt(flag);
	reg.dmt0 |= DM_DOUB1;		// ダブルワードの第1アクセスのときセット
	reg.dmt0 |= FLD_D << 7;
	reg.dmt0 |= DM_EN_MASK;
	reg.dmt0 |= DM_WRITE;
	reg.dmd0 = rD;

	reg.dma1 = reg.dma0 + 4;
	reg.dmt1  = Calcdmt(flag);
	reg.dmt1 |= FLD_D2 << 7;
	reg.dmt1 |= DM_EN_MASK;
	reg.dmt1 |= DM_WRITE;
	reg.dmd1 = rD2;

	reg.dmt2 &= ~DM_VALID;

	if ((flag & DM_DOUB1) == 0) {
		// .d の2ワード目がバスエラー。第1ワードのアクセスは成功した。
		// DMx0 が例外を起こしたアクセスを示す。
		// Manual 6.7.3
		reg.dma0 = reg.dma1;
		reg.dmt0 = reg.dmt1;
		reg.dmd0 = reg.dmd1;
		reg.dmt1 &= ~DM_VALID;
	}

	putlog(1, "WriteDAE64 xip=%x addr=%x", reg.xip, addr);
	ExceptionCore(EXCEP_DATA, ExceptionKind::DATA);
}

// データアクセス例外(xmem)
// xmem には .d は無い
void
MPU88xx0Device::XmemDataException(uint32 addr, uint32 flag)
{
	reg.dma0 = addr & ~0x3;
	reg.dmt0  = Calcdmt(flag);
	reg.dmt0 |= DM_LOCK;		// xmem でセット
	reg.dmt0 |= FLD_D << 7;
	reg.dmt0 |= flag & DM_SIGNED;
	reg.dmt0 |= (flag & DM_EN_MASK) >> (addr & 0x03);

	reg.dma1 = reg.dma0;
	reg.dmt1  = Calcdmt(flag);
	reg.dmt1 |= DM_LOCK;		// xmem でセット
	reg.dmt1 |= FLD_D << 7;
	reg.dmt1 |= flag & DM_SIGNED;
	reg.dmt1 |= (flag & DM_EN_MASK) >> (addr & 0x03);
	reg.dmt1 |= DM_WRITE;
	reg.dmd1 = rD;

	reg.dmt2 &= ~DM_VALID;

	// ミスアラインドはこの例外より前にチェックされていて、
	// ここにはアラインドしかこない。
	if (cmmu[1]->fault_code != m88200::FAULT_CODE_BUSERR
	 || cmmu[1]->acc_read) {
		// アドレス変換フォルトまたは read 時点
		// nop
	} else {
		// xmem の write 時点
		reg.dmt0 &= ~DM_VALID;
	}

	putlog(1, "xmemDAE xip=%x addr=%x", reg.xip, addr);
	ExceptionCore(EXCEP_DATA, ExceptionKind::DATA);
}

// FPxS レジスタをセット
/*static*/ void
MPU88xx0Device::SetFPxS(uint32 *h, uint32 *l, double src, uint32 ts)
{
	// FPHS[12]、FPLS[12] は以下の構造
	//
	//    3                   2                   1                   0
	//  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
	// +-+---------------------+---------------------------------------+
	// |S|   Exponent(11bit)   |     High Order 20 bits of Mantissa    | FPHSn
	// +-+---------------------+---------------------------------------+
	//
	// +---------------------------------------------------------------+
	// |           Low Order bits of Mantissa, or Integer              | FPLSn
	// +---------------------------------------------------------------+
	//
	// ts が 1 (double) の場合は IEEE754 double と同じ。
	// ts が 0 (single) の場合は float を double にキャスト (拡大変換) した
	// ものではないことに注意。
	// S(符号ビット)はそのまま。float の Mantissa (23bit) のうち上位 20bit
	// を FPHSn に、残り 3bit を FPLSn の上位 3bit に。ここまではいい。
	// Exponent は float の指数部(8bit、バイアス=127) の値を *そのまま*
	// 符号拡張(?)する。
	// (float)1.0 は 1.0 * 2^0 なのでバイアス込みの指数部は 127、
	// (float)2.0 は 1.0 * 2^1 なのでバイアス込みの指数部は 128。
	// バイアス込みにした値は符号なし数のはずだが、ここではこれを 11bit に
	// 符号拡張して FPHSn の Exponent 部に格納するようだ。取り出す際に下位
	// 8bit しか取り出さなければ害はないし、符号拡張する方がハードウェア的に
	// 都合がよかったとかだろうか。
	// 仮に普通に double に変換すると double の指数部バイアスは 1023 なので
	// バイアス込みの指数部は先程の例だとそれぞれ 1023、1024 になるが
	// これではないということ。

	union64 u;
	u.q = d2u(src);
	if (ts == 0) {
		// single

		uint32 exp;
		// src は普通の double なのでこの指数部を取り出す
		uint32 double_biased = (u.h >> 20) & 0x7ff;
		if (double_biased == 0x7ff) {
			// Inf, NAN
			exp = 0x7ff00000;
		} else {
			// double のバイアスを外して float のバイアスを足す
			int float_biased = (double_biased - 1023) + 127;
			// この 8bit 値を 11bit に符号拡張して所定の位置へ
			exp = ((int32)(float_biased << 24)) >> 4;
			exp &= 0x7ff00000;
		}

		*h = (u.h & 0x800fffff) | exp;
		*l = u.l;
	} else {
		// double
		*h = u.h;
		*l = u.l;
	}
}

// FPxS1 レジスタをセット
void
MPU88xx0Device::SetFPS1(double src, uint32 t1)
{
	SetFPxS(&reg.fphs1, &reg.fpls1, src, t1);
}

// FPxS2 レジスタをセット
void
MPU88xx0Device::SetFPS2(double src, uint32 t2)
{
	SetFPxS(&reg.fphs2, &reg.fpls2, src, t2);
}

void
MPU88xx0Device::SetFPS1(uint32 src)
{
	reg.fpls1 = src;
}

void
MPU88xx0Device::SetFPS2(uint32 src)
{
	reg.fpls2 = src;
}

// FPR[HL] レジスタに double 値をセット
// - それ以外のフィールドは保存する XXX ここでクリアしてもいいか?
// - XXX Guard, Round, Sticky (, AddOne) はまだない
void
MPU88xx0Device::SetFPRx(double src)
{
	union64 u;
	uint32 exp;

	u.q = d2u(src);

	// Sign は FPRH
	reg.fprh &= ~(FPRH_SIGN | FPRH_MANT);	// Mantissa もついでにクリア
	reg.fprh |= u.h & 0x80000000;

	// Exp  は FPIT。
	// double は 11bit だが FPIT::RESEXP は 12bit なので 1bit 符号拡張する。
	// 符号付き数(正確には負数)の右シフトは実装依存だが gcc, clang は
	// 算術右シフトになる。
	//
	//           3                   2
	//         1 0 9 8 7 6 5 4 3 2 1 0 9
	//        +-+---------------------+-
	// double |S|   Exponent(11bit)   |
	//        +-+---------------------+-
	//
	//        +---------------------+-+-
	// u.h<<1 |  Exponent(11bit)    |X|
	//        +---------------------+-+-
	//
	//        +-----------------------+-
	// exp>>1 |   Exponent(12bit)     |
	//        +-----------------------+-
	exp = u.h << 1;
	exp = (int32)exp >> 1;
	reg.fpit &= ~FPIT_RESEXP;
	reg.fpit |= exp & FPIT_RESEXP;

	// Mantissa は FPRH, FPRL
	// 上位側(FPRH)には最上位(実数桁)の隠しビットを含む。
	reg.fprh |= (u.h & 0x000fffff);
	reg.fprh |= FPRH_1;
	reg.fprl = u.l;
}

// FP Precise 例外。浮動小数点数1つの場合。
void
MPU88xx0Device::FPPreciseException(uint32 cause, double s2)
{
	SetFPS2(s2, m88100opf_FP_T2(reg.opX));
	FPPreciseException(cause);
}

// FP Precise 例外。浮動小数点数2つの場合。
void
MPU88xx0Device::FPPreciseException(uint32 cause, double s1, double s2)
{
	SetFPS1(s1, m88100opf_FP_T1(reg.opX));
	SetFPS2(s2, m88100opf_FP_T2(reg.opX));
	FPPreciseException(cause);
}

// FP Precise 例外。
void
MPU88xx0Device::FPPreciseException(uint32 cause)
{
	assert(cause == FPECR_FIOV   ||
	       cause == FPECR_FUNIMP ||
	       cause == FPECR_FPRV   ||
	       cause == FPECR_FROP   ||
	       cause == FPECR_FDVZ);

	reg.fpecr = cause;

	// FPPT(FP Precise operation Type Register)
	reg.fppt = reg.opX & 0xffe0;		// Opcode, T1, T2, TD は opX と同じ位置
	reg.fppt |= m88100opf_D(reg.opX);	// DEST

	Exception(EXCEP_SFU1_PRECISE);
}

// FP Imprecise 例外。
void
MPU88xx0Device::FPImpreciseException(uint32 cause)
{
	assert(cause == FPECR_FUNF ||
	       cause == FPECR_FOVF ||
	       cause == FPECR_FINX);

	reg.fpecr = cause;

	// FPIT(FP Imprecise operation Type Register)
	reg.fpit &= FPIT_RESEXP;
	reg.fpit |= reg.opX & FPIT_OPCODE;
	reg.fpit |= (reg.opX & 0x00000020) << 5;	// DESTSIZ
	reg.fpit |= (reg.fpcr & 0x0000001f) << 5;	// EFINV,EFDVZ,EFUNF,EFOVR,EFINX
	reg.fpit |= m88100opf_D(reg.opX);			// DEST

	// FPRH(FP Result High Register)
	reg.fprh &= ~FPRH_RNDMODE;
	reg.fprh |= (reg.fpcr & FPCR_RM) << 14;

	Exception(EXCEP_SFU1_IMPRECISE);
}

void
MPU88xx0Device::fpu_unimpl()
{
	printf("%s\n", __func__);
	FPPreciseException(FPECR_FUNIMP);
}

OP_DEF(illegal)
{
	// 本来は不当命令例外?
	putlog(0, "Illegal instruction %08x", (uint32)reg.opX);
}

//
// m88100reg の static 変数
//

// fcr のマスク
/*static*/ const uint32 m88100reg::fcr_mask[11] = {
	M88100::FPECR_MASK,
	0xffffffff,	// fphs1
	0xffffffff,	// fpls1
	0xffffffff,	// fphs2
	0xffffffff,	// fpls2
	M88100::FPPT_MASK,
	M88100::FPRH_MASK,
	0xffffffff,	// fprl
	M88100::FPIT_MASK,
	M88100::FPSR_MASK,
	M88100::FPCR_MASK,
};

/*static*/ const char * const m88100reg::sipname[3] = {
	"sxip", "snip", "sfip",
};

/*static*/ const char * const m88100reg::dmt_en_str[16] = {
	"----",
	"---B",
	"--B-",
	"--HH",
	"-B--",
	"-1-1", // not used normally
	"-11-", // not used normally
	"-111", // not used normally
	"B---",
	"1--1", // not used normally
	"1-1-", // not used normally
	"1-11", // not used normally
	"HH--",
	"11-1", // not used normally
	"111-", // not used normally
	"LLLL",
};
