//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

// MC88200(CMMU)

#include "m88200.h"
#include "mainbus.h"
#include "monitor.h"
#include "mpu88xx0.h"

// static 変数
m88200 *m88200::mbus_master = NULL;

// m88200 コンストラクタ
m88200::m88200(MPU88xx0Device *parent_, uint id_)
	: inherited(OBJ_M88200(id_))
{
	parent = parent_;

	// ID は IDR レジスタへの書き込みで後から変更できるように読めるが、
	// ハードウェアパラメータのはずだし変更する意味はないのと、仮に変更しても
	// 実際訳の分からんことにしかならんと思う。
	// そして PROM も OpenBSD/luna88k も IDR へ書き込んでいないので、やはり
	// あらかじめ決まっているハードウェアパラメータだと思うことにする。
	// それと、コンストラクタ時点で ID が決まってないのは何かと困るという
	// こちらの都合もある。
	id = id_;

	ClearAlias();
	AddAlias(string_format("CMMU%u", id));

	reg_monitor = gMonitorManager->Regist(ID_MONITOR_CMMU(id), this);
	reg_monitor->func = ToMonitorCallback(&m88200::MonitorUpdateReg);
	reg_monitor->SetSize(52, 8);

	atc_monitor = gMonitorManager->Regist(ID_MONITOR_ATC(id), this);
	atc_monitor->func = ToMonitorCallback(&m88200::MonitorUpdateATC);
#if defined(M88200_STAT)
	atc_monitor->SetSize(77, 44 + 5);
#else
	atc_monitor->SetSize(77, 44);
#endif

	cache_monitor = gMonitorManager->Regist(ID_SUBWIN_CACHE(id), this);
	cache_monitor->func = ToMonitorCallback(&m88200::MonitorUpdateCache);
	cache_monitor->SetSize(82, 23);

	// ログ表示用の名前
	sapr.name = "SAPR";
	uapr.name = "UAPR";
}

// m88200 デストラクタ
m88200::~m88200()
{
}

// 初期化
bool
m88200::Init()
{
	if (inherited::Init() == false) {
		return false;
	}

	mainbus = GetMainbusDevice();

	memset(&atc_hash_all[0], 0, atc_hash_all.size());
	for (auto& b : batc) {
		b.lba |= BATC_INVALID;
	}
	for (auto& p : patc) {
		p.lpa |= PATC_INVALID;
	}

	// 暗黙 BATC を初期化
	SetBATC(8, 0xfff00000, 0xfff00000, BWP_S | BWP_WT | BWP_CI | BWP_V);
	SetBATC(9, 0xfff80000, 0xfff80000, BWP_S | BWP_WT | BWP_CI | BWP_V);

	// セットインデックスを初期化
	for (int i = 0; i < setarray.size(); i++) {
		auto& set = setarray[i];
		set.setidx = i;
	}

	return true;
}

// リセット
void
m88200::Reset()
{
	const uint32 Undefined = 0xcccccccc;

	// レジスタ (p6-4, Table6-3)
	// ID, version は別途 Set される。
	command = 0;
	ssr = 0;
	sar = Undefined;
	sctr = 0;
	fault_code = 0;
	fault_addr = Undefined;
	sapr.enable = false;
	sapr.stat = APR_CI;
	uapr.enable = false;
	uapr.stat = APR_CI;

	// BATC, PATC はリセットで初期化されないようだが確認のしようがない。
}

// モニター更新 (CMMU レジスタ)
void
m88200::MonitorUpdateReg(Monitor *, TextScreen& screen)
{
	uint32 baseaddr;
	uint32 reg;
	int x;
	int y;

	baseaddr = 0xfff00000 + (id << 12);

	screen.Clear();
	x = 24;

	y = 0;
	screen.Print(0, y++, "$%08x  IDR:%08x ID=$%02x Type=5(88200) Ver=$%x",
		baseaddr + 0, GetIDR(), id, version);

	reg = GetSSR();
	screen.Print(0, y, "$%08x  SSR:%08x", baseaddr + 0x008, reg);
	screen.Puts(x,      y, TA::OnOff(reg & SSR_CE), "CE");
	screen.Puts(x + 3,  y, TA::OnOff(reg & SSR_BE), "BE");
	screen.Puts(x + 6,  y, TA::OnOff(reg & SSR_WT), "WT");
	screen.Puts(x + 9,  y, TA::OnOff(reg & SSR_SP), "SP");
	screen.Puts(x + 12, y, TA::OnOff(reg & SSR_G),  "G");
	screen.Puts(x + 14, y, TA::OnOff(reg & SSR_CI), "CI");
	screen.Puts(x + 17, y, TA::OnOff(reg & SSR_M),  "M");
	screen.Puts(x + 19, y, TA::OnOff(reg & SSR_U),  "U");
	screen.Puts(x + 21, y, TA::OnOff(reg & SSR_WP), "WP");
	screen.Puts(x + 24, y, TA::OnOff(reg & SSR_BH), "BH");
	screen.Puts(x + 27, y, TA::OnOff(reg & SSR_V),  "V");
	y++;

	screen.Print(0, y++, "$%08x  SAR:%08x", baseaddr + 0x00c, GetSAR());

	reg = GetSCTR();
	screen.Print(0, y, "$%08x SCTR:%08x", baseaddr + 0x104, GetSCTR());
	screen.Puts(x,      y, TA::OnOff(reg & SCTR_PE), "PE");
	screen.Puts(x + 3,  y, TA::OnOff(reg & SCTR_SE), "SE");
	screen.Puts(x + 6,  y, TA::OnOff(reg & SCTR_PR), "PR");
	y++;

	reg = GetPFSR();
	static const char * const codestr[] = {
		"Success",
		"1?",
		"2?",
		"Bus Error",
		"Segment Fault",
		"Page Fault",
		"Supervisor Violation",
		"Write Violation",
	};
	screen.Print(0, y++, "$%08x PFSR:%08x %s", baseaddr + 0x108,
		reg, codestr[fault_code]);

	screen.Print(0, y++, "$%08x PFAR:%08x", baseaddr + 0x10c, GetPFAR());

	for (int i = 0; i < 2; i++) {
		int s = 1 - i;
		reg = GetAPR(s);
		screen.Print(0, y, "$%08x %cAPR:%08x", baseaddr + 0x200 + i * 4,
			s ? 'S' : 'U', reg);
		screen.Print(x, y, "STBA=%05x'000 %c%c%c%c",
			reg >> 12,
			(reg & APR_WT) ? 'T' : '-',
			(reg & APR_G)  ? 'G' : '-',
			(reg & APR_CI) ? 'C' : '-',
			(reg & APR_TE) ? 'E' : '-');
		y++;
	}
}

// モニター更新 (ATC)
void
m88200::MonitorUpdateATC(Monitor *, TextScreen& screen)
{
	int x;
	int y;

	screen.Clear();

	screen.Print(0, 0,  "SAPR  %05x'000 TE=%u      %c%c%c",
		(sapr.addr >> 12) & 0xfffff,
		 sapr.enable ? 1 : 0,
		(sapr.stat & APR_WT) ? 'T' : '-',
		(sapr.stat & APR_G)  ? 'G' : '-',
		(sapr.stat & APR_CI) ? 'C' : '-');

	screen.Print(40, 0, "UAPR  %05x'000 TE=%u      %c%c%c",
		(uapr.addr >> 12) & 0xfffff,
		 uapr.enable ? 1 : 0,
		(uapr.stat & APR_WT) ? 'T' : '-',
		(uapr.stat & APR_G)  ? 'G' : '-',
		(uapr.stat & APR_CI) ? 'C' : '-');

	screen.Puts(0,  1, "<BATC>");
	screen.Puts(0,  2, "No. LBA         PBA       Stat  Hit%");
	screen.Puts(40, 2, "No. LBA         PBA       Stat  Hit%");
	x = 0;
	y = 3;
	uint64 batc_total = 0;
	for (int i = 0; i < 10; i++) {
		if (i == batc.size() / 2) {
			x = 40;
			y = 3;
		}
		if (i < 8) {
			screen.Print(x, y, " %u:", i);
		} else {
			screen.Print(x, y, "(%u)", i);
		}

		const m88200BATC& b = batc[i];
		if (b.IsValid()) {
			screen.Print(x + 4, y, "%c:%04x'0000 %04x'0000 %c%c%c%c",
				b.IsS() ? 'S' : 'U',
				b.lba >> 16, b.pba >> 16,
				(b.stat & DESC_WT) ? 'T' : '-',
				(b.stat & DESC_G)  ? 'G' : '-',
				(b.stat & DESC_CI) ? 'C' : '-',
				(b.stat & DESC_WP) ? 'P' : '-');
		}
		if (batc_hit[i] == 0) {
			screen.Puts(x + 32, y, "--.-%");
		} else {
			screen.Print(x + 32, y, "%4.1f%%",
				(double)batc_hit[i] / translate_total * 100);
			batc_total += batc_hit[i];
		}
		y++;
	}

	screen.Puts(0,  8, "<PATC>");
	screen.Puts(0,  9, "No. LPA         PFA       Stat");
	screen.Puts(40, 9, "No. LPA         PFA       Stat");

	x = 0;
	y = 10;
	for (int i = 0; i < patc.size(); i++) {
		if (i == patc.size() / 2) {
			x = 40;
			y = 10;
		}
		screen.Print(x, y, "%2u:", i);

		const m88200PATC& p = patc[i];
		if (p.IsValid()) {
			screen.Print(x + 4,  y, "%c:%05x'000 %05x'000 %c%c%c%c%c",
				p.IsS() ? 'S' : 'U',
				p.lpa >> 12, p.pfa >> 12,
				(p.stat & DESC_WT) ? 'T' : '-',
				(p.stat & DESC_G)  ? 'G' : '-',
				(p.stat & DESC_CI) ? 'C' : '-',
				(p.stat & DESC_WP) ? 'P' : '-',
				p.m                ? 'M' : '-');
		}
		y++;
	}

	x = 49;
	y++;
	screen.Puts(0, y++, "<Statistics>");
	screen.Print(0, y++, "Translate            %26s",
		format_number(translate_total).c_str());
	screen.Print(0, y, " BATC hit            %26s (",
		format_number(batc_total).c_str());
	if (__predict_false(batc_total == 0)) {
		screen.Puts(x, y, "---.-%)");
	} else {
		screen.Print(x, y, "%5.1f%%)",
			(double)batc_total / translate_total * 100);
	}
	y++;
	screen.Print(0, y, " PATC hit            %26s (",
		format_number(patc_hit).c_str());
	if (__predict_false(patc_hit == 0)) {
		screen.Puts(x, y, "---.-%)");
	} else {
		screen.Print(x, y, "%5.1f%%)",
			(double)patc_hit / translate_total * 100);
	}
	y++;
	screen.Print(0, y, " BATC/PATC miss      %26s (",
		format_number(atc_miss).c_str());
	if (__predict_false(atc_miss == 0)) {
		screen.Puts(x, y, "---.-%)");
	} else {
		screen.Print(x, y, "%5.1f%%)",
			(double)atc_miss / translate_total * 100);
	}
#if defined(M88200_STAT)
	y++;
	screen.Print(0, y++, "PATC create          %26s",
		format_number(stat_patc_create).c_str());
	screen.Print(0, y++, "PATC invalidate      %26s",
		format_number(stat_patc_invalidate).c_str());
	screen.Print(0, y++, "PATC SCR InvAll      %26s",
		format_number(stat_patc_invcmd_all).c_str());
	screen.Print(0, y++, "PATC SCR InvSegment  %26s",
		format_number(stat_patc_invcmd_seg).c_str());
	screen.Print(0, y++, "PATC SCR InvPage     %26s",
		format_number(stat_patc_invcmd_page).c_str());
#endif
}

// モニター更新 (キャッシュ、概要と詳細の両方、GUI から呼ばれる)
// screen.userdata は注目しているセット番号。
void
m88200::MonitorUpdateCache(Monitor *, TextScreen& screen)
{
	uint setidx = screen.userdata;

	// 上半分(概要)
	MonitorCacheOverview(screen, 0, setidx, true);
	// 下半分(セット詳細)
	MonitorCacheSet(screen, 18, setidx);
}

// データキャッシュの特定セットの詳細を TextScreen に出力する。
// y は開始オフセット。
// TextScreen は (70, 5) 必要。
void
m88200::MonitorCacheSet(TextScreen& s, int y, uint setidx)
{
	assertmsg(setidx < setarray.size(), "setidx=%u", setidx);
	const m88200CacheSet& set = setarray[setidx];

	/*
	012345678901234567890123456789012345678901234567890123456789
	L Tag        Status
	0 $11223'344 VV     12345678 12345678 12345678 12345678
	*/
	s.Puts(0, y, "L Tag        Status Word");
	s.Puts(58, y, "Order");
	y++;

	// ラインの古い順に評価して順序を0-3でつける
	int Lorder[4];
	int tmpL = set.L;
	for (int i = 0; i < 4; i++) {
		int line = m88200CacheSet::TryGetOldestLine(tmpL);
		Lorder[line] = 3 - i;
		tmpL = m88200CacheSet::TryUseLine(tmpL, line);
	}

	for (int line = 0; line < 4; line++, y++) {
		TA attr;
		// ステータスによって属性を選択
		switch (set.vv[line]) {
		 case m88200CacheSet::Status::IV:
			attr = TA::Disable;
			break;
		 case m88200CacheSet::Status::EU:
		 case m88200CacheSet::Status::SU:
			attr = TA::Off;
			break;
		 case m88200CacheSet::Status::EM:
			attr = TA::Em;
			break;
		}
		s.Print(0, y, attr, "%u", line);
		static const char statusstr[][4] = { "EU", "EM", "SU", "IV" };
		s.Print(2, y, attr, "$%05x'%03x %s",
			(set.tag[line] >> 12),
			(setidx << 4),
			statusstr[set.vv[line]]);

		if (set.vv[line] == m88200CacheSet::Status::EM) {
			// メモリに対してダーティならボールドにする
			for (uint w = 0; w < 4; w++) {
				uint32 addr;
				uint32 m;
				addr = (set.tag[line] & 0xfffff000) | (setidx << 4) | (w << 2);
				m = (mainbus->Peek1(addr) << 24)
				  | (mainbus->Peek1(addr + 1) << 16)
				  | (mainbus->Peek1(addr + 2) <<  8)
				  | (mainbus->Peek1(addr + 3));
				if (set.word[line * 4 + w] != m) {
					attr = TA::Em;
				} else {
					attr = TA::Off;
				}
				s.Print(20 + w * 9, y, attr, "%08x", set.word[line * 4 + w]);
			}
		} else {
			// Unmodified (or Invalid) ならメモリとの比較は不要
			for (int w = 0; w < 4; w++) {
				s.Print(20 + w * 9, y, "%08x", set.word[line * 4 + w]);
			}
		}

		// 順序
		s.Print(58, y, "%u", Lorder[line]);
	}
}

// データキャッシュの概要を指定の TextScreen に出力する。
// y は開始オフセット。CLI では単独コマンドとして、GUI ではページの一部と
// して描画するためこうなっている。
// cursor で指定された番号のセットは反転表示する。GUI でのカーソル用。
// 負数など範囲外の値を指定すればカーソルは表示されない。
// is_gui は GUI かどうか。CLI では80桁を微妙に越えるのは嫌だしどうせ表示
// だけなので間を詰めてあるが、GUI では 80桁制約はない代わりにマウス操作が
// あるので 1セットごとに間を空けて等間隔にしたい、という違いから。
// TextScreen は CLI なら (70, 17)、GUI なら (82, 17) 必要。
void
m88200::MonitorCacheOverview(TextScreen& s, int y, uint cursor, bool is_gui)
{
	// X ガイド
	for (uint i = 0; i < 16; i++) {
		uint x;
		if (is_gui) {
			x = 3 + i * 5;
		} else {
			x = 3 + i * 4 + (i / 4);
		}
		s.Print(x, y, "+0%x", i);
	}
	y++;

	// Y ガイド
	for (uint i = 0; i < 16; i++) {
		s.Print(0, y + i, "%02x", i * 16);
	}

	for (uint i = 0; i < setarray.size(); i++) {
		const auto& set = setarray[i];
		const char str[] = "EMS-";
		uint col = i % 16;
		uint row = i / 16;
		uint x;
		if (is_gui) {
			x = 3 + col * 5;
		} else {
			x = 3 + col * 4 + col / 4;
		}

		s.Print(x, y + row, TA::OnOff(i == cursor),
			"%c%c%c%c",
			str[set.vv[0]],
			str[set.vv[1]],
			str[set.vv[2]],
			str[set.vv[3]]);
	}
}

// IDR の Version フィールドを設定する
void
m88200::SetVersion(uint version_)
{
	version = version_;
}

// コマンド名
// (0-15 は全部 No Operation なので、16以降のみ)
/*static*/ const char * const
m88200::commandname[] = {
	"No Operation",						// $10
	"No Operation",						// $11
	"No Operation",						// $12
	"No Operation",						// $13
	"DCache Invalidate, Line",			// $14
	"DCache Invalidate, Page",			// $15
	"DCache Invalidate, Segment",		// $16
	"DCache Invalidate, All",			// $17

	"DCache Copyback, Line",			// $18
	"DCache Copyback, Page",			// $19
	"DCache Copyback, Segment",			// $1a
	"DCache Copyback, All",				// $1b
	"DCache Copy&Inv, Line",			// $1c
	"DCache Copy&Inv, Page",			// $1d
	"DCache Copy&Inv, Segment",			// $1e
	"DCache Copy&Inv, All",				// $1f

	"Probe User Address",				// $20
	"Probe User Address",				// $21
	"Probe User Address",				// $22
	"Probe User Address",				// $23
	"Probe Supervisor Address",			// $24
	"Probe Supervisor Address",			// $25
	"Probe Supervisor Address",			// $26
	"Probe Supervisor Address",			// $27

	"Probe User Address",				// $28
	"Probe User Address",				// $29
	"Probe User Address",				// $2a
	"Probe User Address",				// $2b
	"Probe Supervisor Address",			// $2c
	"Probe Supervisor Address",			// $2d
	"Probe Supervisor Address",			// $2e
	"Probe Supervisor Address",			// $2f

	"Invalidate U PATC, Line",			// $30
	"Invalidate U PATC, Page",			// $31
	"Invalidate U PATC, Segment",		// $32
	"Invalidate U PATC, All",			// $33
	"Invalidate S PATC, Line",			// $34
	"Invalidate S PATC, Page",			// $35
	"Invalidate S PATC, Segment",		// $36
	"Invalidate S PATC, All",			// $37

	"Invalidate U PATC, Line",			// $38
	"Invalidate U PATC, Page",			// $39
	"Invalidate U PATC, Segment",		// $3a
	"Invalidate U PATC, All",			// $3b
	"Invalidate S PATC, Line",			// $3c
	"Invalidate S PATC, Page",			// $3d
	"Invalidate S PATC, Segment",		// $3e
	"Invalidate S PATC, All",			// $3f
};

// SCR の読み出し
uint32
m88200::GetSCR() const
{
	// b31-b6 (reserved) の読み出し値は未定義らしい。
	// b5-b0 (Command Code) の読み出し値はマニュアルに記載がないけど
	// たぶんそのまま読めるのかな。
	return command;
}

// SCR への書き込み
void
m88200::SetSCR(uint32 data)
{
	command = data & 0x3f;

	// %00'XXXX No Operation
	// %01'00XX No Operation
	// %01'01gg Data Cache Invalidate
	// %01'10gg Data Cache Copyback to Memory
	// %01'11gg Data Cache Copyback and Invalidate
	// %10'X0XX Probe User Address
	// %10'X1XX Probe Supervisor Address
	// %11'X0gg Invalidate User PATC Descriptors
	// %11'X1gg Invalidate Supervisor PATC Descriptors

	const char *name;
	if (command < 0x10) {
		name = commandname[0];
	} else {
		name = commandname[command - 0x10];
	}
	putlog(1, "SCR  <- $%08x (%s)", data, name);

	switch (command) {
	 case 0x00 ... 0x13:	// No Operation
		return;

	 case 0x14 ... 0x1f:	// Flush Data Cache
		FlushCache();
		return;

	 case 0x20 ... 0x23:	// Probe User Address
	 case 0x28 ... 0x2b:
		putlog(0, "SCR Command: Probe User Address (NOT IMPLEMENTED)");
		return;

	 case 0x24 ... 0x27:	// Probe Supervisor Address
	 case 0x2c ... 0x2f:
		putlog(0, "SCR Command: Probe Supervisor Address (NOT IMPLEMENTED)");
		return;

	 case 0x30 ... 0x3f:	// Invalidate {User,Supervisor} PATC Descriptors
		InvalidatePATC();
		return;
	}
	PANIC("should not reach: command=$%02x", command);
}

// command に応じてデータキャッシュをフラッシュする。SetSCR() の下請け。
// command が $14..$1f でのみ呼ぶこと。
// p3-18, Section 3.7
void
m88200::FlushCache()
{
	uint32 op = command & 0x3c;
	uint32 gg = command & 0x03;
	uint32 addr = GetSAR();
	bool copyback;
	bool invalidate;

	assert(0x14 <= op && op <= 0x1f);

	// op       CopyBack Invalidate
	// %01'01gg false    true       | Data Cache Invalidate
	// %01'10gg true     false      | Data Cache Copyback to Memory
	// %01'11gg true     true       | Data Cache Copyback and invalidate

	// op によって書き戻しと無効化の組み合わせが異なる
	if (op == 0x14) {
		copyback   = false;
		invalidate = true;
	} else if (op == 0x18) {
		copyback   = true;
		invalidate = false;
	} else {
		copyback   = true;
		invalidate = true;
	}

	// 影響範囲
	switch (gg) {
	 case GG_LINE:
		addr &= 0xfffffff0;
		parent->AddCycle(1);	// Table.6-1
		break;
	 case GG_PAGE:
		addr &= 0xfffff000;
		parent->AddCycle(256);	// Table.6-1
		break;
	 case GG_SEG:
		addr &= 0xffc00000;
		parent->AddCycle(1024);	// Table.6-1
		break;
	 case GG_ALL:
		parent->AddCycle(1024);	// Table.6-1
		break;
	 default:
		__unreachable();
	}

	// サイクル(基本部分) Table.6-1
	uint32 cycle = 0;
	if (invalidate) {
		switch (gg) {
		 case GG_LINE:	cycle = 1;		break;
		 case GG_PAGE:	cycle = 256;	break;
		 case GG_SEG:	cycle = 1024;	break;
		 case GG_ALL:	cycle = 256;	break;
		 default:	__unreachable();
		}
	}
	if (copyback) {
		switch (gg) {
		 case GG_LINE:	cycle = 1;		break;
		 case GG_PAGE:	cycle = 256;	break;
		 case GG_SEG:	cycle = 1024;	break;
		 case GG_ALL:	cycle = 1024;	break;
		 default:	__unreachable();
		}
	}
	// XXX ループ中のサイクル数は正しくないかも。よく分からん

	for (auto& set : setarray) {
		for (int line = 0; line < 4; line++) {
			// 条件にマッチするか
			bool match;
			switch (gg) {
			 case GG_LINE:
				match = (addr == (set.tag[line] | (set.setidx << 4)));
				break;
			 case GG_PAGE:
				match = (addr == set.tag[line]);
				break;
			 case GG_SEG:
				match = (addr == (set.tag[line] & 0xffc00001));
				break;
			 case GG_ALL:
				match = true;
				break;
			 default:
				__unreachable();
			}
			if (!match)
				continue;

			// Copyback ならまず EM なエントリを書き戻す。
			if (copyback) {
				if (set.vv[line] == m88200CacheSet::EM) {
					cycle += 7;	// Table.6-1
					CopyBackLine(set, line);
				}
			}

			// Invalidate なら無効化する
			if (invalidate) {
				cycle += 1;	// Table.6-1
				set.Update(line, m88200CacheSet::IV);
			}
		}
	}

	parent->AddCycle(cycle);
}

// issuper, addr から atc_hash_all[] のインデックスを返す。
// (_all でないほうのインデックスは (laddr >> 12) だけなので用意しない)
inline uint32
m88200::addr2hash(bool issuper, uint32 addr) const
{
	uint64 la = (issuper ? 0x1'00000000ULL : 0) | addr;
	return la >> 12;
}

// command に応じて PATC を無効化する。SetSCR() の下請け。
// command が $30..$3f で呼ぶこと。
// p2-9, Section 2.2.4
void
m88200::InvalidatePATC()
{
	uint32 gg = command & 0x03;
	bool s = (command & 0x04);

	if (__predict_true(gg == GG_PAGE)) {
		// 指定の1本だけ無効にする。
		// Page は呼び出し回数が多いのと1本だけならハッシュで引けるので別対応。
#if defined(M88200_STAT)
		stat_patc_invcmd_page++;
#endif

		uint32 la = addr2hash(s, GetSAR());
		uint8 n = atc_hash_all[la];

		if (__predict_true((int8)n > 0)) {
			// PATC#(n-1)
			n--;
			putlog(4, "Invalidate PATCEntry from GG_PAGE n=%u", n);
			InvalidatePATCEntry(patc[n]);
			// 1本ヒットすればこれ以上一致することはないはず
		}
	} else {
		// 指定範囲を無効にする。All と Segment はマスクが違うだけ。
		uint32 addr;
		uint32 mask;

		if (gg == GG_ALL) {
			// S/U 指定したほう全体を無効にする
#if defined(M88200_STAT)
			stat_patc_invcmd_all++;
#endif
			mask = 0;
		} else if (gg == GG_SEG) {
			// S/U 指定したほうの指定セグメント範囲全部を無効にする
#if defined(M88200_STAT)
			stat_patc_invcmd_seg++;
#endif
			mask = 0xffc00000;
		} else {
			// GG_LINE はマニュアルにどうなるか書いてない
			putlog(0, "Undefined Invalidate PATC Line");
			return;
		}

		addr = GetSAR() & mask;
		addr |= (s) ? PATC_S : 0;

		mask |= PATC_S | PATC_INVALID;

		for (int i = 0; i < patc.size(); i++) {
			m88200PATC& p = patc[i];
			if ((p.lpa & mask) == addr) {
				// 無効化
				// XXX 削除した結果穴が空いても詰める処理は未実装
				putlog(4, "Invalidate PATCEntry from GG_* n=%u", i);
				InvalidatePATCEntry(p);
			}
		}
	}
}

// SCTR レジスタへの書き込み
void
m88200::SetSCTR(uint32 data)
{
	sctr = data & (SCTR_PE | SCTR_SE | SCTR_PR);

	// 全 CMMU について、それぞれスヌープ相手になる CMMU リストを更新。
	// どの CMMU の SCTR への書き込みでも毎回全ての CMMU を書き換える。
	std::array<m88200*, 8> cmmu {};
	uint n = cmmu.size();
	for (int i = 0; i < n; i++) {
		cmmu[i] = gMainApp.FindObject<m88200>(OBJ_M88200(i));
	}
	for (int i = 0; i < n; i++) {
		if (cmmu[i] == NULL)
			continue;
		cmmu[i]->other_cmmu.clear();
		for (int j = 0; j < n; j++) {
			if (cmmu[j] == NULL || i == j)
				continue;
			if ((cmmu[j]->sctr & SCTR_SE)) {
				cmmu[i]->other_cmmu.push_back(cmmu[j]);
			}
		}
	}

	std::string msg;
	if ((sctr & SCTR_PE))
		msg += ",PE";
	if ((sctr & SCTR_PR))
		msg += ",PR";
	if (msg.length() > 0) {
		putlog(0, "SCTR <- $%08x (%s NOT IMPLEMENTED)", data, msg.c_str() + 1);
	}
}

// SAPR, UAPR レジスタ値の読み出し (S/U ビット指定)
uint32
m88200::GetAPR(uint issuper) const
{
	if (issuper) {
		return GetAPR(sapr);
	} else {
		return GetAPR(uapr);
	}
}

// SAPR, UAPR レジスタ値の読み出し (実体指定)
uint32
m88200::GetAPR(const m88200APR& xapr) const
{
	uint32 data;

	data  = xapr.addr & APR_ADDR_MASK;
	data |= xapr.stat & (APR_WT | APR_G | APR_CI);
	if (xapr.enable) {
		data |= APR_TE;
	}
	return data;
}

// SAPR, UAPR レジスタへの書き込み (実体指定)
void
m88200::SetAPR(m88200APR& xapr, uint32 data)
{
	bool old_enable = xapr.enable;
	xapr.addr = data & APR_ADDR_MASK;
	xapr.stat = data & (APR_WT | APR_G | APR_CI);
	xapr.enable = data & APR_TE;

	if (old_enable == false && xapr.enable == true) {
		// 変換開始
		putlog(1, "%s <- $%08x (Translation Enabled)", xapr.name, data);
	} else if (old_enable == true && xapr.enable == false) {
		// 変換停止
		putlog(1, "%s <- $%08x (Translation Disabled)", xapr.name, data);
	} else {
		// 変化なし
		putlog(1, "%s <- $%08x", xapr.name, data);
	}
}

// BWP(BATC Write Port) #n への書き込み
void
m88200::SetBWP(uint bn, uint32 data)
{
	assert(bn < 8);
	putlog(1, "BWP%u <- $%08x", bn, data);

	uint32 laddr =  data & BWP_LBA_MASK;
	uint32 paddr = (data & BWP_PBA_MASK) << 13;
	uint32 flags =  data & BWP_FLAG_MASK;

	// BATC の LBA は衝突してはいけない (が、どうなるかは書いてない)。
	// XXX 要実機検証…

	// とりあえず暗黙 BATC (#8, #9) と衝突する設定は無視しておく。
	if (laddr >= 0xfff00000) {
		return;
	}

	// それ以外の BATC (#0-#7) と衝突する設定は先着優先としておく。
	uint32 la = addr2hash((flags & BWP_S), laddr);
	uint8 n = atc_hash_all[la];
	if (n != 0 && ~n != bn) {
		return;
	}

	SetBATC(bn, laddr, paddr, flags);
}

// BATC #n を更新する。
// n は 0-9 (暗黙 BATC も含む)。
// laddr の衝突は呼び出し側で回避してある。
void
m88200::SetBATC(uint n, uint32 laddr, uint32 paddr, uint32 flags)
{
	m88200BATC& b = batc[n];

	// BWP レジスタへの書き込み値と内部データ構造(stat)ではビット位置が違う
	// ことに注意。
	//
	//       31      10   9    8    7    6    5    4    3    2    1    0
	//      +----..-----+----+----+----+----+----+----+----+----+----+----+
	// BWP  |LBA              PBA           | S  | WT | G  | CI | WP | V  |
	//      +----..-----+----+----+----+----+----+----+----+----+----+----+
	//
	//       31      10   9    8    7    6    5    4    3    2    1    0
	//      +----..-----+----+----+----+----+----+----+----+----+----+----+
	// stat | 0      0  | WT | SP | G  | CI | 0  | M  | U  | WP | 0  | V  |
	//      +----..-----+----+----+----+----+----+----+----+----+----+----+

	// 更新前の BATC が有効なら、BATC を書き換える前にハッシュをクリア。
	if (b.IsValid()) {
		uint32 la = addr2hash(b.IsS(), b.lba);
		memset(&atc_hash_all[la], 0, 128);

		// BATC が消えたことによってここを指していた PATC が見えるように
		// なったら、復活させる。
		for (int i = 0; i < patc.size(); i++) {
			const auto& p = patc[i];
			if (p.IsValid() &&
			    p.IsS() == b.IsS() &&
			    (p.lpa & m88200BATC::LBAMASK) == b.Laddr())
			{
				la = addr2hash(p.IsS(), p.lpa);
				atc_hash_all[la] = i + 1;
			}
		}
	}

	memset(&b, 0, sizeof(b));
	b.lba  = laddr;
	b.lba |= (flags & BWP_S);
	if ((flags & BWP_V) == 0) {
		b.lba |= BATC_INVALID;
	}
	b.pba = paddr;
	if ((flags & BWP_WT))
		b.stat |= DESC_WT;
	if ((flags & BWP_G))
		b.stat |= DESC_G;
	if ((flags & BWP_CI))
		b.stat |= DESC_CI;
	if ((flags & BWP_WP)) {
		b.stat |= DESC_WP;
		b.wp = true;
	}

	// 更新後の BATC が有効なら、ハッシュもセット。
	if (b.IsValid()) {
		uint32 la = addr2hash(b.IsS(), b.lba);
		// この領域に PATC があれば追い出す。
		for (int i = la; i < la + 128; i++) {
			uint8 pn = atc_hash_all[i];
			if ((int8)pn > 0) {
				pn--;
				putlog(4, "Invalidate PATCEntry from SetBATC(%u) n=%u", n, pn);
				InvalidatePATCEntry(patc[pn]);
			}
		}
		memset(&atc_hash_all[la], ~n, 128);
	}
	//DumpATC();
}

// CSSP レジスタの読み出し
uint32
m88200::GetCSSP() const
{
	uint32 setidx = (GetSAR() >> 4) & 0xff;
	const m88200CacheSet& set = setarray[setidx];

	uint32 data = 0;
	// L5-L0
	data |= set.L << 24;

	// XXX D3-D0 未実装

	// VV3-VV0
	data |= set.vv[3] << CSSP_VV3_OFFSET;
	data |= set.vv[2] << CSSP_VV2_OFFSET;
	data |= set.vv[1] << CSSP_VV1_OFFSET;
	data |= set.vv[0] << CSSP_VV0_OFFSET;

	return data;
}

// CSSP レジスタへの書き込み
void
m88200::SetCSSP(uint32 data)
{
	uint32 setidx = (GetSAR() >> 4) & 0xff;
	m88200CacheSet& set = setarray[setidx];

	putlog(1, "CSSP <- $%08x (set=$%02x)", data,setidx);

	// L5-L0 書き込み
	set.L = (data >> 24) & 0x3f;

	// XXX D3-D0 未実装

	// VV3-VV0
	set.vv[3] = (m88200CacheSet::Status)((data >> CSSP_VV3_OFFSET) & 3);
	set.vv[2] = (m88200CacheSet::Status)((data >> CSSP_VV2_OFFSET) & 3);
	set.vv[1] = (m88200CacheSet::Status)((data >> CSSP_VV1_OFFSET) & 3);
	set.vv[0] = (m88200CacheSet::Status)((data >> CSSP_VV0_OFFSET) & 3);
}

// S/U 信号線を設定。true なら Super、false なら User。
void
m88200::SetSuper(bool super)
{
	// acc_super は S/U ビットだけだと知っているので OR ではなく代入。
	if (super) {
		acc_super = BusAddr::S;
		acc_apr = &sapr;
		atc_hash = &atc_hash_all[0x10'0000];
	} else {
		acc_super = BusAddr::U;
		acc_apr = &uapr;
		atc_hash = &atc_hash_all[0];
	}
}


//
// 物理バスアクセス
//

// MBus から読み込みを行う。
// paddr は size によって 1, 2, 4 バイト境界に整列していること。
// バスエラーなら呼び出し側で SetFault(FAULT_CODE_BUSERR, paddr) を呼ぶこと。
busdata
m88200::MBusRead(uint32 paddr, int size)
{
	busaddr baddr = busaddr(paddr) | acc_super | busaddr::Size(size);
	busdata bd;

	// m88k システムに接続しているデバイスはすべて、m68030 システムでいう
	// ところのロングワードポートデバイスなので応答は常に 4 バイト分ある。
	bd = mainbus->Read(baddr);

	if (size == 4) {
	} else if (size == 2) {
		if ((paddr & 2) == 0) {
			bd >>= 16;
		} else {
			bd &= 0xffff;
		}
	} else {
		bd >>= (3 - (paddr & 3)) * 8;
		bd &= 0xff;
	}

	parent->AddWait(bd.GetWait());
	return bd;
}

// MBus に書き込みを行う。
// paddr は size によって 1, 2, 4 バイト境界に整列していること。
// バスエラーなら呼び出し側で SetFault(FAULT_CODE_BUSERR, paddr) を呼ぶこと。
busdata
m88200::MBusWrite(uint32 paddr, uint32 data, int size)
{
	busaddr baddr = busaddr(paddr) | acc_super | busaddr::Size(size);
	busdata bd;

	bd = mainbus->Write(baddr, data);
	parent->AddWait(bd.GetWait());
	return bd;
}

// MBus に xmem トランザクションを行う。
// paddr は size によって 1, 4 バイト境界に整列していること (2バイトはない)。
// MBus Acquire された状態で呼び出すこと。
// 成功すれば読み込めた値を返す。
// バスエラーなら呼び出し側で SetFault(FAULT_CODE_BUSERR, paddr) を呼ぶこと。
// read/write どちらでエラーが起きたかは acc_read で判別できる。
// p3-13, Figure3-7 の下半分のメインラインあたりに該当。
busdata
m88200::MBusXmem(uint32 paddr, uint32 data, int size)
{
	busdata fetched;
	busdata bd;

	// Read data with intent to modify
	acc_read = true;
	bd = MBusRead(paddr, size);
	parent->AddWait(bd.GetWait());
	if (__predict_false(bd.IsBusErr())) {
		return bd;
	}
	fetched = bd;

	// Supply data to PBus
	// Reply = Success

	// Write data to memory
	acc_read = false;
	bd = MBusWrite(paddr, data, size);
	parent->AddWait(bd.GetWait());
	if (__predict_false(bd.IsBusErr())) {
		return bd;
	}

	return fetched;
}

//
// アクセス関数
//

template <uint size> uint64
m88200::load(uint32 addr)
{
	uint64 data;

	acc_laddr = addr;
	acc_read  = true;
	if (__predict_false(Translate() == false)) {
		putlog(2, "Translation BusError %c:$%08x",
			(IsSuper() ? 'S' : 'U'), acc_laddr);
		return (uint64)-1;
	}
	if (__predict_false((acc_stat & DESC_CI))) {
		parent->AddCycle(7);	// Table.6-2
		MBusAcquire();
		MBusMakeSnoop(acc_paddrH, 0);
		uint32 paddr = acc_paddrH + acc_paddrL;
		busdata bd = MBusRead(paddr, size);
		MBusRelease();
		if (__predict_false(bd.IsBusErr())) {
			putlog(2, "MBus BusError %c:$%08x",
				(IsSuper() ? 'S' : 'U'), acc_laddr);
			SetFault(FAULT_CODE_BUSERR, paddr);
			return (uint64)-1;
		}
		return bd.Data();
	} else {
		data = CacheRead(acc_paddrH);
		if (__predict_false((int64)data < 0)) {
			putlog(2, "Cache BusError %c:$%08x",
				(IsSuper() ? 'S' : 'U'), acc_laddr);
			return data;
		}

		if (size == 1) {
			switch (acc_paddrL) {
			 case 0:
				return (data >> 24);
			 case 1:
				return (data >> 16) & 0xff;
			 case 2:
				return (data >>  8) & 0xff;
			 case 3:
				return data & 0xff;
			 default:
				PANIC("corrupted acc_paddrL=%u", acc_paddrL);
			}
		} else if (size == 2) {
			if (acc_paddrL == 0) {
				return data >> 16;
			} else {
				return data & 0xffff;
			}
		} else if (size == 4) {
			return data;
		} else {
			PANIC("bits must be 8, 16, 32");
		}
	}
}

uint64
m88200::load_1(uint32 addr)
{
	return load<1>(addr);
}

uint64
m88200::load_2(uint32 addr)
{
	return load<2>(addr);
}

uint64
m88200::load_4(uint32 addr)
{
	return load<4>(addr);
}

template <uint size> uint64
m88200::store(uint32 addr, uint32 data)
{
	acc_laddr = addr;
	acc_read  = false;
	if (__predict_false(Translate() == false)) {
		return (uint64)-1;
	}
	uint32 paddr = acc_paddrH + acc_paddrL;
	if (__predict_false((acc_stat & DESC_CI))) {
		parent->AddCycle(7);	// Table.6-2
		MBusAcquire();
		MBusMakeSnoop(acc_paddrH, 1);
		busdata bd = MBusWrite(paddr, data, size);
		MBusRelease();
		if (__predict_false(bd.IsBusErr())) {
			SetFault(FAULT_CODE_BUSERR, paddr);
			return (uint64)-1;
		}
		return 0;
	} else {
		return CacheWrite(paddr, data, size);
	}
}

uint64
m88200::store_1(uint32 addr, uint32 data)
{
	return store<1>(addr, data);
}

uint64
m88200::store_2(uint32 addr, uint32 data)
{
	return store<2>(addr, data);
}

uint64
m88200::store_4(uint32 addr, uint32 data)
{
	return store<4>(addr, data);
}

template <uint size> uint64
m88200::xmem(uint32 addr, uint32 data)
{
	acc_laddr = addr;
	acc_read  = false;
	if (__predict_false(Translate() == false)) {
		return (uint64)-1;
	}
	uint32 paddr = acc_paddrH + acc_paddrL;
	if (__predict_false((acc_stat & DESC_CI))) {
		// XXX キャッシュ禁止領域だとどうなる?
		MBusAcquire();
		MBusMakeSnoop(acc_paddrH, 1);
		busdata bd = MBusXmem(paddr, data, size);
		MBusRelease();
		if (__predict_false(bd.IsBusErr())) {
			SetFault(FAULT_CODE_BUSERR, paddr);
			return (uint64)-1;
		}
		return bd.Data();
	} else {
		return CacheXmem(paddr, data, size);
	}
}

uint64
m88200::xmem_1(uint32 addr, uint32 data)
{
	return xmem<1>(addr, data);
}

uint64
m88200::xmem_4(uint32 addr, uint32 data)
{
	return xmem<4>(addr, data);
}

//
// アドレス変換
//

// デバッグ用
/*static*/ std::string
m88200::stat2str(uint32 stat)
{
	std::string buf;

	if ((stat & DESC_WT))
		buf += ",WT";
	if ((stat & DESC_G))
		buf += ",G";
	if ((stat & DESC_CI))
		buf += ",CI";
	if ((stat & DESC_M))
		buf += ",M";
	if ((stat & DESC_U))
		buf += ",U";
	if ((stat & DESC_WP))
		buf += ",WP";

	if (buf.empty()) {
		buf = ",0";
	}
	return buf.substr(1);
}

// ATC ハッシュを表示。
void
m88200::DumpATC()
{
	uint8 zero[16] {};

	printf("DumpATC(#%u)\n", id);
	for (uint i = 0; i < atc_hash_all.size(); i += 16) {
		const uint8 *a = &atc_hash_all[i];
		if (memcmp(a, zero, 16) != 0) {
			printf("#%u %c.%08x:", id, ((i >> 20) ? 'S' : 'U'), i << 12);

			for (int j = 0; j < 16; j++) {
				uint8 n = a[j];
				if (__predict_true(n == 0)) {
					printf("   ");
				} else if (__predict_false((int8)n < 0)) {
					printf("~%u ", (uint)(uint8)~n);
				} else {
					printf("%-2u ", n - 1);
				}
			}
			printf("\n");
		}
	}
}

// アドレス変換。
// 事前に acc_laddr, acc_super, acc_read を設定しておくこと。
// 成功すれば acc_paddrH, acc_paddrL をセットして true を返す。
// 失敗ならいろいろセットして(?) false を返す。
// p2-3, Figure2-1
bool
m88200::Translate()
{
	uint32 la;
	uint8 n;

	// Logical address(LA) presented on PBus

	putlog(3, "Translate %c:$%08x", (IsSuper() ? 'S' : 'U'), acc_laddr);

	// 下位2ビットは常にこうなる。
	acc_paddrL = acc_laddr & 3;

	// Select Area Descriptor
	if (SelectAreaDesc() == false) {
		// 変換しない場合でも暗黙 BWP は有効…。
		// マッチまではここで独自に行い、マッチしたら下の BWP へ合流。
		if (__predict_false(acc_laddr >= 0xfff00000) && IsSuper()) {
			n = (acc_laddr < 0xfff80000) ? 8 : 9;
			goto batc;
		}

		// XXX 図のこれはたぶん間違いだよなあ…
		// 誤: Physical Address <= LA[18-2] :: 00
		// 正: Physical Address <= LA[31-2] :: 00
		acc_paddrH = acc_laddr - acc_paddrL;

		parent->AddCycle(1);
		return true;
	}

	// 変換する分だけカウントするのでいいか
	translate_total++;

	// type == Valid ここから

	// 論理アドレスで現在の S/U のハッシュを引く。
	// 実際にはまず BATC を引いて、なければ次に PATC を引くのだが、
	// BATC と PATC が同じアドレスを指すことはなく、BATC が同じところを
	// 指すような指定は (出来るけど結果不定とマニュアルに書いてあるので)
	// 起きないものとすると、アドレスと BATC もしくは PATC は 1:1 で
	// マッピング出来るので、この全域を一回で検索する。
	la = acc_laddr >> 12;
 try_again:
	n = atc_hash[la];
	acc_patc = NULL;

	if (n == 0) {
		// BATC, PATC どちらにも載ってない。
		atc_miss++;
		// patc_miss に合流する
	} else if ((int8)n < 0) {
		// BATC #(~n) でヒットした。
		// n=0xff が BATC#0 を、n=0xf6 が BATC#9 を示す。
		n = ~n;
 batc:
		m88200BATC& b = batc[n];
		assert(b.IsValid());
		putlog(4, " BATC[%u] hit", n);
		batc_hit[n]++;
		if (b.wp && acc_IsWrite()) {
			goto write_violation;
		}

		// Physical Address <= PBA :: LA[18-2] :: 00
		acc_paddrH = b.pba | (acc_laddr & 0x0007fffc);
		acc_stat |= b.stat;
		putlog(4, " BATC hit acc=%s paddr=$%08x",
			stat2str(acc_stat).c_str(), acc_paddrH);

		parent->AddCycle(1);
		return true;
	} else {
		// PATC #(n-1) でヒットした。
		// n=1 が PATC#0 を、n=56 が PATC#55 を示す。
		n--;
		m88200PATC& p = patc[n];
		assert(p.IsValid());
		patc_hit++;

		if (acc_IsWrite() && p.wp) {
			putlog(4, " PATC[%u] hit; stat=%s m=%u wp=%u", n,
				stat2str(p.stat).c_str(), p.m, p.wp);
			goto write_violation;
		}
		if (acc_IsWrite() && p.m == false) {
			// Update M bit
			// フローチャートでは PATC[M] はこの後のテーブルサーチ中
			// の Update Page Descriptor 処理中で更新するように書いて
			// あるように読めるのだが、その通りに実装すると、メモリ
			// 上の Page Descriptor に最初から M ビットが立っていると
			// (というか OpenBSD カーネルが用意した Page Descriptor
			// には立っているので) PATC[M] の更新も行われず、PATC[M]
			// が立っていないので再びここに来てしまい無限ループになる。
			// 誰が間違ってるのか分からないけど、とりあえずここで
			// PATC[M] を立てれば問題は起きない。
			putlog(4, " PATC[%u] hit; Need to update M bit", n);
			p.m = true;

			// この下の PATC miss に合流する
			acc_patc = &p;
			goto patc_miss;
		}

		// Physical Address <= PFA :: LA[11-2] :: 00
		acc_paddrH = p.pfa | (acc_laddr & 0x00000ffc);
		acc_stat |= p.stat;
		putlog(4, " PATC[%u] hit acc=%s paddr=$%08x", n,
			stat2str(acc_stat).c_str(), acc_paddrH);

		parent->AddCycle(1);
		return true;
	}

	// PATC miss
 patc_miss:
	putlog(4, " BATC/PATC miss");

	// PATC にヒットして M bit 更新するものは両方カウントされるけど
	atc_miss++;

	// Table search operation
	if (TableSearch() == true) {
		// テーブルサーチの結果、エントリが見付かって PATC が更新されたので
		// もう一度やり直す。
		// フローチャートでは Select Area Descriptor まで戻るように書いて
		// あるが、PATC ミスによるテーブルサーチでは Area Descriptor と BATC
		// は変化しないので、Area Descriptor の検索は省略。
		goto try_again;
	}

	// Invalid
	// Reply with fault (Figure2-10 だがここではすべて不要)
	return false;

 write_violation:
	// XXX fault_addr 不明
	SetFault(FAULT_CODE_WRITE, 0xcccccccc);
	// Reply with fault (Figure2-10 だがここではすべて不要)
	return false;
}

// Select Area Descriptor.
// 結果が Type=Valid なら true、Type=Untranslated なら false を返す。
// Probe Command からも呼ばれる (はずだが未実装)。
// p2-21, Figure2-5
bool
m88200::SelectAreaDesc()
{
	acc_stat = acc_apr->stat & ACC_STAT_MASK;
	if (acc_apr->enable) {
		// セグメントディスクリプタのアドレスはここで決まるが、
		// 次に行う BATC、PATC サーチでは使わず、それらが全部ミスして
		// テーブルサーチまで来た所で初めて使うので、ここでは表示しない。
		acc_sdaddr = acc_apr->addr | ((acc_laddr >> 20) & ~3U);
		putlog(4, " %s acc=%s", acc_apr->name, stat2str(acc_stat).c_str());
		return true;
	} else {
		putlog(3, " %s acc=%s Untranslated",
			acc_apr->name, stat2str(acc_stat).c_str());
		return false;
	}
}

// Table Search
// Type=Valid なら true、Type=Invalid なら false を返す。Type=Retry はない。
// p2-20, Figure2-4
bool
m88200::TableSearch()
{
	putlog(4, " %s $%08x/%s acc=%s", __func__, acc_laddr,
		(acc_read ? "Read" : "Write"), stat2str(acc_stat).c_str());

	// XXX フローチャートを厳密に追っかけると MBus の Acquire/Release が
	// 対応していないので(orz)、ここでは
	// TableSearch() に入ったところで Acquire、
	// TableSearch() から出るところで Release だけに統一する。

	MBusAcquire();

	if (FetchSegmentDesc() == false) {
		MBusRelease();

		parent->AddCycle(7);	// Table.6-2
		return false;
	}

	if (FetchPageDesc() == false) {
		MBusRelease();

		parent->AddCycle(11);	// Table.6-2
		return false;
	}

	// 中央の TYPE = VALID のところ
	//                       ( )
	//            +----------+ +----------+
	//            |                       |
	//      DESC[U]=0 ||              Otherwise
	//  DESC[M]=0 && WRITE                |
	//            |                       |
	//     UpdatePageDesc                 |
	//            |                       |
	// <- RETRY -( )-- INVALID ---------- | -->
	//            |                       |
	//          VALID                     |
	//            |                       |
	//           ( ) <--------------------+
	//           | |
	//       +---+ +------------+
	//   Otherwise    TableSearch due to PATC miss
	//       |                  |
	//  (Only U/M bits     CreatePATCEntry
	//   Updated)               |
	//       |                  v
	//       +---------------> ( )

	if ((tmp_desc & DESC_U) == 0 ||
	    ((tmp_desc & DESC_M) == 0 && acc_IsWrite()))
	{
		parent->AddCycle(15);	// Table.6-2

		if (UpdatePageDesc() == false) {
			MBusRelease();
			return false;
		}
	} else {
		parent->AddCycle(11);	// Table.6-2
	}

	if (acc_patc == NULL) {
		// Table search due to PATC miss
		CreatePATCEntry();
	}

	MBusRelease();
	return true;
}

// Fetch Segment Descriptor
// Type=Valid なら true、Type=Invalid なら false を返す。Type=Retry はない。
// 公式フローチャートと違って MBus Acquire された状態で呼び出すこと。
// p2-22, Figure2-6
bool
m88200::FetchSegmentDesc()
{
	busdata data;

	MBusMakeSnoop(acc_sdaddr, 0);
	data = MBusRead(acc_sdaddr, 4);
	if (__predict_false(data.IsBusErr())) {
		SetFault(FAULT_CODE_BUSERR, acc_sdaddr);
		return false;
	}
	tmp_desc = data.Data();
	tmp_desc &= (0xfffff000 |
		DESC_WT | DESC_SP | DESC_G | DESC_CI | DESC_WP | DESC_V);

	if (__predict_false((tmp_desc & DESC_V) == 0)) {
		SetFault(FAULT_CODE_SEGMENT, acc_sdaddr);
		putlog(4, "  SD $%08x desc=$%08x SegFault", acc_sdaddr, tmp_desc);
		return false;
	} else {
		// Descriptor is valid

		if (__predict_false((tmp_desc & DESC_SP) && IsUser())) {
			SetFault(FAULT_CODE_SUPERVISOR, acc_sdaddr);
			putlog(4, "  SD $%08x desc=$%08x SupervisorFault",
				acc_sdaddr, tmp_desc);
			return false;
		}

		acc_pdaddr = (tmp_desc & 0xfffff000) | ((acc_laddr >> 10) & 0xffc);
		acc_stat |= tmp_desc & ACC_STAT_MASK;
		putlog(4, "  SD $%08x pdaddr=$%08x stat=%s acc=%s",
			acc_sdaddr, acc_pdaddr,
			stat2str(tmp_desc).c_str(),
			stat2str(acc_stat).c_str());
		return true;
	}
}

// Fetch Page Descriptor
// Type=Valid なら true、Type=Invalid なら false を返す。Type=Retry はない。
// 公式フローチャートと違って MBus Acquire された状態で呼び出すこと。
// p2-23, Figure2-7
bool
m88200::FetchPageDesc()
{
	busdata data;

	MBusMakeSnoop(acc_pdaddr, 0);
	data = MBusRead(acc_pdaddr, 4);
	if (__predict_false(data.IsBusErr())) {
		SetFault(FAULT_CODE_BUSERR, acc_pdaddr);
		return false;
	}
	tmp_desc = data.Data();
	tmp_desc &= (0xfffff000 | DESC_WT | DESC_SP | DESC_G | DESC_CI |
		DESC_M | DESC_U | DESC_WP | DESC_V);

	if (__predict_false((tmp_desc & DESC_V) == 0)) {
		SetFault(FAULT_CODE_PAGE, acc_pdaddr);
		putlog(4, "  PD $%08x desc=$%08x PageFault", acc_pdaddr, tmp_desc);
		return false;
	} else {
		// Descriptor is valid

		if (__predict_false((tmp_desc & DESC_SP) && IsUser())) {
			SetFault(FAULT_CODE_SUPERVISOR, acc_pdaddr);
			putlog(4, "  PD $%08x desc=$%08x SupervisorFault",
				acc_pdaddr, tmp_desc);
			return false;
		}

		acc_stat |= tmp_desc & ACC_STAT_MASK;

		if (__predict_false((acc_stat & DESC_WP) && acc_IsWrite())) {
			// fault_addr は invalid data
			SetFault(FAULT_CODE_WRITE, 0xcccccccc);
			putlog(4, "  PD $%08x desc=$%08x WriteFault", acc_pdaddr, tmp_desc);
			return false;
		} else {
			putlog(4, "  PD $%08x stat=%s acc=%s", acc_pdaddr,
				stat2str(tmp_desc).c_str(),
				stat2str(acc_stat).c_str());
			return true;
		}
	}
}

// Update Page Descriptor
// Type=Valid なら true、Type=Invalid なら false を返す。Type=Retry はない。
// p2-24, Figure2-8
bool
m88200::UpdatePageDesc()
{
	busdata bd;

	if ((tmp_desc & DESC_M) == 0 && acc_IsWrite()) {
		// Update Modified bit and accrue status

		// XXX 図中 PATC[M] をセットするとあるが、ここでは遅い気がする。
		// Translate() 中の Update M bit のところのコメントも参照。

		// XXX 図中 ACC_STATUS[M] は TEMP_DESCR[M] じゃないの?
		tmp_desc |= DESC_M;
	}

	// Update Used bit
	tmp_desc |= DESC_U;

	MBusMakeSnoop(acc_pdaddr, 1);
	bd = MBusWrite(acc_pdaddr, tmp_desc, 4);
	if (__predict_false(bd.IsBusErr())) {
		SetFault(FAULT_CODE_BUSERR, acc_pdaddr);
		return false;
	}
	putlog(4, "  Update PD $%08x desc=$%08x stat=%s",
		acc_pdaddr, tmp_desc, stat2str(tmp_desc).c_str());

	return true;
}

// 指定の PATC を無効にする。
void
m88200::InvalidatePATCEntry(m88200PATC& p)
{
	uint32 la = addr2hash(p.IsS(), p.lpa);

	if (p.IsValid()) {
		uint8 n = atc_hash_all[la];
		// - BATC (n<0) なら、atc_hash は触らず PATC エントリだけ消す。
		//   この PATC が作成された後にここに BATC がセットされたとかなので。
		// - PATC (n>0) なら (自分のはずなので) atc_hash ごと消す。
		if ((int8)n > 0) {
			atc_hash_all[la] = 0;
		}
	}
	//DumpATC();

	p.lpa |= PATC_INVALID;
#if defined(M88200_STAT)
	stat_patc_invalidate++;
#endif
}

// Create PATC Entry
// 公式フローチャートと違って MBus Acquire したまま戻ること。
// p2-25, Figure2-9
void
m88200::CreatePATCEntry()
{
	// 空きがあろうがなかろうが、とにかく一番古いエントリを使う。
	// おそらく実機も空きエントリを前に詰めるような複雑な処理はして
	// ないだろうという推測。

#if defined(M88200_STAT)
	stat_patc_create++;
#endif
	m88200PATC& p = patc[patc_next];
	// 有効なら無効化
	if (p.IsValid()) {
		InvalidatePATCEntry(p);
	}

	// このエントリを潰して作る
	memset(&p, 0, sizeof(p));
	p.lpa = acc_laddr & 0xfffff000;
	p.pfa = tmp_desc & 0xfffff000;
	if (IsSuper()) {
		p.lpa |= PATC_S;
	}
	p.stat = acc_stat;
	putlog(4, "  %s %c:$%08x:$%08x stat=%s", __func__,
		p.IsS() ? 'S' : 'U', (p.lpa & 0xfffff000), p.pfa,
		stat2str(p.stat).c_str());

	// ハッシュの値は PATC#0 が 1。
	uint32 la = p.lpa >> 12;
	atc_hash[la] = patc_next + 1;
	//DumpATC();

	patc_next++;
	if (__predict_false(patc_next >= patc.size())) {
		patc_next = 0;
	}
}

// アドレス変換 (デバッガ用)
busaddr
m88200::TranslatePeek(busaddr addr_) const
{
	const m88200APR *xapr;
	uint32 laddr = addr_.Addr();
	bool issuper = addr_.IsSuper();
	uint32 la;

	// Select Area Descriptor
	if (issuper) {
		xapr = &sapr;
	} else {
		xapr = &uapr;
	}
	if (xapr->enable == 0) {
		return busaddr(laddr);
	}

	la = addr2hash(issuper, laddr);
	uint8 n = atc_hash_all[la];

	if (n == 0) {
		// FALLTHROUGH
	} else if ((int8)n < 0) {
		// BATC #(~n) hit
		n = ~n;
		const m88200BATC& b = batc[n];
		return busaddr(b.pba | (laddr & 0x0007ffff));
	} else {
		// BATC #(n-1) hit
		n--;
		const m88200PATC& p = patc[n];
		return busaddr(p.pfa | (laddr & 0x00000fff));
	}

	// ここからテーブルサーチ
	uint64 data;
	uint32 desc_addr;
	uint32 desc;

	// Peek Segment Descriptor
	desc_addr = xapr->addr | ((laddr >> 20) & ~3U);
	data = mainbus->Peek4(desc_addr);
	if ((int64)data < 0) {
		return BusAddr::BusErr | BusAddr::TableSearched;
	}
	desc = (uint32)data;
	if ((desc & DESC_V) == 0) {
		return BusAddr::BusErr | BusAddr::TableSearched;
	}
	if ((desc & DESC_SP) && issuper == false) {
		return BusAddr::BusErr | BusAddr::TableSearched;
	}

	// Peek Page Descriptor
	desc_addr = (desc & 0xfffff000) | ((laddr >> 10) & 0xffc);
	data = mainbus->Peek4(desc_addr);
	if ((int64)data < 0) {
		return BusAddr::BusErr | BusAddr::TableSearched;
	}
	desc = (uint32)data;
	if ((desc & DESC_V) == 0) {
		return BusAddr::BusErr | BusAddr::TableSearched;
	}
	if ((desc & DESC_SP) && issuper == false) {
		return BusAddr::BusErr | BusAddr::TableSearched;
	}

	busaddr r((desc & 0xfffff000) | (laddr & 0x00000fff));
	r |= BusAddr::TableSearched;
	return r;
}


//
// キャッシュ
//

// このラインの状態を更新
void
m88200CacheSet::Update(uint line, m88200CacheSet::Status status)
{
	vv[line] = status;
	if (__predict_false(status == IV)) {
		// 無効なら
		tag[line] |= TAG_INVALID;
		L = TryUnuseLine(L, line);
	}
}

// CSSP L5-L0 の処理。
// 引数 tmpL の状態から line を最新にしたらどうなるか、を返す。
// 表示処理で一時変数に対して処理が必要なため分離してある。
/*static*/ int
m88200CacheSet::TryUseLine(int tmpL, int line)
{
	// L フィールドは 3bit, 2bit, 1bit で構成され、
	// 他のラインの対応ビットを落として、
	// 自分のラインの対応ビットを全部立てれば、
	// 他のラインの順序を保存した状態で、自分が最新になるように
	// 順序をつけられる。

	if (line == 3) {
		tmpL |= 0b111'00'0;
	} else if (line == 2) {
		tmpL &= 0b011'11'1;
		tmpL |= 0b000'11'0;
	} else if (line == 1) {
		tmpL &= 0b101'01'1;
		tmpL |= 0b000'00'1;
	} else {
		tmpL &= 0b110'10'0;
	}
	return tmpL;
}

// CSSP L5-L0 の処理。
// 引数 tmpL の状態から line を最古にしたらどうなるか、を返す。
/*static*/ int
m88200CacheSet::TryUnuseLine(int tmpL, int line)
{
	// Use のちょうど反転論理
	if (line == 3) {
		tmpL &= 0b000'11'1;
	} else if (line == 2) {
		tmpL |= 0b100'00'0;
		tmpL &= 0b111'00'1;
	} else if (line == 1) {
		tmpL |= 0b010'10'0;
		tmpL &= 0b111'11'0;
	} else {
		tmpL |= 0b001'01'1;
	}
	return tmpL;
}

// CSSP L5-L0 の処理。
// 引数 tmpL の状態で、最新の line を返す。
// 表示処理で一時変数に対して処理が必要なため分離してある。
/*static*/ int
m88200CacheSet::TryGetOldestLine(int tmpL)
{
	if (tmpL < 8) {
		return 3;
	}
	tmpL &= 7;
	if (tmpL < 2) {
		return 2;
	}
	tmpL &= 1;
	if (tmpL == 0) {
		return 1;
	} else {
		return 0;
	}
}

// 更新用に最も古いラインを選んで差し出す
int
m88200CacheSet::SelectOldestLine() const
{
	// なければ、最も古いラインを差し出す
	return TryGetOldestLine(L);
}

// キャッシュの指定の line, word に data を書き込む。
// size は 1, 2, 4 バイト。
void
m88200CacheSet::Write(uint line, uint32 paddr, uint32 data, uint size)
{
	uint32 wordidx = (paddr >> 2) & 0x03;
	uint32 data32;

	if (__predict_true(size == 4)) {
		word[line * 4 + wordidx] = data;
		return;
	}

	data32 = word[line * 4 + wordidx];
	if (size == 2) {
		if ((paddr & 2) == 0) {
			data32 = (data32 & 0x0000ffff) | (data << 16);
		} else {
			data32 = (data32 & 0xffff0000) | data;
		}
	} else {
		switch (paddr & 3) {
		 case 0:
			data32 = (data32 & 0x00ffffff) | (data << 24);
			break;
		 case 1:
			data32 = (data32 & 0xff00ffff) | (data << 16);
			break;
		 case 2:
			data32 = (data32 & 0xffff00ff) | (data << 8);
			break;
		 case 3:
			data32 = (data32 & 0xffffff00) | data;
			break;
		 default:
			__unreachable();
		}
	}
	word[line * 4 + wordidx] = data32;
}

// キャッシュを検索。
// ヒットすれば line 番号(0..3) を返す。ヒットしなければ -1 を返す。
int
m88200CacheSet::Lookup(uint32 tagaddr) const
{
	for (int line = 0; line < 4; line++) {
		if (tag[line] == tagaddr) {
			return line;
		}
	}
	return -1;
}


// キャッシュの指定の set, line にメモリから 1 ライン(4word) 読み込む。
// 成功すれば 0 を返す。
// バスエラーなら fault_code, fault_addr をセットし (uint64)-1 を返す。
uint64
m88200::ReadLine(m88200CacheSet& set, uint line, uint32 tagaddr)
{
	// タグを更新
	set.tag[line] = tagaddr;

	busaddr addr = busaddr(set.tag[line] | (set.setidx << 4)) | acc_super;
	busdata r = mainbus->ReadBurst16(addr, &set.word[line * 4]);
	if (__predict_true(r.IsBusErr() == false)) {
		parent->AddWait(r.GetWait());
		return 0;
	} else {
		addr |= BusAddr::Size4;
		for (int i = 0; i < 4; i++) {
			busdata bd = mainbus->Read(addr);
			parent->AddWait(r.GetWait());
			if (__predict_false(bd.IsBusErr())) {
				SetFault(FAULT_CODE_BUSERR, addr.Addr());
				return (uint64)-1;
			}
			set.word[line * 4 + i] = bd.Data();
			addr += 4;
		}
	}
	return 0;
}

// キャッシュの指定の set, line の1ライン(4word) を書き出す (コピーバック)。
// 成功すれば 0 を返す。
// バスエラーなら fault_code, fault_addr をセットし (uint64)-1 を返す。
uint64
m88200::CopyBackLine(m88200CacheSet& set, uint line)
{
	busaddr addr = busaddr(set.tag[line] | (set.setidx << 4)) | acc_super;
	busdata r = mainbus->WriteBurst16(addr, &set.word[line * 4]);
	if (__predict_true(r.IsBusErr() == false)) {
		parent->AddWait(r.GetWait());
		return 0;
	} else {
		addr |= BusAddr::Size4;
		for (int i = 0; i < 4; i++) {
			busdata bd = mainbus->Write(addr, set.word[line * 4 + i]);
			if (__predict_false(bd.IsBusErr())) {
				SetFault(FAULT_CODE_BUSERR, addr.Addr());
				return (uint64)-1;
			}
			addr += 4;
		}
	}
	return 0;
}

// キャッシュに対して paddr の読み込みを行う。
// paddr は 32bit 境界のアドレスであること。
// 読み込めれば該当の32bitワードを返す。
// 何らかエラーが起きれば (uint64)-1 を返す。エラー要因は fault_code とか参照。
// p.3-8 Figure 3-3
uint64
m88200::CacheRead(uint32 paddr)
{
	int line;
	uint64 rv;

	assert((paddr & 3) == 0);

	// タグとセット番号
	uint32 tagaddr = (paddr & 0xfffff000);
	uint32 setidx  = (paddr >> 4) & 0xff;
	uint32 wordidx = (paddr >> 2) & 0x03;

	m88200CacheSet& set = setarray[setidx];
	putlog(3, "CacheRead paddr=$%08x (set=$%02x)", paddr, setidx);

	line = set.Lookup(tagaddr);
	if (__predict_true(line >= 0)) {
		// Cache Hit
		putlog(4, " CacheRead hit (line=%u,word=%u)", line, wordidx);
		parent->AddCycle(1);
		goto success;
	}

	// Cache Miss

	parent->AddCycle(10);	// Table.6-2

	// reply <- Wait
	MBusAcquire();

	// Select cache line for replacement
	line = set.SelectOldestLine();
	putlog(4, " CacheRead miss (replace line=%u)", line);

	if (set.vv[line] == m88200CacheSet::EM) {
		parent->AddCycle(7);	// Table.6-2
		rv = CopyBackLine(set, line);
		if ((int64)rv < 0) {
			goto error;
		}
	}

	// Mark cache line invalid
	set.Update(line, m88200CacheSet::IV);

	// Read line from memory
	MBusMakeSnoop(paddr, 0);
	rv = ReadLine(set, line, tagaddr);
	if (__predict_false((int64)rv < 0)) {
		goto error;
	}

	MBusRelease();

	// Update cache line
	set.Update(line, m88200CacheSet::SU);

	putlog(4, " CacheRead updated (line=%u,word=%u)", line, wordidx);
 success:
	set.Use(line);
	return set.word[line * 4 + wordidx];

 error:
	MBusRelease();
	return (uint64)-1;
}

// キャッシュに対して paddr への data の書き込みを行う。size は 1, 2, 4 バイト。
// paddr は size に応じた境界にあること。
// 書き込めれば 0、エラーが起きれば (uint64)-1 を返す。
// エラー要因は fault_code とか参照。
// p.3-10 Figure 3-5
uint64
m88200::CacheWrite(uint32 paddr, uint32 data, uint size)
{
	int line;
	busdata bd;
	uint64 rv;

	assert((paddr & (size - 1)) == 0);

	// タグとセット番号
	uint32 tagaddr = (paddr & 0xfffff000);
	uint32 setidx  = (paddr >> 4) & 0xff;

	m88200CacheSet& set = setarray[setidx];
	putlog(3, "CacheWrite paddr=$%08x (set=$%02x)", paddr, setidx);

	line = set.Lookup(tagaddr);
	if (__predict_true(line >= 0)) {
		// Cache Hit
		parent->AddCycle(1);
		return CacheWriteHit(set, line, paddr, data, size);
	}

	// Cache Miss

	parent->AddCycle(14);	// Table.6-2

	// reply <- Wait
	MBusAcquire();

	// Select cache line for replacement
	line = set.SelectOldestLine();
	putlog(4, " CacheWrite miss (replace line=%u)", line);

	if (set.vv[line] == m88200CacheSet::EM) {
		parent->AddCycle(7);	// Table.6-2
		rv = CopyBackLine(set, line);
		if ((int64)rv < 0) {
			goto error;
		}
	}

	// Mark line invalid
	set.Update(line, m88200CacheSet::IV);

	// Read line with intent to modify
	MBusMakeSnoop(paddr, 1);
	rv = ReadLine(set, line, tagaddr);
	if ((int64)rv < 0) {
		goto error;
	}

	// Write data to memory
	bd = MBusWrite(paddr, data, size);
	if (__predict_false(bd.IsBusErr())) {
		SetFault(FAULT_CODE_BUSERR, paddr);
		goto error;
	}

	// Write data into cache
	set.Write(line, paddr, data, size);

	// Mark line exclusive unmodified
	// p3-9 3.4.2.2 CACHE WRITE:CACHE HIT. の本文に writethrough のときは
	// It marks the line as shared unmodified... と書いてある。
	// Fig 3-5 には記載はないが、キャッシュエントリが作られるときも
	// Fig 3-6 のキャッシュヒット時と同じように SU にならないと矛盾する。
	if ((acc_stat & DESC_WT)) {
		set.Update(line, m88200CacheSet::SU);
	} else {
		set.Update(line, m88200CacheSet::EU);
	}
	set.Use(line);

	MBusRelease();
	return 0;

 error:
	MBusRelease();
	return (uint64)-1;
}

// キャッシュがヒットした場合。
// 書き込めれば 0、エラーが起きれば (uint64)-1 を返す。
// p3-11 Figure 3-6
uint64
m88200::CacheWriteHit(m88200CacheSet& set, uint line,
	uint32 paddr, uint32 data, uint size)
{
	if (__predict_false(set.vv[line] == m88200CacheSet::SU)) {
		// Line Shared Unmodified の場合

		putlog(4, " CacheWrite hit shared unmodified (line=%u)", line);

		// WriteThrough と Global は最後の状態変化だけが違う
		if ((acc_stat & (DESC_WT | DESC_G))) {
			// reply = Wait;
			MBusAcquire();

			// Write data to cache
			set.Write(line, paddr, data, size);

			// Write data to memory
			MBusMakeSnoop(paddr, 1);
			busdata bd = MBusWrite(paddr, data, size);
			if (__predict_false(bd.IsBusErr())) {
				MBusRelease();
				SetFault(FAULT_CODE_BUSERR, paddr);
				return (uint64)-1;
			}

			if ((acc_stat & DESC_WT)) {
				// Mark line shared unmodified
				set.Update(line, m88200CacheSet::SU);
			} else {
				// Mark line exclusive unmodified
				set.Update(line, m88200CacheSet::EU);
			}
			set.Use(line);

			MBusRelease();
			return 0;
		}

		// どちらでもない場合は Line Exclusive と同じ処理に落ちる
	} else {
		// Line Exclusive の場合
		putlog(4, " CacheWrite hit exclusive (line=%u)", line);
	}

	// Write data into cache
	putlog(4, " CacheWrite line=%u $%08x sz=%u", line, paddr, size);
	set.Write(line, paddr, data, size);

	// Mark line exclusive modified
	set.Update(line, m88200CacheSet::EM);
	set.Use(line);
	return 0;
}

// paddr への xmem を行う。size は 1, 2, 4 バイト。
// 成功すれば読み出した値、エラーが起きれば (uint64)-1 を返す。
// p3-13, Figure3-7
uint64
m88200::CacheXmem(uint32 paddr, uint32 data, uint size)
{
	int line;
	busdata bd;
	uint64 rv;

	assert((paddr & (size - 1)) == 0);

	// タグとセット番号
	uint32 tagaddr = (paddr & 0xfffff000);
	uint32 setidx  = (paddr >> 4) & 0xff;

	m88200CacheSet& set = setarray[setidx];
	putlog(3, "CacheXmem paddr=$%08x (set=$%02x)", paddr, setidx);

	line = set.Lookup(tagaddr);
	if (__predict_false(line >= 0 && set.vv[line] == m88200CacheSet::EM)) {
		// Cache Hit (and Line exclusive modified)
		putlog(4, " CacheXmem hit and EM (line=%u)", line);

		MBusAcquire();

		parent->AddCycle(7);	// Table.6-2
		MBusMakeSnoop(paddr, 1);
		rv = CopyBackLine(set, line);
		if (__predict_false((int64)rv < 0)) {
			goto error;
		}

		set.Update(line, m88200CacheSet::IV);
	} else {
		// Cache Miss or (Cache Hit but other than exclusive modified)

		if (line >= 0) {
			// Cache Hit and Otherwise (= !EM)
			putlog(4, " CacheXmem hit but !EM (line=%u)", line);
			set.Update(line, m88200CacheSet::IV);
		} else {
			putlog(4, " CacheXmem miss");
		}

		MBusAcquire();
	}

	MBusMakeSnoop(paddr, 1);
	bd = MBusXmem(paddr, data, size);
	MBusRelease();
	if (__predict_false(bd.IsBusErr())) {
		SetFault(FAULT_CODE_BUSERR, paddr);
		return (uint64)-1;
	}
	return bd.Data();

 error:
	MBusRelease();
	return rv;
}

// MBus 使用権を取得する
void
m88200::MBusAcquire()
{
	// アービトレーションのたびに1クロックかかる(?)
	// よく分からんけど、とりあえず、CMMU は一度所有権を持ったら放さない、
	// 別の CMMU がバスリクエストすると所有権はその CMMU に移る、とする。
	if (__predict_false(mbus_master != this)) {
		mbus_master = this;
		parent->AddCycle(1);
	}
}

// 他 CMMU にバススヌープさせるポイント。
// 本来はバスマスタ CMMU が MBus にアドレスと IM を出すと、それを監視して
// いる他 CMMU が必要に応じて反応するのだが、ここでは全部マスタ主導で行う。
// Figure 3-8, 3-9
void
m88200::MBusMakeSnoop(uint32 paddr, uint im)
{
	if ((acc_stat & DESC_G) == 0) {
		return;
	}

	// MBus Status <- Wait (2 clocks)
	// 本来はスヌープをするスレーブ側が 2 クロックかかるのだが、
	// 個別の CMMU がクロックを持っていない実装なのでマスタ側で
	// クロックを消費したことにする。
	parent->AddCycle(2);

	// 本来はここで解放ではなく、スレーブ側が反応したい時にバスリクエストを
	// 出してそれによって手放すのだが、その機構はないので、こちらが自主的に
	// 一旦手放したようにしておく。とは言っても現状 MBusRelease() はダミーで
	// 実際には他の誰かが MBusAcquire() を呼ぶまで握りっぱなしなので、
	// あまり問題ないはず。
	MBusRelease();
	for (const auto other : other_cmmu) {
		other->Snoop(paddr, im);
	}
	MBusAcquire();
}

// バススヌープ (スレーブ側)。
// 本来はバスマスタでない CMMU は、バスマスタが MBus に出すアドレス情報を
// 監視し、必要ならそれに反応するのだが、ここでは全部マスタ主導で行う。
// こっちはそのマスタから呼ばれるスレーブ側。
// Figure 3-8, 3-9
void
m88200::Snoop(uint32 paddr, uint im)
{
	assert((sctr & SCTR_SE));

	uint32 tagaddr = (paddr & 0xfffff000);
	uint32 setidx  = (paddr >> 4) & 0xff;

	m88200CacheSet& set = setarray[setidx];
	int line = set.Lookup(tagaddr);

	if (line >= 0) {
		// Cache hit
		if (set.vv[line] == m88200CacheSet::EM) {
			// Line exclusive modified

			// Assert retry

			// Request MBus
			// 実際にはここで MBus 使用権を要求して獲得するのだが
			// その機構はなくその代わり Acquire を呼ぶだけでいける。
			MBusAcquire();
			uint64 rv = CopyBackLine(set, line);
			MBusRelease();
			if ((int64)rv < 0) {
				// Set CE bit in system status register
				return;
			}

			if (im == 0) {
				// Mark line shared unmodified (if IM=0)
				set.Update(line, m88200CacheSet::SU);
			} else {
				// Mark line invalid (if IM=1)
				set.Update(line, m88200CacheSet::IV);
			}
		}
	}
	// Resume servicing PBus
}
