// vi:set ts=4:

//
// nono
// Copyright (C) 2024 nono project
// Licensed under nono-license.txt
//

#ifndef runx_h
#define runx_h

#include <err.h>
#include <errno.h>
#include <unistd.h>
#include <stdio.h>
#include <sys/ptrace.h>
#include <machine/reg.h>

#define countof(x)	(sizeof(x) / sizeof((x)[0]))

#define DEBUG(lv, fmt...)	do {	\
	if (opt_debug >= (lv)) {	\
		printf("\x1b[33m");	\
		printf(fmt);	\
		printf("\x1b[m");	\
		printf("\n");	\
	}	\
} while (0)

#define TRACE(pc, fmt...)	do {	\
	if (opt_trace) {	\
		printf("\x1b[33m");	\
		printf("%06x: ", pc);	\
		printf(fmt);	\
		printf("\x1b[m");	\
		printf("\n");	\
	}	\
} while (0)

// struct reg *reg は固定。
#define RegR(n)	(reg->r_regs[(n)])
#define RegD(n)	(reg->r_regs[(n)])
#define RegA(n)	(reg->r_regs[8 + (n)])
#define RegPC	(reg->r_pc)
#define RegCCR	(reg->r_sr)

typedef uint8_t		uint8;
typedef uint16_t	uint16;
typedef uint32_t	uint32;
typedef int8_t		int8;
typedef int16_t		int16;
typedef int32_t		int32;

struct reg_emul {
	uint16 sr;		// SR の上位バイトのみ保持。下位(CCR) は持たない。
	uint32 usp;
	uint32 ssp;
	uint32 vbr;
	uint32 caar;
};
// SR+CCR を返す
#define GetSR	(emul.sr | (RegCCR & 0x00ff))
// SR+CCR をセットする
#define SetSR(v_)	do {	\
	uint32 v__ = (v_);	\
	emul.sr = v__ & 0xff00;	\
	RegCCR  = v__ & 0x00ff;	\
} while (0)

/* runx.cpp */
extern uint32 readmem1(uint32);
extern uint32 readmem2(uint32);
extern uint32 readmem4(uint32);
extern void writemem1(uint32, uint32);
extern void writemem2(uint32, uint32);
extern void writemem4(uint32, uint32);

/* doscall.cpp */
extern int  init_doscall();
extern int  doscall(pid_t, uint32, struct reg *);

/* iocscall.cpp */
extern int  iocscall(pid_t, struct reg *);

/* m680x0.cpp */
extern uint32 exception_format0(struct reg *, uint32, uint32);
extern uint32 exception_format2(struct reg *, uint32);
extern uint32 exec_op(pid_t, struct reg *, int, uint32);

/* util.cpp */
extern int  getpgsize();
extern const char *signame(int);

extern int opt_debug;
extern int opt_trace;
extern struct reg_emul emul;

#endif /* !runx_h */
