//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// イベント
//

#pragma once

#include "nono.h"

// ユーザ定義リテラルを使ったサフィックス

// ナノ秒
static inline constexpr uint64 operator"" _nsec(unsigned long long ns)
{
	return ns;
}
static inline constexpr uint64 operator"" _nsec(long double ns)
{
	return ns;
}

// マイクロ秒
static inline constexpr uint64 operator"" _usec(unsigned long long us)
{
	return us * 1000_nsec;
}
static inline constexpr uint64 operator"" _usec(long double us)
{
	return us * 1000_nsec;
}

// ミリ秒
static inline constexpr uint64 operator"" _msec(unsigned long long ms)
{
	return ms * 1000_usec;
}
static inline constexpr uint64 operator"" _msec(long double ms)
{
	return ms * 1000_usec;
}

// 秒
static inline constexpr uint64 operator"" _sec(unsigned long long sec)
{
	return sec * 1000_msec;
}
static inline constexpr uint64 operator"" _sec(long double sec)
{
	return sec * 1000_msec;
}

// コールバック関数の型
class Device;
class Event;
using EventCallback_t = void (Device::*)(Event&);
#define ToEventCallback(f) static_cast<EventCallback_t>(f)


// イベント
//
// イベントは必要なデバイスごとに用意する。おそらく最初から個数が決まって
// いることが多いはずなので、デバイスクラス内のメンバ変数(実体) として
// 宣言し、デバイスのコンストラクタ等で初期化するという流れになるはず。
//
// VM リセットによるイベントタイマーの停止処理は各デバイスの責任で行うこと。
class Event
{
 public:
	Event();
	explicit Event(Device *dev_);
	~Event();

	// このイベントが動作中なら true を返す
	bool IsRunning() const {
		return active;
	}

	// 以下はイベントを呼び出す側がセットする

	// イベントを起こすまでの仮想時刻での相対時間 [nsec]
	// 例えば今から 1msec 後にイベントを起こしたい場合は 1000000 となる。
	uint64 time {};

	Device *dev {};				// デバイス
	EventCallback_t func {};	// コールバック関数
	int code {};				// コールバック関数への引数
	const std::string GetName() const { return name; }
	void SetName(const std::string& val);

	// スケジューラが内部で使う
	bool active {};				// スケジューラにセットされていれば真
	uint64 vtime {};			// イベントを起こす仮想時刻

	uint64 count {};			// コールバックを呼び出した回数
	uint64 last_count {};		// (表示用)前回表示したときの count

 private:
	// イベント名(表示用)
	// '\0' 含まず25文字まで。デバイス名も必要なら含めること。
	std::string name {};
};
