//
// nono
// Copyright (C) 2023 nono project
// Licensed under nono-license.txt
//

//
// 拡張エリアセット
//

// 拡張エリアセットの範囲は $eaff80 〜 $eaffff までの 128バイトで、
// 他は (現状デバイスがないので) バスエラーになる。
// X68kIODevice の構造の都合上、$eae000 〜 $eaffff までの 8KB のアクセスが
// すべてここに来るので、BusIO を使わず自前でさばく必要がある。
// (本来は MuxDevice で分離すべき)

#include "extarea.h"
#include "x68kio.h"

// コンストラクタ
ExtAreaDevice::ExtAreaDevice()
	: inherited(OBJ_EXTAREA)
{
}

// デストラクタ
ExtAreaDevice::~ExtAreaDevice()
{
}

// 初期化
bool
ExtAreaDevice::Init()
{
	x68kio = GetX68kIODevice();

	return true;
}

// リセット
void
ExtAreaDevice::ResetHard(bool poweron)
{
	// リセット時はすべて 0 らしい。(X68030 Inside/Out)
	for (int i = 0; i < 5; i++) {
		x68kio->SetExtarea(i, 0x00);
	}
}

busdata
ExtAreaDevice::Read(busaddr addr)
{
	uint32 reqsize = addr.GetSize();
	uint32 paddr = addr.Addr();
	uint32 offset = paddr & 1;
	uint32 datasize = std::min(2 - offset, reqsize);

	if (paddr >= 0xeaff80) {
		// 拡張エリアセット部分は書き込み専用なので読み出しはすべてバスエラー。
		putlog(2, "Read $%06x.%c (BusErr)",
			paddr, (datasize == 1 ? 'B' : 'W'));
	} else {
		// それ以外の部分も何もないのでバスエラー。
	}

	busdata data = BusData::BusErr;
	// XXX wait
	data |= busdata::Size(datasize);
	return data;
}

busdata
ExtAreaDevice::Write(busaddr addr, uint32 data)
{
	busdata r;

	uint32 reqsize = addr.GetSize();
	uint32 paddr = addr.Addr();
	uint32 offset = paddr & 1;
	uint32 datasize = std::min(2 - offset, reqsize);

	if (paddr >= 0xeaff80) {
		// 拡張エリアセット内は16バイト(8ワード)単位でミラー。
		// 偶数アドレスはバスエラー。
		if (datasize == 1 && offset == 0) {
			r = BusData::BusErr;
		} else {
			uint32 regno = (paddr >> 1) & 7;
			if (regno < 6) {
				putlog(1, "$%06x [%u] <- $%02x", paddr, regno, data & 0xff);
				x68kio->SetExtarea(regno, data);
			} else {
				r = BusData::BusErr;
			}
		}
		if (r.IsBusErr()) {
			putlog(2, "Write $%06x.%c (BusErr)",
				paddr, (datasize == 1 ? 'B' : 'W'));
		}
	} else {
		// それ以外の部分も何もないのでバスエラー。
		r = BusData::BusErr;
	}

	r |= busdata::Size(datasize);
	return data;
}

busdata
ExtAreaDevice::Peek1(uint32 addr)
{
	return BusData::BusErr;
}
