//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// GVRAM
//

// GVRAM はワード単位でホストバイトオーダ配置なので、
// バイトアクセス時は HB() マクロを使用のこと。

#include "gvram.h"
#include "videoctlr.h"

// InsideOut p.135
static const busdata wait = busdata::Wait(9 * 40_nsec);

// コンストラクタ
GVRAMDevice::GVRAMDevice()
	: inherited(OBJ_GVRAM)
{
}

// デストラクタ
GVRAMDevice::~GVRAMDevice()
{
}

// 初期化
bool
GVRAMDevice::Init()
{
	uint devlen = 0x20'0000;	// XXX とりあえず
	mem.reset(new(std::nothrow) uint8[devlen]);
	if ((bool)mem == false) {
		warnx("Cannot allocate %u bytes at %s", devlen, __method__);
		return false;
	}

	return true;
}

inline uint32
GVRAMDevice::Decoder(uint32 addr) const
{
	return addr - baseaddr;
}

busdata
GVRAMDevice::Read(busaddr addr)
{
	uint32 offset = Decoder(addr.Addr());
	busdata data;

	data = *(uint16 *)&mem[offset & ~1U];
	data |= wait;
	data |= BusData::Size2;
	return data;
}

busdata
GVRAMDevice::Write(busaddr addr, uint32 data)
{
	uint32 offset = Decoder(addr.Addr());
	uint32 reqsize = addr.GetSize();
	uint32 datasize = std::min(2 - (offset & 1U), reqsize);
	data >>= (reqsize - datasize) * 8;

	if (datasize == 1) {
		mem[HB(offset)] = data;
	} else {
		*(uint16 *)&mem[offset] = data;
	}

	busdata r = wait;
	r |= busdata::Size(datasize);
	return r;
}

busdata
GVRAMDevice::Peek1(uint32 addr)
{
	uint32 offset = Decoder(addr);
	return mem[HB(offset)];
}
