//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// LUNA のキーボードとマウス
//

#pragma once

#include "keyboard.h"
#include "event.h"

class SIODevice;

class LunaKeyboard : public Keyboard
{
	using inherited = Keyboard;
 public:
	LunaKeyboard();
	~LunaKeyboard() override;

	bool Init() override;

	// シフト状態なら true を返す
	bool IsShift() const override;

	// マウス入力
	void MouseInput(int x, int y, bool rb, bool lb, bool mb) override;

	// マウスデータを VM に送信する MD 処理
	void MouseSendStart() override;

	// ホストからの制御コマンド
	void Command(uint32 data) override;

 private:
	// キーボードの接続と取り外し
	void MDConnect() override;
	void MDDisconnect() override;

	// MD キーコードに変換
	uint KeyToMDKey(uint keystat) const override;

	// キーを1つ処理する
	void KeyInput(uint keystat) override;

	// キーイベントを起こせる空きスロットを計算して、イベントを開始
	void SendStart() override;

	// イベントを開始
	void Start(uint64 period);

	// コールバック
	void Callback(Event& ev);

	// 共通キーコードから LED 番号を取得。
	// 0: かな
	// 1: CAPS
	int Keycode2LED(uint keycode) const;
	// LED 番号から共通キーコードを取得。
	uint LED2Keycode(int ledcode) const;

	// マウスサンプリング
	void MouseSampling();
	void MouseOn();
	void MouseOff();

	DECLARE_MONITOR_CALLBACK(MonitorUpdate);

	// マウスサンプリング
	bool mouse_on {};

	// ブザー
	uint buzzer_msec {};		// コマンドで指定した発音時間 [msec]
	uint buzzer_freq {};		// コマンドで指定した発音周波数 [Hz]
	uint64 buzzer_end {};		// ブザーを鳴らし終わる仮想時刻

	// イベント
	Event event { this };

	Monitor *monitor {};

	SIODevice *sio {};

	// 変換テーブル
	static const uint keycode2lunakey_table[KC_max];

	// LUNA キーコード
	static const uint NoKey					= 0;
	static const uint LunaKey_TAB			= 0x09;
	static const uint LunaKey_CTRL			= 0x0a;
	static const uint LunaKey_kana			= 0x0b;
	static const uint LunaKey_SHIFT_L		= 0x0c;
	static const uint LunaKey_SHIFT_R		= 0x0d;
	static const uint LunaKey_CAPS			= 0x0e;
	static const uint LunaKey_SF			= 0x0f;

	static const uint LunaKey_ESC			= 0x10;
	static const uint LunaKey_BS			= 0x11;
	static const uint LunaKey_enter			= 0x12;
	static const uint LunaKey_space			= 0x14;
	static const uint LunaKey_DEL			= 0x15;
	static const uint LunaKey_XFER			= 0x16;
	static const uint LunaKey_VALID			= 0x17;
	static const uint LunaKey_PF11			= 0x18;
	static const uint LunaKey_PF12			= 0x19;
	static const uint LunaKey_PF13			= 0x1a;
	static const uint LunaKey_PF14			= 0x1b;
	static const uint LunaKey_up			= 0x1c;
	static const uint LunaKey_left			= 0x1d;
	static const uint LunaKey_right			= 0x1e;
	static const uint LunaKey_down			= 0x1f;

	static const uint LunaKey_1				= 0x22;
	static const uint LunaKey_2				= 0x23;
	static const uint LunaKey_3				= 0x24;
	static const uint LunaKey_4				= 0x25;
	static const uint LunaKey_5				= 0x26;
	static const uint LunaKey_6				= 0x27;
	static const uint LunaKey_7				= 0x28;
	static const uint LunaKey_8				= 0x29;
	static const uint LunaKey_9				= 0x2a;
	static const uint LunaKey_0				= 0x2b;
	static const uint LunaKey_minus			= 0x2c;
	static const uint LunaKey_circum		= 0x2d;
	static const uint LunaKey_backslash		= 0x2e;
	static const uint LunaKey_buzzer		= 0x2f;

	static const uint LunaKey_Q				= 0x32;
	static const uint LunaKey_W				= 0x33;
	static const uint LunaKey_E				= 0x34;
	static const uint LunaKey_R				= 0x35;
	static const uint LunaKey_T				= 0x36;
	static const uint LunaKey_Y				= 0x37;
	static const uint LunaKey_U				= 0x38;
	static const uint LunaKey_I				= 0x39;
	static const uint LunaKey_O				= 0x3a;
	static const uint LunaKey_P				= 0x3b;
	static const uint LunaKey_at			= 0x3c;
	static const uint LunaKey_bracketleft	= 0x3d;

	static const uint LunaKey_A				= 0x42;
	static const uint LunaKey_S				= 0x43;
	static const uint LunaKey_D				= 0x44;
	static const uint LunaKey_F				= 0x45;
	static const uint LunaKey_G				= 0x46;
	static const uint LunaKey_H				= 0x47;
	static const uint LunaKey_J				= 0x48;
	static const uint LunaKey_K				= 0x49;
	static const uint LunaKey_L				= 0x4a;
	static const uint LunaKey_semicolon		= 0x4b;
	static const uint LunaKey_colon			= 0x4c;
	static const uint LunaKey_bracketright	= 0x4d;

	static const uint LunaKey_Z				= 0x52;
	static const uint LunaKey_X				= 0x53;
	static const uint LunaKey_C				= 0x54;
	static const uint LunaKey_V				= 0x55;
	static const uint LunaKey_B				= 0x56;
	static const uint LunaKey_N				= 0x57;
	static const uint LunaKey_M				= 0x58;
	static const uint LunaKey_comma			= 0x59;
	static const uint LunaKey_period		= 0x5a;
	static const uint LunaKey_slash			= 0x5b;
	static const uint LunaKey_underscore	= 0x5c;

	static const uint LunaKey_PAD_plus		= 0x61;
	static const uint LunaKey_PAD_minus		= 0x62;
	static const uint LunaKey_PAD_7			= 0x63;
	static const uint LunaKey_PAD_8			= 0x64;
	static const uint LunaKey_PAD_9			= 0x65;
	static const uint LunaKey_PAD_4			= 0x66;
	static const uint LunaKey_PAD_5			= 0x67;
	static const uint LunaKey_PAD_6			= 0x68;
	static const uint LunaKey_PAD_1			= 0x69;
	static const uint LunaKey_PAD_2			= 0x6a;
	static const uint LunaKey_PAD_3			= 0x6b;
	static const uint LunaKey_PAD_0			= 0x6c;
	static const uint LunaKey_PAD_decimal	= 0x6d;
	static const uint LunaKey_PAD_enter		= 0x6e;

	static const uint LunaKey_F1			= 0x72;
	static const uint LunaKey_F2			= 0x73;
	static const uint LunaKey_F3			= 0x74;
	static const uint LunaKey_F4			= 0x75;
	static const uint LunaKey_F5			= 0x76;
	static const uint LunaKey_F6			= 0x77;
	static const uint LunaKey_F7			= 0x78;
	static const uint LunaKey_F8			= 0x79;
	static const uint LunaKey_F9			= 0x7a;
	static const uint LunaKey_F10			= 0x7b;
	static const uint LunaKey_PAD_multiply	= 0x7c;
	static const uint LunaKey_PAD_divide	= 0x7d;
	static const uint LunaKey_PAD_equal		= 0x7e;
	static const uint LunaKey_PAD_comma		= 0x7f;
};
