//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// LUNA* の RAM のうち 16MB に満たないメモリブロックを担当する人
//

// devtable[] は1エントリで 16MB の空間を担当するが LUNA-I のメインメモリは
// 4MB、8MB、12MB と 16MB 未満の構成があり得る。また 16MB 以上にメモリを
// 増設する場合でも ROM の容量チェックが末尾に NopIODevice の番兵を必要と
// するため、ここを 16MB 単位での増減にしていると、最大搭載量は 256MB では
// なく 240MB になってしまう。中途半端な上に分かりづらいので 255MB まで
// 使いたい。そうするとこのデバイスは単に任意のアドレスから 16MB に満たない
// 1ブロックを担当する人であればよくて、それを devtable[] のどこに割り当てる
// かは VM::Init() 側が決めればよい。
//
// なおこのデバイス自身は RAM を持たず、バックエンドの gMainRAM
// (と NopIODevice) への切り替えスイッチなだけ (X68kIO のような立場)。
// また、X68030 の場合は X68kIO がすでに同様の処理を (8KB 単位で) 行って
// いるのと、そもそも RAM がないところは NopIO ではなく BusError という違いも
// あるのでここでは担当しない。

#include "partialram.h"
#include "mainram.h"

// コンストラクタ
PartialRAMDevice::PartialRAMDevice()
	: inherited(OBJ_PARTIAL_RAM)
{
	// こいつはプロキシだし今の所ログも出力しないので、指定できる必要ない。
	ClearAlias();
}

// デストラクタ
PartialRAMDevice::~PartialRAMDevice()
{
}

// 初期化
bool
PartialRAMDevice::Init()
{
	ramdev = GetMainRAMDevice();
	nopdev = GetNopIODevice();

	// SearchDevice() で使うための RAM サイズをここで取得する。
	// ただしこのデバイスがどこに割り当てられるか (或いは割り当てられないか)
	// はこの後の VM_LUNA*::Init() で判断するため、ここではチェックなどは
	// 行わない。
	ram_size = (uint32)ramdev->GetSize();

	return true;
}

inline IODevice *
PartialRAMDevice::SearchDevice(uint32 addr) const
{
	if (__predict_true(addr < ram_size)) {
		return ramdev;
	} else {
		// メモリのないところをアクセスした時のウェイトは無視
		return nopdev;
	}
}

busdata
PartialRAMDevice::Read(busaddr addr)
{
	IODevice *d = SearchDevice(addr.Addr());
	return d->Read(addr);
}

busdata
PartialRAMDevice::Write(busaddr addr, uint32 data)
{
	IODevice *d = SearchDevice(addr.Addr());
	return d->Write(addr, data);
}

busdata
PartialRAMDevice::Peek1(uint32 addr)
{
	IODevice *d = SearchDevice(addr);
	return d->Peek1(addr);
}

bool
PartialRAMDevice::Poke1(uint32 addr, uint32 data)
{
	IODevice *d = SearchDevice(addr);
	return d->Poke1(addr, data);
}
