//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// テキストスクリーンを描画するパネル
//

// wxPanel
//  ↓
// WXTextPanel  テキストフォントを描画するクラス
//  ↓
// WXTextScreen テキストスクリーンを表示するパネル

#pragma once

#include "wxtextpanel.h"
#include "geometric.h"

class WXTextScreen : public WXTextPanel
{
	using inherited = WXTextPanel;

 public:
	// テキストスクリーンは本来単純なキャラクタビットマップ構造のため、
	// クライアント領域の端の1ピクセルまでふんだんに使用している。
	// ウィンドウマネージャ側でこのウィンドウに枠などをつけてくれる場合は
	// あまり問題にならないが、枠などを一切つけないタイプのウィンドウ
	// マネージャでこのウィンドウを表示するとウィンドウの縁に接する文字の
	// 視認性が(文字とウィンドウ外の背景の組み合わせによっては)すこぶる
	// 悪くなる。そのため、テキストスクリーンの周囲に余白をもたせる。
	static const int DefaultPadding = 3;

 public:
	WXTextScreen(wxWindow *parent, const nnSize& screensize);
	~WXTextScreen() override;

	// フォントサイズ変更
	void FontChanged() override;

	// テキストスクリーンを取得 (WXTextScreen を継承せずに更新する人のため)
	TextScreen& GetScreen() { return screen; }

	// パディングサイズ変更
	void SetPadding(int l, int t, int r, int b);
	// パディングサイズ取得
	int GetPaddingLeft() const   { return pad_left; }
	int GetPaddingTop() const    { return pad_top; }
	int GetPaddingRight() const  { return pad_right; }
	int GetPaddingBottom() const { return pad_bottom; }

	// 桁数、行数を取得する
	int GetCol() const { return screen.GetCol(); }
	int GetRow() const { return screen.GetRow(); }

	// 全域を表示するのに必要なピクセルサイズを返す。
	wxSize GetDesiredSize() const;

	// クライアント座標 pos をテキスト座標(桁、行)に変換して返す
	// (パディング領域では戻り値が負数になることに注意)
	wxPoint GetTextPosition(const wxPoint& pos) const;

	// TextScreen のユーザデータ取得、設定
	uint64 GetUserData() const { return screen.userdata; }
	void SetUserData(uint64 value) { screen.userdata = value; }

	// コンテキストメニューイベントを解除
	// (ここのコンテキストメニューを使いたくない人が呼ぶ)
	void DisconnectContextMenu();

 protected:
	TextScreen screen {};
	std::vector<uint16> prevbuf {};

 private:
	void OnSize(wxSizeEvent&);
	void OnContextMenu(wxContextMenuEvent&);
	void OnCopy(wxCommandEvent&);

	void Init(int col, int row);
	void DoSize();
	void InvalidateText();
	void Draw() override;

	// コンテキストメニューを接続
	void ConnectContextMenu();

	int pad_left {};
	int pad_top {};
	int pad_right {};
	int pad_bottom {};

	wxDECLARE_EVENT_TABLE();
};
