//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// Bt454/458
//

#pragma once

#include "device.h"
#include "color.h"

class LunafbDevice;

class BT45xDevice : public IODevice
{
	using inherited = IODevice;

 public:
	BT45xDevice();
	~BT45xDevice() override;

	bool Init() override;
	void ResetHard(bool poweron) override;

	// 設定からプレーン数を取得
	static int GetConfigPlaneCount();

	// プレーン数を取得
	uint GetPlaneCount() const { return nplane; }

	// ホストパレットのアドレスを取得
	const std::vector<Color>& GetHostPalette() const { return hostcolor; }

	// 生パレットを uint32 形式にしたものを返す (GUI 用)
	const std::vector<uint32> GetIntPalette() const;

	// BusIO インタフェース
	// (内蔵 ROM からのアクセス用に特別に public にしてある)
	busdata WritePort(uint32 offset, uint32 data);
 protected:
	static const uint32 NPORT = 4;
	busdata ReadPort(uint32 offset);
	busdata PeekPort(uint32 offset);
	bool PokePort(uint32 offset, uint32 data);

	// レジスタ値からホストパレットを作成
	void MakeHostColor(uint idx);

	// パレット n の色を uint32 表現したものを返す (GUI 用)
	virtual uint32 PaletteToInt(uint n) const = 0;

	// 通知
	void Notify();

	uint32 PeekAddress() const;
	uint32 PeekPalette() const;
	void WriteAddress(uint32 data);
	void WritePalette(uint32 data);
	virtual uint32 ReadControl() = 0;
	virtual void WriteControl(uint32 data) = 0;
	virtual uint32 PeekControl() const = 0;

	// モデル種別。Bt454 なら true。
	bool bt454 {};

	// ビットマップのプレーン数 (1, 4, 8)
	uint nplane {};

	// パレット数 (16 or 256)
	uint palettes {};

	// データの下位ニブルマスク。Bt454 なら LOW_NIBBLE。
	uint8 low_nibble {};

	// Bt458 のパレットアドレスマスク。(Bt454 なら使用しない)
	uint8 readmask {};

	// idx は次にアクセスされるべき color[] へのインデックス。
	// idx = 0 ならパレット0 の R。
	// idx = 4 ならパレット1 の G。
	// Bt454 では R/G/B をカウントするための 3値 2ビットのカウンタ(?)を
	// ADDRa,ADDRb (ADDRa,b) と呼んだりしているようなので、ここではその
	// modulo 3 を数えるカウンタを ab と呼ぶことにする。つまり
	// idx = 0 はパレット 0、ab 0。
	// idx = 4 はパレット 1、ab 1。
	uint idx {};

	// パレットは R, G, B の順に格納されている。
	// color[0] = パレット0 R
	// color[1] = パレット0 G
	// color[2] = パレット0 B
	// color[3] = パレット1 R
	// :
	// color[47] = パレット15 B
	//
	// Bt454 なら 16パレット * 3 = 48エントリ。
	// 色深度は 4ビットで上位4ビットと下位4ビットに同じものを格納する。
	// つまり $4 なら $44、$f なら $ff として格納しておく。
	//
	// Bt458 なら 256パレット * 3 = 768エントリ。
	// 色深度は 8ビットでそのまま格納。
	std::vector<uint8> color {};

	// ホスト用に変換したパレットデータ。
	// ホスト用なので各8ビット、といいつつ元情報 (color[]) が8ビットすべて
	// 有効なので値はそのまま流用する。
	//
	// また 1bpp モードなら R,G,B のうち G だけが使われるので、ここで
	// 展開する。パレットのうち実際には #14 と #15 だけが使われるが、
	// それについてはレンダラ側で処理している。
	// see https://twitter.com/tsutsuii/status/356704682514714625
	std::vector<Color> hostcolor {};

	Monitor *monitor {};

	// ログ出力用
	static const char rgbstr[3];

	LunafbDevice *lunafb {};
};

// Bt454
class BT454Device : public BT45xDevice
{
	using inherited = BT45xDevice;

	static const uint32 LOW_NIBBLE = 0x0f;
 public:
	BT454Device();
	~BT454Device() override;

 private:
	DECLARE_MONITOR_CALLBACK(MonitorUpdate);

	uint32 ReadControl() override;
	void WriteControl(uint32 data) override;
	uint32 PeekControl() const override;

	uint32 PaletteToInt(uint) const override;
};

// Bt458
class BT458Device : public BT45xDevice
{
	using inherited = BT45xDevice;
 public:
	BT458Device();
	~BT458Device() override;

	bool Init() override;

 private:
	DECLARE_MONITOR_CALLBACK(MonitorUpdate);

	uint32 ReadControl() override;
	void WriteControl(uint32 data) override;
	uint32 PeekControl() const override;

	uint32 PaletteToInt(uint) const override;

	void WriteRMask(uint32 data);
	void WriteBMask(uint32 data);
	void WriteCommand(uint32 data);
	void WriteTest(uint32 data);

	uint8 blinkmask {};		// ブリンクマスク
	uint8 command {};		// コマンド
	uint8 testreg {};		// テストレジスタ
};

static inline BT45xDevice *GetBT45xDevice() {
	return Object::GetObject<BT45xDevice>(OBJ_BT45x);
}
