//
// nono
// Copyright (C) 2022 nono project
// Licensed under nono-license.txt
//

//
// プレーン VRAM (垂直 VRAM)
// Lunafb と X68k TVRAM の共通部分
//

#pragma once

#include "device.h"
#include "bitmap.h"
#include <mutex>
#include <vector>

class PlaneVRAMDevice : public IODevice
{
	using inherited = IODevice;

	// 更新情報。
	struct ModifyInfo
	{
		// 1ライン単位の更新フラグ。
		// 仮想 Y 座標に対応するので Lunafb も TVRAM も 1024個。
		std::vector<bool> line {};

		// パレットを更新した場合やスクロールした場合のように、
		// composite に変更がなくても bitmap を更新する必要がある場合は true。
		bool invalidate2 {};
	};

 public:
	PlaneVRAMDevice(int width_, int height_);
	~PlaneVRAMDevice() override;

	bool Init() override;
	void ResetHard(bool poweron) override;

	// プレーン数を取得
	uint GetPlaneCount() const { return nplane; }

	bool Snap();

	// 画面を合成する
	bool Render(BitmapRGBX& dst);

	// VRAM には更新がないが再レンダリングが必要。(パレット変更、スクロール等)
	void Invalidate2();

	// 合成ビットマップを取得する
	const BitmapI8& GetComposite() const { return composite; }

	// 座標位置の情報を出力 (GUI 用)
	virtual void UpdateInfo(TextScreen&, int x, int y) const = 0;

 protected:
	// 全画面の更新フラグを立てる
	void Invalidate();

	// 垂直 VRAM から BitmapI8 バッファを作成する
	void RenderVRAMToComposite(const std::vector<bool>& modified);

	// BitmapI8 バッファから BitmapRGBX を作成する。
	void RenderCompositeToRGBX(BitmapRGBX& dst,
		const ModifyInfo& modified);

	// X 方向にはみ出したときに対応する仮想画面の Y 座標を返す。
	virtual int GetWrapY(int) const = 0;

	// ビットマッププレーン。
	// メモリを確保する時のサイズ計算のため型を uint8 にしているが、
	// 運用はロングワードでホストバイトオーダー。
	// Lunafb では 1枚 256KBで 1 or 4枚。TVRAM では 512KB。
	std::unique_ptr<uint8[]> mem {};

	// プレーン数 (TVRAM なら 4 固定、LUNA は設定による)
	uint nplane {};

	// テキスト画面の合成ビットマップ
	BitmapI8 composite {};

	// VM スレッド内での仮想画面の更新フラグ。
	ModifyInfo dirty {};

	// レンダラスレッドで未処理の行。
	// VM スレッドからレンダラスレッドへの連絡用なので mtx で保護する。
	ModifyInfo pending {};
	std::mutex mtx {};

	// レンダリング時に使う変換テーブル
	std::unique_ptr<uint64[]> deptable {};

	// ホストパレット
	const Color *palette {};

	// スクロール
	int xscroll {};
	int yscroll {};
};

static inline PlaneVRAMDevice *GetPlaneVRAMDevice() {
	return Object::GetObject<PlaneVRAMDevice>(OBJ_PLANEVRAM);
}
