//
// nono
// Copyright (C) 2022 nono project
// Licensed under nono-license.txt
//

//
// ADPCM (MSM6258V)
//

#include "adpcm.h"
#include "dmac.h"
#include "event.h"
#include "mpu.h"
#include "pio.h"
#include "scheduler.h"

// コンストラクタ
ADPCMDevice::ADPCMDevice()
	: inherited(OBJ_ADPCM)
{
}

// デストラクタ
ADPCMDevice::~ADPCMDevice()
{
}

// 初期化
bool
ADPCMDevice::Init()
{
	dmac = GetDMACDevice();
	ppi = GetPPIDevice();

	auto evman = GetEventManager();
	event = evman->Regist(this,
		ToEventCallback(&ADPCMDevice::EventCallback),
		"ADPCM");

	return true;
}

// リセット
void
ADPCMDevice::ResetHard(bool poweron)
{
	playing = false;
	dack = false;

	// ?
	clk = 8000'000;
	div = 512;
}

busdata
ADPCMDevice::ReadPort(uint32 offset)
{
	busdata data;

	switch (offset) {
	 case 0:	// STAT
		data = 0x40;
		if (playing) {
			data |= STAT_PLAY;
		}
		putlog(1, "STAT -> $%02x", data.Data());
		break;
	 case 1:	// DATA
		putlog(0, "Read $%06x (NOT IMPLEMENTED)", mpu->GetPaddr());
		data = 0xff;
		break;
	 default:
		VMPANIC("corrupted offset=%u", offset);
	}

	// InsideOut p.135
	const busdata wait = busdata::Wait(18 * 40_nsec);
	data |= wait;

	data |= BusData::Size1;
	return data;
}

busdata
ADPCMDevice::WritePort(uint32 offset, uint32 data)
{
	switch (offset) {
	 case 0:	// CMD
		putlog(1, "CMD  <- $%02x", data);
		if ((data & REC_START)) {
			putlog(0, "Start recording (NOT IMPLEMENTED)");
		}
		if ((data & PLAY_START)) {
			putlog(0, "Start playing (NOT IMPLEMENTED)");
			StartPlay();
		}
		if ((data & STOP)) {
			putlog(0, "Stop (NOT IMPLEMENTED)");
			playing = false;
		}
		break;
	 case 1:	// DATA
		// 未実装だけど再生中の書き込みは来るのは分かっているので抑えておく
		if ((playing && loglevel >= 2) || playing == false) {
			putlogn("DATA <- $%02x (NOT IMPLEMENTED)", data);
		}
		break;
	 default:
		VMPANIC("corrupted offset=%u", offset);
	}

	// InsideOut p.135
	const busdata wait = busdata::Wait(22 * 40_nsec);

	busdata r = wait;
	r |= BusData::Size1;
	return r;
}

busdata
ADPCMDevice::PeekPort(uint32 offset)
{
	// XXX 未実装
	return 0xff;
}

bool
ADPCMDevice::PokePort(uint32 offset, uint32 data)
{
	return false;
}

// 再生開始
void
ADPCMDevice::StartPlay()
{
	// PPI で設定されているサンプリングレートをここで読み出す。
	// XXX パンは未対応
	uint rate = ppi->GetADPCMRate();
	switch (rate) {
	 case 0:
		div = 1024;
		break;
	 case 1:
	 case 3:		// %11 は %01 と同じになる (Inside p295)
		div = 768;
		break;
	 case 2:
		div = 512;
		break;
	 default:
		VMPANIC("corrupted ADPCM rate=%u", rate);
	}

	playing = true;
	dmac->AssertREQ(3);

	event->time = div / clk;
	scheduler->StartEvent(event);
}

// イベント
void
ADPCMDevice::EventCallback(Event *ev)
{
	// XXX 適当
	if (playing) {
		dmac->AssertREQ(3);
	}
}

// DACK 信号をアサートする (DMAC から呼ばれる)
void
ADPCMDevice::AssertDACK(bool tc_)
{
	dack = true;
	// DRQ を下げる
	dmac->NegateREQ(0);
}

// DACK 信号をネゲートする (DMAC から呼ばれる)
void
ADPCMDevice::NegateDACK()
{
	dack = false;
}

// クロック切り替え。OPM から呼ばれる。
// CT1 = %0 で 8MHz、%1 で 4MHz。
void
ADPCMDevice::SetClock(bool ct1)
{
	if (ct1) {
		clk = 4000'000;
	} else {
		clk = 8000'000;
	}
}
