//
// nono
// Copyright (C) 2024 nono project
// Licensed under nono-license.txt
//

//
// シリアルポートのメイン画面コンソール入出力ドライバ
//

#include "comdriver_cons.h"
#include "console.h"
#include "hostcom.h"

// コンストラクタ
COMDriverConsole::COMDriverConsole(HostDevice *hostdev_)
	: inherited(hostdev_, "console")
{
}

// デストラクタ
COMDriverConsole::~COMDriverConsole()
{
	if (rxpipe.Valid()) {
		hostdev->DelOuter(rxpipe);
	}

	if (console) {
		console->Detach();
	}
	auto keyboard = gMainApp.FindObject<ConsoleKeyboard>(OBJ_KEYBOARD);
	if (keyboard) {
		keyboard->Detach();
	}
}

// ドライバ初期化
bool
COMDriverConsole::InitDriver(bool startup)
{
	int fds[2];
	if (pipe(fds) < 0) {
		errmsg = string_format("pipe: %s", strerror(errno));
		putmsg(1, "%s", errmsg.c_str());
		return false;
	}
	rxpipe = fds[0];
	txpipe = fds[1];

	if (hostdev->AddOuter(rxpipe) < 0) {
		errmsg = string_format("AddOuter: %s", strerror(errno));
		putmsg(1, "%s", errmsg.c_str());
		return false;
	}

	console = GetConsoleDevice();
	console->Attach(this);

	auto keyboard = gMainApp.GetObject<ConsoleKeyboard>(OBJ_KEYBOARD);
	keyboard->Attach(this);

	return true;
}

// 外部への1バイト書き出し
void
COMDriverConsole::Write(uint32 ch)
{
	console->Putchar(ch);
}

// 外部からの1バイト読み込み。
// HostCOM スレッドから呼ばれる。
int
COMDriverConsole::Read()
{
	uint8 buf[1];
	int r;

	r = read(rxpipe, buf, sizeof(buf));
	if (r < 0) {
		putmsg(0, "read: %s", strerror(errno));
		return NODATA;
	}
	if (r == 0) {
		return NODATA;
	}
	return buf[0];
}

// VM からのキー入力。
// ConsoleKeyboard から呼ばれる。この文字をパイプに書き込んで帰る。
void
COMDriverConsole::EnqueueChar(uint charcode)
{
	char buf[1];
	int r;

	buf[0] = charcode;
	r = write(txpipe, buf, sizeof(buf));
	if (r < 0) {
		putmsg(0, "write: %s", strerror(errno));
		return;
	}
	if (r == 0) {
		putmsg(0, "write: short");
		return;
	}
}
