package waf

import "testing"

func TestWAFDetection(t *testing.T) {
	detector := NewWafDetector()
	if detector == nil {
		t.Fatal("expected non-nil wafDetector")
	}

	tests := []struct {
		name        string
		content     string
		expectedWAF string
		shouldMatch bool
	}{
		{
			name:        "Cloudflare WAF",
			content:     "Attention Required! | Cloudflare",
			expectedWAF: "cloudflare",
			shouldMatch: true,
		},
		{
			name:        "ModSecurity WAF",
			content:     "This error was generated by Mod_Security",
			expectedWAF: "modsecurity",
			shouldMatch: true,
		},
		{
			name:        "No WAF",
			content:     "Regular response with no WAF signature",
			expectedWAF: "",
			shouldMatch: false,
		},
		{
			name:        "Wordfence WAF",
			content:     "Generated by Wordfence",
			expectedWAF: "wordfence",
			shouldMatch: true,
		},
		{
			name:        "Sucuri WAF",
			content:     "Access Denied - Sucuri Website Firewall",
			expectedWAF: "sucuri",
			shouldMatch: true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			waf, matched := detector.DetectWAF(tt.content)
			if matched != tt.shouldMatch {
				t.Errorf("expected match=%v, got match=%v", tt.shouldMatch, matched)
			}
			if matched && waf != tt.expectedWAF {
				t.Errorf("expected WAF=%s, got WAF=%s", tt.expectedWAF, waf)
			}
		})
	}
}
