/* pdp11_xq_bootrom.h: DEQNA/DELQA bootrom data
  ------------------------------------------------------------------------------

   Copyright (c) 2003-2008, David T. Hittner

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

   Except as contained in this notice, the name of the author shall not be
   used in advertising or otherwise to promote the sale, use or other dealings
   in this Software without prior written authorization from the author.

  ------------------------------------------------------------------------------

  Modification history:

  26-Mar-03  DTH  Removed 'static' declaration
  23-Mar-03  DTH  Created by extracting from merged DEQNA bootrom dumps

  ------------------------------------------------------------------------------
*/

#ifndef PDP11_XQ_BOOTROM_H
#define PDP11_XQ_BOOTROM_H

/*
  Word 0: NOP
  Word 1: Branch to extended primary boot (EPB)
  Word 2: Branch/Vector to Citizenship tests
  Word 3: Offset from beginning to checksum word

  See INIQNA.MAC for further information on format and contents.

  Boot ROM code is available for each of the respective boards
  which are being simulated (DEQNA, DELQA and DELQA-T)
*/

/*
  Bootrom code is from executing a Boot Diagnostic load command to a real
  DEQNA (M7504) board (thus loading the ROM contents to RAM) and using ODT
  to display it.
*/

uint16 xq_bootrom_deqna[2048] = {
    0x00A0,0x0114,0x016B,0x0DFE,0x0155,0x0000,0x8000,0x8000,  /* 0000 (0000000) */
    0x0400,0xFC00,0x0000,0x0000,0x8000,0x8000,0x0C00,0xFC00,  /* 0010 (0000020) */
    0x0000,0x0000,0x8000,0x2000,0xC000,0x8000,0x15C6,0x03FE,  /* 0020 (0000040) */
    0x1026,0x15F1,0x000C,0x0004,0x0A31,0x0006,0x15F1,0x0208,  /* 0030 (0000060) */
    0x000E,0x0A00,0x0AC0,0x02FE,0x0A31,0x000E,0x0A00,0x0AC0,  /* 0040 (0000100) */
    0x02FE,0x15F1,0x0002,0x000E,0x0A31,0x000E,0x45F7,0x3FFF,  /* 0050 (0000120) */
    0xFFB2,0x2DF7,0xFFAE,0xFFC0,0x0233,0x45F7,0x3FFF,0xFFA8,  /* 0060 (0000140) */
    0x2DF7,0xFFA4,0xFFB2,0x022C,0x45F7,0x3FFF,0xFFA6,0x2DF7,  /* 0070 (0000160) */
    0xFFA2,0xFFA6,0x0225,0x15C4,0x0FFE,0x0A03,0x0A02,0x9C80,  /* 0080 (0000200) */
    0x0400,0x0A82,0x45C0,0xFF00,0x6003,0x209F,0x0006,0x0201,  /* 0090 (0000220) */
    0x0BD2,0x0AC4,0x02F4,0x17C0,0x0006,0x2C03,0x0400,0x0210,  /* 00A0 (0000240) */
    0x005F,0x0408,0x15C2,0x1400,0x09F7,0x0020,0x0BC0,0x0208,  /* 00B0 (0000260) */
    0x1382,0x0202,0x0077,0x08EE,0x25C2,0x01FF,0x8601,0x004A,  /* 00C0 (0000300) */
    0x17C3,0x000A,0x0301,0x004B,0x0000,0x01FE,0x65C1,0x000E,  /* 00D0 (0000320) */
    0x09F7,0x004C,0x55C9,0x0002,0x0A09,0x0977,0x071E,0xFFC0,  /* 00E0 (0000340) */
    0x020C,0x0BCA,0x020C,0x35F2,0x0400,0x0028,0x0302,0x55C9,  /* 00F0 (0000360) */
    0x0400,0x0977,0x0706,0xFF92,0x0302,0x55CA,0x0040,0x15C9,  /* 0100 (0000400) */
    0x0002,0x0A09,0x1240,0x45C0,0x6800,0x25C0,0x1030,0x0302,  /* 0110 (0000420) */
    0x55CA,0x0040,0x1280,0xE5C1,0x000E,0x0087,0x5500,0xFFAA,  /* 0120 (0000440) */
    0x11C3,0x15C0,0x00E0,0x65C3,0x0066,0x10DF,0x0004,0x101F,  /* 0130 (0000460) */
    0x0006,0x119F,0x01F6,0x105F,0x01FA,0x109F,0x01F8,0x65C3,  /* 0140 (0000500) */
    0x000A,0x10DF,0x0014,0x101F,0x0016,0x65C3,0x0012,0x15F1,  /* 0150 (0000520) */
    0x01FC,0xFFFE,0x10DF,0x01FC,0x101F,0x01FE,0x0A32,0x0002,  /* 0160 (0000540) */
    0x0A0A,0x15C0,0x0072,0x6080,0x15C4,0x004A,0x0A10,0x0AC4,  /* 0170 (0000560) */
    0x02FD,0x0977,0x0686,0xFFB6,0x0318,0x0AB2,0x0002,0x25F2,  /* 0180 (0000600) */
    0x0002,0x0002,0x06ED,0x0087,0x55CA,0x0400,0x65C6,0x0004,  /* 0190 (0000620) */
    0x0087,0x17C6,0x01F6,0x17C1,0x01FA,0x17C2,0x01F8,0x55CA,  /* 01A0 (0000640) */
    0x2000,0x0087,0x55CA,0x0800,0x0002,0x8D17,0x0000,0x1043,  /* 01B0 (0000660) */
    0xE5C3,0x000E,0x1084,0x65C4,0x000A,0x15C5,0x0006,0x14C0,  /* 01C0 (0000700) */
    0x9014,0x0AC5,0x02FC,0x1084,0x65C4,0x000A,0x1105,0x0BD4,  /* 01D0 (0000720) */
    0x0204,0x0BD4,0x0202,0x0BCC,0x0311,0x1144,0x15C0,0xFFFF,  /* 01E0 (0000740) */
    0x2014,0x020E,0x2014,0x020C,0x200C,0x0308,0x0209,0x1144,  /* 01F0 (0000760) */
    0x25D4,0x00AA,0x0203,0xA5CC,0x0004,0x8602,0x55CA,0x0001,  /* 0200 (0001000) */
    0x15C0,0xFFFF,0x09F7,0x06D6,0x15F4,0xC03F,0x0002,0x15F4,  /* 0210 (0001020) */
    0xE000,0x0004,0x17C5,0x01FC,0x11C3,0x65C3,0x0034,0x10DF,  /* 0220 (0001040) */
    0x01FC,0x55C9,0x0040,0x1131,0xFFFA,0x0A00,0x1031,0xFFFC,  /* 0230 (0001060) */
    0x15C3,0x0008,0x0AC3,0x02FE,0x1240,0x45C0,0x7C00,0x25C0,  /* 0240 (0001100) */
    0x00F0,0x0202,0x0077,0x0022,0x55CA,0x0800,0x0087,0x65C6,  /* 0250 (0001120) */
    0x0004,0x115F,0x01FC,0x8D17,0x0000,0x0A80,0x03F5,0x1240,  /* 0260 (0001140) */
    0x45C0,0x7C00,0x25C0,0x00F4,0x02EF,0x1300,0x45C0,0x3FFF,  /* 0270 (0001160) */
    0x25C0,0xC000,0x02E9,0x0BF4,0x0008,0x02E6,0x0BF4,0x000A,  /* 0280 (0001200) */
    0x02E3,0x15C9,0x8480,0x11C3,0x65C3,0xFE94,0x10F2,0x0002,  /* 0290 (0001220) */
    0x10F2,0x0006,0x65C3,0x0005,0x10F2,0x0004,0x0A32,0x0008,  /* 02A0 (0001240) */
    0x1083,0x65C3,0x0072,0x15C0,0x0002,0x15C5,0x0006,0x8A13,  /* 02B0 (0001260) */
    0x15C4,0x0007,0x2CB2,0x0004,0x0006,0x0603,0x1CB2,0x0002,  /* 02C0 (0001300) */
    0x0006,0x9E93,0x0006,0x0AB2,0x0006,0x0AC4,0x02F3,0x0AC5,  /* 02D0 (0001320) */
    0x02EE,0x15C4,0x0010,0x8A13,0x0AC4,0x02FD,0x0AC0,0x02E5,  /* 02E0 (0001340) */
    0x0977,0x0518,0xFFC5,0x0303,0x55CA,0x0002,0x0087,0x0AB2,  /* 02F0 (0001360) */
    0x0008,0x25F2,0x01F4,0x0008,0x06D3,0x0A32,0x0008,0x1083,  /* 0300 (0001400) */
    0x65C3,0x0072,0x10C4,0x10F2,0x0002,0x15C0,0x0201,0x8A13,  /* 0310 (0001420) */
    0x0AC0,0x02FD,0x95E3,0x00AB,0x1103,0x0A83,0x0A00,0x0A04,  /* 0320 (0001440) */
    0x0A05,0x8CC0,0x0201,0x0A80,0x9013,0x0A84,0x25C4,0x0007,  /* 0330 (0001460) */
    0x0603,0x0A04,0x65C3,0x0039,0x0A85,0x35C5,0x0007,0x0207,  /* 0340 (0001500) */
    0x65C3,0x0008,0x25C5,0x0030,0x0202,0xE5C3,0x0030,0x25C5,  /* 0350 (0001520) */
    0x0038,0x06E7,0x0115,0x1083,0x65C3,0x0072,0x10F2,0x0002,  /* 0360 (0001540) */
    0x15C5,0x0008,0x15C4,0x0006,0x8BD3,0x15C0,0x0007,0x8A53,  /* 0370 (0001560) */
    0x0AC0,0x02FD,0x0AC4,0x02F8,0x65C3,0x0010,0x0AC5,0x02F2,  /* 0380 (0001600) */
    0x0A32,0x0006,0x1C80,0x0002,0x0977,0x0476,0xFFC0,0x022E,  /* 0390 (0001620) */
    0x1C83,0x0002,0x65C3,0x0001,0x10F2,0x0004,0x1C83,0x0004,  /* 03A0 (0001640) */
    0x1084,0x65C4,0x08F2,0x15C0,0x0006,0x92CC,0x8A54,0x65C3,  /* 03B0 (0001660) */
    0x0008,0x0AC0,0x02FA,0x0A14,0x0A14,0x15C9,0x0001,0x15C0,  /* 03C0 (0001700) */
    0xFFFD,0x09F7,0x04FC,0x1131,0xFFF6,0x0A31,0xFFF8,0x09F7,  /* 03D0 (0001720) */
    0x050C,0x1131,0xFFFA,0x0A31,0xFFFC,0x15C0,0x01FF,0x35C9,  /* 03E0 (0001740) */
    0x8000,0x0207,0x0AC0,0x02FB,0x55CA,0x0200,0x55CA,0x0004,  /* 03F0 (0001760) */
    0x0087,0x0A00,0x0977,0x0512,0x0000,0x4800,0x02F7,0x45C9,  /* 0400 (0002000) */
    0x0001,0x1084,0x65C4,0x08F2,0x0A54,0x0A54,0x0A54,0x15C0,  /* 0410 (0002020) */
    0xFFFD,0x09F7,0x04AC,0x1131,0xFFF6,0x0A31,0xFFF8,0x09F7,  /* 0420 (0002040) */
    0x04BC,0x1131,0xFFFA,0x0A31,0xFFFC,0x15C9,0x0001,0x15C0,  /* 0430 (0002060) */
    0x01FF,0x35C9,0x8000,0x0203,0x0AC0,0x02FB,0x01D5,0x0A00,  /* 0440 (0002100) */
    0x0977,0x04C6,0x0000,0x0000,0x02D1,0x45C9,0x0001,0x15C0,  /* 0450 (0002120) */
    0x0006,0x1084,0x65C4,0x02F2,0x1085,0x65C5,0x08F2,0xA515,  /* 0460 (0002140) */
    0x02C5,0x0AC0,0x02FC,0x0AB2,0x0004,0x0AB2,0x0006,0x25F2,  /* 0470 (0002160) */
    0x0007,0x0006,0x0693,0x25F2,0x0007,0x0006,0x0206,0x65F2,  /* 0480 (0002200) */
    0x0039,0x0004,0x65F2,0x0001,0x0006,0x25F2,0x000E,0x0006,  /* 0490 (0002220) */
    0x0485,0x65F2,0x0080,0x0002,0xA5FA,0x00AB,0x0002,0x0302,  /* 04A0 (0002240) */
    0x0077,0xFEDC,0x0BF2,0x0008,0x0204,0x0AB2,0x0008,0x0077,  /* 04B0 (0002260) */
    0xFEA4,0x55C9,0x0200,0x11C3,0x65C3,0xFC64,0x10F2,0x0002,  /* 04C0 (0002300) */
    0x10F2,0x0006,0x65C3,0x0004,0x10F2,0x0004,0x15F2,0x003C,  /* 04D0 (0002320) */
    0x0008,0x1083,0x65C3,0x08F2,0x1C85,0x0008,0x2CB2,0x0004,  /* 04E0 (0002340) */
    0x0006,0x0603,0x1CB2,0x0002,0x0006,0x9E93,0x0006,0x0AB2,  /* 04F0 (0002360) */
    0x0006,0x0AC5,0x02F3,0x8A13,0x8A13,0x1C80,0x0008,0x0977,  /* 0500 (0002400) */
    0x037C,0x871A,0x1C80,0x0008,0x0977,0x03FE,0x0000,0x2000,  /* 0510 (0002420) */
    0x0216,0x1C80,0x0008,0x1084,0x65C4,0x02F2,0x1085,0x65C5,  /* 0520 (0002440) */
    0x08F2,0xA515,0x0209,0x0AC0,0x02FC,0x65F2,0x0009,0x0008,  /* 0530 (0002460) */
    0x25F2,0x0601,0x0008,0x04CD,0x55CA,0x0008,0x0087,0x25F2,  /* 0540 (0002500) */
    0x0600,0x0008,0x02F9,0x1C83,0x0018,0x45C3,0x07F8,0x25C3,  /* 0550 (0002520) */
    0x6000,0x0303,0x25C3,0x7001,0x02EF,0x45CA,0x4000,0x15C0,  /* 0560 (0002540) */
    0xFFF8,0x09F7,0x035C,0x1131,0xFFF6,0x0A31,0xFFF8,0x1083,  /* 0570 (0002560) */
    0x65C3,0x0020,0x10C4,0x15C0,0x0008,0x15D3,0x8000,0x15D3,  /* 0580 (0002600) */
    0x8000,0x108B,0x65D3,0x000A,0x15D3,0xFFFF,0x0A13,0x0A13,  /* 0590 (0002620) */
    0x0AC0,0x02F3,0x1103,0x55F3,0x0080,0x0002,0x55F3,0x00C0,  /* 05A0 (0002640) */
    0x000E,0x15F3,0xFFFE,0x0012,0x55F3,0x0040,0x001A,0x65F3,  /* 05B0 (0002660) */
    0x0002,0x001C,0x65F3,0x0004,0x0028,0x15F3,0x4000,0x0032,  /* 05C0 (0002700) */
    0x10F3,0x0034,0x65F3,0x003C,0x0034,0x10F3,0x0040,0x65F3,  /* 05D0 (0002720) */
    0x0038,0x0040,0x15F3,0xFFFE,0x0042,0x15F3,0xA000,0x004A,  /* 05E0 (0002740) */
    0x15F3,0xFFFD,0x004E,0x15F3,0x2000,0x0056,0x1131,0xFFFA,  /* 05F0 (0002760) */
    0x0A31,0xFFFC,0x15C0,0x01FF,0x35C9,0x8000,0x0205,0x0AC0,  /* 0600 (0003000) */
    0x02FB,0x55CA,0x0200,0x0109,0x15C0,0x0010,0x0977,0x02FA,  /* 0610 (0003020) */
    0x8000,0x2000,0x0305,0x55CA,0x4000,0x55CA,0x0010,0x0087,  /* 0620 (0003040) */
    0x1083,0x65C3,0x0020,0x15C0,0x0008,0x1CC5,0x0000,0x45C5,  /* 0630 (0003060) */
    0x3FFF,0x25C5,0xC000,0x02EF,0x25C0,0x0004,0x0303,0x25C0,  /* 0640 (0003100) */
    0x0001,0x0207,0x0BF3,0x0008,0x02E6,0x0BF3,0x000A,0x02E3,  /* 0650 (0003120) */
    0x0114,0x25C0,0x0002,0x0305,0x35F3,0x8000,0x0008,0x03DB,  /* 0660 (0003140) */
    0x010C,0x1CC5,0x0008,0x45C5,0x240F,0x25C5,0x0100,0x02D3,  /* 0670 (0003160) */
    0x1CC5,0x000A,0x45C5,0xC000,0x03CE,0x65C3,0x000C,0x0AC0,  /* 0680 (0003200) */
    0x02D4,0x1083,0x65C3,0x000A,0x1084,0x65C4,0x02F2,0x1105,  /* 0690 (0003220) */
    0x24D4,0x02C3,0x24D4,0x02C1,0x24D4,0x02BF,0x0BD4,0x02BD,  /* 06A0 (0003240) */
    0x0BD4,0x02BB,0x2554,0x02B9,0x2554,0x02B7,0x2554,0x02B5,  /* 06B0 (0003260) */
    0x1083,0x65C3,0x0072,0x10C5,0x15C0,0x004B,0x0A13,0x0AC0,  /* 06C0 (0003300) */
    0x02FD,0x1084,0x65C4,0x000A,0x09F7,0x0026,0x65C5,0x0010,  /* 06D0 (0003320) */
    0x09F7,0x001E,0x0977,0x0124,0xFFBC,0x0301,0x0087,0x15C0,  /* 06E0 (0003340) */
    0x1B58,0x35C9,0x2000,0x0313,0x0AC0,0x02FB,0x55CA,0x8000,  /* 06F0 (0003360) */
    0x0087,0x1126,0x15C0,0x0006,0x0A85,0x15C3,0x0007,0x9315,  /* 0700 (0003400) */
    0x0AC3,0x02FD,0x0A84,0x0AC0,0x02F7,0x1584,0x0087,0x0BCA,  /* 0710 (0003420) */
    0x020F,0x55C9,0x0300,0x15C0,0x002E,0x09F7,0x005A,0x15C0,  /* 0720 (0003440) */
    0x003C,0x0977,0x0158,0x8605,0x45CA,0x0200,0x55CA,0x8000,  /* 0730 (0003460) */
    0x0087,0x15C0,0x003C,0x09F7,0x006E,0x03F1,0x0303,0x55CA,  /* 0740 (0003500) */
    0x0020,0x0087,0x15C0,0x003C,0x09F7,0x009A,0x02F8,0x15C0,  /* 0750 (0003520) */
    0x05DC,0x09F7,0x0022,0x15C0,0x05EA,0x0977,0x0120,0x87EF,  /* 0760 (0003540) */
    0x15C0,0x05EA,0x09F7,0x0040,0x03F6,0x02E9,0x15C0,0x05EA,  /* 0770 (0003560) */
    0x09F7,0x0072,0x02E4,0x0087,0x1083,0x65C3,0x08F2,0x1084,  /* 0780 (0003600) */
    0x65C4,0x000A,0x1105,0x1513,0x1513,0x1513,0x1553,0x1553,  /* 0790 (0003620) */
    0x1553,0x15D3,0x0090,0x0A13,0x15D3,0x0001,0x9013,0x0AC0,  /* 07A0 (0003640) */
    0x02FD,0x0A32,0x0002,0x0087,0x0977,0x015E,0x0000,0x2000,  /* 07B0 (0003660) */
    0x0204,0x65CE,0x0002,0x0BCA,0x0087,0x1C80,0x0028,0x45C0,  /* 07C0 (0003700) */
    0xBFFF,0x0208,0x1C80,0x0018,0x35C0,0xBFFF,0x0203,0x45CA,  /* 07D0 (0003720) */
    0x4000,0x01EF,0x0AB2,0x0002,0x25F2,0x000A,0x0002,0x07E9,  /* 07E0 (0003740) */
    0x45CA,0x4000,0x01E8,0x1084,0x65C4,0x02F2,0x1085,0x65C5,  /* 07F0 (0003760) */
    0x08F2,0xA515,0x0202,0x0AC0,0x02FC,0x0087,0x1080,0x65C0,  /* 0800 (0004000) */
    0x0072,0x1026,0x1340,0x0AC0,0x09F7,0x00B6,0x1131,0xFFF6,  /* 0810 (0004020) */
    0x0A31,0xFFF8,0x1540,0x09F7,0x00C4,0x15F4,0xB000,0x0002,  /* 0820 (0004040) */
    0x13B4,0x0004,0x1131,0xFFFA,0x0A31,0xFFFC,0x15C4,0x01FF,  /* 0830 (0004060) */
    0x35C9,0x8000,0x0208,0x0AC4,0x02FB,0x55CA,0x0200,0x0BD6,  /* 0840 (0004100) */
    0x55CA,0x1000,0x011B,0x1D40,0xFFFE,0x0CC0,0x0B00,0x0977,  /* 0850 (0004120) */
    0x00B8,0x0000,0x2000,0x02F3,0x1D40,0xFFFE,0x0CC0,0x0B00,  /* 0860 (0004140) */
    0x1084,0x65C4,0x02F2,0x1583,0xA513,0x02EA,0x0AC0,0x02FC,  /* 0870 (0004160) */
    0x1300,0x45C0,0x3FFF,0x25C0,0xC000,0x02E2,0x0085,0x0A26,  /* 0880 (0004200) */
    0x0C00,0x0B4E,0x6380,0x0B00,0x09F7,0x0036,0x1131,0xFFF6,  /* 0890 (0004220) */
    0x0A31,0xFFF8,0x09F7,0x0046,0x0BD6,0x0303,0x55F4,0x0080,  /* 08A0 (0004240) */
    0x0002,0x1131,0xFFFA,0x0A31,0xFFFC,0x0A00,0x35C9,0x8000,  /* 08B0 (0004260) */
    0x0206,0x0AC0,0x02FB,0x55CA,0x0200,0x00B1,0x0101,0x00A1,  /* 08C0 (0004300) */
    0x0085,0x1083,0x65C3,0x02F2,0x15C4,0x0080,0x15D3,0x52E3,  /* 08D0 (0004320) */
    0x0AC4,0x02FC,0x0977,0x0010,0x0010,0x02F2,0x0087,0x0977,  /* 08E0 (0004340) */
    0x0006,0x0020,0x08F2,0x0087,0x1543,0x6083,0x10C4,0x15D3,  /* 08F0 (0004360) */
    0x8000,0x15D3,0xA000,0x154B,0x6093,0x1013,0x0A13,0x0A13,  /* 0900 (0004400) */
    0x15D3,0x8000,0x15D3,0x2000,0x0085,0x1026,0x0A00,0x1244,  /* 0910 (0004420) */
    0x55C9,0x8080,0x45C4,0x7F11,0x25C4,0x8080,0x0301,0x11C0,  /* 0920 (0004440) */
    0x1C84,0x0020,0x45C4,0x3FFF,0x25C4,0xC000,0x0301,0x11C0,  /* 0930 (0004460) */
    0x1C84,0x0028,0x8002,0x45C4,0x7FFF,0x45C4,0x250F,0x2544,  /* 0940 (0004500) */
    0x0301,0x11C0,0x1C84,0x002A,0x45C4,0xC000,0x0201,0x11C0,  /* 0950 (0004520) */
    0x1C84,0x0010,0x45C4,0x3FFF,0x25C4,0xC000,0x0301,0x11C0,  /* 0960 (0004540) */
    0x1C84,0x0018,0x1126,0x45C4,0x07F9,0x2544,0x0301,0x11C0,  /* 0970 (0004560) */
    0x1584,0x45C4,0xF8FF,0x25F2,0xFFFD,0x0026,0x0202,0x0BD6,  /* 0980 (0004600) */
    0x010D,0x35F2,0x1000,0x0022,0x0301,0x0A04,0x1C83,0x001A,  /* 0990 (0004620) */
    0x45C3,0xFF00,0x60C4,0x2584,0x0301,0x11C0,0x1003,0x0302,  /* 09A0 (0004640) */
    0x55CA,0x4000,0x0085,0x0005,0x15C6,0x1FFE,0x118E,0x13A6,  /* 09B0 (0004660) */
    0xE5CE,0x062E,0x11C3,0x65C3,0x000C,0x10DF,0x0004,0x1240,  /* 09C0 (0004700) */
    0x0101,0x0000,0x09F7,0x0098,0x1385,0x15D5,0x0508,0x15D5,  /* 09D0 (0004720) */
    0x0101,0x8A15,0x8A15,0x15D5,0x0191,0x95D5,0x0002,0x15C2,  /* 09E0 (0004740) */
    0x05EA,0x9095,0x00C2,0x9095,0x15C2,0x000B,0x0A00,0x09F7,  /* 09F0 (0004760) */
    0x012A,0x02E8,0x15C2,0x05EE,0x09F7,0x01B0,0x0226,0x1383,  /* 0A00 (0005000) */
    0x94C4,0x0AC2,0xA117,0x0002,0x0304,0x8BC4,0x02DD,0xE5C2,  /* 0A10 (0005020) */
    0x0004,0x8BCB,0x0302,0xA2C0,0x0218,0x94C0,0x8A80,0x0AC2,  /* 0A20 (0005040) */
    0x0708,0x14C5,0x0BD3,0xE5C2,0x0004,0x0703,0x94D5,0x0AC2,  /* 0A30 (0005060) */
    0x06FD,0x8BC4,0x030F,0x0A03,0x1385,0x95D5,0x000A,0x9015,  /* 0A40 (0005100) */
    0x90D5,0x0A05,0x15C2,0x0003,0x01D2,0x8BC0,0x03BB,0x15C3,  /* 0A50 (0005120) */
    0x0001,0x01F2,0x09F7,0x029A,0x94E6,0x94F6,0x0001,0x0087,  /* 0A60 (0005140) */
    0x55F1,0x0002,0x000E,0x1D85,0x0002,0x1144,0x65C4,0xFFD0,  /* 0A70 (0005160) */
    0x1136,0x0004,0x15C2,0x0010,0x0A14,0x7E82,0x1144,0x65C4,  /* 0A80 (0005200) */
    0xFFF0,0x0A35,0xFFD2,0x1135,0xFFD4,0x0A35,0xFFE2,0x1135,  /* 0A90 (0005220) */
    0xFFE4,0x55F5,0x8000,0xFFD2,0x15C2,0x05EE,0x0C82,0x0B02,  /* 0AA0 (0005240) */
    0x10B5,0xFFD6,0x55F5,0xA000,0xFFE2,0x1D84,0x0004,0x1137,  /* 0AB0 (0005260) */
    0x02D0,0x0A37,0x02CE,0x65C4,0x0010,0x1137,0x02C8,0x0A37,  /* 0AC0 (0005300) */
    0x02C6,0x9C77,0x0000,0x02AE,0x9C77,0x0002,0x02A9,0x9C77,  /* 0AD0 (0005320) */
    0x0004,0x02A4,0x9C77,0x0006,0x029F,0x9C77,0x0008,0x029A,  /* 0AE0 (0005340) */
    0x9C77,0x000A,0x0295,0x8AB7,0x0296,0x45F1,0x0002,0x000E,  /* 0AF0 (0005360) */
    0x55F1,0x0100,0x000E,0x09F7,0x0240,0x0A35,0x0002,0x1DE4,  /* 0B00 (0005400) */
    0x0278,0x1DE4,0x0272,0x1DE4,0x026C,0x15E4,0x0000,0x15E4,  /* 0B10 (0005420) */
    0x0200,0x15E4,0x00AB,0x09F7,0x0026,0x0087,0x1D85,0x0002,  /* 0B20 (0005440) */
    0x1144,0x10A4,0x15E4,0x0160,0x1DE4,0x024E,0x1DE4,0x0248,  /* 0B30 (0005460) */
    0x1DE4,0x0242,0x1DE4,0x022C,0x1DE4,0x0226,0x1DE4,0x0220,  /* 0B40 (0005500) */
    0x65C5,0xFFE0,0x65C2,0x000E,0x2097,0x003C,0x0403,0x15C2,  /* 0B50 (0005520) */
    0x003C,0x0107,0x35C2,0x0001,0x0304,0x55F5,0x0080,0x0002,  /* 0B60 (0005540) */
    0x0A82,0x0C82,0x0B02,0x10B5,0x0006,0x0A35,0x0008,0x0A35,  /* 0B70 (0005560) */
    0x000A,0x1DF1,0x0210,0x0008,0x1DF1,0x020C,0x000A,0x15C4,  /* 0B80 (0005600) */
    0x0084,0x09F7,0x0188,0x860A,0x0304,0x35F5,0x0300,0x0008,  /* 0B90 (0005620) */
    0x02EC,0x45F5,0x0080,0x0002,0x00A4,0x0087,0x45F5,0x0080,  /* 0BA0 (0005640) */
    0x0002,0x35C2,0x0004,0x0301,0x0000,0x0087,0x1D85,0x0002,  /* 0BB0 (0005660) */
    0x65C5,0xFFD0,0x0A35,0x0008,0x0A35,0x000A,0x1DF1,0x01C2,  /* 0BC0 (0005700) */
    0x0004,0x1DF1,0x01BE,0x0006,0x55F1,0x0001,0x000E,0x15C4,  /* 0BD0 (0005720) */
    0x8004,0x09F7,0x0138,0x8618,0x02ED,0x55F1,0x0002,0x000E,  /* 0BE0 (0005740) */
    0x15F7,0x00AB,0x017A,0x15F7,0x0100,0x0176,0x15F7,0x0000,  /* 0BF0 (0005760) */
    0x0172,0x8AB7,0x018A,0x0A00,0x45F1,0x0002,0x000E,0x55F1,  /* 0C00 (0006000) */
    0x0100,0x000E,0x00A4,0x0087,0x45F1,0x0001,0x000E,0x55F1,  /* 0C10 (0006020) */
    0x0002,0x000E,0x1D85,0x0002,0x1D42,0xFFFE,0x45F1,0x0002,  /* 0C20 (0006040) */
    0x000E,0x55F1,0x0100,0x000E,0x25F5,0x0160,0xFFFC,0x0221,  /* 0C30 (0006060) */
    0x8BF7,0x014C,0x020D,0x2DF5,0x012A,0xFFFA,0x02B7,0x2DF5,  /* 0C40 (0006100) */
    0x0120,0xFFF8,0x02B3,0x2DF5,0x0116,0xFFF6,0x02AF,0x0087,  /* 0C50 (0006120) */
    0xA5CD,0x0003,0x02AB,0x1D77,0xFFF6,0x0104,0x1D77,0xFFF8,  /* 0C60 (0006140) */
    0x0100,0x1D77,0xFFFA,0x00FC,0x8A37,0x0114,0x00A4,0x0A00,  /* 0C70 (0006160) */
    0x0087,0x25F5,0x0090,0xFFFC,0x0313,0x25F5,0x0260,0xFFFC,  /* 0C80 (0006200) */
    0x0295,0xA5CD,0x0005,0x0292,0x09F7,0x00AE,0x1D34,0xFFFE,  /* 0C90 (0006220) */
    0xFFF8,0x1D34,0xFFFC,0xFFF6,0x1D34,0xFFFA,0xFFF4,0x011F,  /* 0CA0 (0006240) */
    0x1144,0x6084,0x1143,0x65C3,0xFFD0,0x1CC2,0x0008,0x45C2,  /* 0CB0 (0006260) */
    0xF8FF,0xDCC2,0x000A,0x65C2,0x002E,0x25D4,0x0002,0x0217,  /* 0CC0 (0006300) */
    0x65F5,0x0008,0xFFFE,0x35CC,0x0001,0x0211,0x1143,0x65C3,  /* 0CD0 (0006320) */
    0xFFF0,0x1513,0x1513,0x1513,0x1144,0x65C4,0xFFFC,0x1DE4,  /* 0CE0 (0006340) */
    0x0098,0x1DE4,0x0092,0x1DE4,0x008C,0x09F7,0xFE52,0x0077,  /* 0CF0 (0006360) */
    0xFEBA,0x1DDF,0x006A,0x0018,0x1DDF,0x0066,0x001A,0x1DDF,  /* 0D00 (0006400) */
    0x0062,0x001C,0x55F1,0x0002,0x000E,0x00B4,0x0087,0x0A03,  /* 0D10 (0006420) */
    0x15C2,0x0030,0x3131,0x000E,0x0206,0x0AC3,0x02FB,0x0AC2,  /* 0D20 (0006440) */
    0x02F9,0x00B1,0x0087,0x1C42,0x000E,0x10B1,0x000E,0x35F5,  /* 0D30 (0006460) */
    0x4000,0x0008,0x0301,0x00B1,0x0087,0x1166,0x11C3,0x65C3,  /* 0D40 (0006500) */
    0x0028,0x15C2,0x0018,0x14D5,0x0BD5,0x94D5,0x0AC2,0x02FD,  /* 0D50 (0006520) */
    0x1585,0x1144,0x15C2,0x001C,0x10A4,0x15E4,0x0260,0x0087,  /* 0D60 (0006540) */
    0x00AB,0x0100,0x0000,0x0007,0x0001,0x0303,0x0000,0x0002,  /* 0D70 (0006560) */
    0x0102,0x0700,0x0600,0xFFFF,0xFFFF,0xFFFF,0x0064,0x0501,  /* 0D80 (0006600) */
    0x0000,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,  /* 0D90 (0006620) */
    0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,  /* 0DA0 (0006640) */
    0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,  /* 0DB0 (0006660) */
    0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,  /* 0DC0 (0006700) */
    0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,  /* 0DD0 (0006720) */
    0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,  /* 0DE0 (0006740) */
    0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xE93D,  /* 0DF0 (0006760) */
    0x5501,0xE2FF,0xE5F2,0xF290,0x80AF,0xDD32,0xDE1A,0x1518,  /* 0E00 (0007000) */
    0x0181,0xE21F,0x54E2,0x451A,0xF2A0,0xE2E2,0x01D2,0x1532,  /* 0E10 (0007020) */
    0xC081,0xC032,0xC233,0x32AF,0x31AE,0x30AD,0x1830,0xBE12,  /* 0E20 (0007040) */
    0x0230,0x4101,0xBFBE,0x0102,0x7D41,0x8DEC,0x7E30,0x8E0C,  /* 0E30 (0007060) */
    0xD231,0x7A18,0xD208,0xC2A4,0xE2A4,0xF2EA,0x97D2,0x97C2,  /* 0E40 (0007100) */
    0x0C74,0x01F2,0x7580,0xA022,0x8275,0x75A0,0x0189,0x3275,  /* 0E50 (0007120) */
    0x7528,0x0033,0x97C2,0xEC7D,0x308D,0x0C7E,0x318E,0x18C2,  /* 0E60 (0007140) */
    0x007F,0xA075,0x306D,0x02A2,0x8CD2,0x8375,0x744D,0xF00C,  /* 0E70 (0007160) */
    0x0B74,0x74F2,0xF282,0x2075,0x7500,0x05B8,0xA875,0x1082,  /* 0E80 (0007200) */
    0x9600,0xA1D2,0xA1C2,0xB130,0x74F6,0xF200,0xA875,0x1087,  /* 0E90 (0007220) */
    0x2101,0xB120,0xC20D,0x53AF,0xFC22,0x97D2,0x97C2,0x0C7A,  /* 0EA0 (0007240) */
    0x6821,0xB030,0x8F06,0xD221,0x0102,0x10BE,0x0B02,0xB420,  /* 0EB0 (0007260) */
    0x0175,0xAE9F,0xAD31,0x0130,0x309F,0xD30F,0xAFC2,0xA4D2,  /* 0EC0 (0007300) */
    0xA4C2,0x1031,0x21E5,0x7054,0x5323,0x0321,0x2145,0x2253,  /* 0ED0 (0007320) */
    0x451C,0xF522,0xF522,0xE082,0xB520,0xC204,0x218C,0xE509,  /* 0EE0 (0007340) */
    0x5422,0x03E0,0x0303,0x9003,0x2601,0x93FF,0x31F5,0xEFFE,  /* 0EF0 (0007360) */
    0x9304,0x30F5,0xC2FD,0xD218,0x758C,0x0021,0x0C74,0x4B01,  /* 0F00 (0007400) */
    0x21E5,0x0C54,0x0170,0xB422,0x0304,0x12D2,0xB422,0x0308,  /* 0F10 (0007420) */
    0x13D2,0xD222,0x2214,0x0301,0x0D01,0x3301,0xCB01,0xFB03,  /* 0F20 (0007440) */
    0xEC0C,0xAF30,0xBDBF,0xAFC2,0xA4D2,0xA4C2,0x0290,0x7C00,  /* 0F30 (0007460) */
    0x7B11,0xE400,0xF293,0xE5A3,0xB483,0x0310,0x8375,0xDB00,  /* 0F40 (0007500) */
    0xDCF2,0x20EE,0xFDB4,0x8375,0x534D,0xE322,0x2285,0xE082,  /* 0F50 (0007520) */
    0x0C74,0x75F2,0x0022,0x4B01,0x2285,0xE082,0x5001,0xFFFF,  /* 0F60 (0007540) */
    0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,  /* 0F70 (0007560) */
    0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,  /* 0F80 (0007600) */
    0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,  /* 0F90 (0007620) */
    0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,  /* 0FA0 (0007640) */
    0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,  /* 0FB0 (0007660) */
    0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,  /* 0FC0 (0007700) */
    0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,  /* 0FD0 (0007720) */
    0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,  /* 0FE0 (0007740) */
    0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,  /* 0FF0 (0007760) */
    };
#if 0
/* This ROM image was produced by assembling the INIQNA.MAC file.  
   It was used to debug the DEQNA citizenship test. Since the INIQNA.MAC source
   does not contain a MOP boot, this isn't a useful boot ROM image.  */
uint16 xq_bootrom_deqna_iniqna[2048] = {
    0x00A0,0x0114,0x016B,0x0FFE,0x0155,0x0000,0x8000,0x8000,  /* 0000 (0000000) */
    0x0400,0xFC00,0x0000,0x0000,0x8000,0x8000,0x0C00,0xFC00,  /* 0010 (0000020) */
    0x0000,0x0000,0x8000,0x2000,0xC000,0x8000,0x15C6,0x03FE,  /* 0020 (0000040) */
    0x1026,0x15F1,0x000C,0x0004,0x0A31,0x0006,0x15F1,0x0208,  /* 0030 (0000060) */
    0x000E,0x0A00,0x0AC0,0x02FE,0x0A31,0x000E,0x0A00,0x0AC0,  /* 0040 (0000100) */
    0x02FE,0x15F1,0x0002,0x000E,0x0A31,0x000E,0x45F7,0x3FFF,  /* 0050 (0000120) */
    0xFFB2,0x2DF7,0xFFAE,0xFFC0,0x0233,0x45F7,0x3FFF,0xFFA8,  /* 0060 (0000140) */
    0x2DF7,0xFFA4,0xFFB2,0x022C,0x45F7,0x3FFF,0xFFA6,0x2DF7,  /* 0070 (0000160) */
    0xFFA2,0xFFA6,0x0225,0x15C4,0x0FFE,0x0A03,0x0A02,0x9C80,  /* 0080 (0000200) */
    0x0400,0x0A82,0x45C0,0xFF00,0x6003,0x209F,0x0006,0x0201,  /* 0090 (0000220) */
    0x0BD2,0x0AC4,0x02F4,0x17C0,0x0006,0x2C03,0x0400,0x0210,  /* 00A0 (0000240) */
    0x005F,0x0408,0x15C2,0x1400,0x09F7,0x0020,0x0BC0,0x0208,  /* 00B0 (0000260) */
    0x1382,0x0202,0x0077,0x08FA,0x25C2,0x01FF,0x8601,0x004A,  /* 00C0 (0000300) */
    0x17C3,0x000A,0x0301,0x004B,0x0000,0x01FE,0x65C1,0x000E,  /* 00D0 (0000320) */
    0x09F7,0x004C,0x55C9,0x0002,0x0A09,0x0977,0x072A,0xFFC0,  /* 00E0 (0000340) */
    0x020C,0x0BCA,0x020C,0x35F2,0x0400,0x0028,0x0302,0x55C9,  /* 00F0 (0000360) */
    0x0400,0x0977,0x0712,0xFF92,0x0302,0x55CA,0x0040,0x15C9,  /* 0100 (0000400) */
    0x0002,0x0A09,0x1240,0x45C0,0x6800,0x25C0,0x1030,0x0302,  /* 0110 (0000420) */
    0x55CA,0x0040,0x1280,0xE5C1,0x000E,0x0087,0x5500,0xFFAA,  /* 0120 (0000440) */
    0x17E6,0x001C,0x17E6,0x001E,0x11C3,0x65C3,0x0082,0x10DF,  /* 0130 (0000460) */
    0x001C,0x15DF,0x00E0,0x001E,0x8900,0x159F,0x001E,0x159F,  /* 0140 (0000500) */
    0x001C,0x17C0,0x01F4,0x55C0,0x00E0,0x65C3,0x0008,0x10DF,  /* 0150 (0000520) */
    0x0004,0x101F,0x0006,0x119F,0x01F6,0x105F,0x01FA,0x109F,  /* 0160 (0000540) */
    0x01F8,0x65C3,0x000A,0x10DF,0x0014,0x101F,0x0016,0x65C3,  /* 0170 (0000560) */
    0x0012,0x15F1,0x01FC,0xFFFE,0x10DF,0x01FC,0x101F,0x01FE,  /* 0180 (0000600) */
    0x0A32,0x0002,0x0A0A,0x15C0,0x0072,0x6080,0x15C4,0x004A,  /* 0190 (0000620) */
    0x0A10,0x0AC4,0x02FD,0x0977,0x066E,0xFFB6,0x031C,0x0AB2,  /* 01A0 (0000640) */
    0x0002,0x25F2,0x0002,0x0002,0x06ED,0x0087,0x1D9F,0x0002,  /* 01B0 (0000660) */
    0x01F4,0x0006,0x55CA,0x0400,0x65C6,0x0004,0x0087,0x17C6,  /* 01C0 (0000700) */
    0x01F6,0x17C1,0x01FA,0x17C2,0x01F8,0x55CA,0x2000,0x0087,  /* 01D0 (0000720) */
    0x55CA,0x0800,0x0002,0x8D17,0x0000,0x1043,0xE5C3,0x000E,  /* 01E0 (0000740) */
    0x1084,0x65C4,0x000A,0x15C5,0x0006,0x14C0,0x9014,0x0AC5,  /* 01F0 (0000760) */
    0x02FC,0x1084,0x65C4,0x000A,0x1105,0x0BD4,0x0204,0x0BD4,  /* 0200 (0001000) */
    0x0202,0x0BCC,0x0311,0x1144,0x15C0,0xFFFF,0x2014,0x020E,  /* 0210 (0001020) */
    0x2014,0x020C,0x200C,0x0308,0x0209,0x1144,0x25D4,0x00AA,  /* 0220 (0001040) */
    0x0203,0xA5CC,0x0004,0x8602,0x55CA,0x0001,0x15C0,0xFFFF,  /* 0230 (0001060) */
    0x09F7,0x06B6,0x17C5,0x01FC,0x11C3,0x65C3,0x0024,0x10DF,  /* 0240 (0001100) */
    0x01FC,0x55C9,0x0040,0x1131,0xFFFA,0x0A00,0x1031,0xFFFC,  /* 0250 (0001120) */
    0x15C3,0x0FFF,0x0AC3,0x02FE,0x55CA,0x0800,0x0087,0x65C6,  /* 0260 (0001140) */
    0x0004,0x115F,0x01FC,0x0A80,0x03F7,0x1240,0x35C0,0x0080,  /* 0270 (0001160) */
    0x03F3,0x1300,0x45C0,0x3FFF,0x25C0,0xC000,0x02ED,0x0BF4,  /* 0280 (0001200) */
    0x0008,0x03EA,0x0BF4,0x000A,0x03E7,0x15C9,0x8480,0x8D1F,  /* 0290 (0001220) */
    0x01F4,0x11C3,0x65C3,0xFE88,0x10F2,0x0002,0x10F2,0x0006,  /* 02A0 (0001240) */
    0x65C3,0x0005,0x10F2,0x0004,0x0A32,0x0008,0x1083,0x65C3,  /* 02B0 (0001260) */
    0x0072,0x15C0,0x0002,0x15C5,0x0006,0x8A13,0x15C4,0x0007,  /* 02C0 (0001300) */
    0x2CB2,0x0004,0x0006,0x0603,0x1CB2,0x0002,0x0006,0x9E93,  /* 02D0 (0001320) */
    0x0006,0x0AB2,0x0006,0x0AC4,0x02F3,0x0AC5,0x02EE,0x15C4,  /* 02E0 (0001340) */
    0x0010,0x8A13,0x0AC4,0x02FD,0x0AC0,0x02E5,0x0977,0x0518,  /* 02F0 (0001360) */
    0xFFC5,0x0303,0x55CA,0x0002,0x0087,0x0AB2,0x0008,0x25F2,  /* 0300 (0001400) */
    0x01F4,0x0008,0x06D3,0x0A32,0x0008,0x1083,0x65C3,0x0072,  /* 0310 (0001420) */
    0x10C4,0x10F2,0x0002,0x15C0,0x0201,0x8A13,0x0AC0,0x02FD,  /* 0320 (0001440) */
    0x95E3,0x00AB,0x1103,0x0A83,0x0A00,0x0A04,0x0A05,0x8CC0,  /* 0330 (0001460) */
    0x0201,0x0A80,0x9013,0x0A84,0x25C4,0x0007,0x0603,0x0A04,  /* 0340 (0001500) */
    0x65C3,0x0039,0x0A85,0x35C5,0x0007,0x0207,0x65C3,0x0008,  /* 0350 (0001520) */
    0x25C5,0x0030,0x0202,0xE5C3,0x0030,0x25C5,0x0038,0x06E7,  /* 0360 (0001540) */
    0x0115,0x1083,0x65C3,0x0072,0x10F2,0x0002,0x15C5,0x0008,  /* 0370 (0001560) */
    0x15C4,0x0006,0x8BD3,0x15C0,0x0007,0x8A53,0x0AC0,0x02FD,  /* 0380 (0001600) */
    0x0AC4,0x02F8,0x65C3,0x0010,0x0AC5,0x02F2,0x0A32,0x0006,  /* 0390 (0001620) */
    0x1C80,0x0002,0x0977,0x0476,0xFFC0,0x022E,0x1C83,0x0002,  /* 03A0 (0001640) */
    0x65C3,0x0001,0x10F2,0x0004,0x1C83,0x0004,0x1084,0x65C4,  /* 03B0 (0001660) */
    0x08F2,0x15C0,0x0006,0x92CC,0x8A54,0x65C3,0x0008,0x0AC0,  /* 03C0 (0001700) */
    0x02FA,0x0A14,0x0A14,0x15C9,0x0001,0x15C0,0xFFFD,0x09F7,  /* 03D0 (0001720) */
    0x04FC,0x1131,0xFFF6,0x0A31,0xFFF8,0x09F7,0x050C,0x1131,  /* 03E0 (0001740) */
    0xFFFA,0x0A31,0xFFFC,0x15C0,0x01FF,0x35C9,0x8000,0x0207,  /* 03F0 (0001760) */
    0x0AC0,0x02FB,0x55CA,0x0200,0x55CA,0x0004,0x0087,0x0A00,  /* 0400 (0002000) */
    0x0977,0x0512,0x0000,0x4800,0x02F7,0x45C9,0x0001,0x1084,  /* 0410 (0002020) */
    0x65C4,0x08F2,0x0A54,0x0A54,0x0A54,0x15C0,0xFFFD,0x09F7,  /* 0420 (0002040) */
    0x04AC,0x1131,0xFFF6,0x0A31,0xFFF8,0x09F7,0x04BC,0x1131,  /* 0430 (0002060) */
    0xFFFA,0x0A31,0xFFFC,0x15C9,0x0001,0x15C0,0x01FF,0x35C9,  /* 0440 (0002100) */
    0x8000,0x0203,0x0AC0,0x02FB,0x01D5,0x0A00,0x0977,0x04C6,  /* 0450 (0002120) */
    0x0000,0x0000,0x02D1,0x45C9,0x0001,0x15C0,0x0006,0x1084,  /* 0460 (0002140) */
    0x65C4,0x02F2,0x1085,0x65C5,0x08F2,0xA515,0x02C5,0x0AC0,  /* 0470 (0002160) */
    0x02FC,0x0AB2,0x0004,0x0AB2,0x0006,0x25F2,0x0007,0x0006,  /* 0480 (0002200) */
    0x0693,0x25F2,0x0007,0x0006,0x0206,0x65F2,0x0039,0x0004,  /* 0490 (0002220) */
    0x65F2,0x0001,0x0006,0x25F2,0x000E,0x0006,0x0485,0x65F2,  /* 04A0 (0002240) */
    0x0080,0x0002,0xA5FA,0x00AB,0x0002,0x0302,0x0077,0xFEDC,  /* 04B0 (0002260) */
    0x0BF2,0x0008,0x0204,0x0AB2,0x0008,0x0077,0xFEA4,0x55C9,  /* 04C0 (0002300) */
    0x0200,0x11C3,0x65C3,0xFC58,0x10F2,0x0002,0x10F2,0x0006,  /* 04D0 (0002320) */
    0x65C3,0x0004,0x10F2,0x0004,0x15F2,0x003C,0x0008,0x1083,  /* 04E0 (0002340) */
    0x65C3,0x08F2,0x1C85,0x0008,0x2CB2,0x0004,0x0006,0x0603,  /* 04F0 (0002360) */
    0x1CB2,0x0002,0x0006,0x9E93,0x0006,0x0AB2,0x0006,0x0AC5,  /* 0500 (0002400) */
    0x02F3,0x8A13,0x8A13,0x1C80,0x0008,0x0977,0x037C,0x871A,  /* 0510 (0002420) */
    0x1C80,0x0008,0x0977,0x03FE,0x0000,0x2000,0x0216,0x1C80,  /* 0520 (0002440) */
    0x0008,0x1084,0x65C4,0x02F2,0x1085,0x65C5,0x08F2,0xA515,  /* 0530 (0002460) */
    0x0209,0x0AC0,0x02FC,0x65F2,0x0009,0x0008,0x25F2,0x0601,  /* 0540 (0002500) */
    0x0008,0x04CD,0x55CA,0x0008,0x0087,0x25F2,0x0600,0x0008,  /* 0550 (0002520) */
    0x02F9,0x1C83,0x0018,0x45C3,0x07F8,0x25C3,0x6000,0x0303,  /* 0560 (0002540) */
    0x25C3,0x7001,0x02EF,0x45CA,0x4000,0x15C0,0xFFF8,0x09F7,  /* 0570 (0002560) */
    0x035C,0x1131,0xFFF6,0x0A31,0xFFF8,0x1083,0x65C3,0x0020,  /* 0580 (0002600) */
    0x10C4,0x15C0,0x0008,0x15D3,0x8000,0x15D3,0x8000,0x108B,  /* 0590 (0002620) */
    0x65D3,0x000A,0x15D3,0xFFFF,0x0A13,0x0A13,0x0AC0,0x02F3,  /* 05A0 (0002640) */
    0x1103,0x55F3,0x0080,0x0002,0x55F3,0x00C0,0x000E,0x15F3,  /* 05B0 (0002660) */
    0xFFFE,0x0012,0x55F3,0x0040,0x001A,0x65F3,0x0002,0x001C,  /* 05C0 (0002700) */
    0x65F3,0x0004,0x0028,0x15F3,0x4000,0x0032,0x10F3,0x0034,  /* 05D0 (0002720) */
    0x65F3,0x003C,0x0034,0x10F3,0x0040,0x65F3,0x0038,0x0040,  /* 05E0 (0002740) */
    0x15F3,0xFFFE,0x0042,0x15F3,0xA000,0x004A,0x15F3,0xFFFD,  /* 05F0 (0002760) */
    0x004E,0x15F3,0x2000,0x0056,0x1131,0xFFFA,0x0A31,0xFFFC,  /* 0600 (0003000) */
    0x15C0,0x01FF,0x35C9,0x8000,0x0205,0x0AC0,0x02FB,0x55CA,  /* 0610 (0003020) */
    0x0200,0x0109,0x15C0,0x0010,0x0977,0x02FA,0x8000,0x2000,  /* 0620 (0003040) */
    0x0305,0x55CA,0x4000,0x55CA,0x0010,0x0087,0x1083,0x65C3,  /* 0630 (0003060) */
    0x0020,0x15C0,0x0008,0x1CC5,0x0000,0x45C5,0x3FFF,0x25C5,  /* 0640 (0003100) */
    0xC000,0x02EF,0x25C0,0x0004,0x0303,0x25C0,0x0001,0x0207,  /* 0650 (0003120) */
    0x0BF3,0x0008,0x02E6,0x0BF3,0x000A,0x02E3,0x0114,0x25C0,  /* 0660 (0003140) */
    0x0002,0x0305,0x35F3,0x8000,0x0008,0x03DB,0x010C,0x1CC5,  /* 0670 (0003160) */
    0x0008,0x45C5,0x240F,0x25C5,0x0100,0x02D3,0x1CC5,0x000A,  /* 0680 (0003200) */
    0x45C5,0xC000,0x03CE,0x65C3,0x000C,0x0AC0,0x02D4,0x1083,  /* 0690 (0003220) */
    0x65C3,0x000A,0x1084,0x65C4,0x02F2,0x1105,0x24D4,0x02C3,  /* 06A0 (0003240) */
    0x24D4,0x02C1,0x24D4,0x02BF,0x0BD4,0x02BD,0x0BD4,0x02BB,  /* 06B0 (0003260) */
    0x2554,0x02B9,0x2554,0x02B7,0x2554,0x02B5,0x1083,0x65C3,  /* 06C0 (0003300) */
    0x0072,0x10C5,0x15C0,0x004B,0x0A13,0x0AC0,0x02FD,0x1084,  /* 06D0 (0003320) */
    0x65C4,0x000A,0x09F7,0x0026,0x65C5,0x0010,0x09F7,0x001E,  /* 06E0 (0003340) */
    0x0977,0x0124,0xFFBC,0x0301,0x0087,0x15C0,0x1B58,0x35C9,  /* 06F0 (0003360) */
    0x2000,0x0313,0x0AC0,0x02FB,0x55CA,0x8000,0x0087,0x1126,  /* 0700 (0003400) */
    0x15C0,0x0006,0x0A85,0x15C3,0x0007,0x9315,0x0AC3,0x02FD,  /* 0710 (0003420) */
    0x0A84,0x0AC0,0x02F7,0x1584,0x0087,0x0BCA,0x020F,0x55C9,  /* 0720 (0003440) */
    0x0300,0x15C0,0x002E,0x09F7,0x005A,0x15C0,0x003C,0x0977,  /* 0730 (0003460) */
    0x0158,0x8605,0x45CA,0x0200,0x55CA,0x8000,0x0087,0x15C0,  /* 0740 (0003500) */
    0x003C,0x09F7,0x006E,0x03F1,0x0303,0x55CA,0x0020,0x0087,  /* 0750 (0003520) */
    0x15C0,0x003C,0x09F7,0x009A,0x02F8,0x15C0,0x05DC,0x09F7,  /* 0760 (0003540) */
    0x0022,0x15C0,0x05EA,0x0977,0x0120,0x87EF,0x15C0,0x05EA,  /* 0770 (0003560) */
    0x09F7,0x0040,0x03F6,0x02E9,0x15C0,0x05EA,0x09F7,0x0072,  /* 0780 (0003600) */
    0x02E4,0x0087,0x1083,0x65C3,0x08F2,0x1084,0x65C4,0x000A,  /* 0790 (0003620) */
    0x1105,0x1513,0x1513,0x1513,0x1553,0x1553,0x1553,0x15D3,  /* 07A0 (0003640) */
    0x0090,0x0A13,0x15D3,0x0001,0x9013,0x0AC0,0x02FD,0x0A32,  /* 07B0 (0003660) */
    0x0002,0x0087,0x0977,0x015E,0x0000,0x2000,0x0204,0x65CE,  /* 07C0 (0003700) */
    0x0002,0x0BCA,0x0087,0x1C80,0x0028,0x45C0,0xBFFF,0x0208,  /* 07D0 (0003720) */
    0x1C80,0x0018,0x35C0,0xBFFF,0x0203,0x45CA,0x4000,0x01EF,  /* 07E0 (0003740) */
    0x0AB2,0x0002,0x25F2,0x0064,0x0002,0x07E9,0x45CA,0x4000,  /* 07F0 (0003760) */
    0x01E8,0x1084,0x65C4,0x02F2,0x1085,0x65C5,0x08F2,0xA515,  /* 0800 (0004000) */
    0x0202,0x0AC0,0x02FC,0x0087,0x1080,0x65C0,0x0072,0x1026,  /* 0810 (0004020) */
    0x1340,0x0AC0,0x09F7,0x00B6,0x1131,0xFFF6,0x0A31,0xFFF8,  /* 0820 (0004040) */
    0x1540,0x09F7,0x00C4,0x15F4,0xB000,0x0002,0x13B4,0x0004,  /* 0830 (0004060) */
    0x1131,0xFFFA,0x0A31,0xFFFC,0x15C4,0xFFFF,0x35C9,0x8000,  /* 0840 (0004100) */
    0x0208,0x0AC4,0x02FB,0x55CA,0x0200,0x0BD6,0x55CA,0x1000,  /* 0850 (0004120) */
    0x011B,0x1D40,0xFFFE,0x0CC0,0x0B00,0x0977,0x00B8,0x0000,  /* 0860 (0004140) */
    0x2000,0x02F3,0x1D40,0xFFFE,0x0CC0,0x0B00,0x1084,0x65C4,  /* 0870 (0004160) */
    0x02F2,0x1583,0xA513,0x02EA,0x0AC0,0x02FC,0x1300,0x45C0,  /* 0880 (0004200) */
    0x3FFF,0x25C0,0xC000,0x02E2,0x0085,0x0A26,0x0C00,0x0B4E,  /* 0890 (0004220) */
    0x6380,0x0B00,0x09F7,0x0036,0x1131,0xFFF6,0x0A31,0xFFF8,  /* 08A0 (0004240) */
    0x09F7,0x0046,0x0BD6,0x0303,0x55F4,0x0080,0x0002,0x1131,  /* 08B0 (0004260) */
    0xFFFA,0x0A31,0xFFFC,0x0A00,0x35C9,0x8000,0x0206,0x0AC0,  /* 08C0 (0004300) */
    0x02FB,0x55CA,0x0200,0x00B1,0x0101,0x00A1,0x0085,0x1083,  /* 08D0 (0004320) */
    0x65C3,0x02F2,0x15C4,0x0080,0x15D3,0x52E3,0x0AC4,0x02FC,  /* 08E0 (0004340) */
    0x0977,0x0010,0x0010,0x02F2,0x0087,0x0977,0x0006,0x0020,  /* 08F0 (0004360) */
    0x08F2,0x0087,0x1543,0x6083,0x10C4,0x15D3,0x8000,0x15D3,  /* 0900 (0004400) */
    0xA000,0x154B,0x6093,0x1013,0x0A13,0x0A13,0x15D3,0x8000,  /* 0910 (0004420) */
    0x15D3,0x2000,0x0085,0x1026,0x0A00,0x1244,0x55C9,0x8080,  /* 0920 (0004440) */
    0x45C4,0x7F11,0x25C4,0x8080,0x0301,0x11C0,0x1C84,0x0020,  /* 0930 (0004460) */
    0x45C4,0x3FFF,0x25C4,0xC000,0x0301,0x11C0,0x1C84,0x0028,  /* 0940 (0004500) */
    0x8002,0x45C4,0x7FFF,0x45C4,0x750F,0x2544,0x0301,0x11C0,  /* 0950 (0004520) */
    0x1C84,0x002A,0x45C4,0xC000,0x0201,0x11C0,0x1C84,0x0010,  /* 0960 (0004540) */
    0x45C4,0x3FFF,0x25C4,0xC000,0x0301,0x11C0,0x1C84,0x0018,  /* 0970 (0004560) */
    0x1126,0x45C4,0x07FF,0x2544,0x0301,0x11C0,0x1584,0x45C4,  /* 0980 (0004600) */
    0xF8FF,0x25F2,0xFFFD,0x0026,0x0202,0x0BD6,0x010D,0x35F2,  /* 0990 (0004620) */
    0x1000,0x0022,0x0301,0x0A04,0x1C83,0x001A,0x45C3,0xFF00,  /* 09A0 (0004640) */
    0x60C4,0x2584,0x0301,0x11C0,0x1003,0x0302,0x55CA,0x4000,  /* 09B0 (0004660) */
    0x0085,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 09C0 (0004700) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 09D0 (0004720) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 09E0 (0004740) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 09F0 (0004760) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0A00 (0005000) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0A10 (0005020) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0A20 (0005040) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0A30 (0005060) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0A40 (0005100) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0A50 (0005120) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0A60 (0005140) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0A70 (0005160) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0A80 (0005200) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0A90 (0005220) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0AA0 (0005240) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0AB0 (0005260) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0AC0 (0005300) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0AD0 (0005320) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0AE0 (0005340) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0AF0 (0005360) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0B00 (0005400) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0B10 (0005420) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0B20 (0005440) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0B30 (0005460) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0B40 (0005500) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0B50 (0005520) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0B60 (0005540) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0B70 (0005560) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0B80 (0005600) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0B90 (0005620) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0BA0 (0005640) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0BB0 (0005660) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0BC0 (0005700) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0BD0 (0005720) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0BE0 (0005740) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0BF0 (0005760) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0C00 (0006000) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0C10 (0006020) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0C20 (0006040) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0C30 (0006060) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0C40 (0006100) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0C50 (0006120) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0C60 (0006140) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0C70 (0006160) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0C80 (0006200) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0C90 (0006220) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0CA0 (0006240) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0CB0 (0006260) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0CC0 (0006300) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0CD0 (0006320) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0CE0 (0006340) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0CF0 (0006360) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0D00 (0006400) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0D10 (0006420) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0D20 (0006440) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0D30 (0006460) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0D40 (0006500) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0D50 (0006520) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0D60 (0006540) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0D70 (0006560) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0D80 (0006600) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0D90 (0006620) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0DA0 (0006640) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0DB0 (0006660) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0DC0 (0006700) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0DD0 (0006720) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0DE0 (0006740) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0DF0 (0006760) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E00 (0007000) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E10 (0007020) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E20 (0007040) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E30 (0007060) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E40 (0007100) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E50 (0007120) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E60 (0007140) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E70 (0007160) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E80 (0007200) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E90 (0007220) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0EA0 (0007240) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0EB0 (0007260) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0EC0 (0007300) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0ED0 (0007320) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0EE0 (0007340) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0EF0 (0007360) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F00 (0007400) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F10 (0007420) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F20 (0007440) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F30 (0007460) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F40 (0007500) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F50 (0007520) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F60 (0007540) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F70 (0007560) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F80 (0007600) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F90 (0007620) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0FA0 (0007640) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0FB0 (0007660) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0FC0 (0007700) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0FD0 (0007720) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0FE0 (0007740) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x111A,  /* 0FF0 (0007760) */
    };
#endif
/*
  Bootrom code is from merging of the DELQA (M7516) rom image files:
             23-334e5.bin
         and 23-335e5.bin, 
         offset (0x5000 - 050000), for 4096. bytes.
*/
uint16 xq_bootrom_delqa[2048] = {
    0x00A0,0x0113,0x0166,0x0FFE,0x0150,0x0000,0x8000,0x8000,  /* 0000 (0000000) */
    0x0400,0xFC00,0x0000,0x0000,0x8000,0x8000,0x0C00,0xFC00,  /* 0010 (0000020) */
    0x0000,0x0000,0x8000,0x2000,0xC000,0x15C6,0x03FE,0x1026,  /* 0020 (0000040) */
    0x15F1,0x000C,0x0004,0x0A31,0x0006,0x15F1,0x0208,0x000E,  /* 0030 (0000060) */
    0x0A00,0x0AC0,0x02FE,0x0A31,0x000E,0x0A00,0x0AC0,0x02FE,  /* 0040 (0000100) */
    0x15F1,0x0002,0x000E,0x0A31,0x000E,0x45F7,0x3FFF,0xFFB4,  /* 0050 (0000120) */
    0x2DF7,0xFFB0,0xFFC2,0x022F,0x45F7,0x3FFF,0xFFAA,0x2DF7,  /* 0060 (0000140) */
    0xFFA6,0xFFB4,0x0228,0x45F7,0x3FFF,0xFFA8,0x2DF7,0xFFA4,  /* 0070 (0000160) */
    0xFFA6,0x0221,0x15C4,0x0FFE,0x0A03,0x0A02,0x9C80,0x0400,  /* 0080 (0000200) */
    0x0A82,0x45C0,0xFF00,0x6003,0x0AC4,0x02F8,0x17C0,0x0006,  /* 0090 (0000220) */
    0x2C03,0x0400,0x0210,0x005F,0x0408,0x15C2,0x1400,0x09F7,  /* 00A0 (0000240) */
    0x0020,0x0BC0,0x0208,0x1382,0x0202,0x0077,0x0958,0x25C2,  /* 00B0 (0000260) */
    0x01FF,0x8601,0x004A,0x17C3,0x000A,0x0301,0x004B,0x0000,  /* 00C0 (0000300) */
    0x01FE,0x119F,0x01F4,0x65C1,0x000E,0x35F1,0x8000,0xFFFE,  /* 00D0 (0000320) */
    0x0311,0x55F1,0x2000,0xFFFE,0x15C3,0x01FF,0x0A00,0x0AC0,  /* 00E0 (0000340) */
    0x02FE,0x35F1,0x1C00,0xFFFE,0x0305,0x0AC3,0x02F8,0x15C0,  /* 00F0 (0000360) */
    0x0080,0x012C,0x09F7,0x005E,0x55C9,0x0002,0x45C9,0x0002,  /* 0100 (0000400) */
    0x15C3,0x1FFF,0x0AC3,0x02FE,0x0977,0x074C,0xFFC0,0x020C,  /* 0110 (0000420) */
    0x0BCA,0x020C,0x35F2,0x0400,0x0028,0x0302,0x55C9,0x0400,  /* 0120 (0000440) */
    0x0977,0x0734,0xFF92,0x0302,0x55CA,0x0040,0x15C9,0x0002,  /* 0130 (0000460) */
    0x0A09,0x15C3,0x1FFF,0x0AC3,0x02FE,0x1240,0x45C0,0x6800,  /* 0140 (0000500) */
    0x25C0,0x1030,0x0302,0x55CA,0x0040,0x1280,0xE5C1,0x000E,  /* 0150 (0000520) */
    0x0087,0x5500,0xFFAA,0x17E6,0x001C,0x17E6,0x001E,0x11C3,  /* 0160 (0000540) */
    0x65C3,0x0088,0x10DF,0x001C,0x15DF,0x00E0,0x001E,0x8900,  /* 0170 (0000560) */
    0x159F,0x001E,0x159F,0x001C,0x17C0,0x01F2,0x55C0,0x00E0,  /* 0180 (0000600) */
    0x65C3,0x0008,0x10DF,0x0004,0x101F,0x0006,0x119F,0x01F6,  /* 0190 (0000620) */
    0x105F,0x01FA,0x109F,0x01F8,0x65C3,0x000A,0x10DF,0x0014,  /* 01A0 (0000640) */
    0x101F,0x0016,0x65C3,0x0012,0x15F1,0x01FC,0xFFFE,0x55F1,  /* 01B0 (0000660) */
    0x8000,0xFFFE,0x10DF,0x01FC,0x101F,0x01FE,0x0A32,0x0002,  /* 01C0 (0000700) */
    0x0A0A,0x15C0,0x0072,0x6080,0x15C4,0x004A,0x0A10,0x0AC4,  /* 01D0 (0000720) */
    0x02FD,0x0977,0x0682,0xFFB6,0x031C,0x0AB2,0x0002,0x25F2,  /* 01E0 (0000740) */
    0x0002,0x0002,0x06ED,0x0087,0x1D9F,0x0002,0x01F2,0x0006,  /* 01F0 (0000760) */
    0x55CA,0x0400,0x17C6,0x01F4,0x0087,0x55CA,0x2000,0x17C6,  /* 0200 (0001000) */
    0x01F6,0x17C1,0x01FA,0x17C2,0x01F8,0x0087,0x55CA,0x0800,  /* 0210 (0001020) */
    0x0002,0x8D17,0x0000,0x1043,0xE5C3,0x000E,0x1084,0x65C4,  /* 0220 (0001040) */
    0x000A,0x15C5,0x0006,0x14C0,0x9014,0x0AC5,0x02FC,0x1084,  /* 0230 (0001060) */
    0x65C4,0x000A,0x1105,0x0BD4,0x0204,0x0BD4,0x0202,0x0BCC,  /* 0240 (0001100) */
    0x0311,0x1144,0x15C0,0xFFFF,0x2014,0x020E,0x2014,0x020C,  /* 0250 (0001120) */
    0x200C,0x0308,0x0209,0x1144,0x25D4,0x00AA,0x0203,0xA5CC,  /* 0260 (0001140) */
    0x0004,0x8602,0x55CA,0x0001,0x15C0,0xFFFF,0x09F7,0x06CC,  /* 0270 (0001160) */
    0x17C5,0x01FC,0x11C3,0x65C3,0x0024,0x10DF,0x01FC,0x55C9,  /* 0280 (0001200) */
    0x0040,0x1131,0xFFFA,0x0A00,0x1031,0xFFFC,0x15C3,0x1000,  /* 0290 (0001220) */
    0x0AC3,0x02FE,0x55CA,0x0800,0x0087,0x65C6,0x0004,0x115F,  /* 02A0 (0001240) */
    0x01FC,0x0A80,0x03F7,0x1240,0x35C0,0x0080,0x03F3,0x1300,  /* 02B0 (0001260) */
    0x45C0,0x3FFF,0x25C0,0xC000,0x02ED,0x0BF4,0x0008,0x03EA,  /* 02C0 (0001300) */
    0x0BF4,0x000A,0x03E7,0x55C9,0x0002,0x45C9,0x0002,0x15C9,  /* 02D0 (0001320) */
    0x0400,0x8D1F,0x01F2,0x0977,0x057E,0xFFB6,0x0302,0x0087,  /* 02E0 (0001340) */
    0x11C3,0x65C3,0xFE70,0x10F2,0x0002,0x10F2,0x0006,0x65C3,  /* 02F0 (0001360) */
    0x0005,0x10F2,0x0004,0x0A32,0x0008,0x1083,0x65C3,0x0072,  /* 0300 (0001400) */
    0x15C0,0x0002,0x15C5,0x0006,0x8A13,0x15C4,0x0007,0x2CB2,  /* 0310 (0001420) */
    0x0004,0x0006,0x0603,0x1CB2,0x0002,0x0006,0x9E93,0x0006,  /* 0320 (0001440) */
    0x0AB2,0x0006,0x0AC4,0x02F3,0x0AC5,0x02EE,0x15C4,0x0010,  /* 0330 (0001460) */
    0x8A13,0x0AC4,0x02FD,0x0AC0,0x02E5,0x0977,0x051A,0xFFC5,  /* 0340 (0001500) */
    0x0303,0x55CA,0x0002,0x0087,0x0AB2,0x0008,0x25F2,0x01F4,  /* 0350 (0001520) */
    0x0008,0x06D3,0x45F1,0x8000,0xFFFE,0x0A32,0x0006,0x1084,  /* 0360 (0001540) */
    0x65C4,0x0073,0x1132,0x0008,0x0AC4,0x15C3,0x0030,0x8A14,  /* 0370 (0001560) */
    0x0AC3,0x02FD,0x65F2,0x0008,0x0002,0x1CB2,0x0008,0x0002,  /* 0380 (0001600) */
    0x15F2,0x0030,0x0004,0x8A32,0x000A,0x00B1,0x8C72,0x000A,  /* 0390 (0001620) */
    0x8621,0x8C72,0x000A,0x65F2,0x0008,0x0002,0x011B,0x1C84,  /* 03A0 (0001640) */
    0x0008,0x0AC4,0x15C3,0x0030,0x95D4,0x00FF,0x0AC3,0x02FC,  /* 03B0 (0001660) */
    0x15F2,0x0030,0x0004,0x1CB2,0x0008,0x0002,0x95F2,0x00FF,  /* 03C0 (0001700) */
    0x000A,0x00A1,0x8C72,0x000A,0x8705,0x8C72,0x000A,0x65F2,  /* 03D0 (0001720) */
    0x0008,0x0002,0x1C84,0x0002,0x15C3,0x0007,0x8A34,0xFFF8,  /* 03E0 (0001740) */
    0x9C94,0x000A,0x0AC3,0x02FA,0x0977,0x046C,0xFFE8,0x0302,  /* 03F0 (0001760) */
    0x0077,0x010C,0x0BF2,0x0006,0x0209,0x00A1,0x8C7A,0x0002,  /* 0400 (0002000) */
    0x8608,0x8C3A,0x0002,0x8C3A,0x0002,0x0103,0x00B1,0x8C7A,  /* 0410 (0002020) */
    0x0002,0x1C84,0x0008,0x1083,0x65C3,0x08F2,0x15C0,0x0006,  /* 0420 (0002040) */
    0x9313,0x65C4,0x0008,0x0AC0,0x02FB,0x15C9,0x0001,0x15C0,  /* 0430 (0002060) */
    0xFFFD,0x09F7,0x04EA,0x1131,0xFFF6,0x0A31,0xFFF8,0x09F7,  /* 0440 (0002100) */
    0x04FA,0x1131,0xFFFA,0x0A31,0xFFFC,0x15C0,0x7FFF,0x35C9,  /* 0450 (0002120) */
    0x8000,0x0203,0x0AC0,0x02FB,0x0153,0x0A00,0x0977,0x050A,  /* 0460 (0002140) */
    0x0000,0x4000,0x024D,0x1C84,0x0008,0x0A84,0x1083,0x65C3,  /* 0470 (0002160) */
    0x08F2,0x15C0,0x0006,0x9313,0x65C4,0x0008,0x0AC0,0x02FB,  /* 0480 (0002200) */
    0x45C9,0x8080,0x15C0,0xFFFD,0x09F7,0x0494,0x1131,0xFFF6,  /* 0490 (0002220) */
    0x0A31,0xFFF8,0x15C0,0xFFFD,0x09F7,0x04A0,0x1131,0xFFFA,  /* 04A0 (0002240) */
    0x0A31,0xFFFC,0x15C0,0x7FFF,0x35C9,0x8000,0x0203,0x0AC0,  /* 04B0 (0002260) */
    0x02FB,0x012C,0x0A00,0x0977,0x04B0,0x0000,0x0000,0x0220,  /* 04C0 (0002300) */
    0x45C9,0x0001,0x1084,0x65C4,0x02F2,0x1085,0x65C5,0x08F2,  /* 04D0 (0002320) */
    0x15C0,0x0006,0xA515,0x0214,0x0AC0,0x02FC,0x0AF2,0x0004,  /* 04E0 (0002340) */
    0x0207,0x0BF2,0x0006,0x021C,0x0AB2,0x0006,0x0077,0xFEAE,  /* 04F0 (0002360) */
    0x0BF2,0x0006,0x0202,0x0077,0xFE92,0x00B1,0x0077,0xFEC4,  /* 0500 (0002400) */
    0x55CA,0x0004,0x0105,0x55CA,0x0800,0x0102,0x55CA,0x0204,  /* 0510 (0002420) */
    0x55F1,0x8000,0xFFFE,0x0087,0x003C,0x0312,0x05EA,0xFFFF,  /* 0520 (0002440) */
    0x55F1,0x8000,0xFFFE,0x55C9,0x0200,0x11C3,0x65C3,0xFC26,  /* 0530 (0002460) */
    0x10F2,0x0002,0x10F2,0x0006,0x65C3,0x0004,0x10F2,0x0004,  /* 0540 (0002500) */
    0x11C3,0x65C3,0xFFD6,0x10F2,0x0008,0x1083,0x65C3,0x08F2,  /* 0550 (0002520) */
    0x1E85,0x0008,0x2CB2,0x0004,0x0006,0x0603,0x1CB2,0x0002,  /* 0560 (0002540) */
    0x0006,0x9E93,0x0006,0x0AB2,0x0006,0x0AC5,0x02F3,0x1E80,  /* 0570 (0002560) */
    0x0008,0x0977,0x035E,0x8715,0x1E80,0x0008,0x0977,0x03EA,  /* 0580 (0002600) */
    0x0000,0x2000,0x0301,0x0003,0x1084,0x65C4,0x02F2,0x1085,  /* 0590 (0002620) */
    0x65C5,0x08F2,0x1E80,0x0008,0xA515,0x0203,0x0AC0,0x02FC,  /* 05A0 (0002640) */
    0x0103,0x55CA,0x0008,0x0087,0x65F2,0x0002,0x0008,0x25FA,  /* 05B0 (0002660) */
    0xFFFF,0x0008,0x0302,0x0077,0xFF90,0x15C0,0xFFF8,0x09F7,  /* 05C0 (0002700) */
    0x035E,0x1131,0xFFF6,0x0A31,0xFFF8,0x1083,0x65C3,0x0020,  /* 05D0 (0002720) */
    0x10C4,0x15C0,0x0008,0x15D3,0x8000,0x15D3,0x8000,0x108B,  /* 05E0 (0002740) */
    0x65D3,0x000A,0x15D3,0xFFFF,0x0A13,0x0A13,0x0AC0,0x02F3,  /* 05F0 (0002760) */
    0x1103,0x55F3,0x0080,0x0002,0x55F3,0x00C0,0x000E,0x15F3,  /* 0600 (0003000) */
    0xFFFE,0x0012,0x55F3,0x0040,0x001A,0x65F3,0x0002,0x001C,  /* 0610 (0003020) */
    0x65F3,0x0004,0x0028,0x15F3,0x4000,0x0032,0x10F3,0x0034,  /* 0620 (0003040) */
    0x65F3,0x003C,0x0034,0x10F3,0x0040,0x65F3,0x0038,0x0040,  /* 0630 (0003060) */
    0x15F3,0xFFFE,0x0042,0x15F3,0xA000,0x004A,0x15F3,0xFFFD,  /* 0640 (0003100) */
    0x004E,0x15F3,0x2000,0x0056,0x1131,0xFFFA,0x0A31,0xFFFC,  /* 0650 (0003120) */
    0x15C0,0x7FFF,0x35C9,0x8000,0x0205,0x0AC0,0x02FB,0x55CA,  /* 0660 (0003140) */
    0x0200,0x0109,0x15C0,0x0010,0x0977,0x02FE,0x8000,0x2000,  /* 0670 (0003160) */
    0x0305,0x55CA,0x4000,0x55CA,0x0010,0x0087,0x1083,0x65C3,  /* 0680 (0003200) */
    0x0020,0x15C0,0x0008,0x1CC5,0x0000,0x45C5,0x3FFF,0x25C5,  /* 0690 (0003220) */
    0xC000,0x02EF,0x25C0,0x0004,0x0303,0x25C0,0x0001,0x0207,  /* 06A0 (0003240) */
    0x0BF3,0x0008,0x02E6,0x0BF3,0x000A,0x02E3,0x0114,0x25C0,  /* 06B0 (0003260) */
    0x0002,0x0305,0x35F3,0x8000,0x0008,0x03DB,0x010C,0x1CC5,  /* 06C0 (0003300) */
    0x0008,0x45C5,0x2D0F,0x25C5,0x0000,0x02D3,0x1CC5,0x000A,  /* 06D0 (0003320) */
    0x45C5,0xFC00,0x03CE,0x65C3,0x000C,0x0AC0,0x02D4,0x1083,  /* 06E0 (0003340) */
    0x65C3,0x000A,0x1084,0x65C4,0x02F2,0x1105,0x24D4,0x02C3,  /* 06F0 (0003360) */
    0x24D4,0x02C1,0x24D4,0x02BF,0x0BD4,0x02BD,0x0BD4,0x02BB,  /* 0700 (0003400) */
    0x2554,0x02B9,0x2554,0x02B7,0x2554,0x02B5,0x1083,0x65C3,  /* 0710 (0003420) */
    0x0072,0x10C5,0x15C0,0x004B,0x0A13,0x0AC0,0x02FD,0x1084,  /* 0720 (0003440) */
    0x65C4,0x000A,0x09F7,0x0026,0x65C5,0x0010,0x09F7,0x001E,  /* 0730 (0003460) */
    0x0977,0x0124,0xFFBC,0x0301,0x0087,0x15C0,0xFFFF,0x35C9,  /* 0740 (0003500) */
    0x2000,0x0313,0x0AC0,0x02FB,0x55CA,0x8000,0x0087,0x1126,  /* 0750 (0003520) */
    0x15C0,0x0006,0x0A85,0x15C3,0x0007,0x9315,0x0AC3,0x02FD,  /* 0760 (0003540) */
    0x0A84,0x0AC0,0x02F7,0x1584,0x0087,0x0BCA,0x020F,0x55C9,  /* 0770 (0003560) */
    0x0300,0x15C0,0x002E,0x09F7,0x005A,0x15C0,0x003C,0x0977,  /* 0780 (0003600) */
    0x0152,0x8605,0x45CA,0x0200,0x55CA,0x8000,0x0087,0x15C0,  /* 0790 (0003620) */
    0x003C,0x09F7,0x006E,0x03F1,0x0303,0x55CA,0x0020,0x0087,  /* 07A0 (0003640) */
    0x15C0,0x003C,0x09F7,0x009A,0x02F8,0x15C0,0x05DC,0x09F7,  /* 07B0 (0003660) */
    0x0022,0x15C0,0x05EA,0x0977,0x011A,0x87EF,0x15C0,0x05EA,  /* 07C0 (0003700) */
    0x09F7,0x0040,0x03F6,0x02E9,0x15C0,0x05EA,0x09F7,0x0072,  /* 07D0 (0003720) */
    0x02E4,0x0087,0x1083,0x65C3,0x08F2,0x1084,0x65C4,0x000A,  /* 07E0 (0003740) */
    0x1105,0x1513,0x1513,0x1513,0x1553,0x1553,0x1553,0x15D3,  /* 07F0 (0003760) */
    0x0090,0x0A13,0x15D3,0x0001,0x9013,0x0AC0,0x02FD,0x0A32,  /* 0800 (0004000) */
    0x0002,0x0087,0x0977,0x0162,0x0000,0x2000,0x0204,0x65CE,  /* 0810 (0004020) */
    0x0002,0x0BCA,0x0087,0x1C80,0x0028,0x45C0,0xBFFF,0x0208,  /* 0820 (0004040) */
    0x1C80,0x0018,0x35C0,0xBFFF,0x0203,0x45CA,0x4000,0x01EF,  /* 0830 (0004060) */
    0x0AB2,0x0002,0x25F2,0x0064,0x0002,0x07E9,0x45CA,0x4000,  /* 0840 (0004100) */
    0x01E8,0x1084,0x65C4,0x02F2,0x1085,0x65C5,0x08F2,0xA515,  /* 0850 (0004120) */
    0x0202,0x0AC0,0x02FC,0x0087,0x1080,0x65C0,0x0072,0x1026,  /* 0860 (0004140) */
    0x1340,0x0AC0,0x09F7,0x00B8,0x1131,0xFFF6,0x0A31,0xFFF8,  /* 0870 (0004160) */
    0x1540,0x09F7,0x00C6,0x15F4,0xB000,0x0002,0x13B4,0x0004,  /* 0880 (0004200) */
    0x1131,0xFFFA,0x0A31,0xFFFC,0x15C4,0x1FFF,0x35C9,0x8000,  /* 0890 (0004220) */
    0x0208,0x0AC4,0x02FB,0x55CA,0x0200,0x0BD6,0x55CA,0x1000,  /* 08A0 (0004240) */
    0x0118,0x1D40,0xFFFE,0x0CC0,0x0B00,0x0977,0x00BC,0x0000,  /* 08B0 (0004260) */
    0x2000,0x02F3,0x1D40,0xFFFE,0x0CC0,0x0B00,0x1084,0x65C4,  /* 08C0 (0004300) */
    0x02F2,0x1583,0xA513,0x02EA,0x0AC0,0x02FC,0x25CC,0x52E3,  /* 08D0 (0004320) */
    0x02E5,0x0085,0x0A26,0x0C00,0x0B4E,0x6380,0x0B00,0x09F7,  /* 08E0 (0004340) */
    0x003E,0x1131,0xFFF6,0x0A31,0xFFF8,0x09F7,0x004E,0x0BD6,  /* 08F0 (0004360) */
    0x0303,0x55F4,0x0080,0x0002,0x1131,0xFFFA,0x0A31,0xFFFC,  /* 0900 (0004400) */
    0x15C3,0x01FF,0x0A00,0x35C9,0x8000,0x0208,0x0AC0,0x02FE,  /* 0910 (0004420) */
    0x0AC3,0x02F9,0x55CA,0x0200,0x00B1,0x0101,0x00A1,0x0085,  /* 0920 (0004440) */
    0x1083,0x65C3,0x02F2,0x15C4,0x0080,0x15D3,0x52E3,0x0AC4,  /* 0930 (0004460) */
    0x02FC,0x0977,0x0010,0x0010,0x02F2,0x0087,0x0977,0x0006,  /* 0940 (0004500) */
    0x0020,0x08F2,0x0087,0x1543,0x6083,0x10C4,0x15D3,0x8000,  /* 0950 (0004520) */
    0x15D3,0xA000,0x154B,0x6093,0x1013,0x15D3,0x8000,0x15D3,  /* 0960 (0004540) */
    0x8000,0x15D3,0x8000,0x0A13,0x0085,0x1026,0x0A00,0x1244,  /* 0970 (0004560) */
    0x55C9,0x8080,0x45C4,0x7F01,0x25C4,0x80B0,0x0301,0x11C0,  /* 0980 (0004600) */
    0x1C84,0x0020,0x45C4,0x3FFF,0x25C4,0xC000,0x0301,0x11C0,  /* 0990 (0004620) */
    0x1C84,0x0028,0x8002,0x45C4,0x7FFF,0x45C4,0x7D0F,0x2544,  /* 09A0 (0004640) */
    0x0301,0x11C0,0x1C84,0x002A,0x45C4,0xFC00,0x0201,0x11C0,  /* 09B0 (0004660) */
    0x1C84,0x0010,0x45C4,0x3FFF,0x25C4,0xC000,0x0301,0x11C0,  /* 09C0 (0004700) */
    0x1C84,0x0018,0x1126,0x45C4,0x0FFF,0x2544,0x0301,0x11C0,  /* 09D0 (0004720) */
    0x1584,0x45C4,0xF8FF,0x25F2,0xFFFD,0x0026,0x0202,0x0BD6,  /* 09E0 (0004740) */
    0x010D,0x35F2,0x1000,0x0022,0x0301,0x0A04,0x1C83,0x001A,  /* 09F0 (0004760) */
    0x45C3,0xFF00,0x60C4,0x2584,0x0301,0x11C0,0x1003,0x0302,  /* 0A00 (0005000) */
    0x55CA,0x4000,0x0085,0x0005,0x15C6,0x1FFE,0x118E,0x13A6,  /* 0A10 (0005020) */
    0xE5CE,0x062E,0x11C3,0x65C3,0x000C,0x10DF,0x0004,0x1240,  /* 0A20 (0005040) */
    0x0101,0x0000,0x09F7,0x0098,0x1385,0x15D5,0x2508,0x15D5,  /* 0A30 (0005060) */
    0x0101,0x8A15,0x8A15,0x15D5,0x0191,0x95D5,0x0002,0x15C2,  /* 0A40 (0005100) */
    0x05EA,0x9095,0x00C2,0x9095,0x15C2,0x000B,0x0A00,0x09F7,  /* 0A50 (0005120) */
    0x012A,0x02E8,0x15C2,0x05EE,0x09F7,0x01B0,0x0226,0x1383,  /* 0A60 (0005140) */
    0x94C4,0x0AC2,0xA117,0x0002,0x0304,0x8BC4,0x02DD,0xE5C2,  /* 0A70 (0005160) */
    0x0004,0x8BCB,0x0302,0xA2C0,0x0218,0x94C0,0x8A80,0x0AC2,  /* 0A80 (0005200) */
    0x0708,0x14C5,0x0BD3,0xE5C2,0x0004,0x0703,0x94D5,0x0AC2,  /* 0A90 (0005220) */
    0x06FD,0x8BC4,0x030F,0x0A03,0x1385,0x95D5,0x000A,0x9015,  /* 0AA0 (0005240) */
    0x90D5,0x0A05,0x15C2,0x0003,0x01D2,0x8BC0,0x03BB,0x15C3,  /* 0AB0 (0005260) */
    0x0001,0x01F2,0x09F7,0x029A,0x94E6,0x94F6,0x0001,0x0087,  /* 0AC0 (0005300) */
    0x45F1,0x0002,0x000E,0x1D85,0x0002,0x1144,0x65C4,0xFFD0,  /* 0AD0 (0005320) */
    0x1136,0x0004,0x15C2,0x0010,0x0A14,0x7E82,0x1144,0x65C4,  /* 0AE0 (0005340) */
    0xFFF0,0x0A35,0xFFD2,0x1135,0xFFD4,0x0A35,0xFFE2,0x1135,  /* 0AF0 (0005360) */
    0xFFE4,0x55F5,0x8000,0xFFD2,0x15C2,0x05EE,0x0C82,0x0B02,  /* 0B00 (0005400) */
    0x10B5,0xFFD6,0x55F5,0xA000,0xFFE2,0x1D84,0x0004,0x1137,  /* 0B10 (0005420) */
    0x02D4,0x0A37,0x02D2,0x65C4,0x0010,0x1137,0x02CC,0x0A37,  /* 0B20 (0005440) */
    0x02CA,0x9C77,0x0000,0x02B2,0x9C77,0x0002,0x02AD,0x9C77,  /* 0B30 (0005460) */
    0x0004,0x02A8,0x9C77,0x0006,0x02A3,0x9C77,0x0008,0x029E,  /* 0B40 (0005500) */
    0x9C77,0x000A,0x0299,0x8AB7,0x029A,0x45F1,0x0002,0x000E,  /* 0B50 (0005520) */
    0x55F1,0x0100,0x000E,0x09F7,0x0244,0x0A35,0x0002,0x1DE4,  /* 0B60 (0005540) */
    0x027C,0x1DE4,0x0276,0x1DE4,0x0270,0x15E4,0x0000,0x15E4,  /* 0B70 (0005560) */
    0x0200,0x15E4,0x00AB,0x09F7,0x0026,0x0087,0x1D85,0x0002,  /* 0B80 (0005600) */
    0x1144,0x10A4,0x15E4,0x0160,0x1DE4,0x0252,0x1DE4,0x024C,  /* 0B90 (0005620) */
    0x1DE4,0x0246,0x1DE4,0x0230,0x1DE4,0x022A,0x1DE4,0x0224,  /* 0BA0 (0005640) */
    0x65C5,0xFFE0,0x65C2,0x000E,0x2097,0x003C,0x0403,0x15C2,  /* 0BB0 (0005660) */
    0x003C,0x0107,0x35C2,0x0001,0x0304,0x55F5,0x0080,0x0002,  /* 0BC0 (0005700) */
    0x0A82,0x0C82,0x0B02,0x10B5,0x0006,0x0A35,0x0008,0x0A35,  /* 0BD0 (0005720) */
    0x000A,0x1DF1,0x0214,0x0008,0x1DF1,0x0210,0x000A,0x15C4,  /* 0BE0 (0005740) */
    0x0084,0x09F7,0x0188,0x860A,0x0304,0x35F5,0x0200,0x0008,  /* 0BF0 (0005760) */
    0x02EC,0x45F5,0x0080,0x0002,0x00A4,0x0087,0x45F5,0x0080,  /* 0C00 (0006000) */
    0x0002,0x35C2,0x4004,0x0301,0x0000,0x0087,0x1D85,0x0002,  /* 0C10 (0006020) */
    0x65C5,0xFFD0,0x0A35,0x0008,0x0A35,0x000A,0x1DF1,0x01C6,  /* 0C20 (0006040) */
    0x0004,0x1DF1,0x01C2,0x0006,0x55F1,0x0001,0x000E,0x15C4,  /* 0C30 (0006060) */
    0x8004,0x09F7,0x0138,0x8618,0x02ED,0x55F1,0x0002,0x000E,  /* 0C40 (0006100) */
    0x15F7,0x00AB,0x017E,0x15F7,0x0100,0x017A,0x15F7,0x0000,  /* 0C50 (0006120) */
    0x0176,0x8AB7,0x018E,0x0A00,0x45F1,0x0002,0x000E,0x55F1,  /* 0C60 (0006140) */
    0x0100,0x000E,0x00A4,0x0087,0x45F1,0x0001,0x000E,0x55F1,  /* 0C70 (0006160) */
    0x0002,0x000E,0x1D85,0x0002,0x1D42,0xFFFE,0x45F1,0x0002,  /* 0C80 (0006200) */
    0x000E,0x55F1,0x0100,0x000E,0x25F5,0x0160,0xFFFC,0x0221,  /* 0C90 (0006220) */
    0x8BF7,0x0150,0x020D,0x2DF5,0x012E,0xFFFA,0x02B7,0x2DF5,  /* 0CA0 (0006240) */
    0x0124,0xFFF8,0x02B3,0x2DF5,0x011A,0xFFF6,0x02AF,0x0087,  /* 0CB0 (0006260) */
    0xA5CD,0x0003,0x02AB,0x1D77,0xFFF6,0x0108,0x1D77,0xFFF8,  /* 0CC0 (0006300) */
    0x0104,0x1D77,0xFFFA,0x0100,0x8A37,0x0118,0x00A4,0x0A00,  /* 0CD0 (0006320) */
    0x0087,0x25F5,0x0090,0xFFFC,0x0313,0x25F5,0x0260,0xFFFC,  /* 0CE0 (0006340) */
    0x0295,0xA5CD,0x0005,0x0292,0x09F7,0x00B2,0x1D34,0xFFFE,  /* 0CF0 (0006360) */
    0xFFF8,0x1D34,0xFFFC,0xFFF6,0x1D34,0xFFFA,0xFFF4,0x011F,  /* 0D00 (0006400) */
    0x1144,0x6084,0x1143,0x65C3,0xFFD0,0x1CC2,0x0008,0x45C2,  /* 0D10 (0006420) */
    0xF8FF,0xDCC2,0x000A,0x65C2,0x002E,0x25D4,0x0002,0x0217,  /* 0D20 (0006440) */
    0x65F5,0x0008,0xFFFE,0x35CC,0x0001,0x0211,0x1143,0x65C3,  /* 0D30 (0006460) */
    0xFFF0,0x1513,0x1513,0x1513,0x1144,0x65C4,0xFFFC,0x1DE4,  /* 0D40 (0006500) */
    0x009C,0x1DE4,0x0096,0x1DE4,0x0090,0x09F7,0xFE52,0x0077,  /* 0D50 (0006520) */
    0xFEBA,0x1DDF,0x006E,0x0018,0x1DDF,0x006A,0x001A,0x1DDF,  /* 0D60 (0006540) */
    0x0066,0x001C,0x55F1,0x0002,0x000E,0x00B4,0x0087,0x15C3,  /* 0D70 (0006560) */
    0x3FFF,0x00A1,0x15C2,0x0090,0x3131,0x000E,0x0206,0x0AC3,  /* 0D80 (0006600) */
    0x02FE,0x0AC2,0x02F9,0x00B1,0x0087,0x1C42,0x000E,0x10B1,  /* 0D90 (0006620) */
    0x000E,0x35F5,0x4000,0x0008,0x0301,0x00B1,0x0087,0x1166,  /* 0DA0 (0006640) */
    0x11C3,0x65C3,0x0028,0x15C2,0x0018,0x14D5,0x0BD5,0x94D5,  /* 0DB0 (0006660) */
    0x0AC2,0x02FD,0x1585,0x1144,0x15C2,0x001C,0x10A4,0x15E4,  /* 0DC0 (0006700) */
    0x0260,0x0087,0x00AB,0x0100,0x0000,0x0007,0x0001,0x0303,  /* 0DD0 (0006720) */
    0x0000,0x0002,0x0102,0x0700,0x0600,0x0000,0x0000,0x0000,  /* 0DE0 (0006740) */
    0x0064,0x2501,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0DF0 (0006760) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E00 (0007000) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E10 (0007020) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E20 (0007040) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E30 (0007060) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E40 (0007100) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E50 (0007120) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E60 (0007140) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E70 (0007160) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E80 (0007200) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E90 (0007220) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0EA0 (0007240) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0EB0 (0007260) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0EC0 (0007300) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0ED0 (0007320) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0EE0 (0007340) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0EF0 (0007360) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F00 (0007400) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F10 (0007420) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F20 (0007440) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F30 (0007460) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F40 (0007500) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F50 (0007520) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F60 (0007540) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F70 (0007560) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F80 (0007600) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F90 (0007620) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0FA0 (0007640) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0FB0 (0007660) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0FC0 (0007700) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0FD0 (0007720) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0FE0 (0007740) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x8486,  /* 0FF0 (0007760) */
    };

/*
  Bootrom code is from merging of the DELQA-Plus aka Turbo DELQA (M7516-YM)
  rom image files:
             23-365e5.bin
         and 23-366e5.bin, 
         offset (0x5000 - 050000), for 4096. bytes.
*/
uint16 xq_bootrom_delqat[2048] = {
    0x00A0,0x0113,0x0166,0x0FFE,0x0150,0x0000,0x8000,0x8000,  /* 0000 (0000000) */
    0x0400,0xFC00,0x0000,0x0000,0x8000,0x8000,0x0C00,0xFC00,  /* 0010 (0000020) */
    0x0000,0x0000,0x8000,0x2000,0xC000,0x15C6,0x03FE,0x1026,  /* 0020 (0000040) */
    0x15F1,0x000C,0x0004,0x0A31,0x0006,0x15F1,0x0208,0x000E,  /* 0030 (0000060) */
    0x0A00,0x0AC0,0x02FE,0x0A31,0x000E,0x0A00,0x0AC0,0x02FE,  /* 0040 (0000100) */
    0x15F1,0x0002,0x000E,0x0A31,0x000E,0x45F7,0x3FFF,0xFFB4,  /* 0050 (0000120) */
    0x2DF7,0xFFB0,0xFFC2,0x022F,0x45F7,0x3FFF,0xFFAA,0x2DF7,  /* 0060 (0000140) */
    0xFFA6,0xFFB4,0x0228,0x45F7,0x3FFF,0xFFA8,0x2DF7,0xFFA4,  /* 0070 (0000160) */
    0xFFA6,0x0221,0x15C4,0x0FFE,0x0A03,0x0A02,0x9C80,0x0400,  /* 0080 (0000200) */
    0x0A82,0x45C0,0xFF00,0x6003,0x0AC4,0x02F8,0x17C0,0x0006,  /* 0090 (0000220) */
    0x2C03,0x0400,0x0210,0x005F,0x0408,0x15C2,0x1400,0x09F7,  /* 00A0 (0000240) */
    0x0020,0x0BC0,0x0208,0x1382,0x0202,0x0077,0x0958,0x25C2,  /* 00B0 (0000260) */
    0x01FF,0x8601,0x004A,0x17C3,0x000A,0x0301,0x004B,0x0000,  /* 00C0 (0000300) */
    0x01FE,0x119F,0x01F4,0x65C1,0x000E,0x35F1,0x8000,0xFFFE,  /* 00D0 (0000320) */
    0x0311,0x55F1,0x2000,0xFFFE,0x15C3,0x01FF,0x0A00,0x0AC0,  /* 00E0 (0000340) */
    0x02FE,0x35F1,0x1C00,0xFFFE,0x0305,0x0AC3,0x02F8,0x15C0,  /* 00F0 (0000360) */
    0x0080,0x012C,0x09F7,0x005E,0x55C9,0x0002,0x45C9,0x0002,  /* 0100 (0000400) */
    0x15C3,0x1FFF,0x0AC3,0x02FE,0x0977,0x074C,0xFFC0,0x020C,  /* 0110 (0000420) */
    0x0BCA,0x020C,0x35F2,0x0400,0x0028,0x0302,0x55C9,0x0400,  /* 0120 (0000440) */
    0x0977,0x0734,0xFF92,0x0302,0x55CA,0x0040,0x15C9,0x0002,  /* 0130 (0000460) */
    0x0A09,0x15C3,0x1FFF,0x0AC3,0x02FE,0x1240,0x45C0,0x6800,  /* 0140 (0000500) */
    0x25C0,0x1030,0x0302,0x55CA,0x0040,0x1280,0xE5C1,0x000E,  /* 0150 (0000520) */
    0x0087,0x5500,0xFFAA,0x17E6,0x001C,0x17E6,0x001E,0x11C3,  /* 0160 (0000540) */
    0x65C3,0x0088,0x10DF,0x001C,0x15DF,0x00E0,0x001E,0x8900,  /* 0170 (0000560) */
    0x159F,0x001E,0x159F,0x001C,0x17C0,0x01F2,0x55C0,0x00E0,  /* 0180 (0000600) */
    0x65C3,0x0008,0x10DF,0x0004,0x101F,0x0006,0x119F,0x01F6,  /* 0190 (0000620) */
    0x105F,0x01FA,0x109F,0x01F8,0x65C3,0x000A,0x10DF,0x0014,  /* 01A0 (0000640) */
    0x101F,0x0016,0x65C3,0x0012,0x15F1,0x01FC,0xFFFE,0x55F1,  /* 01B0 (0000660) */
    0x8000,0xFFFE,0x10DF,0x01FC,0x101F,0x01FE,0x0A32,0x0002,  /* 01C0 (0000700) */
    0x0A0A,0x15C0,0x0072,0x6080,0x15C4,0x004A,0x0A10,0x0AC4,  /* 01D0 (0000720) */
    0x02FD,0x0977,0x0682,0xFFB6,0x031C,0x0AB2,0x0002,0x25F2,  /* 01E0 (0000740) */
    0x0002,0x0002,0x06ED,0x0087,0x1D9F,0x0002,0x01F2,0x0006,  /* 01F0 (0000760) */
    0x55CA,0x0400,0x17C6,0x01F4,0x0087,0x55CA,0x2000,0x17C6,  /* 0200 (0001000) */
    0x01F6,0x17C1,0x01FA,0x17C2,0x01F8,0x0087,0x55CA,0x0800,  /* 0210 (0001020) */
    0x0002,0x8D17,0x0000,0x1043,0xE5C3,0x000E,0x1084,0x65C4,  /* 0220 (0001040) */
    0x000A,0x15C5,0x0006,0x14C0,0x9014,0x0AC5,0x02FC,0x1084,  /* 0230 (0001060) */
    0x65C4,0x000A,0x1105,0x0BD4,0x0204,0x0BD4,0x0202,0x0BCC,  /* 0240 (0001100) */
    0x0311,0x1144,0x15C0,0xFFFF,0x2014,0x020E,0x2014,0x020C,  /* 0250 (0001120) */
    0x200C,0x0308,0x0209,0x1144,0x25D4,0x00AA,0x0203,0xA5CC,  /* 0260 (0001140) */
    0x0004,0x8602,0x55CA,0x0001,0x15C0,0xFFFF,0x09F7,0x06CC,  /* 0270 (0001160) */
    0x17C5,0x01FC,0x11C3,0x65C3,0x0024,0x10DF,0x01FC,0x55C9,  /* 0280 (0001200) */
    0x0040,0x1131,0xFFFA,0x0A00,0x1031,0xFFFC,0x15C3,0x1000,  /* 0290 (0001220) */
    0x0AC3,0x02FE,0x55CA,0x0800,0x0087,0x65C6,0x0004,0x115F,  /* 02A0 (0001240) */
    0x01FC,0x0A80,0x03F7,0x1240,0x35C0,0x0080,0x03F3,0x1300,  /* 02B0 (0001260) */
    0x45C0,0x3FFF,0x25C0,0xC000,0x02ED,0x0BF4,0x0008,0x03EA,  /* 02C0 (0001300) */
    0x0BF4,0x000A,0x03E7,0x55C9,0x0002,0x45C9,0x0002,0x15C9,  /* 02D0 (0001320) */
    0x0400,0x8D1F,0x01F2,0x0977,0x057E,0xFFB6,0x0302,0x0087,  /* 02E0 (0001340) */
    0x11C3,0x65C3,0xFE70,0x10F2,0x0002,0x10F2,0x0006,0x65C3,  /* 02F0 (0001360) */
    0x0005,0x10F2,0x0004,0x0A32,0x0008,0x1083,0x65C3,0x0072,  /* 0300 (0001400) */
    0x15C0,0x0002,0x15C5,0x0006,0x8A13,0x15C4,0x0007,0x2CB2,  /* 0310 (0001420) */
    0x0004,0x0006,0x0603,0x1CB2,0x0002,0x0006,0x9E93,0x0006,  /* 0320 (0001440) */
    0x0AB2,0x0006,0x0AC4,0x02F3,0x0AC5,0x02EE,0x15C4,0x0010,  /* 0330 (0001460) */
    0x8A13,0x0AC4,0x02FD,0x0AC0,0x02E5,0x0977,0x051A,0xFFC5,  /* 0340 (0001500) */
    0x0303,0x55CA,0x0002,0x0087,0x0AB2,0x0008,0x25F2,0x01F4,  /* 0350 (0001520) */
    0x0008,0x06D3,0x45F1,0x8000,0xFFFE,0x0A32,0x0006,0x1084,  /* 0360 (0001540) */
    0x65C4,0x0073,0x1132,0x0008,0x0AC4,0x15C3,0x0030,0x8A14,  /* 0370 (0001560) */
    0x0AC3,0x02FD,0x65F2,0x0008,0x0002,0x1CB2,0x0008,0x0002,  /* 0380 (0001600) */
    0x15F2,0x0030,0x0004,0x8A32,0x000A,0x00B1,0x8C72,0x000A,  /* 0390 (0001620) */
    0x8621,0x8C72,0x000A,0x65F2,0x0008,0x0002,0x011B,0x1C84,  /* 03A0 (0001640) */
    0x0008,0x0AC4,0x15C3,0x0030,0x95D4,0x00FF,0x0AC3,0x02FC,  /* 03B0 (0001660) */
    0x15F2,0x0030,0x0004,0x1CB2,0x0008,0x0002,0x95F2,0x00FF,  /* 03C0 (0001700) */
    0x000A,0x00A1,0x8C72,0x000A,0x8705,0x8C72,0x000A,0x65F2,  /* 03D0 (0001720) */
    0x0008,0x0002,0x1C84,0x0002,0x15C3,0x0007,0x8A34,0xFFF8,  /* 03E0 (0001740) */
    0x9C94,0x000A,0x0AC3,0x02FA,0x0977,0x046C,0xFFE8,0x0302,  /* 03F0 (0001760) */
    0x0077,0x010C,0x0BF2,0x0006,0x0209,0x00A1,0x8C7A,0x0002,  /* 0400 (0002000) */
    0x8608,0x8C3A,0x0002,0x8C3A,0x0002,0x0103,0x00B1,0x8C7A,  /* 0410 (0002020) */
    0x0002,0x1C84,0x0008,0x1083,0x65C3,0x08F2,0x15C0,0x0006,  /* 0420 (0002040) */
    0x9313,0x65C4,0x0008,0x0AC0,0x02FB,0x15C9,0x0001,0x15C0,  /* 0430 (0002060) */
    0xFFFD,0x09F7,0x04EA,0x1131,0xFFF6,0x0A31,0xFFF8,0x09F7,  /* 0440 (0002100) */
    0x04FA,0x1131,0xFFFA,0x0A31,0xFFFC,0x15C0,0x7FFF,0x35C9,  /* 0450 (0002120) */
    0x8000,0x0203,0x0AC0,0x02FB,0x0153,0x0A00,0x0977,0x050A,  /* 0460 (0002140) */
    0x0000,0x4000,0x024D,0x1C84,0x0008,0x0A84,0x1083,0x65C3,  /* 0470 (0002160) */
    0x08F2,0x15C0,0x0006,0x9313,0x65C4,0x0008,0x0AC0,0x02FB,  /* 0480 (0002200) */
    0x45C9,0x8080,0x15C0,0xFFFD,0x09F7,0x0494,0x1131,0xFFF6,  /* 0490 (0002220) */
    0x0A31,0xFFF8,0x15C0,0xFFFD,0x09F7,0x04A0,0x1131,0xFFFA,  /* 04A0 (0002240) */
    0x0A31,0xFFFC,0x15C0,0x7FFF,0x35C9,0x8000,0x0203,0x0AC0,  /* 04B0 (0002260) */
    0x02FB,0x012C,0x0A00,0x0977,0x04B0,0x0000,0x0000,0x0220,  /* 04C0 (0002300) */
    0x45C9,0x0001,0x1084,0x65C4,0x02F2,0x1085,0x65C5,0x08F2,  /* 04D0 (0002320) */
    0x15C0,0x0006,0xA515,0x0214,0x0AC0,0x02FC,0x0AF2,0x0004,  /* 04E0 (0002340) */
    0x0207,0x0BF2,0x0006,0x021C,0x0AB2,0x0006,0x0077,0xFEAE,  /* 04F0 (0002360) */
    0x0BF2,0x0006,0x0202,0x0077,0xFE92,0x00B1,0x0077,0xFEC4,  /* 0500 (0002400) */
    0x55CA,0x0004,0x0105,0x55CA,0x0800,0x0102,0x55CA,0x0204,  /* 0510 (0002420) */
    0x55F1,0x8000,0xFFFE,0x0087,0x003C,0x0312,0x05EA,0xFFFF,  /* 0520 (0002440) */
    0x55F1,0x8000,0xFFFE,0x55C9,0x0200,0x11C3,0x65C3,0xFC26,  /* 0530 (0002460) */
    0x10F2,0x0002,0x10F2,0x0006,0x65C3,0x0004,0x10F2,0x0004,  /* 0540 (0002500) */
    0x11C3,0x65C3,0xFFD6,0x10F2,0x0008,0x1083,0x65C3,0x08F2,  /* 0550 (0002520) */
    0x1E85,0x0008,0x2CB2,0x0004,0x0006,0x0603,0x1CB2,0x0002,  /* 0560 (0002540) */
    0x0006,0x9E93,0x0006,0x0AB2,0x0006,0x0AC5,0x02F3,0x1E80,  /* 0570 (0002560) */
    0x0008,0x0977,0x035E,0x8715,0x1E80,0x0008,0x0977,0x03EA,  /* 0580 (0002600) */
    0x0000,0x2000,0x0301,0x0003,0x1084,0x65C4,0x02F2,0x1085,  /* 0590 (0002620) */
    0x65C5,0x08F2,0x1E80,0x0008,0xA515,0x0203,0x0AC0,0x02FC,  /* 05A0 (0002640) */
    0x0103,0x55CA,0x0008,0x0087,0x65F2,0x0002,0x0008,0x25FA,  /* 05B0 (0002660) */
    0xFFFF,0x0008,0x0302,0x0077,0xFF90,0x15C0,0xFFF8,0x09F7,  /* 05C0 (0002700) */
    0x035E,0x1131,0xFFF6,0x0A31,0xFFF8,0x1083,0x65C3,0x0020,  /* 05D0 (0002720) */
    0x10C4,0x15C0,0x0008,0x15D3,0x8000,0x15D3,0x8000,0x108B,  /* 05E0 (0002740) */
    0x65D3,0x000A,0x15D3,0xFFFF,0x0A13,0x0A13,0x0AC0,0x02F3,  /* 05F0 (0002760) */
    0x1103,0x55F3,0x0080,0x0002,0x55F3,0x00C0,0x000E,0x15F3,  /* 0600 (0003000) */
    0xFFFE,0x0012,0x55F3,0x0040,0x001A,0x65F3,0x0002,0x001C,  /* 0610 (0003020) */
    0x65F3,0x0004,0x0028,0x15F3,0x4000,0x0032,0x10F3,0x0034,  /* 0620 (0003040) */
    0x65F3,0x003C,0x0034,0x10F3,0x0040,0x65F3,0x0038,0x0040,  /* 0630 (0003060) */
    0x15F3,0xFFFE,0x0042,0x15F3,0xA000,0x004A,0x15F3,0xFFFD,  /* 0640 (0003100) */
    0x004E,0x15F3,0x2000,0x0056,0x1131,0xFFFA,0x0A31,0xFFFC,  /* 0650 (0003120) */
    0x15C0,0x7FFF,0x35C9,0x8000,0x0205,0x0AC0,0x02FB,0x55CA,  /* 0660 (0003140) */
    0x0200,0x0109,0x15C0,0x0010,0x0977,0x02FE,0x8000,0x2000,  /* 0670 (0003160) */
    0x0305,0x55CA,0x4000,0x55CA,0x0010,0x0087,0x1083,0x65C3,  /* 0680 (0003200) */
    0x0020,0x15C0,0x0008,0x1CC5,0x0000,0x45C5,0x3FFF,0x25C5,  /* 0690 (0003220) */
    0xC000,0x02EF,0x25C0,0x0004,0x0303,0x25C0,0x0001,0x0207,  /* 06A0 (0003240) */
    0x0BF3,0x0008,0x02E6,0x0BF3,0x000A,0x02E3,0x0114,0x25C0,  /* 06B0 (0003260) */
    0x0002,0x0305,0x35F3,0x8000,0x0008,0x03DB,0x010C,0x1CC5,  /* 06C0 (0003300) */
    0x0008,0x45C5,0x2D0F,0x25C5,0x0000,0x02D3,0x1CC5,0x000A,  /* 06D0 (0003320) */
    0x45C5,0xFC00,0x03CE,0x65C3,0x000C,0x0AC0,0x02D4,0x1083,  /* 06E0 (0003340) */
    0x65C3,0x000A,0x1084,0x65C4,0x02F2,0x1105,0x24D4,0x02C3,  /* 06F0 (0003360) */
    0x24D4,0x02C1,0x24D4,0x02BF,0x0BD4,0x02BD,0x0BD4,0x02BB,  /* 0700 (0003400) */
    0x2554,0x02B9,0x2554,0x02B7,0x2554,0x02B5,0x1083,0x65C3,  /* 0710 (0003420) */
    0x0072,0x10C5,0x15C0,0x004B,0x0A13,0x0AC0,0x02FD,0x1084,  /* 0720 (0003440) */
    0x65C4,0x000A,0x09F7,0x0026,0x65C5,0x0010,0x09F7,0x001E,  /* 0730 (0003460) */
    0x0977,0x0124,0xFFBC,0x0301,0x0087,0x15C0,0xFFFF,0x35C9,  /* 0740 (0003500) */
    0x2000,0x0313,0x0AC0,0x02FB,0x55CA,0x8000,0x0087,0x1126,  /* 0750 (0003520) */
    0x15C0,0x0006,0x0A85,0x15C3,0x0007,0x9315,0x0AC3,0x02FD,  /* 0760 (0003540) */
    0x0A84,0x0AC0,0x02F7,0x1584,0x0087,0x0BCA,0x020F,0x55C9,  /* 0770 (0003560) */
    0x0300,0x15C0,0x002E,0x09F7,0x005A,0x15C0,0x003C,0x0977,  /* 0780 (0003600) */
    0x0152,0x8605,0x45CA,0x0200,0x55CA,0x8000,0x0087,0x15C0,  /* 0790 (0003620) */
    0x003C,0x09F7,0x006E,0x03F1,0x0303,0x55CA,0x0020,0x0087,  /* 07A0 (0003640) */
    0x15C0,0x003C,0x09F7,0x009A,0x02F8,0x15C0,0x05DC,0x09F7,  /* 07B0 (0003660) */
    0x0022,0x15C0,0x05EA,0x0977,0x011A,0x87EF,0x15C0,0x05EA,  /* 07C0 (0003700) */
    0x09F7,0x0040,0x03F6,0x02E9,0x15C0,0x05EA,0x09F7,0x0072,  /* 07D0 (0003720) */
    0x02E4,0x0087,0x1083,0x65C3,0x08F2,0x1084,0x65C4,0x000A,  /* 07E0 (0003740) */
    0x1105,0x1513,0x1513,0x1513,0x1553,0x1553,0x1553,0x15D3,  /* 07F0 (0003760) */
    0x0090,0x0A13,0x15D3,0x0001,0x9013,0x0AC0,0x02FD,0x0A32,  /* 0800 (0004000) */
    0x0002,0x0087,0x0977,0x0162,0x0000,0x2000,0x0204,0x65CE,  /* 0810 (0004020) */
    0x0002,0x0BCA,0x0087,0x1C80,0x0028,0x45C0,0xBFFF,0x0208,  /* 0820 (0004040) */
    0x1C80,0x0018,0x35C0,0xBFFF,0x0203,0x45CA,0x4000,0x01EF,  /* 0830 (0004060) */
    0x0AB2,0x0002,0x25F2,0x0064,0x0002,0x07E9,0x45CA,0x4000,  /* 0840 (0004100) */
    0x01E8,0x1084,0x65C4,0x02F2,0x1085,0x65C5,0x08F2,0xA515,  /* 0850 (0004120) */
    0x0202,0x0AC0,0x02FC,0x0087,0x1080,0x65C0,0x0072,0x1026,  /* 0860 (0004140) */
    0x1340,0x0AC0,0x09F7,0x00B8,0x1131,0xFFF6,0x0A31,0xFFF8,  /* 0870 (0004160) */
    0x1540,0x09F7,0x00C6,0x15F4,0xB000,0x0002,0x13B4,0x0004,  /* 0880 (0004200) */
    0x1131,0xFFFA,0x0A31,0xFFFC,0x15C4,0x1FFF,0x35C9,0x8000,  /* 0890 (0004220) */
    0x0208,0x0AC4,0x02FB,0x55CA,0x0200,0x0BD6,0x55CA,0x1000,  /* 08A0 (0004240) */
    0x0118,0x1D40,0xFFFE,0x0CC0,0x0B00,0x0977,0x00BC,0x0000,  /* 08B0 (0004260) */
    0x2000,0x02F3,0x1D40,0xFFFE,0x0CC0,0x0B00,0x1084,0x65C4,  /* 08C0 (0004300) */
    0x02F2,0x1583,0xA513,0x02EA,0x0AC0,0x02FC,0x25CC,0x52E3,  /* 08D0 (0004320) */
    0x02E5,0x0085,0x0A26,0x0C00,0x0B4E,0x6380,0x0B00,0x09F7,  /* 08E0 (0004340) */
    0x003E,0x1131,0xFFF6,0x0A31,0xFFF8,0x09F7,0x004E,0x0BD6,  /* 08F0 (0004360) */
    0x0303,0x55F4,0x0080,0x0002,0x1131,0xFFFA,0x0A31,0xFFFC,  /* 0900 (0004400) */
    0x15C3,0x01FF,0x0A00,0x35C9,0x8000,0x0208,0x0AC0,0x02FE,  /* 0910 (0004420) */
    0x0AC3,0x02F9,0x55CA,0x0200,0x00B1,0x0101,0x00A1,0x0085,  /* 0920 (0004440) */
    0x1083,0x65C3,0x02F2,0x15C4,0x0080,0x15D3,0x52E3,0x0AC4,  /* 0930 (0004460) */
    0x02FC,0x0977,0x0010,0x0010,0x02F2,0x0087,0x0977,0x0006,  /* 0940 (0004500) */
    0x0020,0x08F2,0x0087,0x1543,0x6083,0x10C4,0x15D3,0x8000,  /* 0950 (0004520) */
    0x15D3,0xA000,0x154B,0x6093,0x1013,0x15D3,0x8000,0x15D3,  /* 0960 (0004540) */
    0x8000,0x15D3,0x8000,0x0A13,0x0085,0x1026,0x0A00,0x1244,  /* 0970 (0004560) */
    0x55C9,0x8080,0x45C4,0x7F01,0x25C4,0x80B0,0x0301,0x11C0,  /* 0980 (0004600) */
    0x1C84,0x0020,0x45C4,0x3FFF,0x25C4,0xC000,0x0301,0x11C0,  /* 0990 (0004620) */
    0x1C84,0x0028,0x8002,0x45C4,0x7FFF,0x45C4,0x7D0F,0x2544,  /* 09A0 (0004640) */
    0x0301,0x11C0,0x1C84,0x002A,0x45C4,0xFC00,0x0201,0x11C0,  /* 09B0 (0004660) */
    0x1C84,0x0010,0x45C4,0x3FFF,0x25C4,0xC000,0x0301,0x11C0,  /* 09C0 (0004700) */
    0x1C84,0x0018,0x1126,0x45C4,0x0FFF,0x2544,0x0301,0x11C0,  /* 09D0 (0004720) */
    0x1584,0x45C4,0xF8FF,0x25F2,0xFFFD,0x0026,0x0202,0x0BD6,  /* 09E0 (0004740) */
    0x010D,0x35F2,0x1000,0x0022,0x0301,0x0A04,0x1C83,0x001A,  /* 09F0 (0004760) */
    0x45C3,0xFF00,0x60C4,0x2584,0x0301,0x11C0,0x1003,0x0302,  /* 0A00 (0005000) */
    0x55CA,0x4000,0x0085,0x0005,0x15C6,0x1FFE,0x118E,0x13A6,  /* 0A10 (0005020) */
    0xE5CE,0x062E,0x11C3,0x65C3,0x000C,0x10DF,0x0004,0x1240,  /* 0A20 (0005040) */
    0x0101,0x0000,0x09F7,0x0098,0x1385,0x15D5,0x4B08,0x15D5,  /* 0A30 (0005060) */
    0x0101,0x8A15,0x8A15,0x15D5,0x0191,0x95D5,0x0002,0x15C2,  /* 0A40 (0005100) */
    0x05EA,0x9095,0x00C2,0x9095,0x15C2,0x000B,0x0A00,0x09F7,  /* 0A50 (0005120) */
    0x012A,0x02E8,0x15C2,0x05EE,0x09F7,0x01B0,0x0226,0x1383,  /* 0A60 (0005140) */
    0x94C4,0x0AC2,0xA117,0x0002,0x0304,0x8BC4,0x02DD,0xE5C2,  /* 0A70 (0005160) */
    0x0004,0x8BCB,0x0302,0xA2C0,0x0218,0x94C0,0x8A80,0x0AC2,  /* 0A80 (0005200) */
    0x0708,0x14C5,0x0BD3,0xE5C2,0x0004,0x0703,0x94D5,0x0AC2,  /* 0A90 (0005220) */
    0x06FD,0x8BC4,0x030F,0x0A03,0x1385,0x95D5,0x000A,0x9015,  /* 0AA0 (0005240) */
    0x90D5,0x0A05,0x15C2,0x0003,0x01D2,0x8BC0,0x03BB,0x15C3,  /* 0AB0 (0005260) */
    0x0001,0x01F2,0x09F7,0x029A,0x94E6,0x94F6,0x0001,0x0087,  /* 0AC0 (0005300) */
    0x45F1,0x0002,0x000E,0x1D85,0x0002,0x1144,0x65C4,0xFFD0,  /* 0AD0 (0005320) */
    0x1136,0x0004,0x15C2,0x0010,0x0A14,0x7E82,0x1144,0x65C4,  /* 0AE0 (0005340) */
    0xFFF0,0x0A35,0xFFD2,0x1135,0xFFD4,0x0A35,0xFFE2,0x1135,  /* 0AF0 (0005360) */
    0xFFE4,0x55F5,0x8000,0xFFD2,0x15C2,0x05EE,0x0C82,0x0B02,  /* 0B00 (0005400) */
    0x10B5,0xFFD6,0x55F5,0xA000,0xFFE2,0x1D84,0x0004,0x1137,  /* 0B10 (0005420) */
    0x02D4,0x0A37,0x02D2,0x65C4,0x0010,0x1137,0x02CC,0x0A37,  /* 0B20 (0005440) */
    0x02CA,0x9C77,0x0000,0x02B2,0x9C77,0x0002,0x02AD,0x9C77,  /* 0B30 (0005460) */
    0x0004,0x02A8,0x9C77,0x0006,0x02A3,0x9C77,0x0008,0x029E,  /* 0B40 (0005500) */
    0x9C77,0x000A,0x0299,0x8AB7,0x029A,0x45F1,0x0002,0x000E,  /* 0B50 (0005520) */
    0x55F1,0x0100,0x000E,0x09F7,0x0244,0x0A35,0x0002,0x1DE4,  /* 0B60 (0005540) */
    0x027C,0x1DE4,0x0276,0x1DE4,0x0270,0x15E4,0x0000,0x15E4,  /* 0B70 (0005560) */
    0x0200,0x15E4,0x00AB,0x09F7,0x0026,0x0087,0x1D85,0x0002,  /* 0B80 (0005600) */
    0x1144,0x10A4,0x15E4,0x0160,0x1DE4,0x0252,0x1DE4,0x024C,  /* 0B90 (0005620) */
    0x1DE4,0x0246,0x1DE4,0x0230,0x1DE4,0x022A,0x1DE4,0x0224,  /* 0BA0 (0005640) */
    0x65C5,0xFFE0,0x65C2,0x000E,0x2097,0x003C,0x0403,0x15C2,  /* 0BB0 (0005660) */
    0x003C,0x0107,0x35C2,0x0001,0x0304,0x55F5,0x0080,0x0002,  /* 0BC0 (0005700) */
    0x0A82,0x0C82,0x0B02,0x10B5,0x0006,0x0A35,0x0008,0x0A35,  /* 0BD0 (0005720) */
    0x000A,0x1DF1,0x0214,0x0008,0x1DF1,0x0210,0x000A,0x15C4,  /* 0BE0 (0005740) */
    0x0084,0x09F7,0x0188,0x860A,0x0304,0x35F5,0x0200,0x0008,  /* 0BF0 (0005760) */
    0x02EC,0x45F5,0x0080,0x0002,0x00A4,0x0087,0x45F5,0x0080,  /* 0C00 (0006000) */
    0x0002,0x35C2,0x4004,0x0301,0x0000,0x0087,0x1D85,0x0002,  /* 0C10 (0006020) */
    0x65C5,0xFFD0,0x0A35,0x0008,0x0A35,0x000A,0x1DF1,0x01C6,  /* 0C20 (0006040) */
    0x0004,0x1DF1,0x01C2,0x0006,0x55F1,0x0001,0x000E,0x15C4,  /* 0C30 (0006060) */
    0x8004,0x09F7,0x0138,0x8618,0x02ED,0x55F1,0x0002,0x000E,  /* 0C40 (0006100) */
    0x15F7,0x00AB,0x017E,0x15F7,0x0100,0x017A,0x15F7,0x0000,  /* 0C50 (0006120) */
    0x0176,0x8AB7,0x018E,0x0A00,0x45F1,0x0002,0x000E,0x55F1,  /* 0C60 (0006140) */
    0x0100,0x000E,0x00A4,0x0087,0x45F1,0x0001,0x000E,0x55F1,  /* 0C70 (0006160) */
    0x0002,0x000E,0x1D85,0x0002,0x1D42,0xFFFE,0x45F1,0x0002,  /* 0C80 (0006200) */
    0x000E,0x55F1,0x0100,0x000E,0x25F5,0x0160,0xFFFC,0x0221,  /* 0C90 (0006220) */
    0x8BF7,0x0150,0x020D,0x2DF5,0x012E,0xFFFA,0x02B7,0x2DF5,  /* 0CA0 (0006240) */
    0x0124,0xFFF8,0x02B3,0x2DF5,0x011A,0xFFF6,0x02AF,0x0087,  /* 0CB0 (0006260) */
    0xA5CD,0x0003,0x02AB,0x1D77,0xFFF6,0x0108,0x1D77,0xFFF8,  /* 0CC0 (0006300) */
    0x0104,0x1D77,0xFFFA,0x0100,0x8A37,0x0118,0x00A4,0x0A00,  /* 0CD0 (0006320) */
    0x0087,0x25F5,0x0090,0xFFFC,0x0313,0x25F5,0x0260,0xFFFC,  /* 0CE0 (0006340) */
    0x0295,0xA5CD,0x0005,0x0292,0x09F7,0x00B2,0x1D34,0xFFFE,  /* 0CF0 (0006360) */
    0xFFF8,0x1D34,0xFFFC,0xFFF6,0x1D34,0xFFFA,0xFFF4,0x011F,  /* 0D00 (0006400) */
    0x1144,0x6084,0x1143,0x65C3,0xFFD0,0x1CC2,0x0008,0x45C2,  /* 0D10 (0006420) */
    0xF8FF,0xDCC2,0x000A,0x65C2,0x002E,0x25D4,0x0002,0x0217,  /* 0D20 (0006440) */
    0x65F5,0x0008,0xFFFE,0x35CC,0x0001,0x0211,0x1143,0x65C3,  /* 0D30 (0006460) */
    0xFFF0,0x1513,0x1513,0x1513,0x1144,0x65C4,0xFFFC,0x1DE4,  /* 0D40 (0006500) */
    0x009C,0x1DE4,0x0096,0x1DE4,0x0090,0x09F7,0xFE52,0x0077,  /* 0D50 (0006520) */
    0xFEBA,0x1DDF,0x006E,0x0018,0x1DDF,0x006A,0x001A,0x1DDF,  /* 0D60 (0006540) */
    0x0066,0x001C,0x55F1,0x0002,0x000E,0x00B4,0x0087,0x15C3,  /* 0D70 (0006560) */
    0x3FFF,0x00A1,0x15C2,0x0090,0x3131,0x000E,0x0206,0x0AC3,  /* 0D80 (0006600) */
    0x02FE,0x0AC2,0x02F9,0x00B1,0x0087,0x1C42,0x000E,0x10B1,  /* 0D90 (0006620) */
    0x000E,0x35F5,0x4000,0x0008,0x0301,0x00B1,0x0087,0x1166,  /* 0DA0 (0006640) */
    0x11C3,0x65C3,0x0028,0x15C2,0x0018,0x14D5,0x0BD5,0x94D5,  /* 0DB0 (0006660) */
    0x0AC2,0x02FD,0x1585,0x1144,0x15C2,0x001C,0x10A4,0x15E4,  /* 0DC0 (0006700) */
    0x0260,0x0087,0x00AB,0x0100,0x0000,0x0007,0x0001,0x0303,  /* 0DD0 (0006720) */
    0x0000,0x0002,0x0102,0x0700,0x0600,0x0000,0x0000,0x0000,  /* 0DE0 (0006740) */
    0x0064,0x4B01,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0DF0 (0006760) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E00 (0007000) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E10 (0007020) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E20 (0007040) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E30 (0007060) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E40 (0007100) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E50 (0007120) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E60 (0007140) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E70 (0007160) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E80 (0007200) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0E90 (0007220) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0EA0 (0007240) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0EB0 (0007260) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0EC0 (0007300) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0ED0 (0007320) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0EE0 (0007340) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0EF0 (0007360) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F00 (0007400) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F10 (0007420) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F20 (0007440) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F30 (0007460) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F40 (0007500) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F50 (0007520) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F60 (0007540) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F70 (0007560) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F80 (0007600) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0F90 (0007620) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0FA0 (0007640) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0FB0 (0007660) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0FC0 (0007700) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0FD0 (0007720) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,  /* 0FE0 (0007740) */
    0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x84D2,  /* 0FF0 (0007760) */
    };

#ifdef VM_PDP11
#if 0
/*
Tim Shoppa said:

Hopefully better late than never, but I finally got to digging and
found that the CIQNDC0 sources give an example bootstrap:

                IDENTIFICATION
                --------------

PRODUCT CODE:  AC-T612A-MC

PRODUCT NAME:  CIQNDC0 DEQNA ROM RESIDENT CODE

PRODUCT DATE:  29 June 1984

MAINTAINER:    DIAGNOSTIC ENGINEERING

[...]

4.1     PRIMARY BOOT PROCESS

The primary boot, resident in the host, normally checks for the existence 
of the device it is going to boot from, boots 512. bytes from the device, 
verifies the operation, sets parameters and transfers to the "booted" code.

A suggested method for "checking" for a QNA is as follows:

        Write a "2" (a module reset) into the QNA's CSR at location 
        17774456 or 17774476.

        Read back the CSR and compare against an octal 000062.

        If equal then there is most likely a QNA there.

        If not equal (Bus time-out?) then "sniff" elsewhere.

        Write a "0" back into the CSR to "reset" the reset bit.

If a QNA is present and it is to be used for the boot the first thing 
then:

        A descriptor for a 256. word "receive" is validated in the QNA.

                Data is read into the host starting at location 0
                The descriptor is 8 words long, words 0, 4, 5 and 6 will
                contain operation status. Words 1, 2, 3, and 7 are constant.

        Write a "1010" into the CSR to move the boot code into the QNA's
        internal receive buffer and delay for approximately one second.

        Write a "1000" into the CSR to move the data in the internal
        receive buffer into host memory and delay for approximately one
        second.

        Reset the QNA and check the CSR for proper status.

        Checked the receive descriptor for nominal states.

        The Data transfer is verified.
                                
        If the QNA primary boot code detects a failure at this point the
        host boot is re-entered.

        Transfer is made to the first location of the freshly loaded 
        portion of the QNA BD ROM.

                R1 contains the I/O Page address of the QNA (174440)

                R0 contains a "000000" if the DECnet boot resident on the 
                QNA is to be used. 
                                - or -
                If R0 is greater than "000777" an effective JMP (R0) is 
                executed in lieu of the DECnet boot.


                Location 12 contains a "000000" if the EPB is to halt when 
                an error is detected.
                                - or -
                If loaction 12 is greater than "000000" an effective JMP @12
                is executed in lieu of a "HALT".                

4.1.1   EXPECTED VALUES FOR VERIFICATION

CSR is checked for a nominal state as are the status words in the receive
descriptor.

                Status  Nominal
                ---------------
                CSR     000060 or 010060
                FLAG    14xxxx
                BSW1    14xxxx
                BSW2    14xxxx
                CSW0    177777

      x - don't care bits

Next the actual data transfer is verified by checking the first three bytes 
of the data transferred for standard values. These first locations and their
expected contents are:

                Location        Contents
                --------------------------
                0000-1          000240  (NOP)
                0002               001  (a BR instruction)
                0003               xxx

                0004
                 ..             The QNA Extended Primary Boot (EPB) code
                0777

4.1.2   SAMPLE PRIMARY BOOT CODE

Below is an example of how a primary boot could be implemented on a typical
PDP-11 based host.

Upon entry R0 is coded to indicate which QNA is to be used for the boot. 
The first 32K bytes of the host memory is assumed to be mapped 1 to 1 
physical to logical. The I/O page is mapped to the last 4K.

Assume settings for R0 as follows:      

        R0 - Contains a "4" for QNA#1 @ 174440
             Contains a "5" for QNA#2 @ 174460
PDP-11 code for booting the DEQNA EPB might be as follows:
*/
#endif

#define BOOT_LEN (sizeof(boot_rom)/sizeof(uint16))/* Code size */
#define BOOT_START 020000               /* Initial load Address of the Boot code */
#define BOOT_ENTRY (BOOT_START+0)       /* Start address of Boot code */

static const uint16 boot_rom[] = {
                               //       DESC = 4000                    ; Descriptor List Address
                               //       DECnet = 0                     ; MOP boot request
    0012701, 0174440,          // 0000         mov     #174440,r1      ; assume device is DEQNA #1
    0022700, 0000004,          // 0004         cmp     #4,r0
    0001402,                   // 0010         beq     10$             ; good assumption
    0012701, 0177460,          // 0012         mov     #174460,r1      ; select device at DEQNA #2

    0012761, 0000002, 0000016, // 0016 10$:    mov     #SR,16(r1)      ; assert QNA software reset
    0032761, 0000062, 0000016, // 0024         bit     #<RL|XL|SR>,16(r1); "nominal" status during reset
    0001466,                   // 0032         beq     90$             ;? not a proper response, halt
    0005061, 0000016,          // 0034         clr     16(r1)          ; clear reset state and all others

    0012703, 0004000,          // 0040         mov     #DESC,r3        ; pick an address for descriptor
    0010304,                   // 0044         mov     r3,r4
    0012724, 0100000,          // 0046         mov     #100000,(r4)+   ; Flag word, changed to -1
    0011324,                   // 0052         mov     (r3),(r4)+      ; Valid buffer descriptor, receive
    0005024,                   // 0054         clr     (r4)+           ; location zero
    0012724, 0177400,          // 0056         mov      #-400,(r4)+    ; 256. words or 512. bytes
    0005024,                   // 0062         clr     (r4)+           ; Descriptor status 1, changed to -1
    0005024,                   // 0064         clr     (r4)+           ; Descriptor status 2, changed to -1
    0011324,                   // 0066         mov     (r3),(r4)+      ; Another flag word, changed to -1
    0012714, 0020000,          // 0070         mov     #020000,(r4)    ; end descriptor code
    0010361, 0000004,          // 0074         mov     r3,04(r1)       ; receive descriptor low address
    0005061, 0000006,          // 0100         clr     06(r1)          ; validate a receive descriptor

    0012761, 0001010, 0000016, // 0104         mov     #<EL|BD>,16(r1) ; Instruct QNA to unload the EPB code
    0005000,                   // 0112         clr     r0              ; delay about 60 ms for transfer of
    0077001,                   // 0114         sob     r0,.            ; i8051 contents to receive FIFO
    0012761, 0001000, 0000016, // 0116         mov     #<EL>,16(r1)    ; Complete the EPB unload
    0005000,                   // 0124         clr     r0              ; delay about 60 ms for transfer of
    0077001,                   // 0126         sob     r0,.            ; receive FIFO to host memory
    0012761, 0000002, 0000016, // 0130         mov     #SR,16(r1)      ; reset
    0005061, 0000016,          // 0136         clr     16(r1)          ; Final reset to complete operation

    0012704, 0004014,          // 0142         mov     #<DESC+14>,r4
    0042714, 0037777,          // 0146         bic     #037777,(r4)    ; check if last status word was updated 
    0022714, 0140000,          // 0152         cmp     #140000,(r4)    ;
    0001014,                   // 0156         bne     90$             ;

    0022737, 0000240, 0000000, // 0160         cmp     #240,@#0        ; check for QNA boot block
    0001010,                   // 0166         bne     90$             ;? operation data check
    0122737, 0000001, 0000003, // 0170         cmpb    #001,@#3        ; check for "BR" opcode
    0001004,                   // 0176         bne     90$             ;? operation data check

    0012700, 0000000,          // 0200         mov     #DECnet,r0      ; Load DECnet code (or other code into R0)
    0000137, 0000000,          // 0204         jmp     @#0             ; go to extended primary boot

                               //              ; R0 - set to zero for DECnet boot
                               //              ; R1 - has address of QNA #1 or #2

    0000000,                   // 0210 90$:    halt            ;? QNA error, get back to host boot control?
    0000776,                   // 0212         br      90$
    };
#endif                                                  /* VM_PDP11 */

#endif                                                  /* _PDP11_XQ_BOOTROM_H */
